/**
 * A client-side 802.1x implementation supporting EAP/AKA
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2003, 2004 Chris Hessing
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
* EAPOL Function implementations for supplicant
 * 
 * File: eapaka.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapaka.c,v 1.4 2004/08/20 04:42:28 chessing Exp $
 * $Date: 2004/08/20 04:42:28 $
 * $Log: eapaka.c,v $
 * Revision 1.4  2004/08/20 04:42:28  chessing
 * Set all of the new scanning and WPA code to be disabled by default, so that a 1.0.1 release can be made.  Added additional error message output to AKA code.  Fixed a few memory leaks in the AKA code.
 *
 * Revision 1.3  2004/07/19 02:43:17  chessing
 *
 * Changed things to get rid of Linux specific pieces in the interface_data struct. Fixed up EAP-SIM and EAP-AKA to skip AVPs that we don't support.  (We print a mesage, and move on.)  Added the --enable-radiator-test option to configure EAP-AKA to use the test vectors included with Radiator's AKA module.  (To use this mode, no SIM card is required, but the PCSC library is still needed to build.  Also, some errors will be displayed.)
 *
 * Revision 1.2  2004/06/21 05:19:13  chessing
 *
 * Added a few minor fixes to EAP-AKA support.  Added "allmulti" as a global configuration option.  (By default, allmulti is now on!)
 *
 * Revision 1.1  2004/06/15 03:35:19  chessing
 *
 * New updates including fixes to LEAP (keying now works with wireless) and adding EAP-AKA.
 *
 *
 *******************************************************************/

/*******************************************************************
 *
 * The development of the EAP/AKA support was funded by Internet
 * Foundation Austria (http://www.nic.at/ipa)
 *
 *******************************************************************/


#ifdef EAP_SIM_ENABLE     // Only build this if it has been enabled.

#include <inttypes.h>
#include <stdio.h>
#include <openssl/hmac.h>
#include <openssl/sha.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "winscard.h"
#include "interactive.h"
#include "profile.h"
#include "config.h"
#include "eap.h"
#include "eapaka.h"
#include "../sim/eapsim.h"
#include "../sim/sm_handler.h"
#include "../sim/fips.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "aka.h"

int eapaka_get_username(struct interface_data *thisint)
{
  char *imsi;  
  char realm[25], card_mode=0;
  char *readers, *username;
  struct config_eap_aka *userdata;
  struct generic_eap_data mydata;
  SCARDCONTEXT ctx;
  SCARDHANDLE hdl;

  if ((!thisint) || (!thisint->userdata) || (!thisint->userdata->methods) ||
      (!thisint->userdata->methods->method_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to eapsim_get_username()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_aka *)thisint->userdata->methods->method_data;

  mydata.eap_conf_data = userdata;

  // Initalize our smartcard context, and get ready to authenticate.
  if (sm_handler_init_ctx(&ctx) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize smart card context!\n");
      return XESIMGENERR;
    }

  readers = sm_handler_get_readers(&ctx);
  if (readers == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't list available readers!\n");
      return XESIMGENERR;
    }

  // Connect to the smart card.
  if (sm_handler_card_connect(&ctx, &hdl, readers) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Error connecting to smart card reader!\n");
      return XESIMGENERR;
    }

  // Wait for up to 10 seconds for the smartcard to become ready.
  if (sm_handler_wait_card_ready(&hdl, 10) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Smart Card wasn't ready after 10 seconds!\n");
      return XESIMGENERR;
    }

  imsi = sm_handler_3g_imsi(&hdl, card_mode, userdata->password);
  if (imsi == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error starting smart card, and getting IMSI!\n");
      return XESIMGENERR;
    }

  debug_printf(DEBUG_AUTHTYPES, "SIM IMSI (AKA) : %s\n",imsi);

  if (thisint->userdata->identity != NULL)
    {
      free(thisint->userdata->identity);
    }

  thisint->userdata->identity = (char *)malloc(50);  // 50 should be plenty!
  if (thisint->userdata->identity == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for identity!\n");
      return XEMALLOC;
    }

  username = thisint->userdata->identity;
  userdata->username = username;
  bzero(username, 50);

  username[0] = '1';  // An IMSI should always start with a 1.
  strncpy(&username[1], imsi, 18);

  if (userdata->auto_realm == TRUE)
    {
      bzero(&realm, 25);
      sprintf((char *)&realm, "@mnc%c%c%c.mcc%c%c%c.owlan.org",
	      username[4], username[5], username[6], username[1], username[2],
	      username[3]);

      debug_printf(DEBUG_AUTHTYPES, "Realm Portion : %s\n",realm);
      strcat(username, realm);
    }

  // Close the smartcard, so that we know what state we are in.
  sm_handler_close_sc(&hdl, &ctx);

  free(imsi);
  imsi = NULL;

  free(readers);
  readers = NULL;

  debug_printf(DEBUG_AUTHTYPES, "Username is now : %s\n", username);

  return XENONE;
}


int eapaka_setup(struct generic_eap_data *thisint)
{
  struct aka_eaptypedata *mydata;
  struct config_eap_aka *userdata;
  char *imsi;

  debug_printf(DEBUG_AUTHTYPES, "(EAP-AKA) Initalized\n");

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to eapaka_setup()!\n");
      return XEMALLOC;
    }

  thisint->eap_data = (char *)malloc(sizeof(struct aka_eaptypedata));
  if (thisint->eap_data == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for EAP-AKA specific data structure!\n");
      return XEMALLOC;
    }

  mydata = (struct aka_eaptypedata *)thisint->eap_data;
  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

  mydata->numrands = 0;
  mydata->nonce_mt = NULL;
  mydata->keyingMaterial = NULL;

  thisint->eap_data = (void *)mydata;

  // Initalize our smartcard context, and get ready to authenticate.
  if (sm_handler_init_ctx(&mydata->scntx) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize smart card context!\n");
      return XESIMGENERR;
    }

  mydata->readers = sm_handler_get_readers(&mydata->scntx);
  if (mydata->readers == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get any available readers!\n");
      return XESIMGENERR;
    }

  // Connect to the smart card.
  if (sm_handler_card_connect(&mydata->scntx, &mydata->shdl, mydata->readers) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Error connecting to smart card reader!\n");
      return XESIMGENERR;
    }

  // Wait for up to 20 seconds for the smartcard to become ready.
  if (sm_handler_wait_card_ready(&mydata->shdl, 20) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Smart Card wasn't ready after 20 seconds!\n");
      return XESIMGENERR;
    }

  imsi = sm_handler_3g_imsi(&mydata->shdl, mydata->card_mode, userdata->password);
  if (imsi == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error starting smart card, and getting IMSI!\n");
      return XESIMGENERR;
    }

  if (userdata->username == NULL)
    {
      userdata->username = imsi;
    } else {
      free(imsi);
      imsi = NULL;
    }

  return XENONE;
}


int eapaka_process(struct generic_eap_data *thisint, u_char *dataoffs,
		   int insize, u_char *out, int *outsize)
{
  int packet_offset, outptr, i, value16, retsize, retval;
  struct typelength *typelen;
  struct typelengthres *typelenres;
  struct aka_eaptypedata *mydata;
  char *username, *framecpy, mac_val[16], mac_calc[20], reslen = 0, reallen;
  unsigned char sres[16];
  struct config_eap_aka *userdata;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to eapaka_process()!\n");
      return XEMALLOC;
    }

  mydata = (struct aka_eaptypedata *)thisint->eap_data;
  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

  if (!userdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid user data struct in eapaka_process()!\n");
      return XEMALLOC;
    }

  if ((thisint->tempPwd == NULL) && (userdata->password == NULL))
    {
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-AKA");
      thisint->eapchallenge = NULL;
      
      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;

  if (userdata->username == NULL)
    {
      username = thisint->identity;
    } else {
      username = userdata->username;
    }

  if ((userdata->password == NULL) && (thisint->tempPwd != NULL))
    {
      userdata->password = thisint->tempPwd;
      thisint->tempPwd = NULL;
    }

  *outsize = 0;
  outptr = 0;
  bzero(&mac_calc[0], 16);
  bzero(&mac_val[0], 16);

  switch (dataoffs[0])
    {
    case AKA_IDENTITY:
      debug_printf(DEBUG_AUTHTYPES, "Got AKA_IDENTITY!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_AUTHENTICATION_REJECT:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_AUTHENTICATION_REJECT!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_SYNC_FAILURE:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_SYNC_FAILURE!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!  (And, we should *NEVER* get this!\n");
      break;

    case AKA_NOTIFICATION:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_NOTIFICATION!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_REAUTHENTICATION:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_REAUTHENTICATION!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_CLIENT_ERROR:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_CLIENT_ERROR!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_CHALLENGE:
      debug_printf(DEBUG_AUTHTYPES, "Got AKA_CHALLENGE!\n");
      packet_offset = 3;

      typelen = (struct typelength *)&out[0];
      typelen->type = AKA_CHALLENGE;
      outptr = 3;

      while (packet_offset < insize)
	{
	  switch (dataoffs[packet_offset])
	    {
	    case AT_RAND:
	      retval = aka_do_at_rand(mydata, dataoffs, &packet_offset);
	      if (retval != XENONE) return retval;
	      break;

	    case AT_AUTN:
	      retval = aka_do_at_autn(mydata, dataoffs, &packet_offset);
	      if (retval != XENONE) return retval;
	      break;

	    case AT_IV:
	      debug_printf(DEBUG_AUTHTYPES, "Got an IV (Not supported)\n");
	      aka_skip_not_implemented(dataoffs, &packet_offset);
	      break;

	    case AT_MAC:
	      retval = aka_do_at_mac(thisint, mydata, dataoffs, insize, 
				     &packet_offset, username);
	      if (retval == XEAKASYNCFAIL)
		{
		  return aka_do_sync_fail(mydata, out, outsize);
		} else if (retval != XENONE) return retval;
	      break;
	    }
	}

      if ((reslen % 4) != 0)
	{
	  reallen = reslen + (reslen % 4);
	} else {
	  reallen = reslen;
	}

      // Build the challenge response.
      typelenres = (struct typelengthres *)&out[outptr];
      typelenres->type = AT_RES;
      typelenres->length = (reallen/4)+1;
      typelenres->reserved = htons(reslen*8);
      outptr += 4;

      memcpy((char *)&out[outptr], (char *)&sres[0], reslen);
      outptr += reslen;

      if (reallen > reslen)
	{
	  for (i=0;i<(reallen-reslen);i++)
	    {
	      out[outptr+i] = 0x00;
	    }
	  outptr += (reallen-reslen);
	}

      typelenres = (struct typelengthres *)&out[outptr];
      typelenres->type = AT_MAC;
      typelenres->length = 5;
      typelenres->reserved = 0x0000;
      outptr += 4;

      retsize = outptr+16+5;
      
      framecpy = (char *)malloc(retsize);
      if (framecpy == NULL) return XEMALLOC;

      framecpy[0] = 2;   // This is a response.
      framecpy[1] = thisint->eapid;
      value16 = retsize;
      value16 = htons(value16);

      memcpy((char *)&framecpy[2], &value16, 2);
      framecpy[4] = EAP_TYPE_AKA;
      
      memcpy((char *)&framecpy[5], out, retsize);
      debug_printf(DEBUG_AUTHTYPES, "Preframe :\n");
      debug_hex_dump(DEBUG_AUTHTYPES, framecpy, retsize);
      // Zero out the mac.
      bzero((char *)&framecpy[outptr+5], 16);
      debug_printf(DEBUG_AUTHTYPES, "Frame to hash : \n");
      debug_hex_dump(DEBUG_AUTHTYPES, framecpy, retsize);

      HMAC(EVP_sha1(), (char *)&mydata->K_aut[0], 16, framecpy, retsize, (char *)&mac_calc[0], &i);

      free(framecpy);
      framecpy = NULL;

      debug_printf(DEBUG_AUTHTYPES, "MAC = ");
      debug_hex_printf(DEBUG_AUTHTYPES, (char *)&mac_calc[0], 16);

      memcpy((char *)&out[outptr], (char *)&mac_calc[0], 16);

      // Then, calculate the MAC, and return it.
      outptr +=16;
      break;
	  
    default:
      debug_printf(DEBUG_NORMAL, "Unknown SubType value! (%d)\n", 
		   dataoffs[0]);
      break;
    }
  out[2] = 0;
  *outsize = outptr;

  return XENONE;
}

int eapaka_get_keys(struct interface_data *thisint)
{
  struct aka_eaptypedata *mydata;

  if ((!thisint) || (!thisint->userdata) || (!thisint->userdata->activemethod)
      || (!thisint->userdata->activemethod->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to eapaka_get_keys()!\n");
      return XEMALLOC;
    }

  mydata = (struct aka_eaptypedata *)thisint->userdata->activemethod->eap_data;
  if (thisint->keyingMaterial != NULL)
    {
      free(thisint->keyingMaterial);
    }

  thisint->keyingMaterial = (char *)malloc(64);
  if (thisint->keyingMaterial == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for keying material! (%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  memcpy(thisint->keyingMaterial, mydata->keyingMaterial, 64);
  thisint->keyingLength = 32;

  return XENONE;
}

int eapaka_failed(struct generic_eap_data *thisint)
{
  struct config_eap_aka *userdata;

  if ((!thisint) || (!thisint->eap_conf_data))
    {
      debug_printf(DEBUG_AUTHTYPES, "No valid configuration information in EAP-AKA!  Nothing to do!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

#ifndef NO_PWD_RESET
  if (userdata->password != NULL)
    {
      free(userdata->password);
      userdata->password = NULL;
    }
#endif

  return XENONE;
}

int eapaka_cleanup(struct generic_eap_data *thisint)
{
  struct aka_eaptypedata *mydata;

  debug_printf(DEBUG_AUTHTYPES, "(EAP-AKA) Cleaning up!\n");
  mydata = (struct aka_eaptypedata *)thisint->eap_data;

  sm_handler_close_sc(&mydata->shdl, &mydata->scntx);

  free(mydata);
  mydata = NULL;

  return XENONE;
}

#endif
