/*

    xpuyopuyo - pnetconfig.c  Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Send and receive a game configuration
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <xpuyopuyo.h>
#if USE_NETWORK /* Allow network support? */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <config.h>
#include <pnetint.h>
#include <pconfig.h>
#include <pwindow.h>
#include <pconfigm.h>
#include <psnprintf.h>


int pnet_send_config(pconfig *c) {

   byte packet[P_NET_CONFIG_SIZE];
   word type;
   word size;
   dword fakeaccel;
   int error = 0;
   int reply;

   /* Make sure socket is not NULL */
   if(c->socket == NULL) return(-1);

   /* Notify user what we are doing; this is usually a wait */   
   p_window_set_status(c->window, "Sending opponent the game configuration ...");
   p_window_set_waiting(c->window, 1);
   p_window_idle(c->window);

   /* Setup acceleration */
   fakeaccel = c->accel * 1000000;
   c->accel = fakeaccel / 1000000.0;

   /* Setup the configuration to send */
   *(word *)(packet + 4) = htons(c->dropspeed);
   *(dword *)(packet + 6) = htonl(fakeaccel);
   packet[10] = c->nullifyrocks ? 1 : 0;
   packet[11] = c->indestructibles ? 1 : 0;
   *(word *)(packet + 12) = htons(c->fieldwidth);
   *(word *)(packet + 14) = htons(c->fieldheight);
   *(word *)(packet + 16) = htons(c->fieldmatch);
   packet[18] = c->numcolors;
   
   /* Setup our name to send to them */
   p_strncpy(&packet[32], P_PLAYER_0(c)->name, P_PLAYER_NAME_SIZE);

   /* Send configuration */   
   if(pnet_send_header(c, c->socket, P_NET_CONFIG_ID, P_NET_CONFIG_SIZE) < 0) {
      error = 1;
   } else if(pnet_send_buffer(c, c->socket, packet, P_NET_CONFIG_SIZE) < 0) {
      error = 1;
   } else {
      /* Notify user we'll be waiting awhile */
      #ifdef P_NET_DEBUG_COMM
      printf("send_config:  Sent configuration\n");
      #endif /* P_NET_DEBUG_COMM */
   
      p_window_set_status(c->window, "Waiting for opponent to start game ...");
      p_window_idle(c->window);
   
      /* Wait for a reply */
      reply = 0;
      while(!reply && !error) {
         /* Wait for a reply */
         reply = pnet_ready_to_recv(c, c->socket);
         while(reply == 0) {
            p_window_idle(c->window);
            reply = pnet_ready_to_recv(c, c->socket);
         }

         /* Read the reply */
         if(reply < 0) {
            error = 1;
         } else if(pnet_recv_header(c, c->socket, &type, &size) < 0) {
            error = 1;
         } else {
            if(type == P_NET_CONFIG_ID) {
               /* Receive reply configuration */
               if(pnet_recv_buffer(c, c->socket, packet, P_NET_CONFIG_SIZE) < 0) error = 1;
               else reply = 1;
            } else {
               /* Receive some garbage */
               if(pnet_recv_null(c, c->socket, size) < 0) error = 1;
               else reply = 0;
            } /* Is this a reply configuration? */
         } /* Read something ...  anything ... */
      } /* Loop until config reply is seen */
   
      /* We're ready to begin the game! */
      #ifdef P_NET_DEBUG_COMM
      printf("send_config:  Got reply for configuration\n");
      #endif /* P_NET_DEBUG_COMM */
   
      p_config_network(c);
   } /* Any errors when sending? */
   
   p_window_set_status(c->window, NULL);
   p_window_set_waiting(c->window, 0);
   
   /* Return a result */
   if(error) return(-1);

   /* Pull their name from the reply packet */
   p_player_set_name(P_PLAYER_1(c), &packet[32]);

   return(0);

}


int pnet_recv_config(pconfig *c) {

   byte packet[P_NET_CONFIG_SIZE];
   word type;
   word size;
   dword fakeaccel;
   int error = 0;
   int reply;

   /* Make sure we have a socket */
   if(c->socket == NULL) return(-1);

   /* Notify user what we are doing; this is usually a wait */
   #ifdef P_NET_DEBUG_COMM
   printf("recv_config:  Waiting for configuration\n");
   #endif /* P_NET_DEBUG_COMM */
   
   p_window_set_status(c->window, "Waiting for opponent to start game ...");
   p_window_set_waiting(c->window, 1);
   p_window_idle(c->window);
   
   /* Receive a configuration */   
   reply = 0;
   while(!reply && !error) {
      /* Wait for select() to tell us there is something to read */
      reply = pnet_ready_to_recv(c, c->socket);
      while(reply == 0) {
         p_window_idle(c->window);
         reply = pnet_ready_to_recv(c, c->socket);
      }
      /* Read the data */
      if(reply < 0) {
         error = 1;
      } else if(pnet_recv_header(c, c->socket, &type, &size) < 0) {
         error = 1;
      } else {
         /* We have data; is it of the right type? */
         if(type == P_NET_CONFIG_ID) {
            /* Yes; a config packet; read the packet from the network stream */
            reply = 1;
            if(pnet_recv_buffer(c, c->socket, packet, size) < 0) {
               error = 1;
            } /* Did we read the config packet? */
         } else {
            /* Not a config packet; read it into a null buffer */
            reply = 0;
            if(pnet_recv_null(c, c->socket, size) < 0) {
               error = 1;
            } /* Did we get rid of the packet? */
         } /* Packet received; is it a config packet? */
      } /* Any errors while reading the header? */
   } /* Waiting for config packet */

   /* No errors yet, I hope */
   if(!error) {
   
      /* Tell the user what is happening */
      #ifdef P_NET_DEBUG_COMM
      printf("recv_config:  Got configuration\n");
      #endif /* P_NET_DEBUG_COMM */

      p_window_set_status(c->window, "Responding to game configuration ...");
      p_window_idle(c->window);
   
      /* Setup pconfig struct */
      c->dropspeed = ntohs(*(word *)(packet + 4));
      fakeaccel = ntohl(*(dword *)(packet + 6));
      c->accel = fakeaccel / 1000000.0;
      c->nullifyrocks = packet[10];
      c->indestructibles = packet[11];
      c->fieldwidth = ntohs(*(word *)(packet + 12));
      c->fieldheight = ntohs(*(word *)(packet + 14));
      c->fieldmatch = ntohs(*(word *)(packet + 16));
      c->numcolors = packet[18];
      
      /* Pull their name from the reply packet */
      p_player_set_name(P_PLAYER_1(c), &packet[32]);

      /* Setup our name to send to them */
      p_strncpy(&packet[32], P_PLAYER_0(c)->name, P_PLAYER_NAME_SIZE);

      /* Send reply */
      if(pnet_send_header(c, c->socket, P_NET_CONFIG_ID, P_NET_CONFIG_SIZE) < 0) {
         error = 1;
      } else if(pnet_send_buffer(c, c->socket, packet, P_NET_CONFIG_SIZE) < 0) {
         error = 1;
      } else {
         /* We are done */
         p_config_network(c);

         #ifdef P_NET_DEBUG_COMM
         printf("recv_config:  Sent reply for configuration\n");
         #endif /* P_NET_DEBUG_COMM */
      } /* Any errors sending the reply? */

   } /* Any errors? */      

   p_window_set_status(c->window, NULL);
   p_window_set_waiting(c->window, 0);

   /* Return result */
   if(error) return(-1);
   return(0);

}


#endif /* Allow network? */
