/*

    xpuyopuyo - pfield.c      Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Playing field manipulation code
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdio.h>
#include <stdlib.h>
#include <config.h>
#include <pfield.h>


void p_field_clear(pfield *f) {
/* p_field_clear

   Clears the playing field to P_CLEAR.   */

   int *p;
   int i;

   p = f->data;
   i = f->width * f->height;
   while(i) {
      *p = P_CLEAR;
      p++;
      i--;
   }  
   
}


void p_field_copy(pfield *dest, const pfield *src) {
/* p_field_copy

   Copies src playing field to the destination playing field given.  */

   int *psrc;
   int *pdest;
   int i;

   psrc = src->data;
   pdest = dest->data;
   i = dest->width * dest->height;
   while(i) {
      *pdest = *psrc;
      psrc++;
      pdest++;
      i--;
   }  
   
}


static inline pfield *_p_field_new_uninitialised(int width, int height, int num) {
/* _p_field_new_uninitialised

   Allocates memory for the playing field, and sets the default
   playing field variables.  But does not initialise the field
   itself.  */

   pfield *f;

   /* Allocate memory for field */
   if(!(f = (pfield *)malloc(sizeof(pfield)))) {
      perror("Cannot allocate pfield");
      return(NULL);
   }

   /* Set variables for field */   
   f->width = width;
   f->height = height;
   f->size = width * height;
   f->number = num;
   f->piece = NULL;

   /* Allocate memory for field data */   
   if(!(f->data = (int *)malloc(sizeof(int) * f->size))) {
      perror("Cannot allocate pfield data");
      free(f);
      return(NULL);
   }

   /* Allocate memory for field redraw array */   
   if(!(f->redraw = (int *)malloc(sizeof(int) * f->size))) {
      perror("Cannot allocate pfield redraw");
      free(f);
      return(NULL);
   }

   /* By default, assume we want to redraw everything for a new field */
   p_field_redraw_all(f);
   
   return(f);

}


pfield *p_field_new(int width, int height, int num) {
/* p_field_new

   Creates a new, cleared field of the specified dimensions.
   The field number (specified) should be the same as the 
   player number; that is, 0 or 1.  */

   pfield *f;

   if(!(f = _p_field_new_uninitialised(width, height, num))) return(NULL);
   p_field_clear(f);
   return(f);

}


pfield *p_field_new_copy(const pfield *copy) {
/* p_field_new_copy

   Creates a new field based on information in copy.  */

   pfield *f;

   if(!(f = _p_field_new_uninitialised(copy->width, copy->height, copy->number))) return(NULL);
   p_field_copy(f, copy);
   return(f);

}


void p_field_free(pfield **f) {
/* p_field_free

   Releases the specified field, and sets *f to be a NULL pointer.  */

   if(!f || !*f) return;
   if((*f)->piece) p_field_free(&(*f)->piece);
   free((*f)->redraw);
   free((*f)->data);
   free(*f);
   *f = NULL;

}


int p_field_get(const pfield *f, int x, int y) {
/* p_field_get

   Returns the contents of the field at (x, y), or P_WALL if the coordinates
   specified are out of bounds.  */

   if(x >= f->width || y >= f->height || x < 0 || y < 0) return(P_WALL);
   return(P_FIELD_XY(f, x, y));

}


void p_field_set(pfield *f, int x, int y, int z) {
/* p_field_set

   Sets the contents of the field at (x, y) to the value in z.  Does nothing
   if the coordinates specified are out of bounds.  */

   if(x >= f->width || y >= f->height || x < 0 || y < 0) return;
   P_FIELD_XY(f, x, y) = z;

}


void _p_field_redraw(pfield *f, int x1, int y1, int x2, int y2, int redraw) {

   int *p;
   int i;
   int j;

   if(x2 < x1) {
      x1 = x1 + x2;
      x2 = x1 - x2;
      x1 = x1 - x2;
   }
   if(y2 < y1) {
      y1 = y1 + y2;
      y2 = y1 - y2;
      y1 = y1 - y2;
   }
   
   if(x1 >= f->width || x2 < 0) return;
   if(y1 >= f->height|| y2 < 0) return;
   if(x1 < 0) x1 = 0;
   if(y1 < 0) y1 = 0;
   if(x2 >= f->width) x2 = f->width - 1;
   if(y2 >= f->height)y2 = f->height- 1;
   
   p = f->redraw + y1 * f->width + x1;
   j = y2 - y1 + 1;
   while(j) {
      i = x2 - x1 + 1;
      while(i) {
         *p = redraw;
         p++;
         i--;
      }
      p += f->width - (x2 - x1 + 1);
      j--;
   }
   return;

}


void p_field_redraw_all(pfield *f) {
/* p_field_redraw_all

   Redraw the entire playing field.  Read by pwindow to determine what needs
   to be redrawn in the next screen update.  */

   _p_field_redraw(f, 0, 0, f->width - 1, f->height - 1, 1);
   return;

}


void p_field_redraw_none(pfield *f) {
/* p_field_redraw_none

   Redraw nothing.  Called once the screen has been refreshed, to reset the
   refresh rectangle.   */

   _p_field_redraw(f, 0, 0, f->width - 1, f->height - 1, 0);
   return;
   
}


void p_field_redraw_rect(pfield *f, int x1, int y1, int x2, int y2) {
/* p_field_redraw_rect

   Add the rectangle specified to the redraw frame.  */
   
   _p_field_redraw(f, x1, y1, x2, y2, 1);
   return;
   
}


void p_field_redraw_point(pfield *f, int x, int y) {
/* p_field_redraw_point

   Add the point specified to the redraw frame.  */

   _p_field_redraw(f, x, y, x, y, 1);
   return;

}


