/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "label.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

#include <assert.h>

namespace wftk {

Label::Label(const std::string& text, const Font& font) :
  text_(0),
  labelFont_(font.valid() ? font : Font::textFont()),
  align_(ALIGN_CENTER)
{
  setTextColor(labelFont_.color());

  getResourceBackground("label");

  setText(text);

  setPackingInfo();
}

Label::~Label()
{
  delete text_;
}

void Label::draw(Surface& surf, const Point& p, const Region& r)
{
  Debug::channel(Debug::DRAWING)<<"Label blitting text: \""<<labelText_<<"\""<<Debug::endl;

  Widget::draw(surf, p, r);

  if(!text_)
    text_ = labelFont_.getString(labelText_);

  Point offset(p);

  offset.y += (height() - text_->height()) / 2;
 
  switch(align_)
    {
    case ALIGN_CENTER:
      offset.x += (width()-text_->width())/2;
      break;
    case ALIGN_LEFT:
      break;
    case ALIGN_RIGHT:
      offset.x += width()-text_->width();
      break;
    }

  text_->blit(surf, offset, r);
}

void Label::setText(const std::string& data)
{
  //Debug::out << "setText(" << data << ")" <<Debug::endl;
  labelText_ = data;
  delete text_;
  text_ = 0;
  packingUpdate();
  invalidate();
}
 
void Label::setTextColor(const Color& fontCol)
{
  // defer to the font
  labelFont_.setColor(fontCol);
  delete text_;
  text_ = 0;
  invalidate();
}

void Label::setPackingInfo()
{
  Widget::setPackingInfo();
  packing_info_.y.expand = false;

  if(!labelText_.empty()) {
    // the extra space will allow half a space on either
    // side of the text
    Rect extents = labelFont_.getExtents(labelText_ + " ");

    packing_info_.x.min = extents.w;
    packing_info_.y.min = extents.h;
    if(extents.w > packing_info_.x.pref)
      packing_info_.x.pref = extents.w;
    if(extents.h > packing_info_.y.pref)
      packing_info_.y.pref = extents.h;
  }
}

} // namespace wftk
