#ifndef SIMULATION_CONFIGURATION_H
#define SIMULATION_CONFIGURATION_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include <string>
#include <vector>

using std::string;
using std::vector;

class ConfigurationScope;
class ConfigurationChoice;

/** This class represents a simulation configuration. */
class SimulationConfiguration {
  
  /** Simulation needs to be able to allocate SimulationConfigurations in
      the default case. */
  friend class Simulation;

public:
  /**
     Parse a configuration file and return it.

     @param configFileName The name of the configuration file.  Must be a
     valid file name relative to where the program was run from.

     @param argc The number of arguments in the command line.

     @param argv The arguments from the command line.
 */
  static SimulationConfiguration *parseConfiguration( const string &configFileName,
						      const vector<string> &argumentVector );
  
  /** This method returns additional command line arguments that were
     provided at startup.  */
  const vector<string> &getArguments();

  bool communicationManagerIs( const string &testValue ) const;
  const string getCommunicationManagerType() const;

  /**
     If the EventList value is what is passed in, return true, else, false;
  */
  bool eventListTypeIs( const string &testValue ) const;
  bool eventListOrganizationIs( const string &testValue ) const;

  /**
     Return the string of the EventList section of the config file, or
     "(none)".
  */
  const string getEventListType() const;
  const string getEventListOrganization() const;

  /**
     If the GVTManager value is what is passed in, return true, else, false;
  */
  bool gvtManagerTypeIs( const string &testValue ) const;
  /**
     Return the string of the GVTManager of the config file, or
     "(none)".
  */
  const string getGVTManagerType() const;
  

  bool simulationTypeIs( const string &testValue ) const;
  const string getSimulationType() const;

  /**
     Return the gvt period that was specified.
     @return true if an int was found, false otherwise.
     @param period The integer to write the period into if a period is
     found, unchanged otherwise.
  */
  bool getGVTPeriod( unsigned int &period );

  bool physicalLayerIs( const string &testValue ) const;
  const string getPhysicalLayerType() const;

  bool outputManagerIs( const string &testValue ) const;
  const string getOutputManagerType() const;

  bool spinKeySet( const string &testValue ) const;

  bool schedulerTypeIs( const string &testValue ) const;
  const string getSchedulerType() const;

  bool getNumberOfSimulationManagers( unsigned int &number );

  bool stateManagerTypeIs( const string &testValue ) const;
  const string getStateManagerType() const;

  bool getStatePeriod( unsigned int &period ) const;

  /**
     Returns the list of nodes that it has been requested that this
     simulation execute on.
  */
  vector<string> getNodes() const;

  /**
     Returns the binary name which started this execution.  Presumably we
     will use this binary on every node.
  */
  const string getBinaryName() const;

  ~SimulationConfiguration();

private:
  SimulationConfiguration( const ConfigurationScope *outerScope,
			   const vector<string> &argumentVector );

  class Implementation;
  Implementation *_impl;
};

#endif
