/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <AppKit/NSPrintOperation.h>
#include <PDFKit/PDFPrintView.h>
#include "AppController.h"
#include "ViewPDFDocument.h"


/*
 * Non-Public methods.
 */
@interface ViewPDFDocument(Private)
@end


/*
 * An NSDocument that loads a PDF file and holds a
 * corrensponding PDFDocument object. This object
 * can be used by the application to access the
 * content and properties of the file. The PDFDocument
 * is retained and released by this class.
 */
@implementation ViewPDFDocument

/*
 * Designated initializer.
 */
- (id) init
{
   if ((self = [super init]))
   {
      pdfDocument = nil;
   }

   return self;
}


- (void) dealloc
{
   RELEASE(pdfDocument);
   [super dealloc];
}


/*
 * Dump the outline recursive. Only for debugging.
 */
static void printOutlineItems(NSArray* items, int level)
{
   int i;
   NSEnumerator* e = [items objectEnumerator];
   PDFOutlineItem* anItem;
   
   NSMutableString* pattern = [[NSMutableString alloc] init];
   for (i = 0; i < level; i++)
   {
      [pattern appendString: @"   "];
   }
   [pattern appendString: @"%@"];

   while ((anItem = [e nextObject]))
   {
      NSLog(pattern, [anItem title]);
      if ([anItem hasKids])
      {
         int n = level + 1;
         printOutlineItems([anItem kids], n);
      }
   }

   [pattern release];
}


/*
 * Read a PDF document from a file and create a corresponding
 * PDFDocument object. 
 */
- (BOOL) readFromFile: (NSString*)fileName ofType: (NSString*)docType
{
   pdfDocument = RETAIN([PDFDocument documentFromFile: fileName]);
   // Dump outline (only for testing purpose)
   /*
   if (succeeded)
   {
      if ([pdfDoc hasOutline])
      {
         NSLog(@"**** Document Outline:");
         printOutlineItems([[pdfDoc outline] items], 1);
      }
      else
      {
         NSLog(@"Document has not Outline");
      }
   }
   */

   return [pdfDocument isOk];
}


/*
 * Returns the PDFDocument hold by this instance.
 */
- (PDFDocument*)pdfDocument
{
   return pdfDocument;
}


/*
 * Only for debugging.
 */
- (void) close
{
   NSLog(@"Close Document");
   [super close];
}


/*
 * Create an instance of DocumentWindowController that will
 * control a window that displays this document.
 */ 
- (void) makeWindowControllers
{
   [[AppController sharedController] createAndDisplayWindowForDocument: self];

   NSLog(@"new window controller created");   
}


/*
 * Print the document.
 */
- (void)printShowingPrintPanel:(BOOL)showPanels
{
   PDFPrintView*      printView;
   NSPrintOperation*  printOp;

   NSAssert([self pdfDocument], @"cannot print without a PDFDocument");

   printView = [[PDFPrintView alloc] initWithDocument: [self pdfDocument]];
   
   printOp = [NSPrintOperation printOperationWithView: printView 
                                            printInfo: [self printInfo]];
   [printOp setShowPanels: showPanels];
   [printOp runOperation];
//    [printOp runOperationModalForWindow: [self window];
//                               delegate: nil
//                         didRunSelector: NULL
//                            contextInfo: NULL];
//    [self runModalPrintOperation: printOp
//                        delegate: nil
//                  didRunSelector: NULL
//                     contextInfo: NULL];

   RELEASE(printView);
}

@end
