namespace eval make {
    variable make

    # Define the make array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set make {
	list.reset ""
	list.clear ""
	entry.file ""
	entry.target ""
	entry.change_dir ""
	entry.include_dir ""
	entry.load ""
	entry.jobs ""
	entry.oldfile ""
	entry.fake_target ""
	target_sort increasing
	dialog ""
	file ""
	target ""
	debug ""
	ignore ""
	continue ""
	print_cmds ""
	database ""
	question ""
	implicit ""
	silent ""
	touch ""
	version ""
	working_dir ""
	change_dir ""
	include_dir ""
	jobs ""
	load ""
	oldfile ""
	fake_target ""
    }
}

# make::create --
#
#   Method to create the dialog box for the make command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::create { } {
    global tkWorld
    variable make

    # Build the toplevel dialog without grabbing focus if it
    # does not exist.
    if [winfo exists $make(dialog)] {
	switch -- [wm state $make(dialog)] {
	    normal {
		raise $make(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $make(dialog)
	    }
	}
	focus $make(dialog)
	return
    } else {
	set make(dialog) [dialog::create .make Make]
    }

    # The first tab has file and regexp information, along with
    # the standard command line options.
    set tab1 [tabnotebook::page [dialog::interior $make(dialog)] "Options"]

    # Create the top frame which will contain the entries for
    # the filenames and regular expression. NOTE: These entries are
    # NOT reset by the Reset button.  Let the user highlight and
    # delete the text since they are more often to change options instead
    # of files.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.file_button \
	    -text "File" \
	    -width 7 \
	    -command make::open
    set make(entry.file) [entry $f1.entry_file \
	    -background #ffffff \
	    -width 36 \
	    -textvariable make::make(file)]
    button $f1.target_label \
	    -text "Target" \
	    -width 7 \
	    -command make::target
    set make(entry.target) [entry $f1.entry_target \
	    -background #ffffff \
	    -width 25 \
	    -textvariable make::make(target)]
    button $f1.target_reverse \
	    -text "Reverse" \
	    -width 7 \
	    -command make::reverse
    grid $f1.file_button $f1.entry_file - \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.target_label $f1.entry_target $f1.target_reverse \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Duhhh....
    focus $make(entry.file)

    # Now create all of the on/off checkbuttons.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]

    checkbutton $f2.debug \
	    -text "Debug" \
	    -variable make::make(debug) \
	    -onvalue "d" \
	    -offvalue ""
    checkbutton  $f2.ignore \
	    -text "Ignore" \
	    -variable make::make(ignore) \
	    -onvalue "i" \
	    -offvalue ""
    checkbutton $f2.continue \
	    -text "Continue" \
	    -variable make::make(continue) \
	    -onvalue "k" \
	    -offvalue ""
    checkbutton $f2.print_cmds \
	    -text "Print Commands" \
	    -variable make::make(print_cmds) \
	    -onvalue "n" \
	    -offvalue ""
    checkbutton $f2.database \
	    -text "Print Database" \
	    -variable make::make(database) \
	    -onvalue "p" \
	    -offvalue ""
    checkbutton $f2.question \
	    -text "Question Mode" \
	    -variable make::make(question) \
	    -onvalue "q" \
	    -offvalue ""
    checkbutton $f2.implicit \
	    -text "No Implicit Rules" \
	    -variable make::make(implicit) \
	    -onvalue "r" \
	    -offvalue ""
    checkbutton $f2.silent \
	    -text "Silent Mode" \
	    -variable make::make(silent) \
	    -onvalue "s" \
	    -offvalue ""
    checkbutton $f2.touch \
	    -text "Touch" \
	    -variable make::make(touch) \
	    -onvalue "t" \
	    -offvalue ""
    checkbutton $f2.version \
	    -text "Version" \
	    -variable make::make(version) \
	    -onvalue "v" \
	    -offvalue ""
    checkbutton $f2.working_dir \
	    -text "Working Dir" \
	    -variable make::make(working_dir) \
	    -onvalue "w" \
	    -offvalue ""

    grid $f2.debug $f2.ignore $f2.continue \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.print_cmds $f2.database $f2.question \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.implicit $f2.silent $f2.touch \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.version $f2.working_dir \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Create the first tab
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # The second tab has the advanced make options.
    set tab2 [tabnotebook::page [dialog::interior $make(dialog)] "Advanced"]

    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]

    label $f3.change_dir_label \
	    -text "Change To Dir" \
	    -anchor e \
	    -width 12
    set make(entry.change_dir) [entry $f3.entry_change_dir \
	    -width 30 \
	    -textvariable make::make(change_dir)]
    label $f3.include_dir_label \
	    -text "Include File Dir" \
	    -anchor e \
	    -width 12
    set make(entry.include_dir) [entry $f3.entry_include_dir \
	    -width 30 \
	    -textvariable make::make(include_dir)]
    label $f3.jobs_label \
	    -text "Job Limit" \
	    -anchor e \
	    -width 12
    set make(entry.jobs) [entry $f3.jobs_entry \
	    -width 5 \
	    -textvariable make::make(jobs)]
    label $f3.load_label \
	    -text "Load Limit" \
	    -anchor e \
	    -width 12
    set make(entry.load) [entry $f3.load_entry \
	    -width 10 \
	    -textvariable make::make(load)]
    label $f3.old_label \
	    -text "Old File" \
	    -anchor e \
	    -width 12
    set make(entry.oldfile) [entry $f3.old_entry \
	    -width 30 \
	    -textvariable make::make(oldfile)]
    label $f3.label_fake_target \
	    -text "Fake Target" \
	    -anchor e \
	    -width 12
    set make(entry.fake_target) [entry $f3.entry_fake_target \
	    -background #ffffff \
	    -width 30 \
	    -textvariable make::make(fake_target)]

    grid $f3.change_dir_label $f3.entry_change_dir
    grid $f3.include_dir_label $f3.entry_include_dir
    grid $f3.jobs_label $f3.jobs_entry
    grid $f3.load_label $f3.load_entry
    grid $f3.old_label $f3.old_entry
    grid $f3.label_fake_target $f3.entry_fake_target

    foreach o [grid slaves $f3] {
	grid configure $o \
		-sticky w \
		-padx 2 \
		-pady 2
    }

    # Create the second tab.
    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Define the lists for the reset and clear methods.
    set make(list.reset) "debug ignore continue print_cmds \
	    database question implicit silent touch version \
	    working_dir"
    set make(list.clear) "file target change_dir include_dir jobs \
	    load oldfile fake_target"

}

# make::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::ok { } {
    global tkWorld
    variable make

    # Test for null supplied filename.
    set wd $tkWorld(working_dir)
    if ![string length $make(file)] {
	if {[file exists [set x [file join $wd Makefile]]]} {
	    set make(file) $x
	} elseif {[file exists [set x [file join $wd makefile]]]} {
	    set make(file) $x
	} else {
	    focus $make(entry.file)
	    $make(entry.file) configure -bg red -fg white
	    stderr::log 4001
	    return
	}
    }
    unset wd
	
    # Build the file list (see above) and check for non-valid
    # file names in the list.
    set flist ""
    foreach e $make(file) {
	if [catch {append flist "-f \"[glob $e]\" "} err] {
	    focus $make(entry.file)
	    $make(entry.file) configure -bg red -fg white
	    stderr::log 4002 $e  ;# Using msg for the filename.
	    return
	}
    }
    set flist [string trimright $flist " "]

    # Build the command line argument based on the basic options
    # first which do not have arguments.
    set cmd_arg ""
    foreach x $make(list.reset) {
	if [string length $make($x)] {
	    append cmd_arg $make($x)
	}
    }
    if [string length $cmd_arg] {
	set cmd_arg "-$cmd_arg "
    }

    # Now parse the advanced options which have arguments
    # like -I /home/wesb.
    foreach x $make(list.clear) {
	if [string length $make($x)] {
	    switch $x {
		file -
		target {
		    # Just skip these
		}
		change_dir {
		    # Test for validity on the directory.
		    if ![file isdirectory $make($x)] {
			focus $make(entry.change_dir)
			$make(entry.change_dir) configure -bg red -fg white
			stderr::log 4003 $make($x)
			return
		    }
		    append cmd_arg "-C \"$make($x)\" "
		}
		include_dir {
		    # Test for validity on the directory.
		    if ![file isdirectory $make($x)] {
			focus $make(entry.include_dir)
			$make(entry.include_dir) configure -bg red -fg white
			stderr::log 4003 $make($x)
			return
		    }
		    append cmd_arg "-I \"$make($x)\" "
		}
		jobs {
		    append cmd_arg "-j \"$make($x)\" "
		}
		load {
		    append cmd_arg "-l \"$make($x)\" "
		}
		oldfile {
		    if ![file isfile $make($x)] {
			focus $make(entry.oldfile)
			$make(entry.oldfile) configure -bg red -fg white
			stderr::log 4004 $make($x)
			return
		    }
		    append cmd_arg "-o \"$make($x)\" "
		}
		fake_target {
		    append cmd_arg "-W \"$make($x)\" "
		}
	    }
	}
    }

    # Insert the Tcl command list in the Command Center.
    if [string length $cmd_arg] {
	set cmd_arg [string trimright $cmd_arg " "]
	$tkWorld(cmd_center) insert insert \
		[subst {make $flist $cmd_arg $make(target)}]
    } else {
	$tkWorld(cmd_center) insert insert \
		[subst {make $flist $make(target)}]
    }	

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the
    # validated entries.
    foreach x $make(list.clear) {
	$make(entry.$x) configure -bg white -fg black
    }

    # Give warning on null targets.
    if ![string length $make(target)] {
	$make(entry.target) configure -bg yellow -fg black
	stderr::log 3002 $flist
    }
}

# make::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::reset { } {
    variable make

    # Reset all of the checkbuttons.
    foreach x $make(list.reset) {
	set make($x) ""
    }
}

# make::clear --
#
#   Method to clear entry items and reset their bg/fg properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::clear { } {
    variable make

    # Reset the values of the entries and configure the
    # bg/fg properties.
    foreach x $make(list.clear) {
	set make($x) ""
	$make(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $make(entry.file)
}

# make::help --
#
#   Method to display help on using make.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::help { } {
    global tkWorld

    help::create "help/make.html" "Make Command Help"
}

# make::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::close { } {
    variable make

    balloonhelp::cancel
    destroy $make(dialog)
}

# make::open --
#
#   Method to open a single target for make.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::open { } {
    variable make

    # Delete any existing files in the target entry.
    $make(entry.file) delete 0 end

    # Get the makefile from the open dialog and move the cursor.
    regsub { } [file::select] {} make(file)
    $make(entry.file) icursor end
}

# make::target --
#
#   Method to create a picklist of the available targets for
#   a chosen makefile.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::target { } {
    global tkWorld
    variable make

    # Test for null supplied filename.
    set wd $tkWorld(working_dir)
    if ![string length $make(file)] {
	if {[file exists [set x [file join $wd Makefile]]]} {
	    set make(file) $x
	} elseif {[file exists [set x [file join $wd makefile]]]} {
	    set make(file) $x
	} else {
	    focus $make(entry.file)
	    $make(entry.file) configure -bg red -fg white
	    stderr::log 4001
	    return
	}
    }
    unset wd

    # Open the supplied Makefile. Notice the use of the global open
    # command since the make namespace has make::open.
    if [catch {::open $make(file) r} fid] {
	bell; tk_messageBox \
		-type ok \
		-message $fid \
		-icon info \
		-title "Error Opening Makefile"
	return
    }

    # Remove current targets from the entry.
    $make(entry.target) delete 0 end

    # Now build the target list.
    set target_list ""
    foreach line [lsort -increasing [split [read $fid] \n]] {
	if [regexp "^(\[^:\$\]*):" $line match target] {
	    append target_list "$target "
	}
    }
    set target_list [string trimright $target_list " "]

    # Build the modal dialog which the user selects the allowed
    # targets.  Note the order of these picklist methods cannot
    # change!
    set picklist::picklist(status) 0
    picklist::create $target_list
    picklist::wait

    # Insert the users selection in the entry, move the cursor,
    # and set the focus.
    $make(entry.target) insert insert [picklist::cancel]
    $make(entry.target) icursor end
    focus $make(entry.target)

    # Since the order of the elements is by default ascii increasing,
    # set the sort order to decreasing for a reverse.
    set make(target_sort) decreasing
}

# make::reverse --
#
#   Method to reverse the order of the targets in the entry.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc make::reverse { } {
    variable make

    # Define the current list of targets to reverse.
    set clist [$make(entry.target) get]

    # Delete the list from the entry, reverse the order, and
    # move the cursor. Note the preserving of the space at the end
    # of the list so that a user can append another selection at
    # the end of the entry.
    $make(entry.target) delete 0 end
    $make(entry.target) insert end "[lsort -$make(target_sort) $clist] "
    $make(entry.target) icursor end

    # Set the sort order for the next callback.
    switch $make(target_sort) {
	decreasing {
	    set make(target_sort) increasing
	}
	default {
	    set make(target_sort) decreasing
	}
    }
}


