/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue_epoll.c - epoll()-based event system
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "system.h"

#if defined(HAVE_SYS_EPOLL_H) && defined(HAVE_EPOLL_CREATE)
#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/epoll.h>
#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif
#include <sys/time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "nqueue_epoll.h"
#include "types.h"
#include "thy.h"

static struct epoll_event *events; /**< Events returned by
				      epoll_wait(). */
static int epfd; /**< epoll FD. */
static int *rfdidx; /** FD->returned event lookup table. */

int
thy_nq_init_epoll (void)
{
  epfd = epoll_create (_THY_MAXCONN);
  if (epfd == -1)
    {
      bhc_error ("epoll_create(): %s", strerror (errno));
      return -1;
    }

  events = (struct epoll_event *)bhc_calloc (_THY_MAXCONN + 1,
					     sizeof (struct epoll_event));
  rfdidx = (int *)bhc_calloc (_THY_MAXCONN + 1, sizeof (int));

  return 0;
}

void
thy_nq_fd_control_epoll (int fd, thy_nq_event_t event, int set)
{
  struct epoll_event ev;
  int op = EPOLL_CTL_ADD;

  if (fd < 0)
    return;

  if (fdstates[fd])
    op = EPOLL_CTL_MOD;

  if (fdstates[fd] && thy_nq_act_fds)
    thy_nq_act_fds--;

  if (trigger[fd])
    trigger_count--;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  fdstates[fd] = 0;
  trigger[fd] = 0;

  ev.data.fd = fd;
  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      ev.events = EPOLLIN | EPOLLHUP | EPOLLERR;
      break;
    case THY_NQ_EVENT_OUTPUT:
      ev.events = EPOLLOUT | EPOLLHUP | EPOLLERR;
      break;
    case THY_NQ_EVENT_TRIGGER:
      trigger_count++;
      trigger[fd] = 1;
      return;
    default:
      ev.events = EPOLLIN | EPOLLOUT | EPOLLHUP | EPOLLERR;
      break;
    }

  if (set)
    {
      fdstates[fd] = 1;
      thy_nq_act_fds++;
    }

  epoll_ctl (epfd, (set) ? op : EPOLL_CTL_DEL, fd, &ev);
}

int
thy_nq_fd_check_epoll (int fd, thy_nq_event_t event)
{
  int ridx, rval = 0;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  if (event == THY_NQ_EVENT_TRIGGER && trigger[fd])
    return 1;

  ridx = rfdidx[fd];

  if (events[ridx].data.fd != fd)
    return 0;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      if (events[ridx].events & (POLLIN | POLLHUP | POLLERR))
	rval = 1;
      break;
    case THY_NQ_EVENT_OUTPUT:
      if (events[ridx].events & (POLLOUT | POLLHUP | POLLERR))
	rval = 1;
      break;
    default:
      return 0;
    }

  return rval;
}

int
thy_nq_wait_epoll (long int timeout)
{
  int i, r;

  errno = 0;
  r = epoll_wait (epfd, events, _THY_MAXCONN, timeout * 1000);
  if (r == -1)
    return -1;

  for (i = 0; i < r; i++)
    rfdidx[events[i].data.fd] = i;

  return r;
}
#endif

/* arch-tag: cc56e529-2d15-4b4a-9f79-e41ed64d5335 */
