/* Strip Club - Online/Offline Comic Reader/Archiver
 *
 * Copyright notice for this file:
 *  Copyright (C) 2004 Benjamin Cutler
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef _WIN32
#include <sys/stat.h>
#endif
#include <pcre.h>
#include <SDL/SDL_net.h>
#include <FL/Fl_Box.H>
#include <FL/Fl_Preferences.H>
#include <FL/Fl_Text_Buffer.H>
#include <FL/fl_ask.H>
#include "load.h"
#include "url.h"
#include "http.h"
#include "html.h"

char *CurComic;
Fl_Preferences *Prefs = NULL, *CurComicG = NULL;
cInterface Interface;
PageLinks CurLinks;
eOffType Offline = OFFTYPE_CHECK;
char *OffArray[] = { "Offline", "Missing", "Check" };
Fl_Text_Buffer HelpBuffer;
sURL *Proxy = NULL;

void EditComic(Fl_Widget *w, void *Comic);
void OpenStatsWindow(Fl_Widget *w, void *Comic);
void DonateToComic(Fl_Widget *w, void *Comic);
void OpenStore(Fl_Widget *w, void *Comic);

// import.cpp
void ExportComic(Fl_Widget *w, void *Comic);
void UpdateComics();

int main(int argc, char **argv) {
	Fl_Text_Buffer TextOutputBuffer, UpstreamOutputBuffer, DownstreamOutputBuffer, DebugOutputBuffer;
	int oldw = 0, oldh = 0;

#ifdef _WIN32
	{
		struct stat filestat;
		if (stat("stripclub.exe", &filestat)) {	// Can't find the executable, so the working dir must be wrong (sanity check)
			MOut("Strip Club should not be run from any directory but the location of the executable. Working directory is pointing to the wrong place, perhaps?\n");
			return 1;
		}
	}
#endif

	if(SDL_Init(0)==-1) {
		Out("SDL_Init: %s\n", SDL_GetError());
		exit(1);
	}
	if(SDLNet_Init()==-1) {
		Out("SDLNet_Init: %s\n", SDLNet_GetError());
		exit(2);
	}

	debug = fopen(GetCacheFileName("debug.log", true, "debug"), "w");

	Quit = false;

	CurComic = CurLinks.FirstLink = CurLinks.PrevLink = CurLinks.CurLink = CurLinks.NextLink = CurLinks.LatestLink = NULL;

	InitOutputWindow();
	MakeMiscWindows();
	Prefs = new Fl_Preferences(Fl_Preferences::USER, "racercksproductions", "stripclub");

	TextOutput->buffer(&TextOutputBuffer);
	UpstreamOutput->buffer(&UpstreamOutputBuffer);
	DownstreamOutput->buffer(&DownstreamOutputBuffer);
	DebugOutput->buffer(&DebugOutputBuffer);

	// Attempt to create a proxy
	{
		char *proxystring;
		proxystring = getenv("http_proxy");
		if (!proxystring) {		// No env variable
			if (Prefs->entryExists("http_proxy")) {
				Prefs->get("http_proxy", proxystring, "");
				Out("Using %s for proxy (stored in preferences)\n", proxystring);
				Proxy = URLCreate(proxystring);		// If URLCreate returns NULL, that's ok, it means we probably don't need a proxy
				free(proxystring);
			}
		} else {
			Out("Using %s for proxy (environment variable)\n", proxystring);
			Proxy = URLCreate(proxystring);
		}
	}

#ifndef NOGZIP
	{
		char buf[512];
		snprintf(buf, 512, "gzip -c -d %s >%s", ZREADME, GetCacheFileName(README, true, "temp"));
		system(buf);
	}
#endif

	HelpBuffer.loadfile(
#ifdef NOGZIP
	README
#else
	GetCacheFileName(README, true, "temp")
#endif
);

	strcpy(WindowTitle, SCVERSION);

	Interface.Make();

	Interface.MainWindow->show();

	{
		int remx, remy, remw, remh;
		Prefs->get("remx", remx, 4);
		Prefs->get("remy", remy, 4);
		Prefs->get("remw", remw, 640);
		Prefs->get("remh", remh, 400);
		Interface.MainWindow->resize(remx, remy, remw, remh);
	}

	{
		int tmp;
		Prefs->get("offline", tmp, OFFTYPE_MISSING);
		Offline = (eOffType)tmp;
		Interface.OutputOffline->value(OffArray[Offline]);
	}

	if (!Prefs->entryExists("warning")) {
		SplashWarning->show();
		Prefs->set("warning", 1);
	}

	InitMenu();

	while (Interface.MainWindow->shown() && !Quit) {
		if ((oldw != Interface.MainWindow->w()) || (oldh != Interface.MainWindow->h())) {
			oldw = Interface.MainWindow->w();
			oldh = Interface.MainWindow->h();
			Interface.MainMenu->resize(0, 0, 144, 20);
			Interface.OutputURL->resize(144, 0, oldw - 208, 20);
			Interface.OutputOffline->resize(oldw - 64, 0, 64, 20);
			Interface.ScrollBox->resize(2, 22, oldw - 4, oldh - 46);
			Interface.Progress->resize(2, oldh - 22, oldw - 288, 20);
			Interface.FirstButton->resize(oldw - 284, oldh - 22, 56, 20);
			Interface.PrevButton->resize(oldw - 228, oldh - 22, 56, 20);
			Interface.RefreshButton->resize(oldw - 172, oldh - 22, 56, 20);
			Interface.NextButton->resize(oldw - 116, oldh - 22, 56, 20);
			Interface.LatestButton->resize(oldw - 60, oldh - 22, 56, 20);
			Interface.MainWindow->redraw();
		}
		if (!OutputWindow->shown()) {
			OutputWindow->show();
		}
		Fl::wait();
	}

	SDLNet_Quit();

	SDL_Quit();

	if(CurComicG) {
		delete CurComicG;
	}

	Prefs->set("remx", Interface.MainWindow->x());
	Prefs->set("remy", Interface.MainWindow->y());
	Prefs->set("remw", Interface.MainWindow->w());
	Prefs->set("remh", Interface.MainWindow->h());
	Prefs->set("offline", (int)Offline);

	delete Interface.MainWindow;
	delete OutputWindow;
	delete Prefs;

	fclose(debug);

	remove(GetCacheFileName(TEMPHTML, true, "temp"));
	remove(GetCacheFileName(TEMPTXT, true, "temp"));
#ifndef NOGZIP
	remove(GetCacheFileName(README, true, "temp"));
#endif

	return 0;
}

void InitMenu() {
	Fl_Preferences *TempGroup = NULL;
	int i, comics = Prefs->groups();
	static char **groupnames = NULL;		// Static because the pointers are needed for the callback
	char buf[400], *name, *category;
	Interface.MainMenu->menu(cInterface::menu_MainMenu); 		// Reset the menu in case we're reloading
	if(!comics) {
		MOut("No comics found. Make sure you import/create some.\n");
		if (fl_ask("Run updater to download .comic files off the Strip Club site?")) {
			UpdateComics();
		}
		return;
	}
	if (groupnames) {
		for (i = 0; groupnames[i]; i++) {
			delete groupnames[i];
		}
		delete groupnames;
	}
	groupnames = new char*[comics + 1];
	groupnames[comics] = 0;			// To mark the end of the array
	for (i = 0; i < comics; i++) {
		clonestr(Prefs->group(i), (groupnames + i));
		if(strlen(groupnames[i])) {
			Out("Loading Comic: %s\n", groupnames[i]);
			delete TempGroup;
			TempGroup = new Fl_Preferences(Prefs, groupnames[i]);
			if (TempGroup->entryExists("name")) {
				TempGroup->get("name", name, "");
				TempGroup->get("category", category, "Unsorted");
				snprintf(buf, 399, "Comics/%s/%s/Read", category, name);
				Interface.MainMenu->add(buf, 0, SwitchComic, groupnames[i]);
				snprintf(buf, 399, "Comics/%s/%s/Edit", category, name);
				Interface.MainMenu->add(buf, 0, EditComic, groupnames[i]);
				snprintf(buf, 399, "Comics/%s/%s/Export", category, name);
				Interface.MainMenu->add(buf, 0, ExportComic, groupnames[i]);
				snprintf(buf, 399, "Comics/%s/%s/Stats", category, name);
				Interface.MainMenu->add(buf, 0, OpenStatsWindow, groupnames[i]);
				if (TempGroup->entryExists("donate")) {
					snprintf(buf, 399, "Comics/%s/%s/Donate", category, name);
					Interface.MainMenu->add(buf, 0, DonateToComic, groupnames[i]);
				}
				if (TempGroup->entryExists("store")) {
					snprintf(buf, 399, "Comics/%s/%s/Store", category, name);
					Interface.MainMenu->add(buf, 0, OpenStore, groupnames[i]);
				}
				free(name);
				free(category);
			}
		}
	}
	delete TempGroup;
}

void AddComic() {
	const char *Buffer;
	char *GroupName;
	Buffer = fl_input("Enter a short name for the new comic (eg: sluggy)\n", "comic");
	if (!Buffer || !strlen(Buffer)) {
		return;
	}
	if (!strcmp(Buffer, "hosts")) {
		MOut("Error: \"hosts\" is a reserved value, please select another.\n");
		return;
	}
	clonestr(Buffer, &GroupName);
	if (Prefs->groupExists(GroupName)) {
		if (!fl_ask("Comic already exists! Edit?")) {
			free(GroupName);
			return;
		}
	}
	EditComic(0, GroupName);
	free(GroupName);
}

void EditComic(Fl_Widget *w, void *Comic) {
	Fl_Preferences *EditComic = new Fl_Preferences(Prefs, (char *)Comic);
	char Buffer[400];
	EditComic->get("name", Buffer, "New Comic", 399);
	InputName->value(Buffer);
	EditComic->get("baseurl", Buffer, "http://", 399);
	InputBaseURL->value(Buffer);
	EditComic->get("firsturl", Buffer, "http://", 399);
	InputFirstURL->value(Buffer);
	EditComic->get("imgpattern", Buffer, "<img .*src=\"(.*comics/.*(png|jpg|gif))\".*>", 399);
	InputImgPattern->value(Buffer);
	EditComic->get("linktype", Buffer, "tag", 399);
	if (!strcmp("tag", Buffer)) {
		ChoiceLinkType->value(0);
	} else if (!strcmp("text", Buffer)) {
		ChoiceLinkType->value(1);
	} else if (!strcmp("map", Buffer)) {
		ChoiceLinkType->value(2);
	}
	EditComic->get("prevpattern", Buffer, "<img .*name=\"previous_day\".*>", 399);
	InputPrevLink->value(Buffer);
	EditComic->get("nextpattern", Buffer, "<img .*name=\"next_day\".*>", 399);
	InputNextLink->value(Buffer);
	EditComic->get("relativeurl", Buffer, "", 399);
	InputRelative->value(Buffer);
	if (EditComic->entryExists("useragent")) {
		CheckDefaultUserAgent->value(0);
		EditComic->get("useragent", Buffer, "", 399);
		InputUserAgent->value(Buffer);
		InputUserAgent->activate();
		ChoiceUserAgent->activate();
	} else {
		CheckDefaultUserAgent->value(1);
		InputUserAgent->value(USERAGENT);
		InputUserAgent->deactivate();
		ChoiceUserAgent->deactivate();
	}
	EditComic->get("category", Buffer, "Unsorted", 399);
	InputCategory->value(Buffer);
	EditComic->get("donate", Buffer, "", 399);
	InputDonate->value(Buffer);
	EditComic->get("store", Buffer, "", 399);
	InputStore->value(Buffer);
	EditComicWindow->show();
	while (EditComicWindow->shown()) { Fl::wait(); }
	if (!strcmp("delete", InputName->value())) {
		delete EditComic;
		EditComic = NULL;
		Prefs->deleteGroup((char *)Comic);
	} else if (strcmp("cancel", InputName->value())) {		// We did NOT click "Cancel"
		EditComic->set("name", InputName->value());
		EditComic->set("baseurl", InputBaseURL->value());
		EditComic->set("firsturl", InputFirstURL->value());
		EditComic->set("imgpattern", InputImgPattern->value());
		switch(ChoiceLinkType->value()) {
			case(0): {
				EditComic->set("linktype", "tag");
				break;
			}
			case(1): {
				EditComic->set("linktype", "text");
				break;
			}
			case(2): {
				EditComic->set("linktype", "map");
				break;
			}
		}
		EditComic->set("prevpattern", InputPrevLink->value());
		EditComic->set("nextpattern", InputNextLink->value());
		EditComic->set("relativeurl", InputRelative->value());
		if (CheckDefaultUserAgent->value()) {		// Default is checked
			EditComic->deleteEntry("useragent");
		} else {									// Not checked
			EditComic->set("useragent", InputUserAgent->value());
		}
		EditComic->set("category", InputCategory->value());
		if (strlen(InputDonate->value())) {
			EditComic->set("donate", InputDonate->value());
		} else {
			EditComic->deleteEntry("donate");
		}
		if (strlen(InputStore->value())) {
			EditComic->set("store", InputStore->value());
		} else {
			EditComic->deleteEntry("store");
		}
	}
	if (EditComic) {
		if (!EditComic->entryExists("name")) {		// New comic, but cancel was clicked, so make sure it's not stored
			delete EditComic;
			Prefs->deleteGroup((char *)Comic);
		} else {
			delete EditComic;
		}
	}
	InitMenu();
	Prefs->flush(); 		// Make sure changes are written to disk
}

void SetOffline(int i)
{
	Offline = (eOffType)i;
	Interface.OutputOffline->value(OffArray[Offline]);
}

void OpenStatsWindow(Fl_Widget *w, void *Comic) {
	char *buf;		// We can just use the string value, works fine...
	Fl_Preferences *ComicG = new Fl_Preferences(Prefs, (char *)Comic);
	Prefs->get("headcount", buf, "0");
	OutputTotalHead->value(buf);
	free(buf);
	Prefs->get("getcount", buf, "0");
	OutputTotalGet->value(buf);
	free(buf);
	Prefs->get("gigsdown", buf, "0");
	OutputTotalGigs->value(buf);
	free(buf);
	Prefs->get("bytesdown", buf, "0");
	OutputTotalBytes->value(buf);
	free(buf);
	ComicG->get("headcount", buf, "0");
	OutputComicHead->value(buf);
	free(buf);
	ComicG->get("getcount", buf, "0");
	OutputComicGet->value(buf);
	free(buf);
	ComicG->get("gigsdown", buf, "0");
	OutputComicGigs->value(buf);
	free(buf);
	ComicG->get("bytesdown", buf, "0");
	OutputComicBytes->value(buf);
	free(buf);
	delete ComicG;
	StatsWindow->show();
}

void ChangeSettings() 
{
	char *buf;
	Prefs->get("browser", buf, DEFAULT_BROWSER);
	InputBrowser->value(buf);
	free(buf);
	Prefs->get("http_proxy", buf, getenv("http_proxy"));
	InputHTTPProxy->value(buf);
	free(buf);
	ChoiceDownload->value((int)Offline);
	CheckDebug->value(Debug);
	SettingsWindow->show();
	while(SettingsWindow->shown()) { Fl::wait(); }
	if(strcmp(DEFAULT_BROWSER, InputBrowser->value())) {
		Prefs->set("browser", InputBrowser->value());
	} else {
		Prefs->deleteEntry("browser");
	}
	if(strlen(InputHTTPProxy->value()) && (!getenv("http_proxy") || strcmp(InputHTTPProxy->value(), getenv("http_proxy")))) {
		Prefs->set("http_proxy", InputHTTPProxy->value());
	} else {
		Prefs->deleteEntry("http_proxy");
	}
	// Attempt to reset the proxy
	{
		char *proxystring;
		proxystring = getenv("http_proxy");
		if (!proxystring) {		// No env variable
			if (Prefs->entryExists("http_proxy")) {
				Prefs->get("http_proxy", proxystring, "");
				Out("Using %s for proxy (stored in preferences)\n", proxystring);
				Proxy = URLCreate(proxystring);		// If URLCreate returns NULL, that's ok, it means we probably don't need a proxy
				free(proxystring);
			}
		} else {
			Out("Using %s for proxy (environment variable)\n", proxystring);
			Proxy = URLCreate(proxystring);
		}
	}
	SetOffline(ChoiceDownload->value());
	Debug = CheckDebug->value();
	Prefs->flush();		// Write changes to disk
}


void DonateToComic(Fl_Widget *w, void *Comic) {
	char buf[400], cmd[400], browser[400];
	Fl_Preferences *ComicG = new Fl_Preferences(Prefs, (char *)Comic);
	Prefs->get("browser", browser, DEFAULT_BROWSER, 399);

	ComicG->get("donate", buf, "", 399);

	for (unsigned int i = 0; i < strlen(buf); i++) {
		if (!strncmp((buf + i), "\\&", 2)) {
			memmove(buf + i, buf + i + 1, strlen(buf) - i);
		}
	}
#ifdef WIN32
	snprintf(cmd, 399, "%s %s", browser, buf);
#else
	snprintf(cmd, 399, "%s \"%s\" &", browser, buf);
#endif
	DbgOut("Executing: %s\n", cmd);
	system(cmd);
}

void OpenStore(Fl_Widget *w, void *Comic) {
	char buf[400], cmd[400], browser[400];
	Fl_Preferences *ComicG = new Fl_Preferences(Prefs, (char *)Comic);
	Prefs->get("browser", browser, DEFAULT_BROWSER, 399);

	ComicG->get("store", buf, "", 399);

	for (unsigned int i = 0; i < strlen(buf); i++) {		// Look for "\&" and turn them into "&"
		if (!strncmp((buf + i), "\\&", 2)) {
			memmove(buf + i, buf + i + 1, strlen(buf) - i);
		}
	}
#ifdef WIN32
	snprintf(cmd, 399, "%s %s", browser, buf);
#else
	snprintf(cmd, 399, "%s \"%s\" &", browser, buf);
#endif
	DbgOut("Executing: %s\n", cmd);
	system(cmd);
}

void Bookmark(int cmd) {		// (0) = get, (!= 0) = set
	if (CurComicG) {	// Meaningless if we're not actually looking at anything
		if (cmd) {	
			CurComicG->set("bookmark", CurLinks.CurLink);
		} else {
			char *Link, *Base;
			CurComicG->get("baseurl", Base, "");
			CurComicG->get("bookmark", Link, Base);
			SwitchDate(Link);
			free(Link);
			free(Base);
		}
	}
}
