/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: webdav.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $
 */

constant cvs_version="$Id: webdav.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $";

/* BUGS/PROBLEMS
 * - PUT method does not return correct response (at least on IE)
 * - IE asks for overwritting existing file even if file is not present
 */


inherit "http";

import webdavlib;

#include <macros.h>
#include <classes.h>
#include <attributes.h>

//#define WEBDAV_DEBUG

#ifdef WEBDAV_DEBUG
#define DAV_WERR(s, args...) werror(s+"\n", args)
#else
#define DAV_WERR(s, args...)
#endif

//#define CLASS2 

class steamDAV {
  inherit WebdavHandler;

  object _dav;
  object __fp;

  mapping null_ressources = ([ ]);

  void create(object dav, object fp) {
    _dav = dav;
    __fp = fp;
  }

  void lock(mixed ctx, string fname, mapping lock_data) {
    if ( objectp(ctx) )
      ctx->set_attribute(OBJ_LOCK, lock_data);
    else
      null_ressources[fname] = lock_data | ([ "isnull": 1, ]) ;
  }
  mapping is_locked(mixed ctx, string fname) {
    if ( !objectp(ctx) ) {
      return null_ressources[fname];
    }
    return ctx->query_attribute(OBJ_LOCK);
  }


  object get_object_id() { return _dav->get_object_id(); }
  object this() { return _dav->get_user_object(); }
}

static object __webdavHandler;

mapping handle_OPTIONS(object obj, mapping variables)
{	
    mapping result = ::handle_OPTIONS(obj, variables);
    result->extra_heads += ([ 
	"MS-Author-Via": "DAV",
#ifdef CLASS2
	"DAV": "2",
#else
	"DAV": "1", 
#endif
    ]);
	
    return result;
}

#ifdef CLASS2
mapping handle_LOCK(object obj, mapping variables)
{
  return lock(__request->not_query, __request->request_headers, 
	      __request->body_raw,
	      __webdavHandler, obj);
}

mapping handle_UNLOCK(object obj, mapping variables)
{
  return unlock(__request->not_query, __request->request_headers,
		__request->body_raw,
		__webdavHandler, obj);
}
#endif

static bool move_and_rename(object obj, string name)
{
    string fname, dname;

    if ( name[-1] == '/' )
      name = dirname(name);

    fname = basename(name);
    dname = dirname(name);
    object target = _fp->path_to_object(dname);
    if ( !objectp(target) ) {
      DAV_WERR("No Target directory found at %s", dname);
      return false;
    }
    obj->move(target);
    if ( strlen(fname) > 0 ) 
      obj->set_attribute(OBJ_NAME, fname);
    return true;
}

mapping|void handle_MOVE(object obj, mapping variables)
{
    string destination = __request->request_headers->destination;
    string overwrite   = __request->request_headers->overwrite;

    if ( objectp(obj) && obj->is_locked(DOC_LAST_MODIFIED) )
      return response_locked(__request->not_query);

    if ( !stringp(overwrite) )
	overwrite = "T";
    __request->misc->overwrite = overwrite;
    __request->misc->destination = resolve_destination(
	destination,  __request->request_headers->host);

    int res = 201;
    // create copy variables before calling filesystem module
    if ( mappingp(__request->misc->destination) )
	return __request->misc->destination;
    else if ( stringp(__request->misc->destination) )
	__request->misc["new-uri"] = __request->misc->destination;
    DAV_WERR("Handling move:misc=\n"+sprintf("%O\n", __request->misc));
    
    object dest = _fp->path_to_object(__request->misc["new-uri"]);
    if ( objectp(dest) ) {
      if ( __request->misc->overwrite == "F" ) {
	DAV_WERR("overwritting failed !");
	return low_answer(412, "Pre-Condition Failed");
      }
      else {
	res = 204;
	dest->delete();
      }
    }    
    if ( !move_and_rename(obj, __request->misc["new-uri"]) ) 
      return low_answer(409, "conflict");
    return low_answer(res, "moved");
}

mapping|void handle_MKCOL(object obj, mapping variables)
{
    if ( strlen(__request->body_raw) > 0 )
      return low_answer(415, "unsupported type");
    // todo: read the body ?!
    mapping result = ::handle_MKDIR(obj, variables);
    if ( mappingp(result) && (result->error == 200 || !result->error) )
	return low_answer(201, "Created");
    return result;
}

mapping|void handle_COPY(object obj, mapping variables)
{
    string destination = __request->request_headers->destination;
    string overwrite   = __request->request_headers->overwrite;
    string dest_host;

    if ( !stringp(overwrite) )
	overwrite = "T";
    __request->misc->overwrite = overwrite;
    __request->misc->destination = resolve_destination(
	destination, __request->request_headers->host);
    if ( mappingp(__request->misc->destination) )
	return __request->misc->destination;

    mixed result =  ([ ]); // should now how to copy handle_http();
    object duplicate;

    duplicate = _fp->path_to_object(__request->misc->destination);

    DAV_WERR("Handling COPY:misc=\n"+sprintf("%O\n", __request->misc));
    DAV_WERR("Found dest resource = %s !",
	     (objectp(duplicate) ? "yes" : "no"));
    int res = 201;
    if ( objectp(duplicate) ) {
      if ( __request->misc->overwrite == "F" ) {
	DAV_WERR("overwritting failed !");
	return low_answer(412, "conflict");
      }
      else {
	res = 204;
	duplicate->delete();
      }
    }    
    if ( objectp(obj) ) 
      {
	if ( obj->get_object_class() & CLASS_CONTAINER )
	  duplicate = obj->duplicate(true);
	else
	  duplicate= obj->duplicate();

	// dirname and fname
	if ( !move_and_rename(duplicate, __request->misc->destination) )
	  return low_answer(409, "conflict");
	return low_answer(res, "copied");
      }
    else {
      return low_answer(404, "not found");
    }
}

mapping|void handle_PROPPATCH(object obj, mapping variables)
{
    return proppatch(__request->not_query, __request->request_headers,
		     __request->body_raw, __webdavHandler, obj);
}

mapping|void handle_PROPFIND(object obj, mapping variables)
{
  werror("handle_PROPFIND("+(objectp(obj) ? obj->describe() : "none")+")\n");
  if ( !objectp(obj) )
    return low_answer(404, "not found");
  return propfind(__request->not_query, __request->request_headers, 
		  __request->body_raw, __webdavHandler, obj);
}    

mixed get_property(object obj, Property property)
{
  if ( !objectp(property) )
    error("No property found, null-pointer !");
  DAV_WERR("Get property %s, val=%O, ns=%O", property->get_name(),obj->query_attribute(property->get_name()), property->describe_namespace());
  string pname = property->get_ns_name();
#if 0
  switch( property->get_name() ) {
  case "displayname":
    return obj->query_attribute(OBJ_NAME);
  case "name":
    return obj->get_identifier();
  }
#endif
  mixed res = obj->query_attribute(pname);  
  if ( stringp(res) )
    return replace(res, ({ "<", ">" }), ({ "&lt;", "&gt;" }));
  return res;
}

int set_property(object obj, Property property, mapping namespaces)
{
  string val = property->get_value();
  string xmlns = property->describe_namespace();
  DAV_WERR("Set property %s", property->_sprintf());

  obj->set_attribute(property->get_ns_name(), val);
  return 1;
}

string resolve_redirect(object link)
{
  object res = link->get_link_object();
  return _fp->object_to_filename(res);
}

object get_context(object ctx, string f)
{
  if ( objectp(ctx) )
    return ctx->get_object_byname(f);
  return 0;
}

int is_link(object ctx)
{
  if ( objectp(ctx) && ctx->get_object_class() & CLASS_LINK ) 
    return 1;
  return 0;
}

static mapping 
call_command(string cmd, object obj, mapping variables)
{
    mapping result = ([ ]);

    // overwritten - must not forward requests without trailing /
    DAV_WERR("DAV: %s %s", cmd, __request->not_query);


    function call = this_object()["handle_"+cmd];
    if ( functionp(call) ) {
        result = call(obj, variables);
    }
    else {
	result->error = 501;
	result->data = "Not implemented";
    }
    return result;
}

void create(object fp, bool admin_port)
{
    ::create(fp, admin_port);
    __webdavHandler = steamDAV(this_object(), fp);
    __webdavHandler->get_directory = fp->get_directory;
    __webdavHandler->stat_file = fp->stat_file;
    __webdavHandler->set_property = set_property;
    __webdavHandler->get_property = get_property;
    __webdavHandler->resolve_redirect = resolve_redirect;
    __webdavHandler->get_context = get_context;
    __webdavHandler->is_link = is_link;
}

void respond(object req, mapping result)
{
    if ( stringp(result->data) )
	result->length = strlen(result->data);
    ::respond(req, result);
}
