#include <gtk/gtk.h>
#include <phat/phat.h>
#include "miditab.h"
#include "gui.h"
#include "patch.h"

static GtkVBoxClass* parent_class;

static void midi_tab_class_init(MidiTabClass* klass);
static void midi_tab_init(MidiTab* self);


GType midi_tab_get_type(void)
{
    static GType type = 0;

    if (!type)
    {
	static const GTypeInfo info =
	    {
		sizeof (MidiTabClass),
		NULL,
		NULL,
		(GClassInitFunc) midi_tab_class_init,
		NULL,
		NULL,
		sizeof (MidiTab),
		0,
		(GInstanceInitFunc) midi_tab_init,
	    };

	type = g_type_register_static(GTK_TYPE_VBOX, "MidiTab", &info, 0);
    }

    return type;
}


static void midi_tab_class_init(MidiTabClass* klass)
{
    parent_class = g_type_class_peek_parent(klass);
}


static void chan_cb(PhatSliderButton* button, MidiTab* self)
{
    int val;

    val = phat_slider_button_get_value(button);
    patch_set_channel(self->patch, val-1);
}
		    

static void note_cb(PhatSliderButton* button, MidiTab* self)
{
    int val;

    val = phat_slider_button_get_value(button);
    patch_set_note(self->patch, val);
}


static void note_cb2(PhatSliderButton* button, MidiTab* self)
{
    int val;

    val = phat_slider_button_get_value(button);

    phat_slider_button_set_range(PHAT_SLIDER_BUTTON(self->lower_sb), 0, val);
    phat_slider_button_set_range(PHAT_SLIDER_BUTTON(self->upper_sb), val, 127);
}


static void lower_cb(PhatSliderButton* button, MidiTab* self)
{
    int val;

    val = phat_slider_button_get_value(button);
    patch_set_lower_note(self->patch, val);
}


static void upper_cb(PhatSliderButton* button, MidiTab* self)
{
    int val;

    val = phat_slider_button_get_value(button);
    patch_set_upper_note(self->patch, val);
}


static void range_cb(GtkToggleButton* button, MidiTab* self)
{
	patch_set_range(self->patch,
			gtk_toggle_button_get_active(button));
}


static void range_cb2(GtkToggleButton* button, MidiTab* self)
{
    if (gtk_toggle_button_get_active(button))
    {
	gtk_widget_set_sensitive(self->lower_sb, TRUE);
	gtk_widget_set_sensitive(self->upper_sb, TRUE);
    }
    else
    {
	gtk_widget_set_sensitive(self->lower_sb, FALSE);
	gtk_widget_set_sensitive(self->upper_sb, FALSE);
    }
}


static void connect(MidiTab* self)
{
    g_signal_connect(G_OBJECT(self->chan_sb), "value-changed",
		     G_CALLBACK(chan_cb), (gpointer) self);
    g_signal_connect(G_OBJECT(self->note_sb), "value-changed",
		     G_CALLBACK(note_cb), (gpointer) self);
    g_signal_connect(G_OBJECT(self->note_sb), "value-changed",
		     G_CALLBACK(note_cb2), (gpointer) self);
    g_signal_connect(G_OBJECT(self->lower_sb), "value-changed",
		     G_CALLBACK(lower_cb), (gpointer) self);
    g_signal_connect(G_OBJECT(self->upper_sb), "value-changed",
		     G_CALLBACK(upper_cb), (gpointer) self);
    g_signal_connect(G_OBJECT(self->range_check), "toggled",
		     G_CALLBACK(range_cb), (gpointer) self);
    g_signal_connect(G_OBJECT(self->range_check), "toggled",
		     G_CALLBACK(range_cb2), (gpointer) self);
}


static void block(MidiTab* self)
{
    g_signal_handlers_block_by_func(self->chan_sb, chan_cb, self);
    g_signal_handlers_block_by_func(self->note_sb, note_cb, self);
    g_signal_handlers_block_by_func(self->lower_sb, lower_cb, self);
    g_signal_handlers_block_by_func(self->upper_sb, upper_cb, self);
    g_signal_handlers_block_by_func(self->range_check, range_cb, self);

    /* *_cb2 omitted intentionally */
}


static void unblock(MidiTab* self)
{
    g_signal_handlers_unblock_by_func(self->chan_sb, chan_cb, self);
    g_signal_handlers_unblock_by_func(self->note_sb, note_cb, self);
    g_signal_handlers_unblock_by_func(self->lower_sb, lower_cb, self);
    g_signal_handlers_unblock_by_func(self->upper_sb, upper_cb, self);
    g_signal_handlers_unblock_by_func(self->range_check, range_cb, self);
}


static void midi_tab_init(MidiTab* self)
{
    GtkBox* box = GTK_BOX(self);
    GtkWidget* title;
    GtkWidget* table;
    GtkTable* t;
    GtkWidget* pad;

    self->patch = -1;
    gtk_container_set_border_width(GTK_CONTAINER(self), GUI_BORDERSPACE);

    /* table */
    table = gtk_table_new(7, 4, FALSE);
    t = (GtkTable*) table;
    gtk_box_pack_start(box, table, FALSE, FALSE, 0);
    gtk_widget_show(table);
    
    /* midi title */
    title = gui_title_new("MIDI");
    gtk_table_attach_defaults(t, title, 0, 4, 0, 1);
    gtk_widget_show(title);

    /* indentation */
    pad = gui_hpad_new(GUI_INDENT);
    gtk_table_attach(t, pad, 0, 1, 1, 2, 0, 0, 0, 0);
    gtk_widget_show(pad);
    
    /* midi title padding */
    pad = gui_vpad_new(GUI_TITLESPACE);
    gtk_table_attach(t, pad, 1, 2, 1, 2, 0, 0, 0, 0);
    gtk_widget_show(pad);

    /* range title padding */
    pad = gui_vpad_new(GUI_TITLESPACE);
    gtk_table_attach(t, pad, 1, 2, 5, 6, 0, 0, 0, 0);
    gtk_widget_show(pad);

    /* column spacing */
    pad = gui_hpad_new(GUI_SPACING);
    gtk_table_attach(t, pad, 2, 3, 1, 2, 0, 0, 0, 0);
    gtk_widget_show(pad);

    /* section spacing */
    pad = gui_vpad_new(GUI_SECSPACE);
    gtk_table_attach(t, pad, 0, 1, 3, 4, 0, 0, 0, 0);
    gtk_widget_show(pad);

    /* channel sliderbutton */
    self->chan_sb = phat_slider_button_new_with_range(1, 1, 16, 1, "%.0f");
    phat_slider_button_set_format(PHAT_SLIDER_BUTTON(self->chan_sb), NULL, "Channel:", NULL);
    phat_slider_button_set_threshold(PHAT_SLIDER_BUTTON(self->chan_sb), GUI_THRESHOLD);
    gtk_table_attach_defaults(t, self->chan_sb, 1, 2, 2, 3);
    gtk_widget_show(self->chan_sb);

    /* note sliderbutton */
    self->note_sb = phat_slider_button_new_with_range(60, 0, 127, 1, "%.0f");
    phat_slider_button_set_format(PHAT_SLIDER_BUTTON(self->note_sb), NULL, "Note:", NULL);
    phat_slider_button_set_threshold(PHAT_SLIDER_BUTTON(self->note_sb), GUI_THRESHOLD);
    gtk_table_attach_defaults(t, self->note_sb, 3, 4, 2, 3);
    gtk_widget_show(self->note_sb);

    /* range checkbox */
    title = gui_title_new("Range");
    self->range_check = gtk_check_button_new();
    gtk_container_add(GTK_CONTAINER(self->range_check), title);
    gtk_table_attach_defaults(t, self->range_check, 0, 4, 4, 5);
    gtk_widget_show(title);
    gtk_widget_show(self->range_check);

    /* lower sliderbutton */
    self->lower_sb = phat_slider_button_new_with_range(60, 0, 60, 1, "%.0f");
    phat_slider_button_set_format(PHAT_SLIDER_BUTTON(self->lower_sb), NULL, "Lower:", NULL);
    phat_slider_button_set_threshold(PHAT_SLIDER_BUTTON(self->lower_sb), GUI_THRESHOLD);
    gtk_table_attach_defaults(t, self->lower_sb, 1, 2, 6, 7);
    gtk_widget_show(self->lower_sb);
    gtk_widget_set_sensitive(self->lower_sb, FALSE);

    /* upper sliderbutton */
    self->upper_sb = phat_slider_button_new_with_range(60, 60, 127, 1, "%.0f");
    phat_slider_button_set_format(PHAT_SLIDER_BUTTON(self->upper_sb), NULL, "Upper:", NULL);
    phat_slider_button_set_threshold(PHAT_SLIDER_BUTTON(self->upper_sb), GUI_THRESHOLD);
    gtk_table_attach_defaults(t, self->upper_sb, 3, 4, 6, 7);
    gtk_widget_show(self->upper_sb);
    gtk_widget_set_sensitive(self->upper_sb, FALSE);

    connect(self);
}


GtkWidget* midi_tab_new(void)
{
    return (GtkWidget*) g_object_new(MIDI_TAB_TYPE, NULL);
}


void midi_tab_set_patch(MidiTab* self, int patch)
{
    int chan, note;
    int range, lower, upper;
    
    self->patch = patch;

    if (patch < 0)
	return;

    chan = patch_get_channel(patch);
    note = patch_get_note(patch);
    range = patch_get_range(patch);
    lower = patch_get_lower_note(patch);
    upper = patch_get_upper_note(patch);

    block(self);
    
    phat_slider_button_set_value(PHAT_SLIDER_BUTTON(self->chan_sb), chan+1);
    phat_slider_button_set_value(PHAT_SLIDER_BUTTON(self->note_sb), note);
    phat_slider_button_set_value(PHAT_SLIDER_BUTTON(self->lower_sb), lower);
    phat_slider_button_set_value(PHAT_SLIDER_BUTTON(self->upper_sb), upper);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(self->range_check), range);

    unblock(self);
}
    
    
