/*
 This file is part of slPIM.
 
 Copyright (c) 2001 John J. Ruthe <phaust@users.sourceforge.net>

 This program is free software; you can redistribute it and/or modify it
 under the terms of the GNU General Public License as published by the Free
 Software Foundation; either version 2 of the License, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 more details.

 You should have received a copy of the GNU General Public License along
 with this program; if not, write to the Free Software Foundation, Inc., 675
 Mass Ave, Cambridge, MA 02139, USA. 
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gdbm.h>
#include <pcre.h>
#include "defs.h"

GDBM_FILE dbf;
listitem *dirstack;
dyntab *dircont;
pcre *pattern;
pcre_extra *patt_hints;
char *ABASE_ERROR;
char *slpim_ABASE_filename;

void slpim_ABASE_push_item(listitem **stack, char *name) {
	listitem *t;

	t = malloc(sizeof(listitem));
	t->next = *stack;
	t->name = strdup(name);
	*stack = t;
}

char *slpim_ABASE_pop_item(listitem **stack) {
	listitem *t;
	char *s;

	t = (stack != NULL) ? *stack : NULL;
	if(t != NULL) {
		*stack = t->next;
		s = t->name;
		free(t);
	} else
		s = NULL;
	return(s);
}

dyntab *slpim_ABASE_list_to_array(listitem *list) {
	listitem *l;
	dyntab *t;
	int i;
	
	l = list;
	for(i = 0; l != NULL; i++)
		l = l->next;
	t = malloc(sizeof(dyntab));
	t->ent = malloc(i * sizeof(char*));
	t->i = i;
	l = list;
	for(i = 0; l != NULL; i++) {
		t->ent[i] = strdup(l->name);
		l = l->next;
	}
	
	return(t);
}

/* returns:                                 *
 *  -1 - entry/database doesn't exist       *
 *   0 - if entry is an ordinary entry, or  *
 *   1 - if entry is a directory, or        *
 *   2 - if entry is a special entry, or    *
 *   3 - if entry has non-valid ID.         */
int slpim_ABASE_get_record_status(char *name) {
	datum k;
	int i;

	k.dptr = name;
	k.dsize = strlen(name) + 1;
	if(gdbm_exists(dbf,k)) {
		switch(*name) {
			case 'a':
				i = 0;
				break;
			case 'd':
				i = 1;
				break;
			case 'm':
				i = 2;
				break;
			default:
				i = 3;
		}
	} else
		i = -1;
	
	return(i);
}

void slpim_ABASE_dir_save(char *name, dyntab *cont) {
	datum k, c;
	int i, j;
	char *s;

	for(i = 0, j = 0; i < cont->i; i++)
		if(cont->ent[i] != NULL)
			j += strlen(cont->ent[i]) + 1;
	if(j == 0)
		j = 1;
	s = malloc(j);
	*s = 0;
	for(i = 0; i < cont->i; i++)
		if(cont->ent[i] != NULL) {
			strcat(s, cont->ent[i]);
			if(i < cont->i - 1)
				strcat(s, "&");
		}
	k.dptr = name;
	k.dsize = strlen(name) + 1;
	c.dptr = s;
	c.dsize = j;
	gdbm_store(dbf, k, c, GDBM_REPLACE);
	free(s);
}

/* if name == NULL just free the dynamic table */
int slpim_ABASE_dir_load(char *name, dyntab **cont) {
	char *b, *s, *t;
	datum k, c;
	int i, j;

	if(name != NULL) {
		k.dptr = name;
		k.dsize = strlen(name) + 1;
		c = gdbm_fetch(dbf, k);
		if(c.dptr == NULL)
			return(-1);
	}
		
	if(cont != NULL)
		if(*cont != NULL) {
			for(i = 0; i < (*cont)->i; i++)
				free((*cont)->ent[i]);
			if((*cont)->ent != NULL)
				free((*cont)->ent);
			free(*cont);
		}
	
	if(name == NULL)
		return(0);
	
	*cont = malloc(sizeof(dyntab));
	(*cont)->ent = NULL;

	/* parser */
	s = c.dptr;
	j = i = 0;
	t = strtok_r(s, "&", &b);
	while(t != NULL) {
		if(i == j) {
			j += 256;
			(*cont)->ent = realloc((*cont)->ent, j * sizeof(char*));
		}
		(*cont)->ent[i] = strdup(t);
		i++;
		t = strtok_r(NULL, "&", &b);
	}
	if(i > 0)
		(*cont)->ent = realloc((*cont)->ent, i * sizeof(char*));
	(*cont)->i = i;
			
	free(s);
	return(0);
}

char *slpim_ABASE_dir_name(char *dir) {
	char *s, *n;
	datum k, c;
	
	n = malloc(strlen(dir) + 4);
	strcpy(n, dir);
	strcat(n, " na");
	k.dptr = n;
	k.dsize = strlen(n) + 1;
	
	c = gdbm_fetch(dbf, k);
	s = c.dptr;
	free(n);
	return(s);
}

int slpim_ABASE_dir_rename(char *key, char *name) {
	datum k, c;
	char *n;
	int i;
	
	n = malloc(strlen(key) + 4);
	strcpy(n, key);
	strcat(n, " na");
	k.dptr = n;
	k.dsize = strlen(n) + 1;
	c.dptr = name;
	c.dsize = strlen(name) + 1;
	i = gdbm_store(dbf, k, c, GDBM_REPLACE);
	free(n);
	return(i);
}

int slpim_ABASE_dir_enter(char *dir) {
	datum k;
	int i;
	
	k.dptr = dir;
	k.dsize = strlen(dir) + 1;
	i = gdbm_exists(dbf, k);
	if(!i)
		return(-1);

	slpim_ABASE_push_item(&dirstack, dir);
	slpim_ABASE_dir_load(dir, &dircont);
	return(0);
}

void slpim_ABASE_dir_up(void) {
	char *n;

	if(dirstack->next != NULL) {
		n = slpim_ABASE_pop_item(&dirstack);
	free(n);
	}
	slpim_ABASE_dir_load(dirstack->name, &dircont);
}

char *slpim_ABASE_fetch_field(char *key, int field) {
	char *s, *r, *n, *t;
	datum k, c;
	int i;

	if(field < 256) {
		/* fetch field */
		i = strlen(key) + 4;
		s = malloc(i);
		snprintf(s, i, "%s %02x", key, field);
		k.dptr = s;
		k.dsize = i;
		c = gdbm_fetch(dbf, k);
		free(s);
		r = c.dptr;
		/* return empty string instead of NULL */
		if(r == NULL) {
			r = malloc(1);
			*r = 0;
		}
	} else
		/* virtual field */
		switch(field) {
			case FIELD_NAME:
				s = slpim_ABASE_fetch_field(key, FIELD_FNAME);
				n = slpim_ABASE_fetch_field(key, FIELD_MNAME);
				t = slpim_ABASE_fetch_field(key, FIELD_LNAME);
				i = strlen(s) + strlen(t) + 5;
				r = malloc(i);
				snprintf(r, i, "%s %c. %s", s, *n, t);
				if(s != NULL)
					free(s);
				if(n != NULL)
					free(n);
				if(t != NULL)
					free(t);
				break;
			default:
				r = NULL;
		}

	return(r);
}

int slpim_ABASE_store_field(char *key, int field, char *val) {
	datum k, c;
	char *s;
	int i;

	if(field > 255)
		return(-1);
	
	i = strlen(key) + 4;
	s = malloc(i);
	snprintf(s, i, "%s %02x", key, field);
	k.dptr = s;
	k.dsize = i;

	if(val != NULL) {
		c.dptr = val;
		c.dsize = strlen(val) + 1;
		i = gdbm_store(dbf, k, c, GDBM_REPLACE);
	} else
		i = gdbm_delete(dbf, k);
	
	free(s);
	return(i);
}

char *slpim_ABASE_add_record(char *dir) {
	char *s, *d, *r;
	dyntab *tab;
	datum k, c;
	char z;
	int i;

	s = malloc(11);
	k.dptr = s;
	k.dsize = 11;
	c.dptr = &z;
	c.dsize = 1;
	z = 0;
	for(i = 1; i == 1; ) {
		i = random();
		snprintf(s, 11, "a %08x", i);
		i = gdbm_store(dbf, k, c, GDBM_INSERT);
		if(i == -1) {
			free(s);
			return(NULL);
		}
	}
	
	if(dir == NULL) {
		d = strdup(dirstack->name);
	} else
		d = strdup(dir);
	
	r = strdup(s);
	tab = NULL;
	slpim_ABASE_dir_load(d, &tab);
	i = tab->i;
	tab->ent = realloc(tab->ent, (i + 1) * sizeof(char*));
	tab->ent[i] = s;
	(tab->i)++;
	slpim_ABASE_dir_save(d, tab);
	slpim_ABASE_dir_load(NULL, &tab);
	
	if(dir == NULL)
		slpim_ABASE_dir_load(d, &dircont);

	free(d);
	return(r);
}

/* this is universal record/dir remove function */
int slpim_ABASE_rm_record(char *key, char *dir) {
	dyntab *tab;
	char *s, *d;
	int i, j;
	datum k;

	k.dptr = key;
	k.dsize = strlen(key) + 1;
	i = gdbm_delete(dbf, k);
	if(i != 0)
		return(i);
	
	j = strlen(key) + 4;
	s = malloc(j);
	k.dptr = s;
	k.dsize = j;
	switch(*key) {
		case 'a':
			for(i = 0; i < 256; i++) {
				snprintf(s, j, "%s %02x", key, i);
				gdbm_delete(dbf, k);
			}
			break;
		case 'd':
			snprintf(s, j, "%s na", key);
			gdbm_delete(dbf, k);
			break;
		default:
	} 
	free(s);

	if(dir == NULL) {
		d = strdup(dirstack->name);
	} else
		d = strdup(dir);
	
	tab = NULL;
	slpim_ABASE_dir_load(d, &tab);
	for(i = 0, j = 1; (j != 0) && (i < tab->i); i++)
		j = strcmp(key, tab->ent[i]);
	if(j == 0) {
		i--;
		free(tab->ent[i]);
		for(i++; i < tab->i; i++)
			tab->ent[i - 1] = tab->ent[i];
		(tab->i)--;
		tab->ent = realloc(tab->ent, tab->i * sizeof(char*));
		slpim_ABASE_dir_save(d, tab);
	}
	slpim_ABASE_dir_load(NULL, &tab);
	if(dir == NULL)
		slpim_ABASE_dir_load(d, &dircont);

	free(d);
	return(0);
}

char *slpim_ABASE_mkdir(char *dir, char *name) {
	char *s, *d, *r;
	dyntab *tab;
	datum k, c;
	char z;
	int i;

	s = malloc(11);
	k.dptr = s;
	k.dsize = 11;
	c.dptr = &z;
	c.dsize = 1;
	z = 0;
	for(i = 1; i == 1; ) {
		i = random();
		snprintf(s, 11, "d %08x", i);
		i = gdbm_store(dbf, k, c, GDBM_INSERT);
		if(i == -1) {
			free(s);
			return(NULL);
		}
	}
	
	d = malloc(14);
	snprintf(d, 14, "%s na", s);
	k.dptr = d;
	k.dsize = 14;
	c.dptr = name;
	c.dsize = strlen(name) + 1;
	gdbm_store(dbf, k, c, GDBM_REPLACE);
	free(d);
	
	if(dir == NULL) {
		d = strdup(dirstack->name);
	} else
		d = strdup(dir);
	
	r = strdup(s);
	tab = NULL;
	slpim_ABASE_dir_load(d, &tab);
	i = tab->i;
	tab->ent = realloc(tab->ent, (i + 1) * sizeof(char*));
	tab->ent[i] = s;
	(tab->i)++;
	slpim_ABASE_dir_save(d, tab);
	slpim_ABASE_dir_load(NULL, &tab);
	if(dir == NULL)
		slpim_ABASE_dir_load(d, &dircont);

	free(d);
	return(r);
}

int slpim_ABASE_qsort_compare(const doubletab *a, const doubletab *b) {
	if(*(a->key) == 'd')
		if(*(b->key) != 'd')
			return(-1);
	if(*(b->key) == 'd')
		if(*(b->key) != 'd')
			return(1);
	return(strcmp(a->cont, b->cont));
}

int slpim_ABASE_sort_tab(dyntab **t, int field) {
	doubletab *sorttab;
	int i;

	sorttab = malloc((*t)->i * sizeof(*sorttab));
	for(i = 0; i < (*t)->i; i++) {
		sorttab[i].key = (*t)->ent[i];
		switch(*((*t)->ent[i])) {
			case 'a':
				sorttab[i].cont = slpim_ABASE_fetch_field((*t)->ent[i], field);
				break;
			case 'd':
				sorttab[i].cont = slpim_ABASE_dir_name((*t)->ent[i]);
				break;
			default:
				sorttab[i].cont = strdup("zzzz");
		}
	}

	qsort(sorttab, (*t)->i, sizeof(*sorttab), &slpim_ABASE_qsort_compare);
	
	for(i = 0; i < (*t)->i; i++) {
		free(sorttab[i].cont);
		(*t)->ent[i] = sorttab[i].key;
	}
	free(sorttab);
	
	return(0);
}

int slpim_ABASE_sort(char *dir, int field) {
	dyntab *dtab;
	char *d;
	
	if(dir == NULL) {
		d = strdup(dirstack->name);
	} else
		d = strdup(dir);

	dtab = NULL;
	slpim_ABASE_dir_load(d, &dtab);

	slpim_ABASE_sort_tab(&dtab, field);

	slpim_ABASE_dir_save(d, dtab);
	slpim_ABASE_dir_load(NULL, &dtab);
	return(0);
}

/* if dir == NULL - search in current dir */
int slpim_ABASE_find(char *dir, char *patt, int field, int start, int isnext) {
	dyntab *t;
	int i, j, k;
	char *s;

	if(!isnext) {
		if(pattern)
			free(pattern);
		if(patt_hints)
			free(patt_hints);
		pattern = pcre_compile(patt, 0, &s, &i, NULL);
		if(pattern == NULL) {
			j = strlen(s);
			ABASE_ERROR = malloc(j + 31);
			snprintf(ABASE_ERROR, j + 31, "Error while compiling regexp: %s", s);
			return(-2);
		}
		patt_hints = pcre_study(pattern, 0, &s);
		if(s != NULL) {
			free(pattern);
			j = strlen(s);
			ABASE_ERROR = malloc(j + 31);
			snprintf(ABASE_ERROR, j + 31, "Error while analyzing regexp: %s", s);
			return(-2);
		}
	} else
		if(pattern == NULL) {
			ABASE_ERROR = strdup("No previous search defined.");
			return(-2);
		}

	if(dir != NULL) {
		t = NULL;
		slpim_ABASE_dir_load(dir, &t);
	} else
		t = dircont;
	
	k = 0;
	j = -1;
	i = (start + 1 < t->i) ? start + 1 : 0;
	while(k == 0) {
		if(*(t->ent[i]) == 'a') {
			s = slpim_ABASE_fetch_field(t->ent[i], field);
			j = pcre_exec(pattern, patt_hints, s, strlen(s), 0, 0, NULL, 0);
			if(j < -1) {
				free(s);
				ABASE_ERROR = strdup("Pattern matching failed!");
				return(-2);
			}
			free(s);
		}
		k = (j > -1) || (i == start);
		if(k == 0)
			if(++i == t->i)
				i = 0;
	}

	if(dir != NULL)
		slpim_ABASE_dir_load(NULL, &t);

	if(j > -1)
		return(i);

	return(-1);
}

listitem *slpim_ABASE_query_rec(char *dir, listitem *last, pcre *qpat, pcre *qpat_h, int field) {
	listitem *l, *tl;
	dyntab *t;
	int i, j;
	char *s;

	l = last;
	t = NULL;
	slpim_ABASE_dir_load(dir, &t);

	for(i = 0; i < t->i; i++) {
		if(*(t->ent[i]) == 'a') {
			s = slpim_ABASE_fetch_field(t->ent[i], field);
			j = pcre_exec(qpat, qpat_h, s, strlen(s), 0, 0, NULL, 0);
			free(s);
			if(j < -1) {
				ABASE_ERROR = strdup("Error while matching pattern");
				slpim_ABASE_dir_load(NULL, &t);
				return(l);
			}
			if(j > -1) {
				tl = malloc(sizeof(listitem));
				tl->name = strdup(t->ent[i]);
				tl->next = l;
				l = tl;
			}
		}
	}

	for(i = 0; i < t->i; i++) {
		if(*(t->ent[i]) == 'd') {
			tl = slpim_ABASE_query_rec(t->ent[i], l, qpat, qpat_h, field);
			if(ABASE_ERROR) {
				slpim_ABASE_dir_load(NULL, &t);
				return(tl);
			}
			l = tl;
		}
	}
	
	slpim_ABASE_dir_load(NULL, &t);

	return(l);
}
	
/* recursive query function */
dyntab *slpim_ABASE_query(char *patt, int field) {
	pcre_extra *qpatt_hints;
	pcre *qpattern;
	listitem *l, *tl;
	dyntab *t;
	int i, j;
	char *s;

	qpattern = pcre_compile(patt, 0, &s, &i, NULL);
	if(qpattern == NULL) {
		j = strlen(s);
		ABASE_ERROR = malloc(j + 31);
		snprintf(ABASE_ERROR, j + 31, "Error while compiling regexp: %s", s);
		return(NULL);
	}
	qpatt_hints = pcre_study(qpattern, 0, &s);
	if(s != NULL) {
		free(qpattern);
		j = strlen(s);
		ABASE_ERROR = malloc(j + 31);
		snprintf(ABASE_ERROR, j + 31, "Error while analyzing regexp: %s", s);
		return(NULL);
	}

	l = slpim_ABASE_query_rec("d rootdir0", NULL, qpattern, qpatt_hints, field);

	free(qpattern);
	if(qpatt_hints)
		free(qpatt_hints);

	if(ABASE_ERROR) {
		while(l != NULL) {
			tl = l->next;
			free(l->name);
			free(l);
			l = tl;
		}
		return(NULL);
	}

	t = slpim_ABASE_list_to_array(l);
	while(l != NULL) {
		tl = l->next;
		free(l->name);
		free(l);
		l = tl;
	}

	slpim_ABASE_sort_tab(&t, FIELD_NAME);
	
	return(t);
}
	
int slpim_ABASE_open(void) {
	datum key, cont;
	char c;

	ABASE_ERROR = NULL;
	key.dptr = "d rootdir0";
	key.dsize = 11;
	pattern = NULL;
	patt_hints = NULL;
	
	dbf = gdbm_open(slpim_ABASE_filename, 512, GDBM_WRCREAT, 0600, NULL);
	if(dbf == NULL)
		return(-1);

	if(!gdbm_exists(dbf, key)) {
		c = 0;
		cont.dsize = 1;
		cont.dptr = &c;
		gdbm_store(dbf, key, cont, GDBM_INSERT);
	}

	
	dirstack = NULL;
	dircont = NULL;
	slpim_ABASE_push_item(&dirstack, "d rootdir0");
	slpim_ABASE_dir_load("d rootdir0", &dircont);
	
	return(0);
}

void slpim_ABASE_close(void) {
	char *s;

	/* keep database optimized */
	gdbm_reorganize(dbf);
	
	gdbm_close(dbf);

	slpim_ABASE_dir_load(NULL, &dircont);

	while(dirstack != NULL) {
		s = slpim_ABASE_pop_item(&dirstack);
		free(s);
	}
}

