

/*
#    Sfront, a SAOL to C translator    
#    This file: Code generation: misc
#    Copyright (C) 1999  Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/

#include "tree.h"
#include "parser.tab.h"


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*         void toptree(int) andnd void postscript(void)        */
/*                                                              */
/* These two top-level functions are called by the main() of    */
/* sfront (located in sfmain.c). Toptree() prints out the       */
/* instr functions for a rate (ipass, kpass, or apass), and     */
/* postscript() prints out control functions and main().        */
/*                                                              */
/*______________________________________________________________*/


extern void ipassmake(tnode *);
extern void kpassmake(tnode *);
extern void apassmake(tnode *);

/****************************************************************/
/*   create _ipass, _kpass, and _apass high-level functions     */
/****************************************************************/

void toptree(int mode)

{
  sigsym * sptr;
  tnode * tptr;
  int i;

  /* print functions for non-effects instruments*/

  sptr = instrnametable;
  while (sptr != NULL)
    {
      if (reachableinstrexeff(sptr))
	{
	  currinputwidth = 1;
	  currinstrwidth = sptr->width;
	  curropcodeprefix = currinstancename = sptr->val;
	  currinstrument = sptr;
	  currinstance = NULL;

	  switch(mode) {
	  case PRINTIPASS:
	    ipassmake(sptr->defnode);
	    break;
	  case PRINTKPASS:
	    kpassmake(sptr->defnode);
	    break;
	  case PRINTAPASS:
	    if (sptr->cref->alines)
	      apassmake(sptr->defnode);
	    break;
	  }
	}
      sptr = sptr->next;
    }

  /* set starting point for ninstr[] effects positions  */
  /* this must change if startup implementation changes */
  
  i = (startupinstr != NULL);

  /* print functions for effects instruments*/

  tptr = instances;
  while (tptr != NULL)
    {
      redefstatic(i++);
      currinputwidth = tptr->inwidth;
      currinstrwidth = tptr->sptr->width;
      curropcodeprefix = currinstancename = tptr->val;
      currinstrument = tptr->sptr;
      currinstance = tptr;

      switch(mode) {
      case PRINTIPASS:
	ipassmake(tptr->sptr->defnode);
	break;
      case PRINTKPASS:
	kpassmake(tptr->sptr->defnode);
	break;
      case PRINTAPASS:
	if (tptr->sptr->cref->alines)
	  apassmake(tptr->sptr->defnode);
	break;
      }
      tptr = tptr->next;
    }

  /* restore #defines for control and global code to follow */

  redefnormal();                         
}


extern void printinputgroups(void);
extern void printsystemsinit(void);
extern void printeffectsinit(void);
extern void printshutdown(void);
extern void printmaincontrol(void);

/****************************************************************/
/*    control and global functions that complete the sa.c file  */
/****************************************************************/

void postscript(void)

{
  int active;

  printsaoltables(S_DATA); /* print saol wavetable data                */
  postcorefunctions();     /* global data for core opcodes: writepre.c */
  printinputgroups();      /* functions for complex input[] access     */
  printsystemsinit();      /* system_init(), global inits for sa.c     */
  printeffectsinit();      /* effects_init(), effects chain inits      */
  printshutdown();         /* function for clean exit of sa.c program  */
  printmainloops();        /* loops for instr execution: writemain.c   */
  printmaincontrol();      /* loops for table, tempo and control cmds  */
  makerobust();            /* signal-robust file primitives            */

  /* print the actual main() for sa.c, from lib/csrc library */

  active = (ainflow == ACTIVE_FLOW) || (aoutflow == ACTIVE_FLOW);

  if (active)
    makecallback();
  else
    {
      switch (makeaudiotypeout(aout)) {
      case SAMPLE_SHORT:
	makeruntime();
	break;
      case SAMPLE_FLOAT:
	makeruntimef();
	break;
      }
    }
}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                                                              */
/*        Second level functions called by toptree()            */
/*                                                              */
/*______________________________________________________________*/


/****************************************************************/
/*                  generates ipass functions                   */
/****************************************************************/

void ipassmake(tnode * tbranch)


{
  tnode * tptr = tbranch;
  sigsym * ptr1;
  sigsym * gptr;
  int first = 0;
  char name[STRSIZE];

  currblockrate = IRATETYPE;

  if (currinstance == NULL)
    fprintf(outfile,"void %s_ipass(struct ninstr_types * nstate)\n{\n",
	    currinstancename);
  else
    fprintf(outfile,"void %s_ipass(void)\n{\n", currinstancename);


  fprintf(outfile,"   int i;\n");
  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if ((ptr1->vartype == TABLETYPE) && 
	  ((ptr1->kind == K_NORMAL)|| (ptr1->kind == K_IMPORT)))
	{
	  if (!first)
	    {
	      fprintf(outfile,"   int j;\n");
	      first = 1;
	    }
	  if (ptr1->kind == K_NORMAL)
	    wavegeneratorvar(ptr1);
	}
      ptr1 = ptr1->next;
    }
  fprintf(outfile,"\n");
  fprintf(outfile,"memset(&(NV(0)), 0, %s_ENDVAR*sizeof(float));\n",
	  currinstancename);

  /* later reclaim pointer memory */

  fprintf(outfile,"memset(&(NT(0)), 0, %s_ENDTBL*sizeof(struct tableinfo));\n",
	  currinstancename);

  printdurassign();

  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if (ptr1->vartype != TABLETYPE)
	{
	  switch (ptr1->kind) {
	  case K_PFIELD:
	    fprintf(outfile,"   NV(%s_%s) = \n",
		    currinstancename,ptr1->val);
	    fprintf(outfile,"   NS(iline->p[%s_%s]);\n",
		    currinstancename,ptr1->val);
	    break;
	  case K_IMPORT: case K_IMPORTEXPORT:
	    gptr = getvsym(&globalsymtable,ptr1->val);
	    if (ptr1->rate == IRATETYPE)
	      {
		if (gptr == NULL)
		  {
		    printf("Error: Import has no matching ivar global.\n\n");
		    showerrorplace(ptr1->defnode->linenum, 
				   ptr1->defnode->filename);
		  }
		if ((gptr->vartype != ptr1->vartype)||
		    (gptr->width != truewidth(ptr1->width))||
		    (gptr->rate != ptr1->rate))
		  {
		    printf("Error: Import mismatch with global.\n\n");
		    showerrorplace(ptr1->defnode->linenum, 
				   ptr1->defnode->filename);
		  }
		if (ptr1->tref->mirror != GLOBALMIRROR)
		  fprintf(outfile,
		"   memcpy(&(NV(%s_%s)), &(NG(GBL_%s)), %i*sizeof(float));\n",
			currinstancename,ptr1->val,ptr1->val,
			truewidth(ptr1->width));
	      }
	    break;
	  case K_EXPORT:
	    if ((ptr1->rate == IRATETYPE) &&
		(ptr1->tref->mirror == GLOBALMIRROR) &&
		((ptr1->tref->assigntot == 0) || 
		 (ptr1->tref->varstate)))
	      fprintf(outfile,
		"   memset(&(NG(GBL_%s)), 0, %i*sizeof(float));\n",
			ptr1->val, truewidth(ptr1->width));
	  default:
	    break;
	  }
	}
      ptr1 = ptr1->next;
    }

  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if (ptr1->vartype == TABLETYPE)
	{
	  sprintf(name,"TBL_%s",currinstancename);
	  createtable(ptr1,name, S_INSTR);
	}
      ptr1 = ptr1->next;
    }
  blocktree(tptr->down,PRINTIPASS);
  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if ((ptr1->rate == IRATETYPE)&&
	  ((ptr1->kind==K_EXPORT)||(ptr1->kind==K_IMPORTEXPORT)))
	{
	  gptr = getvsym(&globalsymtable,ptr1->val);
	  if (gptr == NULL)
	    {
	      printf("Error: Import has no matching ivar global.\n\n");
	      showerrorplace(ptr1->defnode->linenum, 
			     ptr1->defnode->filename);
	    }
	  if ((gptr->vartype != ptr1->vartype)||
	      (gptr->width != truewidth(ptr1->width))||
	      (gptr->rate != ptr1->rate))
	    {
	      printf("Error: Import mismatch with global.\n\n");
	      showerrorplace(ptr1->defnode->linenum, 
			     ptr1->defnode->filename);
	    }
	  if ((ptr1->vartype != TABLETYPE) &&
	      (ptr1->tref->mirror != GLOBALMIRROR))
	    fprintf(outfile,
	     "   memcpy(&(NG(GBL_%s)), &(NV(%s_%s)), %i*sizeof(float));\n",
		    ptr1->val,currinstancename,ptr1->val,
		    truewidth(ptr1->width));
	}
      ptr1 = ptr1->next;
    }
  fprintf(outfile,"\n}\n\n");
}


extern void updatedynamictables(tnode *, char *);

/****************************************************************/
/*                  generates kpass functions                   */
/****************************************************************/

void kpassmake(tnode * tbranch)

{
  tnode * tptr = tbranch;
  sigsym * ptr1;
  sigsym * gptr;

  currblockrate = KRATETYPE;

  if (currinstance == NULL)
    fprintf(outfile,"void %s_kpass(struct ninstr_types * nstate)\n{\n\n",
	    currinstancename);
  else
    fprintf(outfile,"void %s_kpass(void)\n{\n\n", currinstancename);

  fprintf(outfile,"   int i;\n\n");

  if (currinstrument->cref->itime)
    fprintf(outfile,
	    "   NS(iline->itime) = ((float)(kcycleidx - NS(iline->kbirth)))*KTIME;\n\n");

  printdurassign();

  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if ((ptr1->rate == KRATETYPE)&&
	  ((ptr1->kind==K_IMPORT)||(ptr1->kind==K_IMPORTEXPORT)))
	{
	  gptr = getvsym(&globalsymtable,ptr1->val);
	  if (gptr != NULL)
	    {
	      if ((gptr->vartype != ptr1->vartype)||
		  (gptr->width != truewidth(ptr1->width))||
		  (gptr->rate != ptr1->rate))
		{
		  printf("Error: Import mismatch with global.\n\n");
		  showerrorplace(ptr1->defnode->linenum, 
				 ptr1->defnode->filename);
		}
	      if (ptr1->tref->mirror != GLOBALMIRROR)
		fprintf(outfile,
	      "   memcpy(&(NV(%s_%s)), &(NG(GBL_%s)), %i*sizeof(float));\n",
		      currinstancename,ptr1->val,ptr1->val,
		      truewidth(ptr1->width));
	    }
	}
      ptr1 = ptr1->next;
    }
  updatedynamictables(tptr, currinstancename);
  blocktree(tptr->down,PRINTKPASS);
  ptr1 = tptr->sptr;
  while (ptr1 != NULL)
    {
      if ((ptr1->rate == KRATETYPE)&&
	  ((ptr1->kind==K_EXPORT)||(ptr1->kind==K_IMPORTEXPORT)))
	{
	  gptr = getvsym(&globalsymtable,ptr1->val);
	  if (gptr == NULL)
	    {
	      printf("Error: Import has no matching ksig global.\n\n");
	      showerrorplace(ptr1->defnode->linenum, 
			     ptr1->defnode->filename);
	    }
	  if ((gptr->vartype != ptr1->vartype)||
	      (gptr->width != truewidth(ptr1->width))||
	      (gptr->rate != ptr1->rate))
	    {
	      printf("Error: Import width mismatch with global.\n\n");
	      showerrorplace(ptr1->defnode->linenum, 
			     ptr1->defnode->filename);
	    }
	  if (ptr1->tref->mirror != GLOBALMIRROR)
	    fprintf(outfile,
	     "   memcpy(&(NG(GBL_%s)), &(NV(%s_%s)), %i*sizeof(float));\n",
		    ptr1->val,currinstancename,ptr1->val,
		    truewidth(ptr1->width));
	}
      ptr1 = ptr1->next;
    }
  fprintf(outfile,"\n}\n\n");
}


/****************************************************************/
/*                  generates apass functions                   */
/****************************************************************/

void apassmake(tnode * tbranch)

{
  tnode * tptr = tbranch;
  tnode * bptr;
  int i;

  currblockrate = ARATETYPE;
  if (currinstance == NULL)
    fprintf(outfile,
	    "void %s_apass(struct ninstr_types * nstate)\n{\n\n",
	    currinstancename);
  else
    {
      fprintf(outfile, "void %s_apass(void)\n{\n\n", currinstancename);

      if (shadowcheck())
	{
	  bptr = currinstance->ibus;
	  while (bptr)
	    {
	      for (i = 0; i < bptr->width; i++)
		{
		  fprintf(outfile, "   STB(BUS_%s + %i) = TB(BUS_%s + %i);\n",
			  bptr->val, i, bptr->val, i);
		  if (bptr->sptr == outputbus)
		    fprintf(outfile, "   TB(BUS_%s + %i) = 0.0F;\n",
			    bptr->val, i);
		}
	      bptr = bptr->next;
	    }
	  fprintf(outfile, "\n");
	}
    }
  blocktree(tptr->down,PRINTAPASS); 
  fprintf(outfile,"}\n\n");

}

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                                                              */
/*        Second level functions called by postscript()         */
/*                                                              */
/*______________________________________________________________*/



/****************************************************************/
/*     prints input[] functions for complex send()/route()'s    */
/****************************************************************/

void printinputgroups(void)

{
  tnode * instptr;
  tnode * pptr;
  int i,old;

  /* create functions for instances */

  instptr = instances;
  while (instptr != NULL)
    {

      if (instptr->usesinput)
	{
	  currinstance = instptr;   /* set inputbusmacro() context */
   
	  fprintf(outfile,"\n\nfloat finput%i(float findex)\n\n{\n",
		 instptr->arrayidx);
	  fprintf(outfile,"   int index = ROUND(findex);\n\n");
	  fprintf(outfile,"   switch(index) { \n");
	  pptr = instptr->ibus;
	  i = 0;
	  while ((pptr != NULL) && (i < instptr->inwidth))
	    {
	      old = i;
	      while ((i<(pptr->width + old))&&(i<instptr->inwidth))
		{
		  fprintf(outfile,"   case %i:\n",i);
		  i++;
		}
	      fprintf(outfile,"   return %s(BUS_%s - %i + index);\n",
		      inputbusmacro(), pptr->val,old);
	      pptr = pptr->next;
	    }
	  fprintf(outfile,"   default:\n");
	  fprintf(outfile,"    return 0.0F;\n");
	  fprintf(outfile,"   }\n}\n\n");
	}

      if (instptr->usesingroup)
	{
	  fprintf(outfile,
		  "\n\nfloat finGroup%i(float findex)\n\n{\n",
		 instptr->arrayidx);
	  fprintf(outfile,"   int index = ROUND(findex);\n\n");
	  fprintf(outfile,"   switch(index) { \n");
	  pptr = instptr->ibus;
	  i = 0;
	  while ((pptr != NULL) && (i < instptr->inwidth))
	    {
	      old = i;
	      while ((i<(pptr->width + old))&&(i<instptr->inwidth))
		{
		  fprintf(outfile,"   case %i:\n",i);
		  i++;
		}
	      fprintf(outfile,"   return %i.0F;\n",
		      old+1);
	      pptr = pptr->next;
	    }
	  fprintf(outfile,"   default:\n");
	  fprintf(outfile,"    return 0.0F;\n");
	  fprintf(outfile,"   }\n}\n\n");
	}
      instptr = instptr->next;
    }

}


extern void printmidiglobals(void);

/****************************************************************/
/*            prints systems_init function                      */
/****************************************************************/

void printsystemsinit(void)

{
  sigsym * iptr;
  int first = 0;

  redefglobal();

  fprintf(outfile,"\n\nvoid system_init(int argc, char **argv)\n");
  fprintf(outfile,"{\n\n   int i;\n\n");

  iptr = globalsymtable;
  while (iptr != NULL)
    {
      if ((iptr->vartype == TABLETYPE) && (iptr->kind == K_NORMAL))
	{
	  if (!first)
	    {
	      fprintf(outfile,"   int j;\n");
	      first = 1;
	    }
	  wavegeneratorvar(iptr);
	}
      iptr = iptr->next;
    }

  if (fixedseed)
    fprintf(outfile,"\n   srand(9753193L);\n");
  else
    fprintf(outfile,"\n   srand(((unsigned int)time(0))|1);\n");
    
  fprintf(outfile,"   asys_argc = argc;\n");
  fprintf(outfile,"   asys_argv = argv;\n\n");

  if (cin)
    {
      fprintf(outfile,"   csys_argc = argc;\n");
      fprintf(outfile,"   csys_argv = argv;\n\n");     
    }

  fprintf(outfile,"\n");

  if (session)
    {
      fprintf(outfile,"   if (signal(SIGINT, SIG_IGN) != SIG_IGN)\n");
      fprintf(outfile,"     if (signal(SIGINT, sigint_handler) == SIG_ERR)\n");
      fprintf(outfile,"        epr(0,NULL,NULL,\"%s\");\n\n",
	      "Can't set up SIGINT signal handler.");
    }

  if (catchsignals)
    {
      if (!session)
	{
	  fprintf(outfile,
		  "\n\n   if (signal(SIGINT, signal_handler) == SIG_ERR)\n");
	  fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n",
		  "Can't set up SIGINT signal handler.");
	}
      fprintf(outfile,"   if (signal(SIGILL, signal_handler) == SIG_ERR)\n");
      fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n",
	      "Can't set up SIGILL signal handler.");
      fprintf(outfile,"   if (signal(SIGABRT, signal_handler) == SIG_ERR)\n");
      fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n",
	      "Can't set up SIGABRT signal handler.");
      fprintf(outfile,"   if (signal(SIGFPE, signal_handler) == SIG_ERR)\n");
      fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n",
	      "Can't set up SIGFPE signal handler.");
      fprintf(outfile,"   if (signal(SIGSEGV, signal_handler) == SIG_ERR)\n");
      fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n",
	      "Can't set up SIGSEGV signal handler.");
      fprintf(outfile,"   if (signal(SIGTERM, signal_handler) == SIG_ERR)\n");
      fprintf(outfile,"      epr(0,NULL,NULL,\"%s\");\n\n",
	      "Can't set up SIGTERM signal handler.");
    }

  printmidiglobals();

  fprintf(outfile,"   memset(&(NV(GBL_STARTVAR)), 0, \n");
  fprintf(outfile,"          (GBL_ENDVAR-GBL_STARTVAR)*sizeof(float));\n");

  /* later reclaim tables */

  fprintf(outfile,"   memset(&(NT(0)), 0, GBL_ENDTBL*sizeof(struct tableinfo));\n");

  if (startupinstr != NULL)
    {
      fprintf(outfile,"   u_startup[0].starttime = 0.0F;\n");
      fprintf(outfile,"   u_startup[0].endtime = MAXENDTIME;\n");
      fprintf(outfile,"   u_startup[0].abstime = 0.0F;\n");
      fprintf(outfile,"   u_startup[0].released = 0;\n");
      fprintf(outfile,"   u_startup[0].turnoff = 0;\n");
      fprintf(outfile,"   u_startup[0].time = 0.0F;\n");
      fprintf(outfile,"   u_startup[0].itime = 0.0F;\n");
      fprintf(outfile,"   u_startup[0].sdur = -1.0F;\n");
      fprintf(outfile,"   u_startup[0].kbirth = kbase;\n\n");

      if (totmidichan)
	fprintf(outfile,"   u_startup[0].numchan = midimasterchannel;\n\n");

      fprintf(outfile,"   u_startup[0].noteon = PLAYING;\n");
      fprintf(outfile,"   u_startup[0].notestate = nextstate;\n");
      fprintf(outfile,"   u_startup[0].nstate = &ninstr[nextstate];\n");
      fprintf(outfile,"   ninstr[nextstate].iline = &u_startup[0];\n");
      fprintf(outfile,"   nextstate++;\n");
      fprintf(outfile,"   startup_ipass(u_startup[0].nstate);\n");
    }
  iptr = globalsymtable;
  currinputwidth = inchannels;
  currinstrwidth = outchannels;
  curropcodeprefix = currinstancename = "GBL";
  currinstrument = NULL;  
  currinstance = NULL;

  while (iptr != NULL)
    {
      if (iptr->vartype == TABLETYPE)
	{
	  if (iptr->kind == K_NORMAL)
	    createtable(iptr,"TBL_GBL", S_GLOBAL);
	}
      iptr = iptr->next;
    }
  curropcodeprefix = currinstancename = NULL;

  redefnormal();

  fprintf(outfile,"   for (busidx=0; busidx<ENDBUS;busidx++)\n");
  fprintf(outfile,"      TB(busidx)=0.0F;\n\n");

  fprintf(outfile,"\n}\n\n");

}

/****************************************************************/
/*            print effects_init function                       */
/****************************************************************/

void printeffectsinit(void)

{
  sigsym * sptr;
  tnode * instptr;
  tnode * pptr;
  tnode * pvalptr;
  int i;

  /* print effects_init */

  redefglobal();
  fprintf(outfile,"\n\nvoid effects_init(void)\n{\n\n");

  sptr = instrnametable;
  while (sptr != NULL)
    {
      if (sptr->effects)
	{
	  i = -1;
	  instptr = instances;
	  while (instptr != NULL)
	    {
	      if (!strcmp(instptr->sptr->val, sptr->val))
		{

		  i++;
		  /* list of parameters in instr definition */
		  pptr = instptr->down->sptr->defnode->down->next->next->next->down;

		  /* list of values for these */
		  pvalptr = instptr->down->down->next->next->next->next->down;

		  while (pptr != NULL)
		    {
		      if (pvalptr == NULL)
			{
			  printf("Error: Send parameter list mismatches instr.\n\n");
			  showerrorplace(instptr->down->down->linenum, 
					 instptr->down->down->filename);
			}
		      fprintf(outfile,"e_%s[%i].p[%s_%s] = ",sptr->val,i,
			      instptr->val, pptr->val);
		      currinputwidth = inchannels;
		      currinstrwidth = outchannels;
		      widthupdate(pvalptr);
		      curropcoderate = ARATETYPE;
		      currtreerate = UNKNOWN;
		      rateupdate(pvalptr);
		      if (pvalptr->width != 1)
			{
			  printf("Error: Unindexed arrays in send parameter list.\n\n");
			  showerrorplace(instptr->down->down->linenum, 
					 instptr->down->down->filename);
			}
		      if (pvalptr->rate != IRATETYPE)
			{
			  printf("Error: Send parameters must be i-rate.\n\n");
			  showerrorplace(instptr->down->down->linenum, 
					 instptr->down->down->filename);
			}
		      if (pvalptr->vartype == TABLETYPE)
			{
			  printf("Error: Table used as send parameter.\n\n");
			  showerrorplace(instptr->down->down->linenum, 
					 instptr->down->down->filename);
			}
		      curropcodeprefix = currinstancename = "GBL";
		      currinstrument = NULL;  
		      currinstance = NULL;
		      currblockrate = IRATETYPE;
		      blocktree(pvalptr->down, PRINTTOKENS);
		      fprintf(outfile,";\n");
		      pptr = pptr->next;
		      pvalptr = pvalptr->next;
		      if (pptr != NULL)
			pptr = pptr->next;
		      if (pvalptr != NULL)
			pvalptr = pvalptr->next;
		    }

		  fprintf(outfile,"e_%s[%i].noteon = TOBEPLAYED;\n",sptr->val,i);

		  /* internal variables */

		  fprintf(outfile,"e_%s[%i].starttime = 0.0F;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].abstime = 0.0F;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].released = 0;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].turnoff = 0;\n",sptr->val,i);

		  /* standard names */

		  fprintf(outfile,"e_%s[%i].time = 0.0F;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].itime = 0.0F;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].sdur = -1.0F;\n",sptr->val,i);
		  fprintf(outfile,"e_%s[%i].kbirth = kbase;\n",sptr->val,i);

		  if (totmidichan)
		    fprintf(outfile,"e_%s[%i].numchan = midimasterchannel;\n",sptr->val,i);
		  
		}
	      instptr = instptr->next;
	    }
	}
      sptr = sptr->next;
    }
  
  fprintf(outfile,"\n\n}\n\n");
  redefnormal();

  
}

/****************************************************************/
/*               prints shut_down function                      */
/****************************************************************/

void printshutdown(void)

{
  sigsym * iptr;

  redefglobal();

  iptr = getvsym(&busnametable,"input_bus");

  fprintf(outfile,"\n\nvoid shut_down(void)\n   {\n\n");
  fprintf(outfile,"  if (graceful_exit)\n");
  fprintf(outfile,"    {\n");
  fprintf(outfile,"      fprintf(stderr, \"\\nShutting down system ... please wait.\\n\");\n");
  fprintf(outfile,"      fprintf(stderr, \"If no termination in 10 seconds, use Ctrl-C or Ctrl-\\\\ to force exit.\\n\");\n");
  fprintf(outfile,"      fflush(stderr);\n");
  fprintf(outfile,"    }\n");

  if ((ainflow == PASSIVE_FLOW) && (aoutflow == PASSIVE_FLOW))
    fprintf(outfile,"   asys_putbuf(&asys_obuf, &obusidx);\n");

  if ((ain == aout) && (iptr != NULL) && (inchannels > 0))
    fprintf(outfile,"   asys_ioshutdown();\n");
  else
    {
      if ((iptr != NULL) && (inchannels > 0))
	fprintf(outfile,"   asys_ishutdown();\n");
      fprintf(outfile,"   asys_oshutdown();\n");
    }

  if (cin)
    fprintf(outfile,"\n   csys_shutdown();\n");

  if (session)
    fprintf(outfile,"   nsys_shutdown();\n");

  fprintf(outfile,"\n   }\n\n");

  if (session)
    {
      fprintf(outfile,"\n");
      fprintf(outfile,"void sigint_handler(int signum)\n {\n");
      fprintf(outfile,"\n");
      fprintf(outfile,"  if (graceful_exit)\n");
      fprintf(outfile,"   exit(130);\n");
      fprintf(outfile,"  graceful_exit = 1;\n");
      fprintf(outfile," }\n\n");
    }

  if (catchsignals)
    {      
      fprintf(outfile,"\n\nvoid signal_handler(int signum)\n {\n\n");
      fprintf(outfile,"   fprintf(stderr, \"\\n\\nRuntime Error\\n\");\n");
      fprintf(outfile,"   fprintf(stderr, \"Abnormal Termination -- \");\n");

      fprintf(outfile,"   switch(signum) {\n");

      if (!session)
	{      
	  fprintf(outfile,"    case SIGINT:\n");
	  fprintf(outfile,"     fprintf(stderr, \"Interrupt\\n\");\n");
	  fprintf(outfile,"     break;\n");
	}

      fprintf(outfile,"    case SIGILL:\n");
      fprintf(outfile,"     fprintf(stderr, \"Illegal instruction\\n\");\n");
      fprintf(outfile,"     break;\n");

      fprintf(outfile,"    case SIGABRT:\n");
      fprintf(outfile,"     fprintf(stderr, \"Aborted\\n\");\n");
      fprintf(outfile,"     break;\n");

      fprintf(outfile,"    case SIGFPE:\n");
      fprintf(outfile,"     fprintf(stderr, \"Floating point exception\\n\");\n");
      fprintf(outfile,"     break;\n");

      fprintf(outfile,"    case SIGSEGV:\n");
      fprintf(outfile,"     fprintf(stderr, \"Segmentation fault\\n\");\n");
      fprintf(outfile,"     break;\n");

      fprintf(outfile,"    case SIGTERM:\n");
      fprintf(outfile,"     fprintf(stderr, \"Terminated\\n\");\n");
      fprintf(outfile,"     break;\n");

      fprintf(outfile,"   }\n");
      fprintf(outfile,"  fprintf(stderr, \"Closing Files ...\");\n");
      fprintf(outfile,"  shut_down();\n");
      fprintf(outfile,"  fprintf(stderr, \" Completed. Exiting.\\n\\n\");\n");
      fprintf(outfile,"  exit(0);\n");
      fprintf(outfile," }\n\n");
    }
}

extern void printtempoloop(char *);
extern void printtableloop(char *);
extern void printcontrolloop(char *);

/****************************************************************/
/*    printmaincontrol: prints control/tempo/table loops        */
/****************************************************************/

void printmaincontrol(void)

{

  fprintf(outfile,"void main_control(void)\n");
  fprintf(outfile,"\n");
  fprintf(outfile,"{\n");

  if (csasl|| allsasl->numtable || abssasl->numtable)
    fprintf(outfile,"   int i;\n");
  if (allsasl->numtable || abssasl->numtable)
    fprintf(outfile,"   int len;\n");

  fprintf(outfile,"\n");

  if (allsasl->numcontrol)
    printcontrolloop("s");
  if (abssasl->numcontrol)
    printcontrolloop("sa");

  if (allsasl->numtable)
    printtableloop("s");
  if (abssasl->numtable)
    printtableloop("sa");
  if (csasl)
    makesasltablesys();

  if (allsasl->numtempo)
    printtempoloop("s");
  if (abssasl->numtempo)
    printtempoloop("sa");


  fprintf(outfile,"}\n");

}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                                                              */
/*     Utility functions called by second-level functions       */
/*                                                              */
/*______________________________________________________________*/


/****************************************************************/
/* timestamp check for table sasl commands: used in kpassmake() */
/****************************************************************/

void updatedynamictables(tnode * iptr, char * prefix)

{

  sigsym * sptr;
  sigsym * gptr;
  tnode * optr;
  int opnum;
  char newprefix[STRSIZE];
  char name[STRSIZE];

  sptr = iptr->sptr;
  while (sptr != NULL)
    {
      if ((sptr->vartype == TABLETYPE)&&(sptr->kind==K_IMPORT))
        {    
	  if ((gptr = getvsym(&globalsymtable, sptr->val)) == NULL)
	    internalerror("writeorc","updatedynamictable");
	  if ((sptr->tref->assigntot || gptr->tref->assigntot) && 
	      ((strlen(sptr->val) <= 5) || strncmp(sptr->val, "_sym_", 5)))
	    {
	      fprintf(outfile, "   if (gtables[TBL_GBL_%s].stamp > ",
		      sptr->val);
	      fprintf(outfile, "NT(TBL_%s_%s).stamp)\n",
		      prefix, sptr->val);
	      fprintf(outfile,"   {\n");
	      sprintf(name,"TBL_%s",prefix);
	      createtable(sptr,name, S_FUTURE);
	      fprintf(outfile,"   }\n");
	    }
	}
      sptr = sptr->next;
    }  
  optr = iptr->optr;
  while (optr != NULL)
    {
      if (optr->ttype == S_OPCALL)
	{
	  sprintf(newprefix,"%s_%s%i",prefix,optr->val,optr->arrayidx);
	  updatedynamictables(optr->sptr->defnode,newprefix);
	}
      else
	{
	  for (opnum = 0; opnum < truewidth(optr->opwidth);opnum++)
	    {
	      sprintf(newprefix,"%s_%soparray%i",prefix,optr->val,(opnum+1));
	      updatedynamictables(optr->sptr->defnode,newprefix);
	    }
	}
      optr = optr->next;
    }
}

/****************************************************************/
/*    prints MIDI global constants: used in system_init()       */
/****************************************************************/

void printmidiglobals(void)

{
  int i, j;
  tnode * tptr;

  if (totmidichan)
    {

      /****************************/
      /* set default MIDI volumes */
      /****************************/

      fprintf(outfile, "    /* MIDI volume */\n\n");
      fprintf(outfile, "   ");
      i = 0;
      for (j = MIDIVOLUMEPOS; j < totmidichan*MIDIFRAMELEN; 
	   j = j + MIDIFRAMELEN)
	{
	  fprintf(outfile, "global[%i].f = ", j);
	  if ((i++) == 3)
	    {
	      fprintf(outfile, "\n   ");
	      i = 0;
	    }
	}
      fprintf(outfile, " 100.0F; \n\n");

      /************************/
      /* set default MIDI Pan */
      /************************/

      fprintf(outfile, "    /* MIDI Pan */\n\n");
      fprintf(outfile, "   ");
      i = 0;
      for (j = MIDIPANPOS; j < totmidichan*MIDIFRAMELEN; 
	   j = j + MIDIFRAMELEN)
	{
	  fprintf(outfile, "global[%i].f = ", j);
	  if ((i++) == 3)
	    {
	      fprintf(outfile, "\n   ");
	      i = 0;
	    }
	}
      fprintf(outfile, " 64.0F; \n\n");

      /*************************/
      /* set default MIDI Expr */
      /*************************/

      fprintf(outfile, "    /* MIDI Expression */\n\n");
      fprintf(outfile, "   ");
      i = 0;
      for (j = MIDIEXPRPOS; j < totmidichan*MIDIFRAMELEN; 
	   j = j + MIDIFRAMELEN)
	{
	  fprintf(outfile, "global[%i].f = ", j);
	  if ((i++) == 3)
	    {
	      fprintf(outfile, "\n   ");
	      i = 0;
	    }
	}
      fprintf(outfile, " 127.0F; \n\n");

      /*************************/
      /* set default MIDI Bend */
      /*************************/

      fprintf(outfile, "    /* MIDI Bend */\n\n");
      fprintf(outfile, "   ");
      i = 0;
      for (j = MIDIBENDPOS; j < totmidichan*MIDIFRAMELEN; 
	   j = j + MIDIFRAMELEN)
	{
	  fprintf(outfile, "global[%i].f = ", j);
	  if ((i++) == 3)
	    {
	      fprintf(outfile, "\n   ");
	      i = 0;
	    }
	}
      fprintf(outfile, " 8192.0F; \n\n");

      /********************************/
      /* set Extended Channel Numbers */
      /********************************/

      fprintf(outfile, "    /* MIDI Ext Channel Number */\n\n   ");
      i = 0;
      for (j = 0; j < totmidichan; j++)
	{
	  fprintf(outfile, "global[%i].f = ", 
		  MIDIEXTPOS + j*MIDIFRAMELEN);
	  if (j < confmidi->midinumchan)
	    {
	      tptr = confmidi->imidiroot;
	      while (tptr != NULL)
		{
		  if (tptr->width == j)
		    {
		      fprintf(outfile, "%i.0F; ", tptr->res);
		      tptr = NULL;
		    }
		  else
		    tptr = tptr->next;
		}
	    }
	  else
	    {
	      if (j < sstrmidi->midinumchan)
		{
		  tptr = sstrmidi->imidiroot;
		  while (tptr != NULL)
		    {
		      if (tptr->width == j)
			{
			  fprintf(outfile, "%i.0F; ", tptr->res);
			  tptr = NULL;
			}
		      else
			tptr = tptr->next;
		    }
		}
	      else
		{
		  /* wrong, fix later */
		  fprintf(outfile, "%i.0F; ",
			  (int)(confmidi->miditracks*MCHAN + j
				- confmidi->midinumchan));
		}
	    }
	  if (++i == 3)
	    {
	      i = 0;
	      fprintf(outfile, "\n   ");
	    }
	}
      fprintf(outfile, "\n\n");
    }

}

/****************************************************************/
/*    prints tempo-changing loop: used in printmaincontrol()    */
/****************************************************************/

void printtempoloop(char * prefix)

{
  fprintf(outfile,"   while ((%stempoidx <=end%stempo)&&\n",
	  prefix, prefix);

  if (strcmp("sa", prefix))
    fprintf(outfile,"        (stempo[stempoidx].t <= scorebeats))\n");
  else
    fprintf(outfile,"        (satempo[satempoidx].t <= absolutetime))\n");

  fprintf(outfile,"   {\n");  
  fprintf(outfile,"    kbase = kcycleidx;\n");
  fprintf(outfile,"    scorebase = scorebeats;\n");
  fprintf(outfile,"    tempo = %stempo[%stempoidx].newtempo;\n",
	  prefix, prefix);
  fprintf(outfile,"    scoremult = 1.666667e-02F*KTIME*tempo;\n");
  if (allsasl->endtimeval)
    {
      fprintf(outfile,"    endkcycle = kbase + (int)\n");
      fprintf(outfile,
	      "      (KRATE*(endtime - scorebase)*(60.0F/tempo));\n");
      if (abssasl->endtimeval)
	{
	  fprintf(outfile, 
		  "     endkcycle = (endkinit > endkcycle) ?\n");
	  fprintf(outfile, 
		  "                  endkinit : endkcycle;\n\n");
	}
    }
  fprintf(outfile,"    %stempoidx++;\n", prefix);
  fprintf(outfile,"    }\n\n");

}


/****************************************************************/
/*    prints table-trigger loop: used in printmaincontrol()     */
/****************************************************************/

void printtableloop(char * prefix)

{
  int lc = 0;

  /* add table free(), but be sure not to free() static memory.*/

  z[lc++]="while ((%stableidx <=end%stable)&&"; 

  if (strcmp("sa", prefix))
    z[lc++]="     (%stable[%stableidx].t <= scorebeats))";
  else
    z[lc++]="     (%stable[%stableidx].t <= absolutetime))";  

  z[lc++]="   {";  
  z[lc++]="     if (NT((i = %stable[%stableidx].gindex)).llmem)";
  z[lc++]="      free(NT(i).t);";
  z[lc++]="     memset(&(NT(i)), 0, sizeof(struct tableinfo));";
  z[lc++]="     if ((len = %stable[%stableidx].size) < 0)"; 
  z[lc++]="       (*%stable[%stableidx].tmake) ();";
  z[lc++]="     else"; 
  z[lc++]="      {"; 
  z[lc++]="        NT(i).stamp = scorebeats;"; 
  z[lc++]="        NT(i).lenf= (float)(NT(i).len = len);";
  z[lc++]="        NT(i).tend = len - 1;";
  z[lc++]="        NT(i).n = NT(i).diff = (len - 1.0F)/len;";
  z[lc++]="        NT(i).t = (float *)(%stable[%stableidx].data);";
  z[lc++]="      }"; 
  z[lc++]="     %stableidx++;";
  z[lc++]="   }";

  printcontrolblock(lc, prefix);
}


/****************************************************************/
/*   prints control-trigger loop: used in printmaincontrol()    */
/****************************************************************/

void printcontrolloop(char * prefix)

{
  fprintf(outfile,"  while (%scontrolidx && (%scontrolidx <=end%scontrol)&&\n",
	  prefix, prefix, prefix);

  if (strcmp("sa", prefix))
    fprintf(outfile,"	 (scontrolidx->t <= scorebeats))\n");
  else
    fprintf(outfile,"	 (sacontrolidx->t <= absolutetime))\n");

  fprintf(outfile,"  {\n");
  fprintf(outfile,"    if (%scontrolidx->siptr < 0)\n", prefix);
  fprintf(outfile,"	global[%scontrolidx->imptr].f = %scontrolidx->imval;\n"
	  ,prefix, prefix);
  fprintf(outfile,"    else\n");
  fprintf(outfile,"	{\n");
  fprintf(outfile,"	  if ( (%scontrolidx->iline != NULL) &&\n", prefix);
  fprintf(outfile,"	       (%scontrolidx->iline->noteon == PLAYING) )\n",
	  prefix);
  fprintf(outfile,"          %scontrolidx->iline->nstate->v[%scontrolidx->imptr].f\n", prefix, prefix);
  fprintf(outfile,"	      = %scontrolidx->imval;\n", prefix);

  if (csasl)
    makesaslcrosscontrol( prefix);

  fprintf(outfile,"	}\n");


  fprintf(outfile,"    %scontrolidx++;\n", prefix);
  fprintf(outfile,"  }\n\n");

}

/****************************************************************/
/*    returns 1 if current context requires bus shadowing       */
/****************************************************************/

int shadowcheck(void)

{
  return (useshadowbus && (currinstance != NULL) &&
	  currinstance->sptr->cref->inmirror);
}

/****************************************************************/
/*        returns TB or STB as needed for context               */
/****************************************************************/

char * inputbusmacro(void)

{
  return ((useshadowbus && (currinstance != NULL) && 
	   currinstance->sptr->cref->inmirror) ? "STB" : "TB");
}

/****************************************************************/
/*                  diagnostic function -- not used             */
/****************************************************************/

void ptokens(tnode * tbranch)

{
  tnode * tptr = tbranch;

  while (tptr != NULL)
    {
      if ((tptr->down == NULL))
	{
	  if (tptr->val[0]!='<')
	    fprintf(outfile," %s ",tptr->val);
	  if ((tptr->val[0]==';')||(tptr->val[0]=='{'))
	    fprintf(outfile,"\n");
	}
      else
	{
	  ptokens(tptr->down);
	}
      
      tptr = tptr->next;
    }
}

