// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "line.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Line::Line(const Line& a) : Marker(a)
{
  p1 = a.p1;
  p2 = a.p2;
  p1Arrow = a.p1Arrow;
  p2Arrow = a.p2Arrow;
}

Line::Line(FrameBase* p,
	   const Vector& ptr1, const Vector& ptr2, int a1, int a2,
	   const char* clr, int w, const char* f, 
	   const char* t, unsigned short prop, const char* c, 
	   const List<Tag>& tag) 
  : Marker(p, ptr1, clr, w, f, t, prop, c, tag)
{
  p1 = ptr1;
  p2 = ptr2;
  p1Arrow = a1;
  p2Arrow = a2;

  strcpy(type,"line");
  handle = new Vector[2];
  numHandle = 2;

  updateBBox();
}

void Line::updateBBox()
{
  center = (p2-p1)/2 + p1;

  // bound marker

  bbox = BBox(p1 * parent->refToCanvas, p2 * parent->refToCanvas);

  // generate handles in canvas coords

  handle[0] = p1 * parent->refToCanvas;
  handle[1] = p2 * parent->refToCanvas;

  // make room for handles/arrows

  bbox.expand(5);

  // calculate overall bbox

  calcAllBBox();
}

void Line::updateCoords(const Matrix& m)
{
  p1*=m;
  p2*=m;

  Marker::updateCoords(m);
}

void Line::move(const Vector& v)
{
  p1+=v;
  p2+=v;
  updateBBox();

  doCallBack(&moveCB);
}

void Line::moveTo(const Vector& v)
{
  // v is the new location of the center

  Vector diff = v - center;
  p1+=diff;
  p2+=diff;
  updateBBox();

  doCallBack(&moveCB);
}

void Line::edit(const Vector& v, int h)
{
  switch (h) {
  case 1:
    p1 = v;
    break;
  case 2:
    p2 = v;
    break;
  }
  updateBBox();
  doCallBack(&editCB);
}

#if __GNUC__ >= 3
void Line::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;

  ostringstream str;
  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  if (p1Arrow)
    psArrow(a, (b-a).normalize());

  if (p2Arrow)
    psArrow(b, (a-b).normalize());
}
#else
void Line::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;

  char buf[256];
  ostrstream str(buf,256);
  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);

  if (p1Arrow)
    psArrow(a, (b-a).normalize());

  if (p2Arrow)
    psArrow(b, (a-b).normalize());
}
#endif

int Line::isIn(const Vector& v)
{
  // do this in canvas coords, not ref coords

  Vector l1 = p1*parent->refToCanvas;
  Vector l2 = p2*parent->refToCanvas;
  double a = (l2-l1).angle();

  Matrix m = Translate(-l1) * Rotate(a);
  Vector end = l2*m;
  Vector vv = v*m;
  return (vv[0]>0 && vv[0]<end[0] && vv[1]>-3 && vv[1]<3);
}

void Line::setPoints(const Vector& v1, const Vector& v2)
{
  p1 = v1;
  p2 = v2;
  updateBBox();

  doCallBack(&editCB);
}

void Line::setArrows(int w1, int w2)
{
  p1Arrow = w1 ? 1 : 0;
  p2Arrow = w2 ? 1 : 0;
  updateBBox();
}

// Private

void Line::render(Drawable drawable, const Matrix& matrix, double zoom,
		  RenderMode mode)
{
  setGC(mode);

  Vector a = (p1 * matrix).round();
  Vector b = (p2 * matrix).round();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)b[0], (int)b[1]);

  if (p1Arrow)
    renderArrow(drawable, a, (b-a).normalize());

  if (p2Arrow)
    renderArrow(drawable, b, (a-b).normalize());
}

// list

void Line::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v1 = ptr->mapFromRef(p1,sys);
      Vector v2 = ptr->mapFromRef(p2,sys);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v1 = ptr->mapFromRef(p1,sys,sky);
	    Vector v2 = ptr->mapFromRef(p2,sys,sky);
	    str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		<< ',' << v2[0] << ',' << v2[1] << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra1[16], ra2[16];
	    char dec1[16], dec2[16];
	    {
	      ptr->mapFromRef(p1,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra1 >> dec1;
	    }
	    {
	      ptr->mapFromRef(p2,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra2 >> dec2;
	    }
	    str << type << '(' << ra1 << ',' << dec1 << ',' 
		<< ra2 << ',' << dec2 << ')';
	  }
	  break;
	}
      }
      else {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	    << ',' << v2[0] << ',' << v2[1] << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Line::listProperties(ostream& str, char delim)
{
  if (delim != ';') {
    str << " # line=" << p1Arrow << ' ' << p2Arrow;
    listProps(str);
  }
}

void Line::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listSAOtngPre(str,delim);

      Vector v1 = ptr->mapFromRef(p1,IMAGE);
      Vector v2 = ptr->mapFromRef(p2,IMAGE);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ')';

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listSAOtngPre(str,delim);

      switch (format) {
      case DEGREES:
	{
	  Vector v1 = ptr->mapFromRef(p1,sys,sky);
	  Vector v2 = ptr->mapFromRef(p2,sys,sky);
	  str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	      << v2[0] << ',' << v2[1] << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  char ra1[16], ra2[16];
	  char dec1[16], dec2[16];
	  {
	    ptr->mapFromRef(p1,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra1 >> dec1;
	  }
	  {
	    ptr->mapFromRef(p2,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra2 >> dec2;
	  }
	  str << type << '(' << ra1 << ',' << dec1 << ',' 
	      << ra2 << ',' << dec2 << ')';
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
    break;
  }
}
