/*
*  Rubrica
*  file: search.c
*  
*  Copyright (C) 2000-2003 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <gnome.h>

#include "rubrica.h"
#include "search.h"
#include "prefer.h"
#include "card.h"

static gint items = 0;

void rubrica_search_perform_search(GtkWidget *search_dialog)
{
  Rubrica *rubrica;
  GtkWidget *entry;
  GtkWidget *search_view;
  GtkWidget *data_view;
  GtkWidget *label;
  RubricaSearchData *data;

  g_return_if_fail(search_dialog != NULL);
  
  rubrica = rubrica_get_current_rubrica();
  entry = g_object_get_data(G_OBJECT(search_dialog), "search_text");
  label = g_object_get_data(G_OBJECT(search_dialog), "found");
  search_view = g_object_get_data(G_OBJECT(search_dialog), "view");
  data_view = rubrica_get_view(rubrica);
  
  g_return_if_fail(entry != NULL);
  g_return_if_fail(data_view  != NULL);

  items = 0;
  data = g_malloc (sizeof(RubricaSearchData));

  data->str = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
  data->model = gtk_tree_view_get_model(GTK_TREE_VIEW(search_view));
  gtk_list_store_clear(GTK_LIST_STORE(data->model));

  g_print("\nRubrica: search %s", data->str);
  rubrica_search_find_data(search_dialog, data_view, search_view, data);
  
  gtk_label_set_text(GTK_LABEL(label), g_strdup_printf("%d", items));  
  g_print("\nRubrica: %d item(s) found", items);

  g_free(data->str);
  g_free(data);
}


void rubrica_search_find_data(GtkWidget *dialog, GtkWidget *data_view, 
			      GtkWidget* search_view, RubricaSearchData *data)
{
  GtkTreeModel *view_model;

  view_model = gtk_tree_view_get_model(GTK_TREE_VIEW(data_view));

  switch (rubrica_search_get_search_type())
    {
    case SEARCH_CARD:
      data->column = CARD_COLUMN;
      break;
      
    case SEARCH_FIRST_NAME:
      data->column = FIRST_NAME_COLUMN;
      break;
      
    case SEARCH_LAST_NAME:
      data->column = LAST_NAME_COLUMN;
      break;
      
    case SEARCH_PROFESSION:
      data->column = PROFESSION_COLUMN;
      break;
      
    case SEARCH_CITY:
      data->column = CITY_COLUMN;
      break;
      
    case SEARCH_COUNTRY:
      data->column = COUNTRY_COLUMN;
      break;
      
    case SEARCH_WEB_ADDRESS:
      data->column = WEB_CARD_COLUMN;
      break;

    case SEARCH_EMAIL_ADDRESS:
      data->column = EMAIL_CARD_COLUMN;
      break;
      
    case SEARCH_TELEPHONE:
      data->column = TELEPHONE_COLUMN;
      break;
      
    case SEARCH_COMPANY:
      data->column = COMPANY_COLUMN;
      break;
      
    case SEARCH_CATEGORIES:
      data->column = CATEGORIES_COLUMN;
      break;

    default:
      data->column = CARD_COLUMN;
      break;            
    }

  if (rubrica_search_get_search_case_sensitive(dialog))
    {
      if (rubrica_search_get_search_mode_incomplete(dialog))
	gtk_tree_model_foreach(GTK_TREE_MODEL(view_model), 
			       rubrica_search_case_sensitive_complete, data);
      else
	gtk_tree_model_foreach(GTK_TREE_MODEL(view_model), 
			       rubrica_search_case_sensitive_incomplete, data);
    }
  else
    {
      if (rubrica_search_get_search_mode_incomplete(dialog))
	gtk_tree_model_foreach(GTK_TREE_MODEL(view_model), 
			       rubrica_search_case_insensitive_incomplete, 
			       data);
      else
	gtk_tree_model_foreach(GTK_TREE_MODEL(view_model), 
			       rubrica_search_case_insensitive_complete,
			       data);
    }
}

gboolean
rubrica_search_case_sensitive_complete(GtkTreeModel *view_model,
				       GtkTreePath *path,
				       GtkTreeIter *iter, gpointer data)
{
  RubricaSearchData *search_data = data;
  GtkTreeIter search_iter;
  gchar *buffer;
  gchar *card;
  
  gtk_tree_model_get(view_model, iter, 
		     CARD_COLUMN, &card,
		     search_data->column, &buffer, 
		     -1);
  
  if (strcmp(buffer, search_data->str) == 0)
    {
      gchar *path_str;

      path_str = gtk_tree_path_to_string(path);

      gtk_list_store_append(GTK_LIST_STORE(search_data->model), &search_iter);
      gtk_list_store_set(GTK_LIST_STORE(search_data->model), &search_iter, 
			 SEARCH_RESULT_CARD, card,
			 SEARCH_RESULT_FOUND, buffer, 
			 SEARCH_RESULT_PATH, path_str, 
			 -1);
      items++;
    }
    
  return FALSE;  
}

gboolean 
rubrica_search_case_insensitive_complete(GtkTreeModel *view_model, 
					 GtkTreePath *path,
					 GtkTreeIter *iter, gpointer data)
{
  RubricaSearchData *search_data = data;
  GtkTreeIter search_iter;
  gchar *buffer;
  gchar *card;
  
  gtk_tree_model_get(view_model, iter, 
		     CARD_COLUMN, &card,
		     search_data->column, &buffer, 
		     -1);

  if (strcasecmp(buffer, search_data->str) == 0)
    {
      gchar *path_str;

      path_str = gtk_tree_path_to_string(path);
      
      gtk_list_store_append(GTK_LIST_STORE(search_data->model), &search_iter);
      gtk_list_store_set(GTK_LIST_STORE(search_data->model), &search_iter, 
			 SEARCH_RESULT_CARD, card,
			 SEARCH_RESULT_FOUND, buffer, 
			 SEARCH_RESULT_PATH, path_str, 
			 -1);
      items++;
    }

  return FALSE;  
}


gboolean 
rubrica_search_case_sensitive_incomplete(GtkTreeModel *view_model,
					 GtkTreePath *path,
					 GtkTreeIter *iter, gpointer data)
{
  RubricaSearchData *search_data = data;
  GtkTreeIter search_iter;
  gchar *buffer;
  gchar *tmp;
  gchar *card;
  gint len;
  
  gtk_tree_model_get(view_model, iter, 
		     CARD_COLUMN, &card,
		     search_data->column, &buffer, 
		     -1);

  len = strlen(search_data->str);
  tmp = g_strdup(buffer);

  while (*tmp)
    {
      if (strncmp(tmp, search_data->str, len) == 0)
	{
	  gchar *path_str;
	  
	  path_str = gtk_tree_path_to_string(path);
	  
	  gtk_list_store_append(GTK_LIST_STORE(search_data->model),
				&search_iter);
	  gtk_list_store_set(GTK_LIST_STORE(search_data->model), &search_iter, 
			     SEARCH_RESULT_CARD, card,
			     SEARCH_RESULT_FOUND, buffer, 
			     SEARCH_RESULT_PATH, path_str, 
			     -1);
	  items++;
	  break;
	}
      else
	tmp++;
    }
    
  return FALSE;  
}


gboolean 
rubrica_search_case_insensitive_incomplete(GtkTreeModel *view_model, 
					   GtkTreePath *path,
					   GtkTreeIter *iter, gpointer data)
{
  RubricaSearchData *search_data = data;
  GtkTreeIter search_iter;
  gchar *buffer;
  gchar *tmp;
  gchar *card;
  gint len;

  gtk_tree_model_get(view_model, iter, 
		     CARD_COLUMN, &card,
		     search_data->column, &buffer, 
		     -1);
  tmp = g_strdup(buffer);
  len = strlen(search_data->str);
    
  while (*tmp)
    {
      if (strncasecmp(tmp, search_data->str, len) == 0)
	{
	  gchar *path_str;
	  
	  path_str = gtk_tree_path_to_string(path);
	  
	  gtk_list_store_append(GTK_LIST_STORE(search_data->model),
				&search_iter);
	  gtk_list_store_set(GTK_LIST_STORE(search_data->model), &search_iter, 
			     SEARCH_RESULT_CARD, card,
			     SEARCH_RESULT_FOUND, buffer, 
			     SEARCH_RESULT_PATH, path_str, 
			     -1);
	  items++;
	  break;
	}
      else
	tmp++;
    }
    
  return FALSE;  
}


gboolean rubrica_search_get_search_case_sensitive(GtkWidget *dialog)
{
  GtkWidget *check;

  check = g_object_get_data(G_OBJECT(dialog), "case_check");
  g_return_val_if_fail(check != NULL, FALSE);

  return  gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check));
}

gboolean rubrica_search_get_search_mode_incomplete(GtkWidget *dialog)
{
  GtkWidget *check;

  check = g_object_get_data(G_OBJECT(dialog), "mode_check");
  g_return_val_if_fail(check != NULL, FALSE);

  return  gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check));
}

RubricaSearchType rubrica_search_get_search_type(void)
{
  return rubrica_preferences_get_search_type();
}


void rubrica_search_set_search_case_sensitive(gboolean bool)
{
  GConfClient *client;

  client = gconf_client_get_default();  
  gconf_client_set_bool(client, RUBRICA_KEY_SEARCH_SENSITIVE, bool, NULL);
  g_object_unref(client);
}

void rubrica_search_set_search_mode_incomplete(gboolean bool)
{
  GConfClient *client;

  client = gconf_client_get_default();  
  gconf_client_set_bool(client, RUBRICA_KEY_SEARCH_COMPLETE, bool, NULL);
  g_object_unref(client);
}

void rubrica_search_set_search_type(RubricaSearchType type)
{
  GConfClient *client;

  client = gconf_client_get_default();  
  gconf_client_set_int(client, RUBRICA_KEY_SEARCH_TYPE, type, NULL);
  g_object_unref(client);
}

