/*
*  program name: Rubrica -- http://digilander.iol.it/nfragale/download.html
*  file: prefer_int.c
*  
*  build the graphic interface for preferences dialog
*
*  Copyright (C) 2000-2003 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <gnome.h>
#include <gconf/gconf-client.h>
#include <gconf/gconf-value.h>
#include <gconf/gconf-schema.h>

#include "prefer.h"
#include "prefer_cb.h"
#include "prefer_int.h"
#include "prefer_configure.h"
#include "ui.h"
#include "rubrica_type.h"
#include "dialog.h"
#include "utils.h"


static GtkWidget*   dialog = NULL;
static GtkWidget*   notebook = NULL;
static GtkTooltips* tooltips;


RubricaPreferItem file[] = {
  {
    N_("Defaults"), "folder.png", TRUE, rubrica_preferences_default, NULL
  },
  {
    N_("File Settings"), "filesett.png", TRUE, 
    rubrica_preferences_file_settings, NULL
  },
  {
    NULL
  }    
};


RubricaPreferItem general[] =  {
/*  {N_("Search and Order"), "search.png",  TRUE, rubrica_preferences_search, NULL}, */
  {
    N_("Misc"), "misc.png", TRUE, rubrica_preferences_misc, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem view[] =  {
  {
    N_("Card columns view"), "view-column.png", TRUE, 
    rubrica_preferences_view_cards_columns, NULL
  },
  {
    N_("Personal Data view"), "personal.png", TRUE, 
    rubrica_preferences_view_personal_data, NULL
  },
  {
    N_("Company Data view"), "company.png", TRUE, 
    rubrica_preferences_view_company_data, NULL
  },
  {
    N_("Notes view"), "notes.png", TRUE, 
    rubrica_preferences_view_note_data, NULL
  },
  {
    N_("Accounts view"), "accounts.png", TRUE, 
    rubrica_preferences_view_account_columns, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem crypto[] =  {
  {
    N_("Cryptography"),   "crypto.png",  TRUE, rubrica_not_implemented, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem translation[] =  {
  {
    N_("Rubrica Dictionary"),   "dictionary.png",  TRUE, 
    rubrica_preferences_dictionary, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem rpi[] =  {
  /* label, pixmap file, notebook page flag, 
     notebook page's function builder, children 
  */
  {
    N_("File"), "file.png", TRUE, rubrica_preferences_file_info, file
  },
  {
    N_("General"), "general.png", TRUE,
    rubrica_preferences_general_info, general
  },
  {
    N_("View"), "view.png", TRUE, rubrica_preferences_view_info, view
  },
  {
    N_("Cryptography"), "crypto.png", TRUE, 
    rubrica_preferences_crypto_info, crypto
  },
  {
    N_("Translation"), "translation.png", TRUE, 
    rubrica_preferences_trans_info,   translation
  },
  {
    NULL
  }
};


   
RubricaPreferItem card_columns[] = {
  /* label, key, default value, not used, not used */  
  {
    N_("Show categories pixmap column"), 
    RUBRICA_KEY_SHOW_CARD_COLUMN_PIXMAP, TRUE, NULL, NULL
  },
  {
    N_("Show card column"),       
    RUBRICA_KEY_SHOW_CARD_COLUMN_CARD, TRUE, NULL, NULL
  },
  {
    N_("Show first name column"), 
    RUBRICA_KEY_SHOW_CARD_COLUMN_FIRST_NAME, TRUE, NULL, NULL
  },
  {
    N_("Show last name column"),  
    RUBRICA_KEY_SHOW_CARD_COLUMN_LAST_NAME, TRUE,  NULL, NULL
  },
  {
    N_("Show profession column"),  
    RUBRICA_KEY_SHOW_CARD_COLUMN_PROFESSION, FALSE, NULL, NULL
  },
  {
    N_("Show city column"),        
    RUBRICA_KEY_SHOW_CARD_COLUMN_CITY, FALSE, NULL, NULL
  },
  {
    N_("Show country column"),    
    RUBRICA_KEY_SHOW_CARD_COLUMN_COUNTRY, FALSE, NULL, NULL
  },
  {
    N_("Show web column"),        
    RUBRICA_KEY_SHOW_CARD_COLUMN_WEB, TRUE,  NULL, NULL
  },
  {
    N_("Show email column"),      
    RUBRICA_KEY_SHOW_CARD_COLUMN_EMAIL, TRUE,  NULL, NULL
  },
  {
    N_("Show telephone column"),  
    RUBRICA_KEY_SHOW_CARD_COLUMN_TELEPHONE, FALSE, NULL, NULL
  },
  {
    N_("Show company column"),    
    RUBRICA_KEY_SHOW_CARD_COLUMN_COMPANY, FALSE, NULL, NULL
  },
  {
    N_("Show assigment column"),  
    RUBRICA_KEY_SHOW_CARD_COLUMN_ASSIGMENT, FALSE, NULL, NULL
  },
  {
    N_("Show categories column"), 
    RUBRICA_KEY_SHOW_CARD_COLUMN_CATEGORIES, FALSE, NULL, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem account_columns[] = {
  /* label, key, default value, not used, not used */  
  {
    N_("Show site column"),        
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_SITE, TRUE, NULL, NULL
  },
  {
    N_("Show web url column"),     
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_WEB, TRUE, NULL, NULL
  },
  {
    N_("Show email column"),       
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_EMAIL, FALSE, NULL, NULL
  },
  {
    N_("Show login column"),       
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_LOGIN, TRUE, NULL, NULL
  },
  {
    N_("Show password column"),    
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_PWORD, TRUE, NULL, NULL
  },
  {
    N_("Show description column"), 
    RUBRICA_KEY_SHOW_ACCOUNT_COLUMN_DESC,  FALSE, NULL, NULL
  },
  {NULL}
};


RubricaPreferItem personal_data[] = {
  {
    N_("Show first name"),    
    RUBRICA_KEY_SHOW_DATA_FIRST_NAME, TRUE, NULL, NULL
  },
  {
    N_("Show middle name"),   
    RUBRICA_KEY_SHOW_DATA_MIDDLE_NAME, FALSE, NULL, NULL
  },
  {
    N_("Show last name"),     
    RUBRICA_KEY_SHOW_DATA_LAST_NAME, TRUE, NULL, NULL
  },
/*
  {
    N_("Show nickname"),      
    RUBRICA_KEY_SHOW_DATA_NICKNAME, TRUE, NULL, NULL
  },
*/
  {
    N_("Show profession"),    
    RUBRICA_KEY_SHOW_DATA_PROFESSION, TRUE, NULL, NULL
  },
  {
    N_("Show name prefix"),   
    RUBRICA_KEY_SHOW_DATA_NAME_PREFIX, FALSE, NULL, NULL
  },
  {
    N_("Show title"),         
    RUBRICA_KEY_SHOW_DATA_TITLE, FALSE, NULL, NULL
  },
  {
    N_("Show birthday"),      
    RUBRICA_KEY_SHOW_DATA_BIRTHDAY, TRUE, NULL, NULL}
  ,
  {
    N_("Show street"),        
    RUBRICA_KEY_SHOW_DATA_STREET, TRUE, NULL, NULL
  },
  {
    N_("Show street number"), 
    RUBRICA_KEY_SHOW_DATA_STREET_NUM, FALSE, NULL, NULL
  },
  {
    N_("Show zip code"),      
    RUBRICA_KEY_SHOW_DATA_ZIP_CODE, FALSE, NULL, NULL
  },
  {
    N_("Show city"),          
    RUBRICA_KEY_SHOW_DATA_CITY, TRUE, NULL, NULL
  },
  {
    N_("Show province"),      
    RUBRICA_KEY_SHOW_DATA_PROVINCE, TRUE, NULL, NULL
  },
  {
    N_("Show state"),         
    RUBRICA_KEY_SHOW_DATA_STATE, FALSE, NULL, NULL
  },
  {
    N_("Show country"),       
    RUBRICA_KEY_SHOW_DATA_COUNTRY, TRUE, NULL, NULL
  },
  {
    N_("Show web urls"),      
    RUBRICA_KEY_SHOW_DATA_WEB, TRUE, NULL, NULL
  },
  {
    N_("Show emails"),        
    RUBRICA_KEY_SHOW_DATA_EMAIL, TRUE, NULL, NULL
  },
  {
    N_("Show telephone numbers"), 
    RUBRICA_KEY_SHOW_DATA_TELEPHONE, TRUE, NULL, NULL
  },
  {
    NULL
  }
};


RubricaPreferItem company_data[] = {
  {
    N_("Show company name"),   
    RUBRICA_KEY_SHOW_DATA_COMPANY, TRUE, NULL, NULL
  },
  {
    N_("Show company street"), 
    RUBRICA_KEY_SHOW_DATA_JOB_STREET, TRUE,  NULL, NULL
  },
  {
    N_("Show street number"),  
    RUBRICA_KEY_SHOW_DATA_JOB_STREET_NUM, FALSE, NULL, NULL
  },
  {
    N_("Show zip code"),       
    RUBRICA_KEY_SHOW_DATA_JOB_ZIP_CODE, FALSE, NULL, NULL
  },
  {
    N_("Show city"),           
    RUBRICA_KEY_SHOW_DATA_JOB_CITY, TRUE,  NULL, NULL
  },
  {
    N_("Show province"),       
    RUBRICA_KEY_SHOW_DATA_JOB_PROVINCE, TRUE,  NULL, NULL
  },
  {
    N_("Show country"),        
    RUBRICA_KEY_SHOW_DATA_JOB_COUNTRY, TRUE,  NULL, NULL
  },
  {
    N_("Show web url"),        
    RUBRICA_KEY_SHOW_DATA_JOB_WEB, TRUE, NULL, NULL
  },
  {
    N_("Show email"),          
    RUBRICA_KEY_SHOW_DATA_JOB_EMAIL, TRUE, NULL, NULL
  },
  {
    N_("Show operator"),       
    RUBRICA_KEY_SHOW_DATA_OPERATOR,  TRUE, NULL, NULL
  },
  {
    N_("Show fax"),            
    RUBRICA_KEY_SHOW_DATA_FAX, TRUE, NULL, NULL
  },
  {
    N_("Show green"),          
    RUBRICA_KEY_SHOW_DATA_GREEN, TRUE, NULL, NULL
  },
  {
    N_("Show customer care"),  
    RUBRICA_KEY_SHOW_DATA_CUSTOMER_CARE, TRUE, NULL, NULL
  },
  {
    N_("Show company notes"),  
    RUBRICA_KEY_SHOW_DATA_JOB_NOTE, FALSE, NULL, NULL
  },
  {
    N_("Show contact assigment"), 
    RUBRICA_KEY_SHOW_DATA_ASSIGMENT, TRUE, NULL, NULL
  },
  {
    N_("Show organizzation"),  
    RUBRICA_KEY_SHOW_DATA_ORG, FALSE, NULL, NULL
  },
  {
    N_("Show department"),     
    RUBRICA_KEY_SHOW_DATA_DEP, FALSE, NULL, NULL
  },
  {
    N_("Show sub department"), 
    RUBRICA_KEY_SHOW_DATA_SUB_DEP, FALSE, NULL, NULL
  },
  {
    N_("Show secretary (or company contact)"), 
    RUBRICA_KEY_SHOW_DATA_SECRETARY, FALSE, NULL, NULL
  },
  {
    N_("Show secretary telephone"), 
    RUBRICA_KEY_SHOW_DATA_SECR_PHONE, FALSE, NULL, NULL
  },
  {
    NULL
  }
};

RubricaPreferItem note_data[] = {
  {
    N_("Show if contact is married"),   
    RUBRICA_KEY_SHOW_DATA_MARRIED, TRUE, NULL, NULL
  },
  {
    N_("Show spouse's name"),          
    RUBRICA_KEY_SHOW_DATA_SPOUSE, TRUE,  NULL, NULL
  },
  {
    N_("Show spouse's birthday"),      
    RUBRICA_KEY_SHOW_DATA_SPOUSE_BIRTH, TRUE, NULL, NULL
  },
  {
    N_("Show marriage anniversary"),   
    RUBRICA_KEY_SHOW_DATA_ANNIVERSARY, TRUE, NULL, NULL
  },
  {
    N_("Show if contact has children"), 
    RUBRICA_KEY_SHOW_DATA_HAS_CHILDREN, FALSE, NULL, NULL
  },
  {
    N_("Show children's notes"),        
    RUBRICA_KEY_SHOW_DATA_CHILDREN, FALSE, NULL, NULL
  },
  {
    N_("Show contact's hobbies"),      
    RUBRICA_KEY_SHOW_DATA_HOBBIES, TRUE, NULL, NULL
  },
  {
    N_("Show notes"),                   
    RUBRICA_KEY_SHOW_DATA_NOTE, TRUE, NULL, NULL
  },
  {N_("Show contact's public key"),    
   RUBRICA_KEY_SHOW_DATA_PUBLIC_KEY, TRUE, NULL, NULL
  },
  {
    NULL
  }
};


GtkWidget* rubrica_preferences_get_preferencesbox(void)
{
  return (GtkWidget*) dialog;
}


GtkWidget* rubrica_preferences_get_notebook(void)
{
  return (GtkWidget*) notebook;
}


/* create the model for preferences index tree 
*/
GtkTreeModel*
rubrica_preferences_create_tree_model(GConfClient *client)
{
  GtkTreeStore *model;
  GtkTreeIter iter;
  RubricaPreferItem *mainitem = rpi;

  model = gtk_tree_store_new(RP_LAST_COLUMN, 
			     GDK_TYPE_PIXBUF,  /* pixmap      */
			     G_TYPE_STRING,    /* string      */ 
			     G_TYPE_BOOLEAN,   /* have_page   */
			     G_TYPE_INT);      /* page number */

  while(mainitem->label)
    {
      RubricaPreferItem *child = mainitem->children;
      GdkPixbuf *image;
      gint page;

      if (mainitem->func)
	page = mainitem->func(client, _(mainitem->label));
      else 
	page = -2;  /* -1 is used in gtk_tree_store_set 
		       to identify list's end */
      
      image = rubrica_ui_pixbuf_new(mainitem->str);

      gtk_tree_store_append(model, &iter, NULL);
      gtk_tree_store_set(model, &iter, 
			 RP_PIXMAP_COLUMN, image, 
			 RP_STRING_COLUMN, _(mainitem->label), 
			 RP_HAVE_PAGE,     mainitem->bool,
			 RP_PAGE_COLUMN,   page, 
			 -1);
			 
      while (child->label)
	{
	  GtkTreeIter child_iter;

	  if (child->func)
	    page = child->func(client, _(child->label));
	  else 
	    page = -2;  /* -1 is used in gtk_tree_store_set 
			   to identify list's end */
	  
	  image = rubrica_ui_pixbuf_new(child->str);
	  
	  gtk_tree_store_append(model, &child_iter, &iter);
	  gtk_tree_store_set(model, &child_iter, 
			     RP_PIXMAP_COLUMN, image, 
			     RP_STRING_COLUMN, _(child->label), 
			     RP_HAVE_PAGE,     child->bool,
			     RP_PAGE_COLUMN,   page, 
			     -1);	  

	  child++;
	}

      mainitem++;
    }

  return GTK_TREE_MODEL(model);
}


/* create the view for preferences tree 
*/
GtkWidget*
rubrica_preferences_create_new_tree_view(GtkTreeModel *model, 
					 GtkWidget *notebook)
{
  GtkWidget *treeview;
  GtkTreeSelection *selection;

  treeview  = gtk_tree_view_new_with_model(model);
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_widget_show(treeview);

  g_object_unref(G_OBJECT(model));

  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_columns_autosize(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  rubrica_preferences_add_columns(GTK_TREE_VIEW(treeview), PREF_TREE);
  gtk_tree_view_expand_all(GTK_TREE_VIEW(treeview));

  g_signal_connect(G_OBJECT(selection), "changed", 
		   G_CALLBACK(rubrica_preferences_row_selected), treeview);
  
  return treeview;
}


/* create a list model for card's column manage
 */
GtkTreeModel*
rubrica_preferences_create_list_model(GConfClient *client, 
				      RubricaTreeType type)
{
  GtkListStore *model;
  GtkTreeIter iter;  
  RubricaPreferItem *listitem = NULL; 

  model = gtk_list_store_new(RP_VIEW_LAST_COLUMN,
			     G_TYPE_BOOLEAN,
			     G_TYPE_STRING, 
			     G_TYPE_STRING);

  switch (type)
    {
    case PREF_CARD_LIST:
      listitem = card_columns;
      break;

    case PREF_ACCOUNT_LIST:
      listitem = account_columns;
      break;

    case PREF_PERSONAL_DATA_LIST:
      listitem = personal_data;
      break;

    case PREF_COMPANY_DATA_LIST:
      listitem = company_data;     
	break;

    case PREF_NOTE_DATA_LIST:
      listitem = note_data;
      break;

    default:
      g_warning("can't create model in "
		"rubrica_preferences_create_list_model(). Type is: %d",
		(RubricaTreeType) type);

      return NULL;;
      break;
    }

  while (listitem->label)
    {
      GConfValue  *value;
      gchar *key;
      gboolean bool;
      
      key = listitem->str;
      value = gconf_client_get(client, key, NULL);
      if (value == NULL || value->type != GCONF_VALUE_BOOL)
	{
	  bool = listitem->bool;
	}
      else 
	bool = gconf_client_get_bool(client, key, NULL); 
      
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 RP_VIEW_CHECK_COLUMN, bool,
			 RP_VIEW_LABEL_COLUMN, _(listitem->label), 
			 RP_VIEW_KEY_COLUMN,   key,
			 -1);
      listitem++;
    }
    
  return GTK_TREE_MODEL(model);    
}


/* create a list view using a card or account or data model 
*/
GtkWidget*
rubrica_preferences_create_list_view_with_model(GtkTreeModel *model, 
						RubricaTreeType type)
{
  GtkWidget *listview;
  GtkTreeSelection *selection;

  listview  = gtk_tree_view_new_with_model(model);
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(listview));
  gtk_widget_show(listview);

  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  if (type == PREF_TREE)
    {
      gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(listview), FALSE);
      gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(listview), FALSE);
    }
  else
    {
      gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(listview), TRUE);
      gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(listview), TRUE);
    }

  rubrica_preferences_add_columns(GTK_TREE_VIEW(listview), type);
  gtk_tree_view_expand_all(GTK_TREE_VIEW(listview));

  return listview;
}


/* add the rigth columns to the list view 
*/
void 
rubrica_preferences_add_columns(GtkTreeView *tree, RubricaTreeType type)
{
  GtkCellRenderer *cell;
  GtkTreeModel *model;
  GtkTreeViewColumn *column;
  gint col;

  model = gtk_tree_view_get_model(tree);

  if (type == PREF_TREE)
    {
      cell = gtk_cell_renderer_pixbuf_new();
      gtk_tree_view_insert_column_with_attributes(tree, -1, NULL, cell, 
						  "pixbuf",
						  RP_PIXMAP_COLUMN, NULL);
      
      cell = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes(tree, -1, NULL, cell, 
						  "text", 
						  RP_STRING_COLUMN, NULL); 
    }
  else
    {
      cell = gtk_cell_renderer_toggle_new();
      col  = gtk_tree_view_insert_column_with_attributes(tree, -1, 
							 _("Show"), 
							 cell, "active", 
							 RP_VIEW_CHECK_COLUMN,
							 NULL); 
      column = gtk_tree_view_get_column(tree, col-1);
      gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column),
				      GTK_TREE_VIEW_COLUMN_FIXED);
      gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 50);
      gtk_tree_view_column_set_max_width (GTK_TREE_VIEW_COLUMN(column), 50);
     
      g_signal_connect(G_OBJECT(cell), "toggled", 
		       G_CALLBACK(rubrica_preferences_manage_columns), model);
      
      cell = gtk_cell_renderer_text_new();
      gtk_tree_view_insert_column_with_attributes(tree, -1, 
						  _("Description"), 
						  cell, "text", 
						  RP_VIEW_LABEL_COLUMN, NULL); 
    }
}


/*    list/tree view callbacks 
*/
void 
rubrica_preferences_manage_columns(GtkCellRendererToggle *cell, 
				   gchar *path_str, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path;
  GConfClient *client;
  gboolean status;      /* check status */
  gchar *key;

  model  = (GtkTreeModel *) data;
  path   = gtk_tree_path_new_from_string(path_str);
  client = gconf_client_get_default ();

  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter,
		      RP_VIEW_CHECK_COLUMN, &status, 
		      RP_VIEW_KEY_COLUMN,   &key, 
		      -1);
  status ^= 1;
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, RP_VIEW_CHECK_COLUMN, 
		     status, -1);
  gconf_client_set_bool (client, key, status, NULL);
}


/*    list/tree view callbacks 
*/
void 
rubrica_preferences_row_selected(GtkTreeSelection *select, gpointer data)
{
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));
  GtkTreeIter iter;
  gint page;
  gboolean have_page;
  
  if (gtk_tree_selection_get_selected(select, &model, &iter))
    {
      gtk_tree_model_get(model, &iter, 
			 RP_HAVE_PAGE,   &have_page, 
			 RP_PAGE_COLUMN, &page, 
			 -1);

      if (have_page)
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);      
    }  
}


/*    preferences interface creation
*/
GtkWidget*
create_preferences_dialog(GtkWidget *app, GConfClient *client)
{
  GtkWidget *tree;
  GtkTreeModel *model;

  GtkWidget *dialog_vbox;
  GtkWidget *hbox;

  dialog = gtk_dialog_new_with_buttons(_("Rubrica Preferences"), 
				       GTK_WINDOW(app), GTK_DIALOG_MODAL,
				       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				       NULL);
  notebook = gtk_notebook_new ();
  tooltips = gtk_tooltips_new ();
  model    = rubrica_preferences_create_tree_model(client);
  tree     = rubrica_preferences_create_new_tree_view(model, notebook);  

  rubrica_utils_add_tooltip(tooltips);

  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
  rubrica_preferences_reset_errors();
  
  gtk_dialog_set_has_separator (GTK_DIALOG(dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE);

  dialog_vbox = GTK_DIALOG (dialog)->vbox;
  gtk_container_set_border_width (GTK_CONTAINER (dialog_vbox), 2);
  gtk_widget_show (dialog_vbox);

  hbox = rubrica_ui_box_new(HBOX, FALSE, 2, 2);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), hbox, TRUE, TRUE, 2);

  gtk_box_pack_start(GTK_BOX(hbox), tree, TRUE, TRUE, 2);
  gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 2);

  gtk_widget_show (notebook);

  if (rubrica_preferences_there_are_errors())
    {
      gtk_window_set_modal (GTK_WINDOW (dialog), FALSE);
      rubrica_dialog_error_loading_preferences_keys();      
      gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);      
    }

  g_object_set_data (G_OBJECT (dialog), "preferences_dialog", dialog);
  g_object_set_data (G_OBJECT (dialog), "treeview", tree);
  g_object_set_data (G_OBJECT (dialog), "notebook", notebook);
  g_object_set_data (G_OBJECT (dialog), "tooltips", tooltips);

  return dialog;
}


GtkWidget*
rubrica_preferences_build_notebook_info_page(gchar *frame_label, 
					     gchar *pixfile, gchar *str)
{
  GtkWidget *frame;
  GtkWidget *framebox;
  GtkWidget *imagebox;
  GtkWidget *image;
  GtkWidget *label;

  frame = rubrica_ui_frame_new(frame_label);
  framebox = rubrica_ui_default_box_new(VBOX);
  gtk_container_add (GTK_CONTAINER (frame), framebox);
  
  imagebox = rubrica_ui_default_box_new(HBOX);
  gtk_box_pack_start(GTK_BOX(framebox), imagebox, FALSE, FALSE, 2);
  
  image = rubrica_ui_image_new(pixfile);
  gtk_box_pack_start(GTK_BOX(imagebox), image, FALSE, FALSE, 2);

  label = rubrica_ui_label_new(str);
  gtk_box_pack_start(GTK_BOX(framebox), label, FALSE, FALSE, 2);

  return frame;
}


GtkWidget*
rubrica_preferences_build_notebook_list_page(GConfClient *client, 
					     gchar *frame_label, 
					     gchar *str, RubricaTreeType type)
{
  GtkWidget *frame;

  GtkTreeModel *model;
  GtkWidget *list;
  GtkWidget *box_frame;
  GtkWidget *box_label;
  GtkWidget *label;
  GtkWidget *scrollwin;
  GtkWidget *sep;
  
  
  frame = rubrica_ui_frame_new(frame_label);
  box_frame = gtk_vbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(frame), box_frame);
  gtk_widget_show(box_frame);
  
  model = rubrica_preferences_create_list_model(client, type);
  list  = rubrica_preferences_create_list_view_with_model(model, type);

  g_object_unref(G_OBJECT(model));
  
  scrollwin = rubrica_ui_scrolled_window_new();

  sep = gtk_hseparator_new();
  gtk_widget_show(sep);

  box_label = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(str);
  gtk_box_pack_start(GTK_BOX(box_label), label, FALSE, FALSE, 5);
  gtk_widget_show(box_label);
  gtk_widget_show(label);

  gtk_box_pack_start(GTK_BOX(box_frame), box_label, FALSE, FALSE, 5);
  gtk_box_pack_start(GTK_BOX(box_frame), sep,  FALSE, FALSE, 5);  
  gtk_box_pack_start(GTK_BOX(box_frame), scrollwin,  TRUE, TRUE, 5);  
  gtk_container_add(GTK_CONTAINER(scrollwin), list);

  return frame;
}


int 
rubrica_preferences_file_info(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;

  str = g_strdup(_("In file pages you can customize the default folder\n"
		   "that rubrica will use to save addressbooks, you can\n"
		   "set the default addressbook that rubrica will load\n"
		   "when start, you can choose backup type,can choose to\n"
		   "use autosave and other..."));

  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "file-info.png", 
						       str);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int 
rubrica_preferences_general_info(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;

  str = g_strdup(_("In general pages you can choose how cards will be\n" 
		   "ordered (alphabetic order or by creation date),\n"
		   "increasing order or decreasing order. Can set the\n"
		   "search rules, some decoratives properties"));

  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "general-info.png", 
						       str);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int
rubrica_preferences_view_info(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;

  str = g_strdup(_("In view pages you can choose what rubrica will\n"
		   "show to you in cards/account window and data window.\n"
		   "Can select the card's columns to show.\nCan select the" 
		   "card's data to show."));

  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "view-info.png", 
						       str); 

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int 
rubrica_preferences_crypto_info(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;

  str = g_strdup(_("In cryptographic pages you can choose to protect your\n"
		   "data using a cryptographic routine.\n"
		   "Some crypt algorithms are available:\n"
		   "(TODO: list here)"));
    
  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "crypto-info.png", 
						       str);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int 
rubrica_preferences_trans_info(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;
 
  str = g_strdup(_("In translation pages you can choose if rubrica must use\n"
		   "one default dictionary (Italian or English) or can build\n"
		   "your own dictionary.\n"
		   "Dictionary is used in rubrica preview to show card's\n"
		   "fields in your native language."));
    
  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "translation-info.png",
						       str);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int 
rubrica_not_implemented(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;

  str = g_strdup(_("Sorry, this features is not yet implemented ;-)\n"));
    
  frame = rubrica_preferences_build_notebook_info_page(frame_label, 
						       "info.png", 
						       str);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  g_free(str);
  return page;
}


int 
rubrica_preferences_default(GConfClient *client, gchar *frame_label)
{
  GtkWidget *file_frame;
  gint page;

  GtkWidget *box_file;
  GtkWidget *box_addbook;
  GtkWidget *box_folder;

  GtkWidget *show_frame;
  GtkWidget *addbook_frame;
  GtkWidget *folder_frame;

  GtkWidget *check_path;
  GtkWidget *check_addbook;
  GtkWidget *check_folder;

  GtkWidget *fileentry_addbook;
  GtkWidget *fileentry_folder;
  GtkWidget *entry_addbook;
  GtkWidget *entry_folder;

  gchar *str;


  file_frame = rubrica_ui_frame_new(frame_label);

  /*    SHOW PATH 
	show path check button
  */
    
  box_file = rubrica_ui_box_new(VBOX, FALSE, 0, 5);
  gtk_container_add (GTK_CONTAINER (file_frame), box_file);

  show_frame =  rubrica_ui_frame_new(_("Show"));
  gtk_box_pack_start (GTK_BOX (box_file), show_frame, FALSE, FALSE, 2);

  check_path = rubrica_ui_check_button_new(_("_Show addressbook's path"));
  gtk_tooltips_set_tip(tooltips, check_path, 
		       _("Show the path of current addressbook "), NULL);
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_SHOW, check_path);
  gtk_container_add(GTK_CONTAINER(show_frame), check_path);

  rubrica_preferences_configure_check(client, check_path, RUBRICA_KEY_SHOW); 

  /*    -----------------------------------------------------------------------
	load address book at rubrica start
	use a default folder to save addressbooks

	- check_addbook   
	    - true:  load addressbook at start
	    - false: start with void addressbook

	    - entry: addressbook file name
	
	- check_folder
	    - true:  use the setted directory as default folder
	    - false  save addressbook in current directory

	    - entry: directory name
	-----------------------------------------------------------------------
  */

  /*    ADDRESS BOOK
   */
  addbook_frame = rubrica_ui_frame_new(_("Default address book"));
  gtk_box_pack_start (GTK_BOX (box_file), addbook_frame, FALSE, FALSE, 2);

  box_addbook = rubrica_ui_default_box_new(VBOX);
  gtk_container_add (GTK_CONTAINER (addbook_frame), box_addbook);

  /*    CHECK BUTTON
	CHECK ADDRESS BOOK
   */
  check_addbook = rubrica_ui_check_button_new(_("_Load this addressbook "
						"at start"));
  gtk_box_pack_start (GTK_BOX (box_addbook), check_addbook, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_HAVE_BOOK, check_addbook);

  /*    GNOME FILE ENTRY
	ADDRESS BOOK
  */
  fileentry_addbook = gnome_file_entry_new ("history_addressbook", NULL);
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_BOOK, fileentry_addbook);
  g_object_set_data(G_OBJECT(check_addbook), RUBRICA_KEY_BOOK, 
		    fileentry_addbook);

  gtk_box_pack_start(GTK_BOX(box_addbook), fileentry_addbook, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (fileentry_addbook), 2);
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (fileentry_addbook), TRUE);
  gtk_widget_show (fileentry_addbook);

  /*    FILE ENTRY
	ADDRESS BOOK
  */  
  entry_addbook = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY
					      (fileentry_addbook));
  g_object_set_data (G_OBJECT (dialog), "entry_addbook", entry_addbook);

  if (gconf_client_get_bool(client, RUBRICA_KEY_HAVE_BOOK, NULL))
    str = g_strdup(gconf_client_get_string(client, RUBRICA_KEY_BOOK, NULL));
  else
    str = g_strdup_printf("%s/.rubrica/rubrica.rub", g_get_home_dir());      
  
  gtk_entry_set_text (GTK_ENTRY (entry_addbook), str);
  gtk_widget_show (entry_addbook);
  g_free(str);  

  /*      GConf settings
   */
  rubrica_preferences_configure_checked_entry(client, check_addbook, 
					      fileentry_addbook, 
					      entry_addbook, 
					      RUBRICA_KEY_HAVE_BOOK, 
					      RUBRICA_KEY_BOOK);

  /*    FOLDER
   */
  folder_frame = rubrica_ui_frame_new(_("Default folder"));
  gtk_box_pack_start (GTK_BOX (box_file), folder_frame, FALSE, FALSE, 2);

  box_folder = rubrica_ui_default_box_new(VBOX);
  gtk_container_add (GTK_CONTAINER (folder_frame), box_folder);

  /*    CHECK FOLDER
   */
  check_folder =rubrica_ui_check_button_new(_("_Save addressbooks in this"
					       " folder by default"));
  gtk_box_pack_start (GTK_BOX (box_folder), check_folder, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_HAVE_FOLDER, check_folder);

  /*    GNOME FILE ENTRY
	FOLDER
  */
  fileentry_folder = gnome_file_entry_new ("history_folder", 
					   _("Choose a folder"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_FOLDER, fileentry_folder);

  gtk_box_pack_start (GTK_BOX (box_folder), fileentry_folder, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (fileentry_folder), 2);
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (fileentry_folder), TRUE);
  gtk_widget_show (fileentry_folder);

  /*    FILE ENTRY
	FOLDER
  */ 
  entry_folder =gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(fileentry_folder));
  g_object_set_data (G_OBJECT (dialog), "entry_folder", entry_folder);
  gtk_widget_show (entry_folder);

  if (gconf_client_get_bool(client, RUBRICA_KEY_HAVE_FOLDER, NULL))
    str = g_strdup(gconf_client_get_string(client, RUBRICA_KEY_FOLDER, NULL));
  else
    str = g_strdup_printf("%s/.rubrica", g_get_home_dir());  
    
  gtk_entry_set_text (GTK_ENTRY (entry_folder), str);
  g_free(str);  

  /*    GConf settings
  */
  rubrica_preferences_configure_checked_entry(client,
					      check_folder, fileentry_folder, 
					      entry_folder,  
					      RUBRICA_KEY_HAVE_FOLDER, 
					      RUBRICA_KEY_FOLDER);


  /*    SHOW PATH CALLBACK
  */
  g_signal_connect (G_OBJECT (check_path), "toggled",
		    G_CALLBACK (on_check_button_toggled), RUBRICA_KEY_SHOW);

  /*    LOAD DEFAULT ADDRESS BOOK CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_addbook), "toggled",
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_HAVE_BOOK);
  g_signal_connect (G_OBJECT (entry_addbook), "changed",
		    G_CALLBACK (on_entry_changed), RUBRICA_KEY_BOOK);
  
  /*    USE DEFAULT FOLDER CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_folder), "toggled",
		    G_CALLBACK (on_check_button_toggled),
		    RUBRICA_KEY_HAVE_FOLDER);
  g_signal_connect (G_OBJECT (entry_folder), "changed",
		    G_CALLBACK (on_entry_changed), RUBRICA_KEY_FOLDER);


  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), file_frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), file_frame);
  
  return page;
}


int rubrica_preferences_file_settings(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  GtkWidget *vbox;
  gint page;

  GtkWidget *box_backup; 
  GtkWidget *box_radio_backup;
  GtkWidget *box_main_compress;
  GtkWidget *box_compression;
  GtkWidget *box_compression_rate;
  GtkWidget *box_autosave;
  GtkWidget *box_check_autosave;
  GtkWidget *box_spin_autosave;

  GtkWidget *backup_frame;
  GtkWidget *compress_frame;
  GtkWidget *autosave_frame;

  GSList *radio_backup_group = NULL;
  GtkWidget *radio_backup_inc;
  GtkWidget *radio_backup_classic;

  GtkWidget *check_backup;
  GtkWidget *check_compress;
  GtkWidget *check_autosave;

  GtkWidget *label_compress_rate;
  GtkObject *spin_rate_adj;
  GtkWidget *spin_rate;

  GtkWidget *label_autosave1;
  GtkWidget *label_autosave2;
  GtkObject *spin_autosave_adj;
  GtkWidget *spin_autosave;


  frame = rubrica_ui_frame_new(frame_label);
  vbox  = rubrica_ui_default_box_new(VBOX);
  gtk_container_add (GTK_CONTAINER (frame), vbox);
  
  /*    Backup:
	- do backup before overwrite 

	default: don't do backup

	CHECK BUTTON
	BACKUP ALWAYS
  */ 
  backup_frame = rubrica_ui_frame_new(_("Backup"));
  gtk_box_pack_start (GTK_BOX (vbox), backup_frame, FALSE, FALSE, 2);

  box_backup = rubrica_ui_box_new(VBOX, TRUE, 0, 2);
  gtk_container_add (GTK_CONTAINER (backup_frame), box_backup);

  /*    CHECK BUTTON BACKUP 
  */
  check_backup = rubrica_ui_check_button_new(_("_Make backup of previous file "
						"before overwrite"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_BACKUP, check_backup);
  gtk_box_pack_start (GTK_BOX (box_backup), check_backup, TRUE, TRUE, 0);

  /*    RADIO BACKUP INCREMENT
  */
  box_radio_backup = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_box_pack_start (GTK_BOX (box_backup), box_radio_backup, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (dialog), "box_radio_backup", box_radio_backup);

  radio_backup_inc = gtk_radio_button_new_with_mnemonic(radio_backup_group,
							_("in_cremental "
							  "backup"));
  radio_backup_group = gtk_radio_button_group(GTK_RADIO_BUTTON
					      (radio_backup_inc));

  g_object_set_data(G_OBJECT (dialog), "inc_bak", radio_backup_inc);

  gtk_box_pack_start(GTK_BOX(box_radio_backup),radio_backup_inc,TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, radio_backup_inc, 
			_("Example: The backup of addressbook.rub "
			  "file will be addressbook.rub1, if "
			  "addressbook.rub1 exists then backup "
			  "will be addressbook.rub2, and so on."), NULL);
  gtk_widget_show (radio_backup_inc);

  /*    RADIO BACKUP CLASSIC
  */
  radio_backup_classic = gtk_radio_button_new_with_mnemonic(radio_backup_group,
							    _("cl_assic "
							      "backup"));
  radio_backup_group = gtk_radio_button_group (GTK_RADIO_BUTTON
					       (radio_backup_classic));
  g_object_set_data (G_OBJECT (dialog),"bak_bak", 
		     radio_backup_classic);

  gtk_box_pack_start (GTK_BOX (box_radio_backup), radio_backup_classic, 
		      TRUE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, radio_backup_classic, 
			_("The backup of addressbook.rub will "
			  "be addressbook.bak"), NULL);
  gtk_widget_show (radio_backup_classic);

  /* GConf settings
   */
  rubrica_preferences_configure_check_radio(client,
					    check_backup, radio_backup_inc, 
					    radio_backup_classic, 
					    RUBRICA_KEY_BACKUP, 
					    RUBRICA_KEY_BACKUP_TYPE_INC);

  /*    Compress addressbook:
	- compression check button        
	- compression rate spin button

	default: not compress, compression rate: 0

	CHECK BUTTON
	COMPRESS ADDRESSBOOK FILE
  */
  compress_frame = rubrica_ui_frame_new(_("Compression"));
  gtk_box_pack_start (GTK_BOX (vbox), compress_frame, FALSE, FALSE, 2);

  box_main_compress = rubrica_ui_box_new(VBOX, TRUE, 0, 2);
  gtk_container_add (GTK_CONTAINER (compress_frame), box_main_compress);

  box_compression = rubrica_ui_box_new(HBOX, TRUE, 0, 2);
  gtk_box_pack_start (GTK_BOX (box_main_compress), box_compression, FALSE, 
		      FALSE, 0);

  /*    CHECK BUTTON COMPRESS ADDRESSBOOK FILE
  */
  check_compress = rubrica_ui_check_button_new(_("Com_press addressbook"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_COMPRESS, check_compress);
  gtk_box_pack_start(GTK_BOX (box_compression), check_compress, TRUE, TRUE, 0);

  box_compression_rate = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_box_pack_end(GTK_BOX(box_main_compress), box_compression_rate,
		   FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (check_compress), "spin_box", 
		     box_compression_rate);

  label_compress_rate = rubrica_ui_label_new (_("Compression _rate:"));
  gtk_box_pack_start(GTK_BOX(box_compression_rate), label_compress_rate,
		     FALSE, FALSE, 0);
  gtk_misc_set_alignment (GTK_MISC (label_compress_rate), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label_compress_rate), 3, 0);
  gtk_widget_show (label_compress_rate);

  /*    SPIN BUTTON
	COMPRESSION RATE
  */  
  spin_rate_adj = gtk_adjustment_new (0, 0, 9, 1, 1, 1);
  spin_rate = gtk_spin_button_new (GTK_ADJUSTMENT (spin_rate_adj), 1, 0);
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_RATE, spin_rate);
  gtk_label_set_mnemonic_widget(GTK_LABEL(label_compress_rate), spin_rate);

  gtk_box_pack_start(GTK_BOX(box_compression_rate), spin_rate, FALSE, FALSE,5);
  gtk_tooltips_set_tip (tooltips, spin_rate, _("0 not compressed, "
					       "9 max compression rate"),
			NULL);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin_rate), TRUE);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spin_rate), FALSE);
  gtk_widget_show (spin_rate);

  /*    GConf settings
   */
  rubrica_preferences_configure_spin(client, check_compress, spin_rate,  
				     RUBRICA_KEY_COMPRESS, RUBRICA_KEY_RATE);

  /*    Automatic save:
	- autosave check button        
	- autosave spin button

	default: autosave: no, time elapsed: 0

	CHECK BUTTON
	AUTOSAVE ADDRESSBOOK FILE
  */
  autosave_frame = rubrica_ui_frame_new(_("Autosave"));
  gtk_box_pack_start (GTK_BOX (vbox), autosave_frame, FALSE, FALSE, 2);

  box_autosave = rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (autosave_frame), box_autosave);

  box_check_autosave = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_box_pack_start (GTK_BOX (box_autosave), box_check_autosave, FALSE, 
		      FALSE, 0);

  /*    CHECK BUTTON AUTOSAVE ADDRESSBOOK FILE
  */
  check_autosave = rubrica_ui_check_button_new(_("_enable autosave"));
  g_object_set_data(G_OBJECT (dialog), RUBRICA_KEY_AUTOSAVE, check_autosave);
  gtk_box_pack_start (GTK_BOX (box_check_autosave), check_autosave, TRUE,
		      TRUE, 0);

  box_spin_autosave = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  g_object_set_data(G_OBJECT (check_autosave), "spin_box", box_spin_autosave);

  gtk_box_pack_start (GTK_BOX (box_autosave), box_spin_autosave, FALSE, 
		      FALSE, 0);

  label_autosave1 = rubrica_ui_label_new(_("Sa_ve addressbooks ever"));
  gtk_box_pack_start (GTK_BOX (box_spin_autosave), label_autosave1, FALSE,
		      FALSE, 0);
  gtk_misc_set_padding (GTK_MISC (label_autosave1), 3, 0);
  gtk_widget_show (label_autosave1);

  /*    SPIN BUTTON
	SAVE EVERY xx MINUTES
  */  
  spin_autosave_adj = gtk_adjustment_new (0, 0, 100, 1, 10, 10);
  spin_autosave = gtk_spin_button_new(GTK_ADJUSTMENT(spin_autosave_adj), 1, 0);
  gtk_box_pack_start (GTK_BOX (box_spin_autosave), spin_autosave, 
		      FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_WAIT_TIME, spin_autosave);
  gtk_label_set_mnemonic_widget(GTK_LABEL(label_autosave1), spin_autosave);
  gtk_widget_show (spin_autosave);

  label_autosave2 = rubrica_ui_label_new (_("minutes"));
  gtk_misc_set_padding (GTK_MISC (label_autosave2), 5, 0);
  gtk_box_pack_start (GTK_BOX (box_spin_autosave), label_autosave2,
		      FALSE, FALSE, 0);
  gtk_widget_show (label_autosave2);

  /*    GConf settings
   */
  rubrica_preferences_configure_spin(client,
				     check_autosave, spin_autosave, 
				     RUBRICA_KEY_AUTOSAVE, 
				     RUBRICA_KEY_WAIT_TIME);

  gconf_client_notify_add(client, RUBRICA_KEY_WAIT_TIME, 
			  rubrica_preferences_notify_autosave_time_changes, 
			  spin_autosave, NULL, NULL);

  /*    BACKUP CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_backup), "toggled",
		    G_CALLBACK (on_check_button_toggled), RUBRICA_KEY_BACKUP);

  g_signal_connect (G_OBJECT (radio_backup_inc), "toggled",
		    G_CALLBACK (on_radio_button_toggled), 
		    RUBRICA_KEY_BACKUP_TYPE_INC);
  g_signal_connect (G_OBJECT (radio_backup_classic), "toggled",
		    G_CALLBACK (on_radio_button_toggled), 
		    RUBRICA_KEY_BACKUP_TYPE_BAK);

  /*    COMPRESS CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_compress), "toggled",
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_COMPRESS);
  g_signal_connect (G_OBJECT (spin_rate), "activate",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_RATE);
  g_signal_connect (G_OBJECT (spin_rate), "changed",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_RATE);

  /*    AUTOSAVE CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_autosave), "toggled",
		    G_CALLBACK (on_check_button_toggled),
		    RUBRICA_KEY_AUTOSAVE);
  g_signal_connect (G_OBJECT (spin_autosave), "activate",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_WAIT_TIME);
  g_signal_connect (G_OBJECT (spin_autosave), "changed",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_WAIT_TIME);


  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);

  return page;
}


int rubrica_preferences_search(GConfClient *client, gchar *frame_label)
{
  GtkWidget *general_frame;
  gint page;

  GtkWidget *box_general;
  GtkWidget *search_vbox;  
  GtkWidget *search_hbox;  

  GtkWidget *sort_frame;
  GtkWidget *search_frame;

  GtkWidget *direction;
  GtkWidget *check_case; /* case sensitive (true) insensitive (false) search */
  GtkWidget *check_type; /* search on complete (true) 
			    incomplete (false) string */

  GtkWidget *search_on_menu;
  GtkWidget *search_on_label;
  RubricaOptionType search_on_option[] = {
    {"", N_("Card"),          SEARCH_CARD},
    {"", N_("First Name"),    SEARCH_FIRST_NAME},
    {"", N_("Last Name"),     SEARCH_LAST_NAME},
    {"", N_("Profession"),    SEARCH_PROFESSION},
    {"", N_("City"),          SEARCH_CITY},
    {"", N_("Country"),       SEARCH_COUNTRY},
    {"", N_("Web Address"),   SEARCH_WEB_ADDRESS},
    {"", N_("Email Address"), SEARCH_EMAIL_ADDRESS},
    {"", N_("Telephone"),     SEARCH_TELEPHONE},
    {"", N_("Company"),       SEARCH_COMPANY},
    {"", N_("Categories"),    SEARCH_CATEGORIES},
    {NULL, NULL}
  };

  GtkWidget *sort_label;
  GtkWidget *sort_box;
  GtkWidget *sort_menu;
  RubricaOptionType sort_option[] = {
    {"", N_("in alphabetic order"), NO_GROUP},
    {"", N_("by creation date"),    NO_GROUP},
    {"", N_("by last change date"), NO_GROUP},
    {NULL, NULL}
  };


  general_frame = rubrica_ui_frame_new(frame_label);
  box_general = rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (general_frame), box_general);

  /*    SORT 
  */
  sort_frame = rubrica_ui_frame_new(_("Ordering"));
  gtk_box_pack_start (GTK_BOX (box_general), sort_frame, FALSE, FALSE, 2);

  sort_box = rubrica_ui_box_new(HBOX, FALSE, 2, 2);
  gtk_container_add (GTK_CONTAINER (sort_frame), sort_box);

  /*    CHECK BUTTON: ASCENDING/DESCENDIG SORT
  */
  direction = rubrica_ui_pack_in_box_check_button_new(sort_box, 
						      _("Sort is asce_nding"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_SORT_DIRECTION, direction);
  gtk_container_set_border_width (GTK_CONTAINER (direction), 2);
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for descending sort"), NULL);

  rubrica_preferences_configure_check(client, direction, 
				      RUBRICA_KEY_SORT_DIRECTION); 

  /*        SORT MENU

            - SORT BY DATE (CREATION, or LAST CHANGE), ALPHABETICALLY
	    default: alphabetic order
  */
  sort_label = rubrica_ui_label_new(_("_Sort cards"));
  gtk_misc_set_padding (GTK_MISC (sort_label), 20, 0);
  gtk_box_pack_start(GTK_BOX(sort_box), sort_label, FALSE, FALSE, 2);
  
  sort_menu = rubrica_ui_option_menu_new(sort_option);
  gtk_box_pack_start(GTK_BOX(sort_box), sort_menu, FALSE, FALSE, 2);  
  gtk_label_set_mnemonic_widget (GTK_LABEL (sort_label), sort_menu);

  rubrica_preferences_configure_option_menu(client, sort_menu, 
					    RUBRICA_KEY_SORT_TYPE);


  /*    SEARCH
  */
  search_frame = rubrica_ui_frame_new(_("Search"));
  gtk_box_pack_start (GTK_BOX (box_general), search_frame, FALSE, FALSE, 2);

  search_vbox = rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  search_hbox = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (search_frame), search_vbox);
  gtk_box_pack_start(GTK_BOX(search_vbox), search_hbox, FALSE, FALSE, 2);

  /*    SEARCH CASE SENSITIVE/INSENSITIVE
  */
  check_case = rubrica_ui_check_button_new(_("_Search is case sensitive"));
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for case insensitive search"), NULL);
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_SEARCH_SENSITIVE,check_case);
  gtk_box_pack_start(GTK_BOX(search_hbox), check_case, FALSE, FALSE, 2);

  rubrica_preferences_configure_check(client, check_case, 
				      RUBRICA_KEY_SEARCH_SENSITIVE); 

  /*    SEARCH COMPLETE/INCOMPLETE STRING
  */
  check_type = rubrica_ui_check_button_new(_("Search on _complete string"));
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for search incomplete string"), NULL);
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_SEARCH_COMPLETE, check_case);
  gtk_box_pack_start(GTK_BOX(search_hbox), check_type, FALSE, FALSE, 2);

  rubrica_preferences_configure_check(client, check_type, 
				      RUBRICA_KEY_SEARCH_COMPLETE);

  /*    SEARCH TYPE (where perform search)
  */
  search_on_label = rubrica_ui_label_new(_("Search is performed on:"));
  search_on_menu  = rubrica_ui_option_menu_new(search_on_option);
  gtk_misc_set_padding (GTK_MISC (search_on_label), 20, 0);
  gtk_box_pack_start(GTK_BOX(search_hbox), search_on_label, FALSE, FALSE, 2);
  gtk_box_pack_start(GTK_BOX(search_hbox), search_on_menu, FALSE, FALSE, 2);  
  rubrica_preferences_configure_option_menu(client, search_on_menu, 
					    RUBRICA_KEY_SEARCH_TYPE);
  

  /*    SORT CALLBACKS
   */
  g_signal_connect (G_OBJECT (direction), "toggled", 
		    G_CALLBACK (on_check_button_toggled),
		    RUBRICA_KEY_SORT_DIRECTION); 

  g_signal_connect (G_OBJECT (sort_menu), "changed", 
		    G_CALLBACK (on_option_menu_changed),
		    RUBRICA_KEY_SORT_TYPE); 

  /*     SEARCH CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_case), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_SEARCH_SENSITIVE); 

  g_signal_connect (G_OBJECT (check_type), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_SEARCH_COMPLETE); 

  g_signal_connect (G_OBJECT (search_on_menu), "changed", 
		    G_CALLBACK (on_option_menu_changed),
		    RUBRICA_KEY_SEARCH_TYPE); 


  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), general_frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), general_frame);

  return page;
}



int rubrica_preferences_misc(GConfClient *client, gchar *frame_label)
{
  GtkWidget *general_frame;
  gint page;

  GtkWidget *box_general;
  GtkWidget *box_size;
  GtkWidget *box_exit;
  GtkWidget *box_tooltips;
  GtkWidget *search_vbox;  
  GtkWidget *search_hbox;  
  GtkWidget *search_hbox2;  

  GtkWidget *sort_frame;
  GtkWidget *search_frame;

  GtkWidget *size_frame;
  GtkWidget *exit_frame;
  GtkWidget *tooltips_frame;

  GtkWidget *check_exit;
  GtkWidget *check_tooltips;

  GtkObject *spin_width_adj;
  GtkWidget *spin_width;
  GtkObject *spin_length_adj;
  GtkWidget *spin_length;

  GtkWidget *label_width;
  GtkWidget *label_length; 

  GtkWidget *toolbar_frame;
  GtkWidget *box_toolbar;
  GtkWidget *option_label;
  GtkWidget *option_menu;
  RubricaOptionType toolbar_option[] = {
    {"", N_("only icons"), NO_GROUP},
    {"", N_("only text"),  NO_GROUP},
    {"", N_("both text and icons"), NO_GROUP},
    {"", N_("both text and icons alongside each other"), NO_GROUP},
    // {"", N_("according to Gnome default"), NO_GROUP},
    {NULL}
  };
  
  GtkWidget *direction;
  GtkWidget *check_case; /* case sensitive (true) insensitive (false) search */
  GtkWidget *check_type; /* search on complete (true) 
			    incomplete (false) string */

  GtkWidget *search_on_menu;
  GtkWidget *search_on_label;
  RubricaOptionType search_on_option[] = {
    {"", N_("Card"),          SEARCH_CARD},
    {"", N_("First Name"),    SEARCH_FIRST_NAME},
    {"", N_("Last Name"),     SEARCH_LAST_NAME},
    {"", N_("Profession"),    SEARCH_PROFESSION},
    {"", N_("City"),          SEARCH_CITY},
    {"", N_("Country"),       SEARCH_COUNTRY},
    {"", N_("Web Address"),   SEARCH_WEB_ADDRESS},
    {"", N_("Email Address"), SEARCH_EMAIL_ADDRESS},
    {"", N_("Telephone"),     SEARCH_TELEPHONE},
    {"", N_("Company"),       SEARCH_COMPANY},
    {"", N_("Categories"),    SEARCH_CATEGORIES},
    {NULL, NULL}
  };

  GtkWidget *sort_label;
  GtkWidget *sort_box;
  GtkWidget *sort_menu;
  RubricaOptionType sort_option[] = {
    {"", N_("in alphabetic order"), NO_GROUP},
    {"", N_("by creation date"),    NO_GROUP},
    {"", N_("by last change date"), NO_GROUP},
    {NULL, NULL}
  };


  general_frame = rubrica_ui_frame_new(frame_label);
  box_general = rubrica_ui_box_new(VBOX, FALSE, 0, 3);
  gtk_container_add (GTK_CONTAINER (general_frame), box_general);


  /*    SORT 
  */
  sort_frame = rubrica_ui_frame_new(_("Ordering"));
  gtk_box_pack_start (GTK_BOX (box_general), sort_frame, FALSE, FALSE, 3);

  sort_box = rubrica_ui_box_new(HBOX, FALSE, 2, 3);
  gtk_container_add (GTK_CONTAINER (sort_frame), sort_box);

  /* 
     CHECK BUTTON: ASCENDING/DESCENDIG SORT
  */
  direction = rubrica_ui_pack_in_box_check_button_new(sort_box, 
						      _("Sort is asce_nding"));
  g_object_set_data (G_OBJECT (dialog), 
		     RUBRICA_KEY_SORT_DIRECTION, direction);
  gtk_container_set_border_width (GTK_CONTAINER (direction), 2);
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for descending sort"), NULL);

  rubrica_preferences_configure_check(client, direction, 
				      RUBRICA_KEY_SORT_DIRECTION); 

  /*        SORT MENU

            - SORT BY DATE (CREATION, or LAST CHANGE), ALPHABETICALLY
	    default: alphabetic order
  */
  sort_label = rubrica_ui_label_new(_("_Sort cards"));
  gtk_misc_set_padding (GTK_MISC (sort_label), 20, 0);
  gtk_box_pack_start(GTK_BOX(sort_box), sort_label, FALSE, FALSE, 3);
  
  sort_menu = rubrica_ui_option_menu_new(sort_option);
  gtk_box_pack_start(GTK_BOX(sort_box), sort_menu, FALSE, FALSE, 3);  
  gtk_label_set_mnemonic_widget (GTK_LABEL (sort_label), sort_menu);

  rubrica_preferences_configure_option_menu(client, sort_menu, 
					    RUBRICA_KEY_SORT_TYPE);


  /*    SEARCH
  */
  search_frame = rubrica_ui_frame_new(_("Search"));
  gtk_box_pack_start (GTK_BOX (box_general), search_frame, FALSE, FALSE, 2);

  search_vbox = rubrica_ui_box_new(VBOX, FALSE, 0, 3);
  search_hbox = rubrica_ui_box_new(HBOX, FALSE, 0, 3);
  search_hbox2 = rubrica_ui_box_new(HBOX, FALSE, 0, 3);
  gtk_container_add (GTK_CONTAINER (search_frame), search_vbox);
  gtk_box_pack_start(GTK_BOX(search_vbox), search_hbox, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(search_vbox), search_hbox2, FALSE, FALSE, 3);

  /*    SEARCH CASE SENSITIVE/INSENSITIVE
  */
  check_case = rubrica_ui_check_button_new(_("_Search is case sensitive"));
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for case insensitive search"), NULL);
  g_object_set_data (G_OBJECT (dialog), 
		     RUBRICA_KEY_SEARCH_SENSITIVE, check_case);
  gtk_box_pack_start(GTK_BOX(search_hbox), check_case, FALSE, FALSE, 2);

  rubrica_preferences_configure_check(client, check_case, 
				      RUBRICA_KEY_SEARCH_SENSITIVE); 

  /*    SEARCH COMPLETE/INCOMPLETE STRING
  */
  check_type = rubrica_ui_check_button_new(_("Search on _complete string"));
  gtk_tooltips_set_tip (tooltips, direction, 
			_("Uncheck for search incomplete string"), NULL);
  g_object_set_data (G_OBJECT (dialog), 
		     RUBRICA_KEY_SEARCH_COMPLETE, check_case);
  gtk_box_pack_start(GTK_BOX(search_hbox), check_type, FALSE, FALSE, 3);

  rubrica_preferences_configure_check(client, check_type, 
				      RUBRICA_KEY_SEARCH_COMPLETE);

  /*    SEARCH TYPE (where perform search)
  */
  search_on_label = rubrica_ui_label_new(_("Search is performed on:"));
  search_on_menu  = rubrica_ui_option_menu_new(search_on_option);
  gtk_box_pack_start(GTK_BOX(search_hbox2), search_on_label, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(search_hbox2), search_on_menu, FALSE, FALSE, 3);  
  rubrica_preferences_configure_option_menu(client, search_on_menu,
					    RUBRICA_KEY_SEARCH_TYPE);
  

  /*     WINDOW SIZE FRAME
  */
  size_frame = rubrica_ui_frame_new(_("Main window's dimensions"));
  gtk_box_pack_start (GTK_BOX (box_general), size_frame, FALSE, FALSE, 3);
  
  box_size   = rubrica_ui_box_new(HBOX, FALSE, 0, 3);
  gtk_container_add (GTK_CONTAINER (size_frame), box_size);

  /* lunghezza */
  label_width =  rubrica_ui_label_new(_("_Width dimension:"));
  gtk_misc_set_alignment (GTK_MISC (label_width), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label_width), 5, 0);

  spin_width_adj = gtk_adjustment_new (0, 0, 1200, 1, 10, 10);
  spin_width = gtk_spin_button_new (GTK_ADJUSTMENT (spin_width_adj), 1, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin_width), TRUE);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spin_width), FALSE);
  gtk_widget_show (spin_width);

  gtk_box_pack_start(GTK_BOX(box_size), label_width, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(box_size), spin_width, FALSE, FALSE, 3);
  g_object_set_data(G_OBJECT (dialog), "spin_width", spin_width);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_width), spin_width);


  /* altezza */
  label_length =  rubrica_ui_label_new (_("_Length dimension:"));
  gtk_misc_set_alignment (GTK_MISC (label_length), 0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label_length), 5, 0);

  spin_length_adj = gtk_adjustment_new (0, 0, 800, 1, 10, 10);
  spin_length = gtk_spin_button_new (GTK_ADJUSTMENT (spin_length_adj), 1, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin_length), TRUE);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spin_length), FALSE);
  gtk_widget_show (spin_length);

  gtk_box_pack_start(GTK_BOX(box_size), label_length, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(box_size), spin_length, FALSE, FALSE, 3);
  g_object_set_data(G_OBJECT (dialog), "spin_length", spin_length);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_length), spin_length);

  rubrica_preferences_configure_size_spin(client, spin_width,  
					  RUBRICA_KEY_WIDTH); 
  rubrica_preferences_configure_size_spin(client, spin_length,
					  RUBRICA_KEY_LENGTH); 


  /* EXIT FRAME
   */
  exit_frame = rubrica_ui_frame_new(_("Exit"));
  gtk_box_pack_start (GTK_BOX (box_general), exit_frame, FALSE, FALSE, 3);

  box_exit = rubrica_ui_box_new(HBOX, TRUE, 0, 3);
  gtk_container_add (GTK_CONTAINER (exit_frame), box_exit);

  /*    CHECK BUTTON
	
	EXIT

	default: ask before quit
  */
  check_exit = rubrica_ui_check_button_new(_("E_xit from rubrica without "
					     "confirming"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_EXIT, 
		     check_exit);
  gtk_box_pack_start (GTK_BOX (box_exit), check_exit, TRUE, TRUE, 3);

  /*  	GConf settings
  */
  rubrica_preferences_configure_check(client, check_exit, RUBRICA_KEY_EXIT);


  /*    ENABLE TOOLTIPS
	show tooltips -- default: true
  */
  tooltips_frame = rubrica_ui_frame_new(_("Tooltips"));
  g_object_set_data (G_OBJECT (dialog), "tooltips_frame", 
		     tooltips_frame);
  gtk_box_pack_start (GTK_BOX (box_general), tooltips_frame, FALSE, FALSE, 3);

  box_tooltips = rubrica_ui_box_new(HBOX, TRUE, 0, 3);
  gtk_container_add (GTK_CONTAINER (tooltips_frame), box_tooltips);

  /*    CHECK BUTTON SHOW TIPS
  */
  check_tooltips = rubrica_ui_check_button_new(_("Enable _Tooltips"));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_TOOLTIPS, 
		     check_tooltips);
  g_object_set_data (G_OBJECT (check_tooltips), "tooltips", tooltips);
  gtk_box_pack_start (GTK_BOX (box_tooltips), check_tooltips, TRUE, TRUE, 3);

  /* 	GConf settings
   */
  rubrica_preferences_configure_check(client, check_tooltips, 
				      RUBRICA_KEY_TOOLTIPS);
  gconf_client_notify_add(client, RUBRICA_KEY_TOOLTIPS, 
			  rubrica_preferences_notify_tooltips_changes, 
			  check_tooltips, NULL, NULL);  
  
  /*    TOOLBAR settings
	toolbar style -- default: show both text and icons
  */

  toolbar_frame = rubrica_ui_frame_new(_("Toolbar")); 
  g_object_set_data (G_OBJECT (dialog), "toolbar_frame", 
		     toolbar_frame);
  gtk_box_pack_start (GTK_BOX (box_general), toolbar_frame, FALSE, FALSE, 2);  

  box_toolbar = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (toolbar_frame), box_toolbar);

  option_label = rubrica_ui_label_new(_("Toolbar show:"));
  gtk_box_pack_start(GTK_BOX(box_toolbar), option_label, FALSE, FALSE, 2);
  
  option_menu = rubrica_ui_option_menu_new(toolbar_option);
  gtk_box_pack_start(GTK_BOX(box_toolbar), option_menu, FALSE, FALSE, 5);
  

 /* 	GConf settings
  */   
  rubrica_preferences_configure_option_menu(client, option_menu, 
					    RUBRICA_KEY_TOOLBAR_STYLE);	
					    
  gconf_client_notify_add(client, RUBRICA_KEY_TOOLBAR_STYLE, 
			  rubrica_preferences_notify_toolbar_style_changes,
			  option_menu, NULL, NULL); 
		  

  /*    SORT CALLBACKS
   */
  g_signal_connect (G_OBJECT (direction), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_SORT_DIRECTION); 

  g_signal_connect (G_OBJECT (sort_menu), "changed", 
		    G_CALLBACK (on_option_menu_changed), 
		    RUBRICA_KEY_SORT_TYPE); 

  /*     SEARCH CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_case), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_SEARCH_SENSITIVE); 

  g_signal_connect (G_OBJECT (check_type), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_SEARCH_COMPLETE); 

  g_signal_connect (G_OBJECT (search_on_menu), "changed", 
		    G_CALLBACK (on_option_menu_changed), 
		    RUBRICA_KEY_SEARCH_TYPE); 

  /*    SPIN BUTTONS CALLBACKS
   */
  g_signal_connect (G_OBJECT (spin_width), "activate",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_WIDTH);
  g_signal_connect (G_OBJECT (spin_width), "changed",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_WIDTH);

  g_signal_connect (G_OBJECT (spin_length), "activate",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_LENGTH);  
  g_signal_connect (G_OBJECT (spin_length), "changed",
		    G_CALLBACK (on_spin_activate), RUBRICA_KEY_LENGTH);  

  /*    CHECK BUTTONS CALLBACKS
  */
  g_signal_connect (G_OBJECT (check_exit), "toggled",
		    G_CALLBACK (on_check_button_toggled), RUBRICA_KEY_EXIT);

  g_signal_connect (G_OBJECT (check_tooltips), "toggled", 
		    G_CALLBACK (on_check_button_toggled), 
		    RUBRICA_KEY_TOOLTIPS);

  /*    TOOLBAR STYLE CALLBACK
   */
   g_signal_connect (G_OBJECT (option_menu), "changed", 
		    G_CALLBACK (on_option_menu_changed), 
		    RUBRICA_KEY_TOOLBAR_STYLE);  
  
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), general_frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), general_frame);

  return page;
}



int 
rubrica_preferences_view_cards_columns(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;
  gchar *label;

  str = g_strdup(_("Check the columns that you want to see in card view"));
  label = g_strdup(_(frame_label));
  frame = rubrica_preferences_build_notebook_list_page(client, label, str, 
						       PREF_CARD_LIST);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);
  g_free(label);
  g_free(str);

  return page;
}


int 
rubrica_preferences_view_account_columns(GConfClient *client, 
					 gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;
  gchar *label;

  str = g_strdup(_("Check the columns that you want to see in account view"));
  label = g_strdup(_(frame_label));
  frame = rubrica_preferences_build_notebook_list_page(client, label, str, 
						       PREF_ACCOUNT_LIST);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);
  g_free(label);
  g_free(str);

  return page;  
}


int
rubrica_preferences_view_personal_data(GConfClient *client, 
				       gchar *frame_label) 
{
  GtkWidget *frame;
  gint page;
  gchar *str;
  gchar *label;

  str = g_strdup(_("Check the items that you want to see in Data window"));
  label = g_strdup(_(frame_label));
  frame = rubrica_preferences_build_notebook_list_page(client, label, str, 
						       PREF_PERSONAL_DATA_LIST);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);
  g_free(label);
  g_free(str);

  return page;    
}


int 
rubrica_preferences_view_company_data(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *str;
  gchar *label;

  str = g_strdup(_("Check the items that you want to see in Data window"));
  label = g_strdup(_(frame_label));
  frame = rubrica_preferences_build_notebook_list_page(client, label, str, 
						       PREF_COMPANY_DATA_LIST);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);
  g_free(label);
  g_free(str);

  return page;   
}


int 
rubrica_preferences_view_note_data(GConfClient *client, gchar *frame_label)
{
  GtkWidget *frame;
  gint page;
  gchar *label;
  gchar *str;

  str = g_strdup(_("Check the items that you want to see in Data window"));
  label = g_strdup(_(frame_label));
  frame = rubrica_preferences_build_notebook_list_page(client, label, str, 
						       PREF_NOTE_DATA_LIST);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame);
  g_free(label);
  g_free(str);

  return page;   
}


int 
rubrica_preferences_dictionary(GConfClient *client, gchar *frame_label)
{
  gint page;
  GtkWidget *frame;

  GtkWidget *dictionary_frame;
  GtkWidget *notebook_frame;

  GSList *user_lang = NULL;
  GSList *def_lang = NULL;
  GtkWidget *user_dict;
  GtkWidget *def_dict;
  GtkWidget *italian;
  GtkWidget *english;  

  GtkWidget *fileentry;
  GtkWidget *entry;

  GtkWidget *notebook_dict;

  GtkWidget *load;
  GtkWidget *save;
  GtkWidget *show_rules;

  GtkWidget *box_dict;
  GtkWidget *box_def_dict;
  GtkWidget *box_io;
  GtkWidget *buttonbox_io;

  GtkWidget *dictionary_box;
  GtkWidget *notebook_box;
  GtkWidget *box_frame;
  GtkWidget *frame_personal_data;
  GtkWidget *frame_address;
  GtkWidget *frame_net;
  GtkWidget *frame_work;
  GtkWidget *frame_notes;

  GtkWidget *personal_tab;
  GtkWidget *address_tab;
  GtkWidget *net_tab;
  GtkWidget *work_tab;
  GtkWidget *notes_tab;

  GtkWidget *label1, *label2, *label3;
  gchar *path;

  frame     = rubrica_ui_frame_new(frame_label);
  box_frame = rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  gtk_container_add(GTK_CONTAINER(frame), box_frame);

  /*    dictionary frame
   */
  dictionary_frame = rubrica_ui_frame_new(_("Dictionaries")); 
  gtk_box_pack_start (GTK_BOX (box_frame), dictionary_frame, FALSE, FALSE, 2); 
 

  /*    boxes
   */
  dictionary_box = rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (dictionary_frame), dictionary_box);

  /*    notebook frame
   */  
  notebook_frame = rubrica_ui_frame_new(_("User dictionary"));
  gtk_box_pack_start (GTK_BOX (box_frame), notebook_frame, FALSE, FALSE, 2);

  
  /*    choose dictionary box
  */
  box_dict = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  gtk_box_pack_start (GTK_BOX (dictionary_box), box_dict, TRUE, TRUE, 0);

  label1 = rubrica_ui_label_new(_("Rubrica will use:  "));
  gtk_box_pack_start (GTK_BOX (box_dict), label1, FALSE, TRUE, 0);

  /*    choose a default dictionary box
  */
  box_def_dict = rubrica_ui_box_new(HBOX, FALSE, 0, 2);
  g_object_set_data (G_OBJECT (dialog), "box_def_dict", box_def_dict);
  gtk_box_pack_start (GTK_BOX (dictionary_box), box_def_dict, TRUE, TRUE, 0);

  label2 = rubrica_ui_label_new(_("Use as my default dictionary:"));
  gtk_box_pack_start(GTK_BOX(box_def_dict), label2, FALSE, TRUE, 0);


  /*     RADIO BUTTONS
	 
         - DEFAULT DICTIONARY
	 - EXTERNAL DICTIONARY
	 
	 default: use one of the default dictionary (italian or english)
  */
  user_dict = gtk_radio_button_new_with_mnemonic(user_lang,
						_("an exter_nal dictionary"));
  user_lang = gtk_radio_button_group(GTK_RADIO_BUTTON(user_dict));
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_EXTERNAL, user_dict);

  gtk_box_pack_start (GTK_BOX (box_dict), user_dict, FALSE, TRUE, 3);
  gtk_tooltips_set_tip (tooltips, user_dict, 
			_("uses an external dictionary when show preview"), 
			NULL);
  gtk_widget_show (user_dict);

  def_dict = gtk_radio_button_new_with_mnemonic(user_lang,
						_("a de_fault dictionary"));
  user_lang = gtk_radio_button_group(GTK_RADIO_BUTTON(def_dict));
  g_object_set_data(G_OBJECT (dialog),RUBRICA_KEY_DEFAULT, def_dict);
  gtk_box_pack_start (GTK_BOX (box_dict), def_dict, FALSE, TRUE, 3);
  gtk_tooltips_set_tip(tooltips, def_dict, 
		       _("uses one of the following default dictionaries"),
		       NULL);
  gtk_widget_show (def_dict);

  
  /*    RADIO BUTTONS
    
        - ITALIAN DICTIONARY
	- ENGLISH DICTIONARY

	default: use italian dictionary
  */
  italian = gtk_radio_button_new_with_mnemonic(def_lang, _("_Italian"));
  def_lang = gtk_radio_button_group(GTK_RADIO_BUTTON(italian));
  g_object_set_data (G_OBJECT (dialog), RUBRICA_KEY_ITALIAN, italian);
  gtk_box_pack_start(GTK_BOX(box_def_dict), italian, FALSE, TRUE, 3);
  gtk_widget_show (italian);

  english = gtk_radio_button_new_with_mnemonic(def_lang, _("_English"));
  def_lang = gtk_radio_button_group(GTK_RADIO_BUTTON(english));
  g_object_set_data(G_OBJECT(dialog), RUBRICA_KEY_ENGLISH, english);
  gtk_box_pack_start(GTK_BOX(box_def_dict), english, FALSE, TRUE, 3);
  gtk_widget_show (english);


  /*      dictionary notebook 
  */
  notebook_box =  rubrica_ui_box_new(VBOX, FALSE, 0, 2);
  gtk_container_add (GTK_CONTAINER (notebook_frame), notebook_box);
  g_object_set_data(G_OBJECT (dialog), "notebook_box", notebook_box);
  
  notebook_dict = gtk_notebook_new ();
  g_object_set_data(G_OBJECT (dialog), "notebook_dict", notebook_dict);
  gtk_box_pack_start(GTK_BOX(notebook_box), notebook_dict, TRUE, TRUE, 2);
  gtk_widget_show (notebook_dict);

  /* notebook pages */

  /*  personal page
   */
  frame_personal_data = rubrica_preferences_do_personal_page();
  gtk_container_add (GTK_CONTAINER (notebook_dict), frame_personal_data);
  personal_tab = gtk_label_new (_("Personal"));
  gtk_widget_show (personal_tab);
  gtk_notebook_set_tab_label(GTK_NOTEBOOK (notebook_dict),
			     gtk_notebook_get_nth_page(GTK_NOTEBOOK 
						       (notebook_dict), 0), 
			     personal_tab);
  /*  address page
   */
  frame_address = rubrica_preferences_do_address_page();
  gtk_container_add (GTK_CONTAINER (notebook_dict), frame_address);
  address_tab = gtk_label_new (_("Address"));
  gtk_widget_show (address_tab);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook_dict), 
			      gtk_notebook_get_nth_page(GTK_NOTEBOOK
							(notebook_dict), 1), 
			      address_tab);

  /*  communications page
   */
  frame_net = rubrica_preferences_do_communications_page();
  gtk_container_add (GTK_CONTAINER (notebook_dict), frame_net);
  net_tab = gtk_label_new (_("Communications"));
  gtk_widget_show (net_tab);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook_dict), 
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (notebook_dict), 2), 
			      net_tab);

  /*  work/company page
   */
  frame_work = rubrica_preferences_do_work_page();
  gtk_container_add (GTK_CONTAINER (notebook_dict), frame_work);
  gtk_widget_show (frame_work);
  work_tab = gtk_label_new (_("Work"));
  gtk_widget_show(work_tab);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook_dict), 
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK 
							 (notebook_dict), 3), 
			      work_tab);

  /*    frame notes
  */
  frame_notes = rubrica_preferences_do_note_page();
  gtk_container_add (GTK_CONTAINER (notebook_dict), frame_notes);
  gtk_widget_show (frame_notes);
  notes_tab = gtk_label_new (_("Notes"));
  gtk_widget_show (notes_tab);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook_dict),  
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK 
							 (notebook_dict), 4), 
			      notes_tab);


  /* INPUT/OUTPUT BOX
   */
  path = g_strdup_printf("%s%s.rubrica/dictionary",g_get_home_dir(), 
			 G_DIR_SEPARATOR_S);
  
  box_io = gtk_hbox_new (FALSE, 3);
  gtk_widget_show (box_io);
  gtk_box_pack_start (GTK_BOX (notebook_box), box_io, FALSE, FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (box_io), 2);

  label3 = rubrica_ui_label_new(_("User dictionary:"));
  gtk_box_pack_start(GTK_BOX (box_io), label3, FALSE, TRUE, 3);

  fileentry = gnome_file_entry_new ("dict_history", NULL);
  gnome_file_entry_set_default_path(GNOME_FILE_ENTRY (fileentry), path);
  gtk_box_pack_start(GTK_BOX(box_io), fileentry, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (fileentry), 2);
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (fileentry), TRUE);
  gtk_widget_show (fileentry);

  entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(fileentry));
  g_object_set_data(G_OBJECT (dialog), "dict_entry", entry);
  gtk_widget_show (entry);

  buttonbox_io = gtk_hbox_new(FALSE, 3);
  gtk_widget_show (buttonbox_io);
  gtk_box_pack_start (GTK_BOX (notebook_box), buttonbox_io, FALSE, TRUE, 3);
  gtk_container_set_border_width (GTK_CONTAINER (buttonbox_io), 3);

  /*    load and save dictionary buttons
  */
  load = rubrica_ui_pixbutton_new(_("_Load"), "load.png");
  gtk_box_pack_start(GTK_BOX(buttonbox_io), load, FALSE, TRUE, 3);
  gtk_tooltips_set_tip(tooltips, load, _("Load dictionary and show it"), NULL);

  save = rubrica_ui_pixbutton_new(_("_Save"), "save.png");
  gtk_box_pack_start(GTK_BOX(buttonbox_io), save, FALSE, TRUE, 3);
  gtk_tooltips_set_tip (tooltips, save, _("Save dictionary"), NULL);

  show_rules = rubrica_ui_pixbutton_new(_("Show _Rules"), "help.png");
  gtk_box_pack_start(GTK_BOX(buttonbox_io), show_rules, FALSE, TRUE, 3);
  gtk_tooltips_set_tip(tooltips,show_rules, _("Show translation rules"), NULL);


  /* ---------------------------------------------------------------------
     GConf settings
     ---------------------------------------------------------------------  */
  rubrica_preferences_configure_radio_dictionary(client, user_dict, def_dict,  
						 RUBRICA_KEY_EXTERNAL,
						 RUBRICA_KEY_DEFAULT);  
  rubrica_preferences_configure_radio(client, 2, 
				      italian, RUBRICA_KEY_ITALIAN,  
				      english, RUBRICA_KEY_ENGLISH); 

  rubrica_preferences_configure_entry(client, entry, 
				      RUBRICA_KEY_USER_DICTIONARY);
  /* ---------------------------------------------------------------------  */

  g_signal_connect(G_OBJECT (user_dict), "toggled",
		   G_CALLBACK(on_radio_button_toggled), RUBRICA_KEY_EXTERNAL);
  g_signal_connect(G_OBJECT(def_dict), "toggled",
		   G_CALLBACK(on_radio_button_toggled), RUBRICA_KEY_DEFAULT);

  g_signal_connect(G_OBJECT(italian), "toggled",
		   G_CALLBACK(on_radio_button_toggled), RUBRICA_KEY_ITALIAN);
  g_signal_connect(G_OBJECT(english), "toggled",
		   G_CALLBACK(on_radio_button_toggled), RUBRICA_KEY_ENGLISH);


  /*      notebook dictionary buttons callbacks 
  */
  g_signal_connect(G_OBJECT(load), "clicked",
		   G_CALLBACK(on_load_dictionary_clicked), entry);
  g_signal_connect(G_OBJECT(save), "clicked",
		   G_CALLBACK(on_save_dictionary_clicked), entry);
  g_signal_connect(G_OBJECT(show_rules), "clicked",
		   G_CALLBACK(on_show_rules_clicked), NULL);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, NULL); 
  page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), frame); 

  g_free(path);

  return page;
}


GtkWidget* rubrica_preferences_do_personal_page(void)
{
  GtkWidget *frame;
  GtkWidget *table;

  GtkWidget *entry_first_name;
  GtkWidget *entry_middle_name;
  GtkWidget *entry_last_name;
  GtkWidget *entry_profession;
  GtkWidget *entry_name_prefix;
  GtkWidget *entry_title;
  GtkWidget *entry_card;
  GtkWidget *entry_data;
  GtkWidget *entry_categories;


  frame = rubrica_ui_frame_new(_("Personal dictionary"));

  table = gtk_table_new (5, 4, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 10);

  /*    dictionary --  personal data
   */
  rubrica_ui_tabled_label_new(table, _("First name"),    0, 1, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Last name"),     0, 1, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Name Prefix"),   0, 1, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Personal Data"), 0, 1, 3, 4); 
  rubrica_ui_tabled_label_new(table, _("Categories"),    0, 1, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Middle name"),   2, 3, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Profession"),    2, 3, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Title"),         2, 3, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Card"),          2, 3, 3, 4);

  /*    personal dictionary entry
  */
  entry_first_name  = rubrica_ui_tabled_entry_new(table, 1, 2, 0, 1);
  entry_last_name   = rubrica_ui_tabled_entry_new(table, 1, 2, 1, 2);
  entry_name_prefix = rubrica_ui_tabled_entry_new(table, 1, 2, 2, 3);
  entry_data        = rubrica_ui_tabled_entry_new(table, 1, 2, 3, 4);
  entry_categories  = rubrica_ui_tabled_entry_new(table, 1, 2, 4, 5);
  entry_middle_name = rubrica_ui_tabled_entry_new(table, 3, 4, 0, 1);
  entry_profession  = rubrica_ui_tabled_entry_new(table, 3, 4, 1, 2);
  entry_title       = rubrica_ui_tabled_entry_new(table, 3, 4, 2, 3);
  entry_card        = rubrica_ui_tabled_entry_new(table, 3, 4, 3, 4);

  g_object_set_data(G_OBJECT (dialog), "entry_first_name", entry_first_name);
  g_object_set_data(G_OBJECT (dialog), "entry_middle_name",entry_middle_name);
  g_object_set_data(G_OBJECT (dialog), "entry_last_name",  entry_last_name);
  g_object_set_data(G_OBJECT (dialog), "entry_profession", entry_profession);
  g_object_set_data(G_OBJECT (dialog), "entry_name_prefix", entry_name_prefix);
  g_object_set_data(G_OBJECT (dialog), "entry_title",       entry_title);
  g_object_set_data(G_OBJECT (dialog), "entry_card", entry_card);
  g_object_set_data(G_OBJECT (dialog), "entry_data", entry_data);
  g_object_set_data(G_OBJECT (dialog), "entry_categories", entry_categories);

  /*    personal
  */
  g_signal_connect (G_OBJECT (entry_first_name), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_FIRST_NAME);
  g_signal_connect (G_OBJECT (entry_middle_name), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_MIDDLE_NAME);
  g_signal_connect (G_OBJECT (entry_last_name), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_LAST_NAME);
  g_signal_connect (G_OBJECT (entry_profession), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_PROFESSION);
  g_signal_connect (G_OBJECT (entry_name_prefix), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_NAME_PREFIX);
  g_signal_connect (G_OBJECT (entry_title), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TITLE);
  g_signal_connect (G_OBJECT (entry_card), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_CARD);
  g_signal_connect (G_OBJECT (entry_categories), "changed",
		    G_CALLBACK (on_dictionary_entry_changed), 
		    RUBRICA_KEY_DICT_CARD_GROUP);
  
  return frame;
}


GtkWidget *rubrica_preferences_do_address_page(void)
{
  GtkWidget *frame;
  GtkWidget *table;

  GtkWidget *entry_address;
  GtkWidget *entry_street;
  GtkWidget *entry_street_number;
  GtkWidget *entry_zip_code;
  GtkWidget *entry_city;
  GtkWidget *entry_province;
  GtkWidget *entry_country;
  GtkWidget *entry_state;

  frame = rubrica_ui_frame_new(_("Address  dictionary"));

  table = gtk_table_new (5, 4, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 3);
  gtk_widget_show (table);

  rubrica_ui_tabled_label_new(table, _("Street"),        0, 1, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Street number"), 2, 3, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Zip code"),      0, 1, 1, 2);
  rubrica_ui_tabled_label_new(table, _("City"),          2, 3, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Province"),      0, 1, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Country"),       0, 1, 3, 4);
  rubrica_ui_tabled_label_new(table, _("State"),         2, 3, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Address"),       2, 3, 3, 4);

  /*    address dictionary entry
  */
  entry_street        = rubrica_ui_tabled_entry_new(table, 1, 2, 0, 1);
  entry_street_number = rubrica_ui_tabled_entry_new(table, 3, 4, 0, 1);
  entry_city          = rubrica_ui_tabled_entry_new(table, 3, 4, 1, 2);
  entry_state         = rubrica_ui_tabled_entry_new(table, 3, 4, 2, 3);
  entry_address       = rubrica_ui_tabled_entry_new(table, 3, 4, 3, 4);
  entry_zip_code      = rubrica_ui_tabled_entry_new(table, 1, 2, 1, 2);
  entry_province      = rubrica_ui_tabled_entry_new(table, 1, 2, 2, 3);
  entry_country       = rubrica_ui_tabled_entry_new(table, 1, 2, 3, 4);

  g_object_set_data(G_OBJECT (dialog), "entry_street", entry_street);
  g_object_set_data(G_OBJECT(dialog), "entry_street_number",
		    entry_street_number);
  g_object_set_data(G_OBJECT (dialog), "entry_city", entry_city);
  g_object_set_data(G_OBJECT (dialog), "entry_state", entry_state);
  g_object_set_data(G_OBJECT (dialog), "entry_address", entry_address);
  g_object_set_data(G_OBJECT (dialog), "entry_zip_code", entry_zip_code);
  g_object_set_data(G_OBJECT (dialog), "entry_province", entry_province);
  g_object_set_data(G_OBJECT (dialog), "entry_country", entry_country);

  g_signal_connect (G_OBJECT (entry_address), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_ADDRESS);
  g_signal_connect (G_OBJECT (entry_street), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_STREET);
  g_signal_connect (G_OBJECT (entry_street_number), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_STREET_NUMBER);
  g_signal_connect (G_OBJECT (entry_zip_code), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_ZIP_CODE);
  g_signal_connect (G_OBJECT (entry_city), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_CITY);
  g_signal_connect (G_OBJECT (entry_province), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_PROVINCE);
  g_signal_connect (G_OBJECT (entry_country), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_COUNTRY);
  g_signal_connect (G_OBJECT (entry_state), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_STATE);
  
  return frame;
}


GtkWidget* rubrica_preferences_do_communications_page(void)
{ 
  GtkWidget *frame;
  GtkWidget *table;

  GtkWidget *entry_web_address;
  GtkWidget *entry_web;
  GtkWidget *entry_email_address;
  GtkWidget *entry_email;

  GtkWidget *entry_tel_list;
  GtkWidget *entry_telephone;
  GtkWidget *entry_type;
  GtkWidget *entry_tel_work;
  GtkWidget *entry_tel_home;  
  GtkWidget *entry_tel_fax;  
  GtkWidget *entry_tel_cell; 
  GtkWidget *entry_tel_other; 

  frame = rubrica_ui_frame_new(_("Communications dictionary"));

  table = gtk_table_new (6, 4, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 3);
  gtk_widget_show (table);

  /*                                                         x    y */
  rubrica_ui_tabled_label_new(table, _("Web address"),     0, 1, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Web"),             2, 3, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Email address"),   0, 1, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Email"),           2, 3, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Telephone list"),  0, 1, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Telephone "),      2, 3, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Type"),            0, 1, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Work telephone"),  2, 3, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Home telephone"),  0, 1, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Fax"),             2, 3, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Cellular"),        0, 1, 5, 6);
  rubrica_ui_tabled_label_new(table, _("Other telephone"), 2, 3, 5, 6);


  /*    entries
  */
  entry_web_address   = rubrica_ui_tabled_entry_new(table, 1, 2, 0, 1);
  entry_web           = rubrica_ui_tabled_entry_new(table, 3, 4, 0, 1);
  entry_email_address = rubrica_ui_tabled_entry_new(table, 1, 2, 1, 2);
  entry_email         = rubrica_ui_tabled_entry_new(table, 3, 4, 1, 2);
  entry_tel_list      = rubrica_ui_tabled_entry_new(table, 1, 2, 2, 3);
  entry_telephone     = rubrica_ui_tabled_entry_new(table, 3, 4, 2, 3);
  entry_type          = rubrica_ui_tabled_entry_new(table, 1, 2, 3, 4);
  entry_tel_work      = rubrica_ui_tabled_entry_new(table, 3, 4, 3, 4);
  entry_tel_home      = rubrica_ui_tabled_entry_new(table, 1, 2, 4, 5);
  entry_tel_fax       = rubrica_ui_tabled_entry_new(table, 3, 4, 4, 5);
  entry_tel_cell      = rubrica_ui_tabled_entry_new(table, 1, 2, 5, 6);
  entry_tel_other     = rubrica_ui_tabled_entry_new(table, 3, 4, 5, 6);

  g_object_set_data(G_OBJECT (dialog), "entry_web_address", entry_web_address);
  g_object_set_data(G_OBJECT (dialog), "entry_web", entry_web);
  g_object_set_data(G_OBJECT (dialog), "entry_email", entry_email);
  g_object_set_data(G_OBJECT (dialog), "entry_email_address",
		    entry_email_address);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_list",  entry_tel_list);
  g_object_set_data (G_OBJECT (dialog), "entry_telephone", entry_telephone);
  g_object_set_data (G_OBJECT (dialog), "entry_type",      entry_type);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_work",  entry_tel_work);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_home",  entry_tel_home);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_fax",   entry_tel_fax);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_cell",  entry_tel_cell);
  g_object_set_data (G_OBJECT (dialog), "entry_tel_other", entry_tel_other);


  /*    net
  */
  g_signal_connect (G_OBJECT (entry_web_address), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_WEB_ADDRESS);
  g_signal_connect (G_OBJECT (entry_web), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_WEB);
  g_signal_connect (G_OBJECT (entry_email_address), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_EMAIL_ADDRESS);
  g_signal_connect (G_OBJECT (entry_email), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_EMAIL);
  
  /*    telephone
  */
  g_signal_connect (G_OBJECT (entry_tel_list), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_LIST);
  g_signal_connect (G_OBJECT (entry_telephone), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_NUMBER);
  g_signal_connect (G_OBJECT (entry_type), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_TYPE);

  g_signal_connect (G_OBJECT (entry_tel_work), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_WORK);
  g_signal_connect (G_OBJECT (entry_tel_home), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_HOME);
  g_signal_connect (G_OBJECT (entry_tel_fax), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_FAX);
  g_signal_connect (G_OBJECT (entry_tel_cell), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_CELL);
  g_signal_connect (G_OBJECT (entry_tel_other), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_TELEPHONE_OTHER);

  return frame;
}


GtkWidget* rubrica_preferences_do_work_page(void)
{ 
  GtkWidget *frame;
  GtkWidget *table;

  GtkWidget *entry_company_name;
  GtkWidget *entry_operator;
  GtkWidget *entry_green;
  GtkWidget *entry_company;
  GtkWidget *entry_customer_care;

  GtkWidget *entry_assigment;
  GtkWidget *entry_organization;
  GtkWidget *entry_sub_dep;
  GtkWidget *entry_secretary_phone;
  GtkWidget *entry_department;
  GtkWidget *entry_secretary;
  GtkWidget *entry_work;


  /*    company frame
  */
  frame = rubrica_ui_frame_new(_("Work dictionary"));


  table = gtk_table_new (5, 4, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 3);
  gtk_widget_show (table);

  rubrica_ui_tabled_label_new(table, _("Company Name"),  0, 1, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Customer care"), 2, 3, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Company"),       0, 1, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Operator"),      2, 3, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Green"),         0, 1, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Assigment"),     2, 3, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Department"),    0, 1, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Secretary"),     2, 3, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Work"),          0, 1, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Organization"),  2, 3, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Sub department"),  0, 1, 5, 6);
  rubrica_ui_tabled_label_new(table, _("Secretary Phone"), 2, 3, 5, 6);

  entry_company_name    = rubrica_ui_tabled_entry_new(table, 1, 2, 0, 1);
  entry_customer_care   = rubrica_ui_tabled_entry_new(table, 3, 4, 0, 1);
  entry_company         = rubrica_ui_tabled_entry_new(table, 1, 2, 1, 2);
  entry_operator        = rubrica_ui_tabled_entry_new(table, 3, 4, 1, 2);
  entry_green           = rubrica_ui_tabled_entry_new(table, 1, 2, 2, 3);
  entry_assigment       = rubrica_ui_tabled_entry_new(table, 3, 4, 2, 3);
  entry_department      = rubrica_ui_tabled_entry_new(table, 1, 2, 3, 4);
  entry_secretary       = rubrica_ui_tabled_entry_new(table, 3, 4, 3, 4);
  entry_work            = rubrica_ui_tabled_entry_new(table, 1, 2, 4, 5);
  entry_organization    = rubrica_ui_tabled_entry_new(table, 3, 4, 4, 5);
  entry_sub_dep         = rubrica_ui_tabled_entry_new(table, 1, 2, 5, 6);
  entry_secretary_phone = rubrica_ui_tabled_entry_new(table, 3, 4, 5, 6);

  g_object_set_data(G_OBJECT(dialog),"entry_company_name", entry_company_name);
  g_object_set_data(G_OBJECT(dialog), "entry_operator", entry_operator);
  g_object_set_data(G_OBJECT(dialog), "entry_green", entry_green);
  g_object_set_data(G_OBJECT(dialog), "entry_company", entry_company);
  g_object_set_data(G_OBJECT(dialog), "entry_customer_care", 
		    entry_customer_care);
  g_object_set_data(G_OBJECT(dialog), "entry_assigment", entry_assigment);
  g_object_set_data(G_OBJECT(dialog),"entry_organization", entry_organization);
  g_object_set_data(G_OBJECT(dialog), "entry_sub_dep", entry_sub_dep);
  g_object_set_data(G_OBJECT(dialog), "entry_secretary_phone",
		    entry_secretary_phone);
  g_object_set_data(G_OBJECT(dialog), "entry_department", entry_department);
  g_object_set_data(G_OBJECT(dialog), "entry_secretary", entry_secretary);
  g_object_set_data(G_OBJECT(dialog), "entry_work", entry_work);


  g_signal_connect (G_OBJECT (entry_company_name), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_COMPANY_NAME);
  g_signal_connect (G_OBJECT (entry_operator), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_OPERATOR);
  g_signal_connect (G_OBJECT (entry_green), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_GREEN);
  g_signal_connect (G_OBJECT (entry_company), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_COMPANY);
  g_signal_connect (G_OBJECT (entry_customer_care), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_CUSTOMER_CARE);
  
  g_signal_connect (G_OBJECT (entry_work), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_WORK);
  g_signal_connect (G_OBJECT (entry_assigment), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_ASSIGMENT);
  g_signal_connect (G_OBJECT (entry_organization), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_ORGANIZATION);
  g_signal_connect (G_OBJECT (entry_department), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_DEPARTMENT);
  g_signal_connect (G_OBJECT (entry_sub_dep), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_SUB_DEPARTMENT);
  g_signal_connect (G_OBJECT (entry_secretary), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_SECRETARY);
  g_signal_connect (G_OBJECT (entry_secretary_phone), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_SECRETARY_PHONE);
  
  return frame;
}


GtkWidget* rubrica_preferences_do_note_page(void)
{ 
  GtkWidget *frame;
  GtkWidget *table;

  GtkWidget *entry_anniversary;
  GtkWidget *entry_spouse_name;
  GtkWidget *entry_married;
  GtkWidget *entry_children;
  GtkWidget *entry_hobby;
  GtkWidget *entry_public_key;
  GtkWidget *entry_notes;
  GtkWidget *entry_child_notes;


  GtkWidget *entry_birthday;
  GtkWidget *entry_other_notes;

  frame = rubrica_ui_frame_new(_("Notes dictionary"));

  table = gtk_table_new (5, 4, FALSE);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 3);
  gtk_widget_show (table);

  rubrica_ui_tabled_label_new(table, _("Spouse Name"),    0, 1, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Birthday"),       0, 1, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Children"),       0, 1, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Hobby"),          0, 1, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Note"),           0, 1, 4, 5);
  rubrica_ui_tabled_label_new(table, _("Married"),        2, 3, 0, 1);
  rubrica_ui_tabled_label_new(table, _("Anniversary"),    2, 3, 1, 2);
  rubrica_ui_tabled_label_new(table, _("Children notes"), 2, 3, 2, 3);
  rubrica_ui_tabled_label_new(table, _("Other Notes"),    2, 3, 3, 4);
  rubrica_ui_tabled_label_new(table, _("Public Key"),     2, 3, 4, 5);

  entry_spouse_name = rubrica_ui_tabled_entry_new(table, 1, 2, 0, 1);
  entry_birthday    = rubrica_ui_tabled_entry_new(table, 1, 2, 1, 2);
  entry_children    = rubrica_ui_tabled_entry_new(table, 1, 2, 2, 3);
  entry_hobby       = rubrica_ui_tabled_entry_new(table, 1, 2, 3, 4);
  entry_notes       = rubrica_ui_tabled_entry_new(table, 1, 2, 4, 5);
  entry_married     = rubrica_ui_tabled_entry_new(table, 3, 4, 0, 1);
  entry_anniversary = rubrica_ui_tabled_entry_new(table, 3, 4, 1, 2);
  entry_child_notes = rubrica_ui_tabled_entry_new(table, 3, 4, 2, 3);
  entry_other_notes = rubrica_ui_tabled_entry_new(table, 3, 4, 3, 4);
  entry_public_key  = rubrica_ui_tabled_entry_new(table, 3, 4, 4, 5);

  g_object_set_data(G_OBJECT(dialog), "entry_anniversary", entry_anniversary);
  g_object_set_data(G_OBJECT(dialog), "entry_spouse_name", entry_spouse_name);
  g_object_set_data(G_OBJECT(dialog), "entry_married",     entry_married);
  g_object_set_data(G_OBJECT(dialog), "entry_children",    entry_children);
  g_object_set_data(G_OBJECT(dialog), "entry_hobby",       entry_hobby);
  g_object_set_data(G_OBJECT(dialog), "entry_public_key",  entry_public_key);
  g_object_set_data(G_OBJECT(dialog), "entry_notes",       entry_notes);
  g_object_set_data(G_OBJECT(dialog), "entry_child_notes", entry_child_notes);
  g_object_set_data(G_OBJECT(dialog), "entry_birthday",    entry_birthday);
  g_object_set_data(G_OBJECT(dialog), "entry_other_notes", entry_other_notes);
  

  g_signal_connect (G_OBJECT (entry_spouse_name), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_SPOUSE_NAME);
  g_signal_connect (G_OBJECT (entry_married), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_MARRIED);
  g_signal_connect (G_OBJECT (entry_anniversary), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_ANNIVERSARY);
  g_signal_connect (G_OBJECT (entry_children), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_CHILDREN);
  g_signal_connect (G_OBJECT (entry_child_notes), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_CHILDREN_NOTES);
  g_signal_connect (G_OBJECT (entry_hobby), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_HOBBY);
  g_signal_connect (G_OBJECT (entry_notes), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_NOTES);
  g_signal_connect (G_OBJECT (entry_public_key), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_PUBLIC_KEY);
  g_signal_connect (G_OBJECT (entry_birthday), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_BIRTHDAY);
  g_signal_connect (G_OBJECT (entry_other_notes), "changed",
		    G_CALLBACK (on_dictionary_entry_changed),
		    RUBRICA_KEY_DICT_OTHER_NOTES);

  return frame;
}
