/****h* ROBODoc/Analyser
 * NAME
 *   Analyser -- Functions to scan source and collect headers
 * FUNCTION
 *   This module provides the functions to scan a sourcefile and
 *   collect all the headers.  The general call sequence is as follows
 *
 *   RB_Analyse_Document
 *   +> RB_Grab_Header
 *      +> RB_Find_Marker
 *         +> RB_Is_Begin_Marker
 *      +> RB_Find_Header_Name
 *      +> RB_Function_Name
 *      +> RB_Module_Name
 *      +> RB_Find_End_Marker
 *         +> RB_Is_End_Marker
 *
 *****
 * $Id: analyser.c,v 1.41 2003/12/30 17:39:36 gumpu Exp $
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>

#include "robodoc.h"
#include "globals.h"
#include "headers.h"
#include "headertypes.h"
#include "items.h"
#include "util.h"
#include "folds.h"
#include "links.h"
#include "analyser.h"
#include "document.h"
#include "file.h"
#include "part.h"


#ifdef DMALLOC
#include <dmalloc.h>
#endif

static int          RB_ToBeAdded( struct RB_Document *document,
                                  struct RB_header *header );
static char        *RB_Find_Header_Name( void );
static struct RB_header *RB_Grab_Header( FILE * sourcehandle );
static char        *RB_Function_Name( char *header_name );
static char        *RB_Module_Name( char *header_name );
static struct RB_HeaderType* RB_Find_Marker( FILE* document, int* is_internal );
static int          RB_Find_End_Marker( FILE * document, struct RB_header *new_header );
struct RB_HeaderType* RB_AnalyseHeaderType( char **cur_char, int *is_internal );


/****f* Analyser/RB_Analyse_Document
 * FUNCTION
 *   Scan all the sourcefiles of all parts of a document for
 *   headers.  Store these headers in each part (RB_Part).
 * SYNOPSIS
 *   void RB_Analyse_Document( struct RB_Document* arg_document )
 * INPUTS
 *   o document -- document to be analysed.
 * RESULT
 *   Each part will contain the headers that were found in the
 *   sourcefile of the part.
 * SOURCE
 */

void
RB_Analyse_Document( struct RB_Document *arg_document )
{
    struct RB_Part     *a_part;
    struct RB_Filename *a_filename;
    FILE               *filehandle;

    for ( a_part = arg_document->parts; a_part; a_part = a_part->next )
    {
        struct RB_header   *new_header = NULL;

        a_filename = a_part->filename;
        RB_Say( "analysing %s\n", RB_Get_Fullname( a_filename ) );
        RB_SetCurrentFile( RB_Get_Fullname( a_filename ) );

        RB_Header_Lock_Reset();
        filehandle = RB_Open_Source( a_part );
        line_number = 0;

        for ( new_header = RB_Grab_Header( filehandle );
              new_header; new_header = RB_Grab_Header( filehandle ) )
        {
            if ( RB_ToBeAdded( arg_document, new_header ) )
            {
                RB_Part_Add_Header( a_part, new_header );
            }
            else
            {
                RB_Free_Header( new_header );
            }
        }
        fclose( filehandle );
    }
}

/*****/


/****f* Analyser/RB_ToBeAdded
 * FUNCTION
 *   Test whether or not a header needs to be added to the
 *   list of headers. This implements the options 
 *      --internal 
 *    and
 *      --internalonly
 * INPUTS
 *   o document  -- a document (to determine the options)
 *   o header    -- a header
 * RESULT
 *   TRUE  -- Add header
 *   FALSE -- Don't add header
 * SOURCE
 */

static int
RB_ToBeAdded( struct RB_Document *document, struct RB_header *header )
{
    int                 add = FALSE;

    if ( header->is_internal )
    {
        if ( ( document->actions & DO_INCLUDE_INTERNAL ) ||
                ( document->actions & DO_INTERNAL_ONLY ) )
        {
            add = TRUE;
        }
        else
        {
            add = FALSE;
        }
    }
    else
    {
        if ( document->actions & DO_INTERNAL_ONLY )
        {
            add = FALSE;
        }
        else
        {
            add = TRUE;
        }
    }
    return add;
}

/******/



/****f* Analyser/RB_Grab_Header
 * FUNCTION
 *   Grab a header from a source file, that is scan a source file
 *   until the start of a header is found.  Then search for the end
 *   of a header and store all the lines in between.
 * SYNPOPSIS
 *   struct RB_header* RB_Grab_Header( FILE * sourcehandle )
 * INPUTS
 *   o sourcehandle -- an opened source file.
 * OUTPUT
 *   o sourcehandle -- will point to the line following the end marker.
 * RESULT
 *   0 if no header was found, or a pointer to a new header otherwise.
 * SOURCE
 */

static struct RB_header *
RB_Grab_Header( FILE * sourcehandle )
{
    struct RB_header   *new_header = NULL;
    int                 is_internal = 0;
    struct RB_HeaderType* header_type = RB_Find_Marker( sourcehandle, &is_internal );

    if ( header_type )
    {
        long                previous_line = 0;

        new_header = RB_Alloc_Header(  );
        new_header->htype = header_type;
        new_header->is_internal = is_internal;
        if ( ( new_header->name = RB_Find_Header_Name(  ) ) != NULL )
        {
            RB_Say( "found header [line %5d]: \"%s\"\n",
                    line_number, new_header->name );
            if ( ( new_header->function_name =
                   RB_Function_Name( new_header->name ) ) == NULL )
            {
                RB_Panic( "Can't determine the \"function\" name.\n" );
            }
            if ( ( new_header->module_name =
                   RB_Module_Name( new_header->name ) ) == NULL )
            {
                RB_Panic( "Can't determine the \"module\" name.\n" );
            }
            previous_line = line_number;
            if ( RB_Find_End_Marker( sourcehandle, new_header ) == 0 )
            {
                RB_Panic( "found header on line %d with name \"%s\""
                          " but I can't find the end marker\n",
                          previous_line, new_header->name );
            }
        }
        else
        {
            RB_Panic( "found header marker but no name\n" );
        }
    }
    return new_header;
}

/*******/

/****f* Analyser/RB_Module_Name
 * FUNCTION
 *   Get the module name from the header name.  The header name will be
 *   something like
 *
 *     module/functionname.
 *
 * SYNPOPSIS
 *   char* RB_Module_Name( char *header_name )
 * INPUTS
 *   o header_name -- a pointer to a nul terminated string.
 * RESULT
 *   Pointer to the modulename.  You're responsible for freeing it.
 * SEE ALSO
 *   RB_Function_Name()
 * SOURCE
 */

static char        *
RB_Module_Name( char *header_name )
{
    char               *cur_char;
    char                c;
    char               *name = NULL;

    assert( header_name );

    for ( cur_char = header_name; *cur_char && *cur_char != '/'; ++cur_char );
    if ( *cur_char )
    {
        c = *cur_char;
        *cur_char = '\0';
        name = RB_StrDup( header_name );
        *cur_char = c;
    }
    return name;
}

/******/



/****f* Analyser/RB_Function_Name
 * FUNCTION
 *   A header name is consists of two parts. The module name and
 *   the function name. This returns a pointer to the function name.
 *   The name "function name" is a bit obsolete. It is really the name
 *   of any of objects that can be documented; classes, methods,
 *   variables, functions, projects, etc.
 * SYNOPSIS
 *   char *RB_NamePart(char *header_name)
 * SOURCE
 */

static char        *
RB_Function_Name( char *header_name )
{
    char               *cur_char;
    char               *name;

    name = NULL;
    if ( ( cur_char = header_name ) != NULL )
    {
        for ( ; *cur_char != '\0'; ++cur_char )
        {
            if ( '/' == *cur_char )
            {
                ++cur_char;
                if ( *cur_char )
                {
                    name = cur_char;
                    break;
                }
            }
        }
    }
    if ( name )
    {
        char               *temp;
        temp = ( char * ) malloc( ( strlen( name ) + 1 ) * sizeof( char ) );
        strcpy( temp, name );
        return temp;
    }
    else
    {
        return ( name );
    }
}

/*** RB_Name_Part ***/


/****f* Analyser/RB_Find_Marker
 * NAME
 *   RB_Find_Marker -- Search for header marker in document.
 * SYNOPSIS
 *   header_type = RB_Find_Marker (document)
 *             int RB_Find_Marker (FILE *)
 * FUNCTION
 *   Read document file line by line, and search each line for 
 *   any of the headers defined in the array  header_markers (OR
 *   if using the -rh switch, robo_head)
 * INPUTS
 *   document - pointer to the file to be searched.
 *   the gobal buffer line_buffer.
 * OUTPUT
 *   o document will point to the line after the line with 
 *     the header marker.
 *   o is_internal will be TRUE if the header is an internal
 *     header.
 * RESULT
 *   o header type
 * BUGS
 *   Bad use of feof(), fgets().
 * SEE ALSO
 *   RB_Find_End_Marker, robo_head, DO_ROBO_HEAD
 * SOURCE
 */

static struct RB_HeaderType*
RB_Find_Marker( FILE* document, int* is_internal )
{
    int                 found;
    char               *cur_char;
    struct RB_HeaderType* header_type = 0;

    cur_char = NULL;
    found = FALSE;
    while ( !feof( document ) && !found )
    {
        *line_buffer = '\0';
        fgets( line_buffer, MAX_LINE_LEN, document );
        if ( !feof( document ) )
        {
            line_number++;
            found = RB_Is_Begin_Marker( line_buffer, &cur_char );
            if ( found )
            {
                header_type = RB_AnalyseHeaderType( &cur_char, is_internal );
                RB_Say( "found header marker of type %s\n", header_type->indexName );
            }
        }
    }

    return header_type;
}

/******** END RB_Find_Marker ******/


/****f* Analyser/RB_AnalyseHeaderType
 * FUNCTION
 *   Determine the type of the header.
 * INPUTS
 *   o cur_char -- pointer to the header type character
 * OUTPUT
 *   o is_internal -- indicates if it is an internal header or not.*
 *   o cur_char -- points to the header type character
 * RESULT
 *   o pointer to a RB_HeaderType
 * SOURCE
 */ 

struct RB_HeaderType* RB_AnalyseHeaderType( char **cur_char, int *is_internal )
{
    struct RB_HeaderType*  headertype = 0;

    *is_internal = RB_IsInternalHeader( **cur_char );

    if ( *is_internal ) 
    {
        /* Skip the character */
        ++( *cur_char );
    }
    headertype = RB_FindHeaderType( **cur_char );
    if ( !headertype )
    {
        RB_Panic( "Undefined headertype (%c)\n", **cur_char );
    }

    return headertype;
}

/*******/



/****f* Analyser/RB_Find_End_Marker
 * FUNCTION
 *   Scan and store all lines from a source file until
 *   an end marker is found.
 * SYNOPSIS
 *   int RB_Find_End_Marker( 
 *       FILE* document, struct RB_header *new_header )
 * INPUTS
 *   o document -- a pointer to an opened source file.
 * OUTPUT
 *   o new_header -- the lines of source code will be added
 *                   here.
 * RESULT
 *   o TRUE  -- an end marker was found.
 *   o FALSE -- no end marker was found while scanning the
 *              source file.
 * SOURCE
 */

static int
RB_Find_End_Marker( FILE * document, struct RB_header *new_header )
{
    int                 found = FALSE;
    unsigned int        no_lines = 0;
    unsigned int        max_no_lines = 10;
    char              **lines = NULL;
    char              **new_lines = NULL;
    char               *dummy;

    lines = ( char ** ) calloc( max_no_lines, sizeof( char * ) );
    while ( fgets( line_buffer, MAX_LINE_LEN, document ) != NULL )
    {
        ++line_number;          /* global linecounter, koessi */
        if ( RB_Is_Begin_Marker( line_buffer, &dummy ) )
        {
            /* Bad... found a begin marker but was expecting to
               find an end marker.  Panic... */
            found = FALSE;
            return found;
        }
        else if ( RB_Is_End_Marker( line_buffer ) )
        {
            found = TRUE;
            break;
        }
        else
        {
            unsigned int        n;
            char               *line;

            line = RB_StrDup( line_buffer );
            n = strlen( line );
            assert( n > 0 );
            assert( line[n - 1] == '\n' );
            /* Strip CR */
            line[n - 1] = '\0';
            lines[no_lines] = line;
            ++no_lines;
            if ( no_lines == max_no_lines )
            {
                max_no_lines *= 2;
                new_lines = realloc( lines, max_no_lines * sizeof( char * ) );
                assert( new_lines );
                lines = new_lines;
            }
        }
    }

    new_header->no_lines = no_lines;
    new_header->lines = lines;

    return found;
}

/******/

/****f* Analyser/RB_Find_Header_Name   [3.0b]
 * FUNCTION
 *   Searches the line buffer for the header name.
 *   It assumes that the header name follows after the
 *   header marker, seperated by one or more spaces, and terminated
 *   by one or more spaces or a '\n'.
 *   It allocates an array of chars and copies the name to this array.
 * SYNOPSIS
 *   result = RB_Find_Header_Name ()
 *      char *RB_Find_Header_Name ()
 * INPUTS
 *   the gobal buffer line_buffer.
 * RESULT
 *   pointer to the allocated array of chars that contains the name,
 *   terminated with a '\0'.
 *   NULL if no header name was found.
 * MODIFICATION HISTORY
 *   8. August 1995      --  optimized by koessi
 * SEE ALSO
 *   RB_WordLen(), RB_StrDup()
 * SOURCE
 */

static char        *
RB_Find_Header_Name( void )
{
    char               *cur_char;

    cur_char = line_buffer;
    skip_while( *cur_char != '*' );
    skip_while( !isspace( *cur_char ) );
    skip_while( isspace( *cur_char ) );
    if ( *cur_char )
    {
        char               *end_char, old_char;

        end_char = cur_char + RB_WordLen( cur_char );
        old_char = *end_char;
        *end_char = '\0';
        cur_char = RB_StrDup( cur_char );
        *end_char = old_char;
        return ( cur_char );
    }
    return ( NULL );
}

/*****  RB_Find_Header_Name  *****/

