/* -*- mode: C; c-basic-offset: 4 -*- */
#include <pygobject.h>
#include "pygnomevfs-private.h"
#include <libgnomevfs/gnome-vfs-utils.h>
#include <pyorbit.h>

static GHashTable *monitor_hash;
static gint monitor_id_counter = 0;

static PyObject *pygnomevfs_exc;
static PyObject *pygnomevfs_not_found_exc;
static PyObject *pygnomevfs_generic_exc;
static PyObject *pygnomevfs_internal_exc;
static PyObject *pygnomevfs_bad_parameters_exc;
static PyObject *pygnomevfs_not_supported_exc;
static PyObject *pygnomevfs_io_exc;
static PyObject *pygnomevfs_corrupted_data_exc;
static PyObject *pygnomevfs_wrong_format_exc;
static PyObject *pygnomevfs_bad_file_exc;
static PyObject *pygnomevfs_too_big_exc;
static PyObject *pygnomevfs_no_space_exc;
static PyObject *pygnomevfs_read_only_exc;
static PyObject *pygnomevfs_invalid_uri_exc;
static PyObject *pygnomevfs_not_open_exc;
static PyObject *pygnomevfs_invalid_open_mode_exc;
static PyObject *pygnomevfs_access_denied_exc;
static PyObject *pygnomevfs_too_many_open_files_exc;
static PyObject *pygnomevfs_eof_exc;
static PyObject *pygnomevfs_not_a_directory_exc;
static PyObject *pygnomevfs_in_progress_exc;
static PyObject *pygnomevfs_interrupted_exc;
static PyObject *pygnomevfs_file_exists_exc;
static PyObject *pygnomevfs_loop_exc;
static PyObject *pygnomevfs_not_permitted_exc;
static PyObject *pygnomevfs_is_directory_exc;
static PyObject *pygnomevfs_no_memory_exc;
static PyObject *pygnomevfs_host_not_found_exc;
static PyObject *pygnomevfs_invalid_host_name_exc;
static PyObject *pygnomevfs_host_has_no_address_exc;
static PyObject *pygnomevfs_login_failed_exc;
static PyObject *pygnomevfs_cancelled_exc;
static PyObject *pygnomevfs_directory_busy_exc;
static PyObject *pygnomevfs_directory_not_empty_exc;
static PyObject *pygnomevfs_too_many_links_exc;
static PyObject *pygnomevfs_read_only_file_system_exc;
static PyObject *pygnomevfs_not_same_file_system_exc;
static PyObject *pygnomevfs_name_too_long_exc;
static PyObject *pygnomevfs_service_not_available_exc;
static PyObject *pygnomevfs_service_obsolete_exc;
static PyObject *pygnomevfs_protocol_error_exc;
static PyObject *pygnomevfs_no_master_browser_exc;
#if 0
static PyObject *pygnomevfs_no_default_exc;
static PyObject *pygnomevfs_no_handler_exc;
static PyObject *pygnomevfs_parse_exc;
static PyObject *pygnomevfs_launch_exc;
#endif

gboolean
pygnome_vfs_result_check(GnomeVFSResult result)
{
    switch(result)
    {
    case GNOME_VFS_OK:
	return FALSE;
    case GNOME_VFS_ERROR_NOT_FOUND:
	PyErr_SetObject(pygnomevfs_not_found_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_GENERIC:
	PyErr_SetObject(pygnomevfs_generic_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INTERNAL:
	PyErr_SetObject(pygnomevfs_internal_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_BAD_PARAMETERS:
	PyErr_SetObject(pygnomevfs_bad_parameters_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_SUPPORTED:
	PyErr_SetObject(pygnomevfs_not_supported_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IO:
	PyErr_SetObject(pygnomevfs_io_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_CORRUPTED_DATA:
	PyErr_SetObject(pygnomevfs_corrupted_data_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_WRONG_FORMAT:
	PyErr_SetObject(pygnomevfs_wrong_format_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_BAD_FILE:
	PyErr_SetObject(pygnomevfs_bad_file_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_BIG:
	PyErr_SetObject(pygnomevfs_too_big_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_SPACE:
	PyErr_SetObject(pygnomevfs_no_space_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_READ_ONLY:
	PyErr_SetObject(pygnomevfs_read_only_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_URI:
	PyErr_SetObject(pygnomevfs_invalid_uri_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_OPEN:
	PyErr_SetObject(pygnomevfs_not_open_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	PyErr_SetObject(pygnomevfs_invalid_open_mode_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_ACCESS_DENIED:
	PyErr_SetObject(pygnomevfs_access_denied_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	PyErr_SetObject(pygnomevfs_too_many_open_files_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_EOF:
	PyErr_SetObject(pygnomevfs_eof_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	PyErr_SetObject(pygnomevfs_not_a_directory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IN_PROGRESS:
	PyErr_SetObject(pygnomevfs_in_progress_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INTERRUPTED:
	PyErr_SetObject(pygnomevfs_interrupted_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_FILE_EXISTS:
	PyErr_SetObject(pygnomevfs_file_exists_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LOOP:
	PyErr_SetObject(pygnomevfs_loop_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_PERMITTED:
	PyErr_SetObject(pygnomevfs_not_permitted_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IS_DIRECTORY:
	PyErr_SetObject(pygnomevfs_is_directory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_MEMORY:
	PyErr_SetObject(pygnomevfs_no_memory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	PyErr_SetObject(pygnomevfs_host_not_found_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	PyErr_SetObject(pygnomevfs_invalid_host_name_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	PyErr_SetObject(pygnomevfs_host_has_no_address_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LOGIN_FAILED:
	PyErr_SetObject(pygnomevfs_login_failed_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_CANCELLED:
	PyErr_SetObject(pygnomevfs_cancelled_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	PyErr_SetObject(pygnomevfs_directory_busy_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	PyErr_SetObject(pygnomevfs_directory_not_empty_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	PyErr_SetObject(pygnomevfs_too_many_links_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	PyErr_SetObject(pygnomevfs_read_only_file_system_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	PyErr_SetObject(pygnomevfs_not_same_file_system_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NAME_TOO_LONG:
	PyErr_SetObject(pygnomevfs_name_too_long_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	PyErr_SetObject(pygnomevfs_service_not_available_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	PyErr_SetObject(pygnomevfs_service_obsolete_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	PyErr_SetObject(pygnomevfs_protocol_error_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	PyErr_SetObject(pygnomevfs_no_master_browser_exc, NULL);
	return TRUE;
#if 0	
    case GNOME_VFS_ERROR_NO_DEFAULT:
	PyErr_SetObject(pygnomevfs_no_default_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_HANDLER:
	PyErr_SetObject(pygnomevfs_no_handler_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_PARSE:
	PyErr_SetObject(pygnomevfs_parse_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LAUNCH:
	PyErr_SetObject(pygnomevfs_launch_exc, NULL);
	return TRUE;
#endif	
    default:
	return FALSE;
    }
}

GnomeVFSResult pygnome_vfs_exception_check(void)
{
    if (!PyErr_Occurred()) {
	return -1;
    }
    
    if (PyErr_ExceptionMatches(pygnomevfs_not_found_exc)) {
	return GNOME_VFS_ERROR_NOT_FOUND;
    } else if (PyErr_ExceptionMatches(pygnomevfs_generic_exc)) {
	return GNOME_VFS_ERROR_GENERIC;
    } else if (PyErr_ExceptionMatches(pygnomevfs_internal_exc)) {
	return GNOME_VFS_ERROR_INTERNAL;
    } else if (PyErr_ExceptionMatches(pygnomevfs_bad_parameters_exc)) {
	return GNOME_VFS_ERROR_BAD_FILE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_supported_exc)) {
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_io_exc)) {
	return GNOME_VFS_ERROR_IO;
    } else if (PyErr_ExceptionMatches(pygnomevfs_corrupted_data_exc)) {
	return GNOME_VFS_ERROR_CORRUPTED_DATA;
    } else if (PyErr_ExceptionMatches(pygnomevfs_wrong_format_exc)) {
	return GNOME_VFS_ERROR_WRONG_FORMAT;
    } else if (PyErr_ExceptionMatches(pygnomevfs_bad_file_exc)) {
	return GNOME_VFS_ERROR_BAD_FILE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_big_exc)) {
	return GNOME_VFS_ERROR_TOO_BIG;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_space_exc)) {
	return GNOME_VFS_ERROR_NO_SPACE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_read_only_exc)) {
	return GNOME_VFS_ERROR_READ_ONLY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_uri_exc)) {
	return GNOME_VFS_ERROR_INVALID_URI;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_open_exc)) {
	return GNOME_VFS_ERROR_NOT_OPEN;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_open_mode_exc)) {
	return GNOME_VFS_ERROR_INVALID_OPEN_MODE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_access_denied_exc)) {
	return GNOME_VFS_ERROR_ACCESS_DENIED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_many_open_files_exc)) {
	return GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES;
    } else if (PyErr_ExceptionMatches(pygnomevfs_eof_exc)) {
	return GNOME_VFS_ERROR_EOF;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_a_directory_exc)) {
	return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_in_progress_exc)) {
	return GNOME_VFS_ERROR_IN_PROGRESS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_interrupted_exc)) {
	return GNOME_VFS_ERROR_INTERRUPTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_file_exists_exc)) {
	return GNOME_VFS_ERROR_FILE_EXISTS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_loop_exc)) {
	return GNOME_VFS_ERROR_LOOP;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_permitted_exc)) {
	return GNOME_VFS_ERROR_NOT_PERMITTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_is_directory_exc)) {
	return GNOME_VFS_ERROR_IS_DIRECTORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_memory_exc)) {
	return GNOME_VFS_ERROR_NO_MEMORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_host_not_found_exc)) {
	return GNOME_VFS_ERROR_HOST_NOT_FOUND;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_host_name_exc)) {
	return GNOME_VFS_ERROR_INVALID_HOST_NAME;
    } else if (PyErr_ExceptionMatches(pygnomevfs_host_has_no_address_exc)) {
	return GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_login_failed_exc)) {
	return GNOME_VFS_ERROR_LOGIN_FAILED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_cancelled_exc)) {
	return GNOME_VFS_ERROR_CANCELLED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_directory_busy_exc)) {
	return GNOME_VFS_ERROR_DIRECTORY_BUSY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_directory_not_empty_exc)) {
	return GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_many_links_exc)) {
	return GNOME_VFS_ERROR_TOO_MANY_LINKS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_read_only_file_system_exc)) {
	return GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_same_file_system_exc)) {
	return GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM;
    } else if (PyErr_ExceptionMatches(pygnomevfs_name_too_long_exc)) {
	return GNOME_VFS_ERROR_NAME_TOO_LONG;
    } else if (PyErr_ExceptionMatches(pygnomevfs_service_not_available_exc)) {
	return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_service_obsolete_exc)) {
	return GNOME_VFS_ERROR_SERVICE_OBSOLETE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_protocol_error_exc)) {
	return GNOME_VFS_ERROR_PROTOCOL_ERROR;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_master_browser_exc)) {
	return GNOME_VFS_ERROR_NO_MASTER_BROWSER;
    }
    
    return -2;
}

static PyObject *
pygvfs_create(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "open_mode", "exclusive", "perm", NULL };
    PyObject *uri;
    GnomeVFSOpenMode open_mode = GNOME_VFS_OPEN_NONE;
    gboolean exclusive = FALSE;
    guint perm = 0666;
    GnomeVFSHandle *handle;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O|iii:gnome.vfs.create",
				     kwlist, &uri, &open_mode, &exclusive,
				     &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_create_uri(&handle, pygnome_vfs_uri_get(uri),
				      open_mode, exclusive, perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_create(&handle, PyString_AsString(uri),
				  open_mode, exclusive, perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    return pygnome_vfs_handle_new(handle);
}

static PyObject *
pygvfs_get_file_info(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "options", NULL };
    PyObject *uri;
    GnomeVFSFileInfo *finfo;
    GnomeVFSFileInfoOptions options = GNOME_VFS_FILE_INFO_DEFAULT;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O|i:gnome.vfs.get_file_info",
				     kwlist, &uri, &options))
	return NULL;

    finfo = gnome_vfs_file_info_new();
    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_get_file_info_uri(pygnome_vfs_uri_get(uri), finfo,
					     options);
    else if (PyString_Check(uri))
	result = gnome_vfs_get_file_info(PyString_AsString(uri), finfo,
					 options);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }

    if (pygnome_vfs_result_check(result)) {
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }
    return pygnome_vfs_file_info_new(finfo);
}

static PyObject *
pygvfs_set_file_info(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "info", "mask", NULL };
    PyObject *uri;
    PyGnomeVFSFileInfo *finfo;
    GnomeVFSSetFileInfoMask mask = GNOME_VFS_SET_FILE_INFO_NONE;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "OO!i:gnome.vfs.set_file_info",
				     kwlist, &uri,
                                     &PyGnomeVFSFileInfo_Type, &finfo,
                                     &mask))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_set_file_info_uri(pygnome_vfs_uri_get(uri),
                                             finfo->finfo, mask);
    else if (PyString_Check(uri))
	result = gnome_vfs_set_file_info(PyString_AsString(uri),
                                         finfo->finfo, mask);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }
    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_truncate(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "length", NULL };
    PyObject *uri, *py_length;
    GnomeVFSFileSize length;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "OO:gnome.vfs.truncate",
				     kwlist, &uri, &py_length))
	return NULL;

    length = PyLong_Check(py_length) ? PyLong_AsUnsignedLongLong(py_length)
	: PyInt_AsLong(py_length);
    if (PyErr_Occurred()) return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_truncate_uri(pygnome_vfs_uri_get(uri), length);
    else if (PyString_Check(uri))
	result = gnome_vfs_truncate(PyString_AsString(uri), length);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_make_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "perm", NULL };
    PyObject *uri;
    gint perm;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "Oi:gnome.vfs.make_directory", kwlist,
				     &uri, &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_make_directory_for_uri(pygnome_vfs_uri_get(uri),
						  perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_make_directory(PyString_AsString(uri), perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_remove_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.remove_directory", kwlist,
				     &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_remove_directory_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_remove_directory(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_unlink(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.unlink", kwlist, &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_unlink_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_unlink(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_exists(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *py_uri;
    GnomeVFSURI *uri = NULL;
    gboolean exists;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O:gnome.vfs.exists",
				     kwlist, &py_uri))
	return NULL;

    if (PyObject_TypeCheck(py_uri, &PyGnomeVFSURI_Type))
	uri = gnome_vfs_uri_ref(pygnome_vfs_uri_get(py_uri));
    else if (PyString_Check(py_uri))
	uri = gnome_vfs_uri_new(PyString_AsString(py_uri));

    if (!uri) {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }
    exists = gnome_vfs_uri_exists(uri);
    gnome_vfs_uri_unref(uri);

    return PyInt_FromLong(exists);
}

static PyObject *
pygvfs_get_mime_type(PyObject *self, PyObject *args)
{
    char *text_uri, *mime;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.get_mime_type",
			 &text_uri))
	return NULL;

    mime = gnome_vfs_get_mime_type(text_uri);
    if (mime)
	return PyString_FromString(mime);
    else {
	PyErr_SetString(PyExc_RuntimeError,
			"there was an error reading the file");
	return NULL;
    }
}

static PyObject *
pygvfs_get_mime_type_for_data(PyObject *self, PyObject *args)
{
    char *data;
    char const *mime;
    int data_size, data_size1 = -G_MAXINT;
    
    if(!PyArg_ParseTuple(args, "s#|i:gnome.vfs.get_mime_type_for_data",
			 &data, &data_size, &data_size1))
	return NULL;
    if (data_size1 != -G_MAXINT)
	PyErr_Warn(PyExc_DeprecationWarning, "ignoring deprecated argument data_size");
    mime = gnome_vfs_get_mime_type_for_data(data, data_size);
    if (mime)
	return PyString_FromString(mime);
    else {
	PyErr_SetString(PyExc_RuntimeError,
			"there was an error reading the file");
	return NULL;
    }
}

static PyObject *
pygvfs_mime_get_icon(PyObject *self, PyObject *args)
{
    char *mime_type;
    const char *retval;
        
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_get_icon",
			 &mime_type))
	return NULL;
        
    retval = gnome_vfs_mime_get_icon(mime_type);
    if (retval == NULL) {
            Py_INCREF(Py_None);
            return Py_None;
    }
    return PyString_FromString(retval);
}

static PyObject *
pygvfs_mime_set_icon(PyObject *self, PyObject *args)
{
    char *mime_type, *filename;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "ss:gnome.vfs.mime_set_icon",
			 &mime_type, &filename))
	return NULL;

    result = gnome_vfs_mime_set_icon(mime_type, filename);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_get_description(PyObject *self, PyObject *args)
{
    char *mime_type;
    const char *description;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_get_description",
			 &mime_type))
	return NULL;
    description = gnome_vfs_mime_get_description(mime_type);
    if (description)
	return PyString_FromString(description);
    else {
        Py_INCREF(Py_None);
        return Py_None;
    }
}

static PyObject *
pygvfs_mime_set_description(PyObject *self, PyObject *args)
{
    char *mime_type, *description;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "ss:gnome.vfs.mime_set_description",
			 &mime_type, &description))
	return NULL;

    result = gnome_vfs_mime_set_description(mime_type, description);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_can_be_executable(PyObject *self, PyObject *args)
{
    char *mime_type;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_can_be_executable",
			 &mime_type))
	return NULL;

    return PyInt_FromLong(gnome_vfs_mime_can_be_executable(mime_type));
}

static PyObject *
pygvfs_mime_set_can_be_executable(PyObject *self, PyObject *args)
{
    char *mime_type;
    gboolean new_value;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "si:gnome.vfs.mime_set_description",
			 &mime_type, &new_value))
	return NULL;

    result = gnome_vfs_mime_set_can_be_executable(mime_type, new_value);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static void
pygvfs_monitor_marshal(GnomeVFSMonitorHandle *handle,
		       const gchar *monitor_uri,
		       const gchar *info_uri,
		       GnomeVFSMonitorEventType event_type,
		       PyGVFSCustomNotify *cunote)
{
    PyObject *retobj;
    PyGILState_STATE state;
    
    state = pyg_gil_state_ensure();

    if (cunote->data)
	retobj = PyEval_CallFunction(cunote->func, "(ssiO)", monitor_uri,
				     info_uri, event_type, cunote->data);
    else
	retobj = PyObject_CallFunction(cunote->func, "(ssi)", monitor_uri,
				       info_uri, event_type);

    if (retobj == NULL) {
	PyErr_Print();
	PyErr_Clear();
    }
    
    Py_XDECREF(retobj);
    
    pyg_gil_state_release(state);
}

static PyObject*
pygvfs_monitor_add(PyObject *self, PyObject *args)
{
    char *text_uri;
    int monitor_type;
    PyObject *callback;
    PyObject *extra = NULL;
    PyGVFSCustomNotify *cunote;
    GnomeVFSMonitorHandle *handle;
    GnomeVFSResult result;
    gint monitor_id;
    
    if (!PyArg_ParseTuple(args, "siO|O:gnome.vfs.monitor_add",
			  &text_uri, &monitor_type,
			  &callback, &extra)) {
	return NULL;
    }
    
    if (!PyCallable_Check(callback)) {
        PyErr_SetString(PyExc_TypeError, "third argument not callable");
        return NULL;
    }
    
    cunote = g_new0(PyGVFSCustomNotify, 1);
    cunote->func = callback;
    cunote->data = extra;
    Py_INCREF(cunote->func);
    Py_XINCREF(cunote->data);
    result = gnome_vfs_monitor_add(&handle, text_uri, monitor_type,
		   (GnomeVFSMonitorCallback)pygvfs_monitor_marshal,
		   cunote);

    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }

    do
        monitor_id = ++monitor_id_counter;
    while (g_hash_table_lookup(monitor_hash, GINT_TO_POINTER(monitor_id)));

    g_hash_table_insert(monitor_hash,
			GINT_TO_POINTER(monitor_id),
			handle);
    
    return PyInt_FromLong(monitor_id);
}

static PyObject*
pygvfs_monitor_cancel(PyObject *self, PyObject *args)
{
    gint monitor_id;
    GnomeVFSMonitorHandle *handle;
    
    if (!PyArg_ParseTuple(args, "i:gnome.vfs.monitor_cancel",
			  &monitor_id)) {
	return NULL;
    }
    
    handle = g_hash_table_lookup(monitor_hash, GINT_TO_POINTER(monitor_id));
    if (handle == NULL) {
	PyErr_SetString(PyExc_ValueError, "Invalid monitor id");
	return NULL;
    }

    gnome_vfs_monitor_cancel(handle);
    g_hash_table_remove(monitor_hash, GINT_TO_POINTER(monitor_id));
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_read_entire_file(PyObject *self, PyObject *args)
{
    GnomeVFSResult  result;
    char           *uri;
    char           *file_contents;
    int             file_size;
    PyObject       *rv;

    if (!PyArg_ParseTuple(args, "s:gnome.vfs.read_entire_file", &uri))
	return NULL;
    result = gnome_vfs_read_entire_file (uri, &file_size, &file_contents);
    if (pygnome_vfs_result_check(result))
	return NULL;
    rv = PyString_FromStringAndSize(file_contents, file_size);
    g_free(file_contents);
    return rv;
}

static PyObject *
pygvfs_mime_application_new(GnomeVFSMimeApplication *mimeapp)
{
    PyObject *uri_schemes;
    GList *l;
    int i;

    uri_schemes = PyList_New(g_list_length(mimeapp->supported_uri_schemes));
    for (i = 0, l = mimeapp->supported_uri_schemes; l; ++i, l = l->next)
	PyList_SET_ITEM(uri_schemes, i, PyString_FromString((const char *) l->data));

    return Py_BuildValue("sssOONO", mimeapp->id, mimeapp->name,
			 mimeapp->command,
			 mimeapp->can_open_multiple_files? Py_True : Py_False, 
			 mimeapp->expects_uris? Py_True : Py_False,
			 uri_schemes,
			 mimeapp->requires_terminal? Py_True : Py_False);
}

static PyObject *
pygvfs_mime_get_default_application(PyObject *self, PyObject *args)
{
    const char *mime_type;
    GnomeVFSMimeApplication *mimeapp;
    PyObject *retval;

    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_get_default_application",
			 &mime_type))
	return NULL;
    if (!(mimeapp = gnome_vfs_mime_get_default_application(mime_type))) {
        Py_INCREF(Py_None);
        return Py_None;
    }
    retval = pygvfs_mime_application_new(mimeapp);
    gnome_vfs_mime_application_free(mimeapp);
    return retval;
}


static gint
pygvfs_xfer_progress_callback(GnomeVFSXferProgressInfo *info, gpointer _data)
{
    PyGVFSCustomNotify *data = _data;
    PyObject *py_info, *callback_return;
    gint retval;

    py_info = pygnome_vfs_xfer_progress_info_new(info);
    if (data->data)
	callback_return = PyObject_CallFunction(data->func, "OO", py_info, data->data);
    else
	callback_return = PyObject_CallFunction(data->func, "O", py_info);

      /* because the programmer may keep a reference to the
       * VFSXferProgressInfo python wrapper but we don't own the
       * VFSXferProgressInfo itself, we remove the link from the
       * python wrapper to the C structure */
    pygnome_vfs_xfer_progress_info_set(py_info, NULL);

    if (callback_return == NULL)
	return GNOME_VFS_XFER_ERROR_ACTION_ABORT;
    if (!PyInt_Check(callback_return)) {
	PyErr_SetString(PyExc_TypeError, "progress callback must return an int");
	return GNOME_VFS_XFER_ERROR_ACTION_ABORT;
    }
    retval = PyInt_AsLong(callback_return);
    Py_DECREF(callback_return);
    return retval;
}

static PyObject *
pygvfs_xfer_uri(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "source_uri", "target_uri", "xfer_options",
			      "error_mode", "overwrite_mode",
			      "progress_callback", "data", NULL };
    PyObject *source_uri, *target_uri;
    int xfer_options = -1, error_mode = -1, overwrite_mode = -1;
    PyGVFSCustomNotify custom_data = {NULL, NULL};
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O!O!iiiO|O:gnome.vfs.xfer_uri",
				     kwlist,
				     &PyGnomeVFSURI_Type, &source_uri,
				     &PyGnomeVFSURI_Type, &target_uri,
				     &xfer_options, &error_mode, &overwrite_mode,
				     &custom_data.func, &custom_data.data))
	return NULL;

    if (!PyCallable_Check(custom_data.func)) {
	PyErr_SetString(PyExc_TypeError, "progress_callback must be callable");
	return NULL;
    }

    result = gnome_vfs_xfer_uri(pygnome_vfs_uri_get(source_uri),
				pygnome_vfs_uri_get(target_uri),
				xfer_options,  error_mode, overwrite_mode,
				pygvfs_xfer_progress_callback, &custom_data);
    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static gboolean
_pygvfs_uri_sequence_to_glist(PyObject *seq, GList **list)
{
    int len, i;
    PyObject *item;

    if (!PySequence_Check(seq))
	return FALSE;
    *list = NULL;
    len = PySequence_Length(seq);
    for (i = 0; i < len; ++i) {
	item = PySequence_GetItem(seq, i);
	if (!PyObject_TypeCheck(item, &PyGnomeVFSURI_Type)) {
	    Py_DECREF(item);
	    if (*list)
		g_list_free(*list);
	    return FALSE;
	}
	*list = g_list_append(*list, pygnome_vfs_uri_get(item));
	Py_DECREF(item);
    }
    return TRUE;
}

static PyObject *
pygvfs_xfer_uri_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "source_uri_list", "target_uri_list", "xfer_options",
			      "error_mode", "overwrite_mode",
			      "progress_callback", "data", NULL };
    PyObject *py_source_uri_list, *py_target_uri_list;
    GList *source_uri_list = NULL, *target_uri_list = NULL;
    int xfer_options = -1, error_mode = -1, overwrite_mode = -1;
    PyGVFSCustomNotify custom_data = {NULL, NULL};
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "OOiiiO|O:gnome.vfs.xfer_uri_list",
				     kwlist,
				     &py_source_uri_list, &py_target_uri_list,
				     &xfer_options, &error_mode, &overwrite_mode,
				     &custom_data.func, &custom_data.data))
	return NULL;

    if (!_pygvfs_uri_sequence_to_glist(py_source_uri_list, &source_uri_list)) {
	PyErr_SetString(PyExc_TypeError, "source_uri_list "
			" must be a sequence of gnome.vfs.URI");
	return NULL;
    }
    if (!_pygvfs_uri_sequence_to_glist(py_target_uri_list, &target_uri_list)) {
	PyErr_SetString(PyExc_TypeError, "target_uri_list "
			" must be a sequence of gnome.vfs.URI");
	g_list_free(source_uri_list);
	return NULL;
    }


    if (!PyCallable_Check(custom_data.func)) {
	PyErr_SetString(PyExc_TypeError, "progress_callback must be callable");
	g_list_free(source_uri_list);
	g_list_free(target_uri_list);
	return NULL;
    }

    result = gnome_vfs_xfer_uri_list(source_uri_list, target_uri_list,
				     xfer_options,  error_mode, overwrite_mode,
				     pygvfs_xfer_progress_callback, &custom_data);
    g_list_free(source_uri_list);
    g_list_free(target_uri_list);
    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_xfer_delete_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "delete_uri_list",
			      "xfer_options", "error_mode",
			      "progress_callback", "data", NULL };
    PyObject *py_delete_uri_list;
    GList *delete_uri_list = NULL;
    int xfer_options = -1, error_mode = -1;
    PyGVFSCustomNotify custom_data = {NULL, NULL};
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "OiiO|O:gnome.vfs.xfer_delete_list",
				     kwlist,
				     &py_delete_uri_list,
				     &error_mode, &xfer_options,
				     &custom_data.func, &custom_data.data))
	return NULL;

    if (!_pygvfs_uri_sequence_to_glist(py_delete_uri_list, &delete_uri_list)) {
	PyErr_SetString(PyExc_TypeError, "delete_uri_list "
			" must be a sequence of gnome.vfs.URI");
	return NULL;
    }


    if (!PyCallable_Check(custom_data.func)) {
	PyErr_SetString(PyExc_TypeError, "progress_callback must be callable");
	g_list_free(delete_uri_list);
	return NULL;
    }

    result = gnome_vfs_xfer_delete_list(delete_uri_list,
					error_mode, xfer_options,
					pygvfs_xfer_progress_callback, &custom_data);
    g_list_free(delete_uri_list);
    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_get_default_action_type(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "s:gnome.vfs.mime_get_default_action_type",
				     kwlist, &mime_type))
	return NULL;
    return PyInt_FromLong(gnome_vfs_mime_get_default_action_type(mime_type));
}

static PyObject *
pygvfs_mime_action_new(GnomeVFSMimeAction *action)
{
    switch (action->action_type)
    {
    case GNOME_VFS_MIME_ACTION_TYPE_NONE:
	return Py_BuildValue("(iO)", action->action_type, Py_None);

    case GNOME_VFS_MIME_ACTION_TYPE_APPLICATION:
	return Py_BuildValue("(iN)", action->action_type,
			     pygvfs_mime_application_new(action->action.application));

    case GNOME_VFS_MIME_ACTION_TYPE_COMPONENT: {
	CORBA_any any;
	PyObject *component;
	any._type = TC_Bonobo_ServerInfo;
	any._value = action->action.application;
	component = pyorbit_demarshal_any(&any);
	if (!component) {
	    PyErr_SetString(PyExc_TypeError, "unable to convert Bonobo_ServerInfo of component");
	    return NULL;
	}
	return Py_BuildValue("(iN)", action->action_type, component);
    }
    default:
	PyErr_SetString(PyExc_ValueError, "unknown action type returned");
	return NULL;
    }
}

static PyObject *
pygvfs_mime_get_default_action(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    GnomeVFSMimeAction *action;
    PyObject *retval;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "s:gnome.vfs.mime_get_default_action",
				     kwlist, &mime_type))
	return NULL;
    if (!(action = gnome_vfs_mime_get_default_action(mime_type))) {
        Py_INCREF(Py_None);
        return Py_None;
    }
    retval = pygvfs_mime_action_new(action);
    gnome_vfs_mime_action_free(action);
    return retval;
}

static PyObject *
pygvfs_mime_get_default_component(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    Bonobo_ServerInfo *component;
    PyObject *py_component;
    CORBA_any any;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "s:gnome.vfs.mime_get_default_component",
				     kwlist, &mime_type))
	return NULL;
    if (!(component = gnome_vfs_mime_get_default_component(mime_type))) {
        Py_INCREF(Py_None);
        return Py_None;
    }
    any._type = TC_Bonobo_ServerInfo;
    any._value = component;
    py_component = pyorbit_demarshal_any(&any);
    if (!component) {
	PyErr_SetString(PyExc_TypeError, "unable to convert Bonobo_ServerInfo of component");
	return NULL;
    }
    CORBA_free(component);
    return py_component;
}

static PyObject *
pygvfs_mime_components_list_new(GList *list)
{
    PyObject *retval;
    Bonobo_ServerInfo *component;
    PyObject *py_component;
    CORBA_any any;
    guint i, len = g_list_length(list);
    
    retval = PyList_New(len);
    for (i = 0; list; ++i, list = list->next) {
	g_assert(i < len);
	component = (Bonobo_ServerInfo *) list->data;
	any._value = component;
	py_component = pyorbit_demarshal_any(&any);
	if (!component) {
	    PyErr_SetString(PyExc_TypeError, "unable to convert Bonobo_ServerInfo of component");
	    Py_DECREF(retval);
	    return NULL;
	}
	PyList_SET_ITEM(retval, i, py_component);
    }
    return retval;
}

static PyObject *
pygvfs_mime_get_short_list_components(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    GList *list;
    PyObject *py_list;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "s:gnome.vfs.mime_get_short_list_components",
				     kwlist, &mime_type))
	return NULL;
    list = gnome_vfs_mime_get_short_list_components(mime_type);
    py_list = pygvfs_mime_components_list_new(list);
    gnome_vfs_mime_component_list_free(list);
    return py_list;
}

static PyObject *
pygvfs_mime_get_all_components(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    GList *list;
    PyObject *py_list;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "s:gnome.vfs.mime_get_all_components",
				     kwlist, &mime_type))
	return NULL;
    list = gnome_vfs_mime_get_all_components(mime_type);
    py_list = pygvfs_mime_components_list_new(list);
    gnome_vfs_mime_component_list_free(list);
    return py_list;
}

static PyObject *
pygvfs_mime_applications_list_new(GList *list)
{
    PyObject *retval;
    PyObject *py_app;
    guint i, len = g_list_length(list);
    
    retval = PyList_New(len);
    for (i = 0; list; ++i, list = list->next) {
	g_assert(i < len);
	py_app = pygvfs_mime_application_new((GnomeVFSMimeApplication *) list->data);
	PyList_SET_ITEM(retval, i, py_app);
    }
    return retval;
}

static PyObject *
pygvfs_mime_get_short_list_applications(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    GList *list;
    PyObject *py_list;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "s:gnome.vfs.mime_get_short_list_applications",
				     kwlist, &mime_type))
	return NULL;
    list = gnome_vfs_mime_get_short_list_applications(mime_type);
    py_list = pygvfs_mime_applications_list_new(list);
    gnome_vfs_mime_application_list_free(list);
    return py_list;
}

static PyObject *
pygvfs_mime_get_all_applications(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", NULL };
    char *mime_type;
    GList *list;
    PyObject *py_list;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "s:gnome.vfs.mime_get_all_applications",
				     kwlist, &mime_type))
	return NULL;
    list = gnome_vfs_mime_get_all_applications(mime_type);
    py_list = pygvfs_mime_applications_list_new(list);
    gnome_vfs_mime_application_list_free(list);
    return py_list;
}

static PyObject *
pygvfs_mime_set_default_action_type(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "action_type", NULL };
    char *mime_type;
    GnomeVFSMimeActionType action_type;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "si:gnome.vfs.mime_set_default_action_type",
				     kwlist, &mime_type, &action_type))
	return NULL;
    result = gnome_vfs_mime_set_default_action_type(mime_type, action_type);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_set_default_application(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_id", NULL };
    char *mime_type, *application_id;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_set_default_application",
				     kwlist, &mime_type, &application_id))
	return NULL;
    result = gnome_vfs_mime_set_default_application(mime_type, application_id);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_set_default_component(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "component_iid", NULL };
    char *mime_type, *component_iid;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_set_default_component",
				     kwlist, &mime_type, &component_iid))
	return NULL;
    result = gnome_vfs_mime_set_default_component(mime_type, component_iid);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static int
string_list_converter(PyObject *in, void *out)
{
    guint len, i;
    GList *list = NULL;
    PyObject *item;

    if (!PySequence_Check(in)) {
	PyErr_SetString(PyExc_TypeError, "argument must be a sequence");
	return 0;
    }
    len = PySequence_Length(in);
    for (i = 0; i < len; ++i) {
	item = PySequence_GetItem(in, i);
	if (!PyString_Check(item)) {
	    Py_DECREF(item);
	    g_list_free(list);
	    return 0;
	}
	list = g_list_append(list, PyString_AsString(item));
	Py_DECREF(item);
    }
    *((GList **) out) = list;
    return 1;
}

static PyObject *
pygvfs_mime_set_short_list_applications(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_ids", NULL };
    char *mime_type;
    GList *application_ids;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "sO&:gnome.vfs.mime_set_short_list_applications",
				     kwlist, &mime_type,
				     string_list_converter, &application_ids))
	return NULL;
    result = gnome_vfs_mime_set_short_list_applications(mime_type, application_ids);
    g_list_free(application_ids);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_set_short_list_components(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "component_iids", NULL };
    char *mime_type;
    GList *component_iids;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "sO&:gnome.vfs.mime_set_short_list_components",
				     kwlist, &mime_type,
				     string_list_converter, &component_iids))
	return NULL;
    result = gnome_vfs_mime_set_short_list_components(mime_type, component_iids);
    g_list_free(component_iids);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *
pygvfs_mime_add_application_to_short_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_id", NULL };
    char *mime_type, *application_id;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_add_application_to_short_list",
				     kwlist, &mime_type, &application_id))
	return NULL;
    result = gnome_vfs_mime_add_application_to_short_list(mime_type, application_id);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_remove_application_from_short_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_id", NULL };
    char *mime_type, *application_id;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_remove_application_from_short_list",
				     kwlist, &mime_type, &application_id))
	return NULL;
    result = gnome_vfs_mime_remove_application_from_short_list(mime_type, application_id);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_add_component_to_short_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "component_iid", NULL };
    char *mime_type, *component_iid;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_add_component_to_short_list",
				     kwlist, &mime_type, &component_iid))
	return NULL;
    result = gnome_vfs_mime_add_component_to_short_list(mime_type, component_iid);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_remove_component_from_short_list(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "component_iid", NULL };
    char *mime_type, *component_iid;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_remove_component_from_short_list",
				     kwlist, &mime_type, &component_iid))
	return NULL;
    result = gnome_vfs_mime_remove_component_from_short_list(mime_type, component_iid);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_add_extension(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "extension", NULL };
    char *mime_type, *extension;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_add_extension",
				     kwlist, &mime_type, &extension))
	return NULL;
    result = gnome_vfs_mime_add_extension(mime_type, extension);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_remove_extension(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "extension", NULL };
    char *mime_type, *extension;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "ss:gnome.vfs.mime_remove_extension",
				     kwlist, &mime_type, &extension))
	return NULL;
    result = gnome_vfs_mime_remove_extension(mime_type, extension);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_extend_all_applications(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_ids", NULL };
    char *mime_type;
    GList *application_ids;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "sO&:gnome.vfs.mime_extend_all_applications",
				     kwlist, &mime_type,
				     string_list_converter, &application_ids))
	return NULL;
    result = gnome_vfs_mime_extend_all_applications(mime_type, application_ids);
    g_list_free(application_ids);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_remove_from_all_applications(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "mime_type", "application_ids", NULL };
    char *mime_type;
    GList *application_ids;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "sO&:gnome.vfs.mime_remove_from_all_applications",
				     kwlist, &mime_type,
				     string_list_converter, &application_ids))
	return NULL;
    result = gnome_vfs_mime_remove_from_all_applications(mime_type, application_ids);
    g_list_free(application_ids);
    if (pygnome_vfs_result_check(result))
	return NULL;
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_application_new_from_id(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "id", NULL };
    char *id;
    GnomeVFSMimeApplication *app;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "s:gnome.vfs.mime_application_new_from_id",
				     kwlist, &id))
	return NULL;
    app = gnome_vfs_mime_application_new_from_id(id);

    if (!app) {
	  /* not sure if NULL is a valid return value, but we handle
	   * it just in case... */
	PyErr_SetString(PyExc_ValueError, "unknown application id");
	return NULL;
    }
    return pygvfs_mime_application_new(app);
}

static PyMethodDef pygnomevfs_functions[] = {
    { "create", (PyCFunction)pygvfs_create, METH_VARARGS|METH_KEYWORDS },
    { "get_file_info", (PyCFunction)pygvfs_get_file_info,
      METH_VARARGS|METH_KEYWORDS },
    { "set_file_info", (PyCFunction)pygvfs_set_file_info,
      METH_VARARGS|METH_KEYWORDS },
    { "truncate", (PyCFunction)pygvfs_truncate, METH_VARARGS|METH_KEYWORDS },
    { "make_directory", (PyCFunction)pygvfs_make_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "remove_directory", (PyCFunction)pygvfs_remove_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "unlink", (PyCFunction)pygvfs_unlink, METH_VARARGS|METH_KEYWORDS },
    { "exists", (PyCFunction)pygvfs_exists, METH_VARARGS|METH_KEYWORDS },
    { "get_mime_type", (PyCFunction)pygvfs_get_mime_type, METH_VARARGS },
    { "get_mime_type_for_data", (PyCFunction)pygvfs_get_mime_type_for_data,
      METH_VARARGS },
    { "mime_get_icon", (PyCFunction)pygvfs_mime_get_icon, METH_VARARGS },
    { "mime_set_icon", (PyCFunction)pygvfs_mime_set_icon, METH_VARARGS },
    { "mime_get_description", (PyCFunction)pygvfs_mime_get_description, 
      METH_VARARGS },
    { "mime_set_description", (PyCFunction)pygvfs_mime_set_description, 
      METH_VARARGS },
    { "mime_can_be_executable", (PyCFunction)pygvfs_mime_can_be_executable, 
      METH_VARARGS },
    { "mime_set_can_be_executable", (PyCFunction)pygvfs_mime_set_can_be_executable,
      METH_VARARGS },
    { "monitor_add", pygvfs_monitor_add, METH_VARARGS},
    { "monitor_cancel", pygvfs_monitor_cancel, METH_VARARGS},
    { "read_entire_file", pygvfs_read_entire_file, METH_VARARGS},
    { "mime_get_default_application", (PyCFunction)pygvfs_mime_get_default_application, 
      METH_VARARGS },
    { "xfer_uri", (PyCFunction)pygvfs_xfer_uri,  METH_VARARGS|METH_KEYWORDS },
    { "xfer_uri_list", (PyCFunction)pygvfs_xfer_uri_list,  METH_VARARGS|METH_KEYWORDS },
    { "xfer_delete_list", (PyCFunction)pygvfs_xfer_delete_list,  METH_VARARGS|METH_KEYWORDS },
    { "mime_get_default_action_type", (PyCFunction)pygvfs_mime_get_default_action_type,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_default_action", (PyCFunction)pygvfs_mime_get_default_action,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_default_component", (PyCFunction)pygvfs_mime_get_default_component,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_short_list_components", (PyCFunction)pygvfs_mime_get_short_list_components,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_short_list_applications", (PyCFunction)pygvfs_mime_get_short_list_applications,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_all_applications", (PyCFunction)pygvfs_mime_get_all_applications,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_get_all_components", (PyCFunction)pygvfs_mime_get_all_components,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_set_default_action_type", (PyCFunction)pygvfs_mime_set_default_action_type,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_set_default_application", (PyCFunction)pygvfs_mime_set_default_application,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_set_default_component", (PyCFunction)pygvfs_mime_set_default_component,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_set_short_list_applications", (PyCFunction)pygvfs_mime_set_short_list_applications,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_set_short_list_components", (PyCFunction)pygvfs_mime_set_short_list_components,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_add_application_to_short_list", (PyCFunction)pygvfs_mime_add_application_to_short_list,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_remove_application_from_short_list",
      (PyCFunction)pygvfs_mime_remove_application_from_short_list, METH_VARARGS|METH_KEYWORDS },
    { "mime_add_component_to_short_list", (PyCFunction)pygvfs_mime_add_component_to_short_list,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_remove_component_from_short_list", (PyCFunction)pygvfs_mime_remove_component_from_short_list,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_add_extension", (PyCFunction)pygvfs_mime_add_extension,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_remove_extension", (PyCFunction)pygvfs_mime_remove_extension,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_extend_all_applications", (PyCFunction)pygvfs_mime_extend_all_applications,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_remove_from_all_applications", (PyCFunction)pygvfs_mime_remove_from_all_applications,
      METH_VARARGS|METH_KEYWORDS },
    { "mime_application_new_from_id", (PyCFunction)pygvfs_mime_application_new_from_id,
      METH_VARARGS|METH_KEYWORDS },
    { NULL, NULL, 0 }
};

static void
register_constants(PyObject *module)
{
#define regconst(x) PyModule_AddIntConstant(module, #x, GNOME_VFS_ ## x)
    regconst(FILE_FLAGS_NONE);
    regconst(FILE_FLAGS_SYMLINK);
    regconst(FILE_FLAGS_LOCAL);
    regconst(FILE_TYPE_UNKNOWN);
    regconst(FILE_TYPE_REGULAR);
    regconst(FILE_TYPE_DIRECTORY);
    regconst(FILE_TYPE_FIFO);
    regconst(FILE_TYPE_SOCKET);
    regconst(FILE_TYPE_CHARACTER_DEVICE);
    regconst(FILE_TYPE_BLOCK_DEVICE);
    regconst(FILE_TYPE_SYMBOLIC_LINK);
    regconst(FILE_INFO_FIELDS_NONE);
    regconst(FILE_INFO_FIELDS_TYPE);
    regconst(FILE_INFO_FIELDS_PERMISSIONS);
    regconst(FILE_INFO_FIELDS_FLAGS);
    regconst(FILE_INFO_FIELDS_DEVICE);
    regconst(FILE_INFO_FIELDS_INODE);
    regconst(FILE_INFO_FIELDS_LINK_COUNT);
    regconst(FILE_INFO_FIELDS_SIZE);
    regconst(FILE_INFO_FIELDS_BLOCK_COUNT);
    regconst(FILE_INFO_FIELDS_IO_BLOCK_SIZE);
    regconst(FILE_INFO_FIELDS_ATIME);
    regconst(FILE_INFO_FIELDS_MTIME);
    regconst(FILE_INFO_FIELDS_CTIME);
    regconst(FILE_INFO_FIELDS_SYMLINK_NAME);
    regconst(FILE_INFO_FIELDS_MIME_TYPE);
    regconst(PERM_SUID);
    regconst(PERM_SGID);
    regconst(PERM_STICKY);
    regconst(PERM_USER_READ);
    regconst(PERM_USER_WRITE);
    regconst(PERM_USER_EXEC);
    regconst(PERM_USER_ALL);
    regconst(PERM_GROUP_READ);
    regconst(PERM_GROUP_WRITE);
    regconst(PERM_GROUP_EXEC);
    regconst(PERM_GROUP_ALL);
    regconst(PERM_OTHER_READ);
    regconst(PERM_OTHER_WRITE);
    regconst(PERM_OTHER_EXEC);
    regconst(PERM_OTHER_ALL);
    regconst(FILE_INFO_DEFAULT);
    regconst(FILE_INFO_GET_MIME_TYPE);
    regconst(FILE_INFO_FORCE_FAST_MIME_TYPE);
    regconst(FILE_INFO_FORCE_SLOW_MIME_TYPE);
    regconst(FILE_INFO_FOLLOW_LINKS);
    regconst(SET_FILE_INFO_NONE);
    regconst(SET_FILE_INFO_NAME);
    regconst(SET_FILE_INFO_PERMISSIONS);
    regconst(SET_FILE_INFO_OWNER);
    regconst(SET_FILE_INFO_TIME);
    regconst(DIRECTORY_VISIT_DEFAULT);
    regconst(DIRECTORY_VISIT_SAMEFS);
    regconst(DIRECTORY_VISIT_LOOPCHECK);
    regconst(OPEN_NONE);
    regconst(OPEN_READ);
    regconst(OPEN_WRITE);
    regconst(OPEN_RANDOM);
    regconst(SEEK_START);
    regconst(SEEK_CURRENT);
    regconst(SEEK_END);
    regconst(MONITOR_FILE);
    regconst(MONITOR_DIRECTORY);
    regconst(MONITOR_EVENT_CHANGED);
    regconst(MONITOR_EVENT_DELETED);
    regconst(MONITOR_EVENT_STARTEXECUTING);
    regconst(MONITOR_EVENT_STOPEXECUTING);
    regconst(MONITOR_EVENT_CREATED);
    regconst(MONITOR_EVENT_METADATA_CHANGED);
    regconst(MIME_APPLICATION_ARGUMENT_TYPE_URIS);
    regconst(MIME_APPLICATION_ARGUMENT_TYPE_PATHS);
    regconst(MIME_APPLICATION_ARGUMENT_TYPE_URIS_FOR_NON_FILES);
    regconst(XFER_DEFAULT);
    regconst(XFER_FOLLOW_LINKS);
    regconst(XFER_RECURSIVE);
    regconst(XFER_SAMEFS);
    regconst(XFER_DELETE_ITEMS);
    regconst(XFER_EMPTY_DIRECTORIES);
    regconst(XFER_NEW_UNIQUE_DIRECTORY);
    regconst(XFER_REMOVESOURCE);
    regconst(XFER_USE_UNIQUE_NAMES);
    regconst(XFER_LINK_ITEMS);
    regconst(XFER_FOLLOW_LINKS_RECURSIVE);
    regconst(XFER_PROGRESS_STATUS_OK);
    regconst(XFER_PROGRESS_STATUS_VFSERROR);
    regconst(XFER_PROGRESS_STATUS_OVERWRITE);
    regconst(XFER_PROGRESS_STATUS_DUPLICATE);
    regconst(XFER_OVERWRITE_MODE_ABORT);
    regconst(XFER_OVERWRITE_MODE_QUERY);
    regconst(XFER_OVERWRITE_MODE_REPLACE);
    regconst(XFER_OVERWRITE_MODE_SKIP);
    regconst(XFER_OVERWRITE_ACTION_ABORT);
    regconst(XFER_OVERWRITE_ACTION_REPLACE);
    regconst(XFER_OVERWRITE_ACTION_REPLACE_ALL);
    regconst(XFER_OVERWRITE_ACTION_SKIP);
    regconst(XFER_OVERWRITE_ACTION_SKIP_ALL);
    regconst(XFER_ERROR_MODE_ABORT);
    regconst(XFER_ERROR_MODE_QUERY);
    regconst(XFER_ERROR_ACTION_ABORT);
    regconst(XFER_ERROR_ACTION_RETRY);
    regconst(XFER_ERROR_ACTION_SKIP);
    regconst(XFER_PHASE_INITIAL);
    regconst(XFER_CHECKING_DESTINATION);
    regconst(XFER_PHASE_COLLECTING);
    regconst(XFER_PHASE_READYTOGO);
    regconst(XFER_PHASE_OPENSOURCE);
    regconst(XFER_PHASE_OPENTARGET);
    regconst(XFER_PHASE_COPYING);
    regconst(XFER_PHASE_MOVING);
    regconst(XFER_PHASE_READSOURCE);
    regconst(XFER_PHASE_WRITETARGET);
    regconst(XFER_PHASE_CLOSESOURCE);
    regconst(XFER_PHASE_CLOSETARGET);
    regconst(XFER_PHASE_DELETESOURCE);
    regconst(XFER_PHASE_SETATTRIBUTES);
    regconst(XFER_PHASE_FILECOMPLETED);
    regconst(XFER_PHASE_CLEANUP);
    regconst(XFER_PHASE_COMPLETED);
#undef regconst
}

static void initialize_exceptions (PyObject *d)
{
    pygnomevfs_exc = PyErr_NewException ("gnome.vfs.Error",
                                                 PyExc_RuntimeError, NULL);
    PyDict_SetItemString(d, "Error", pygnomevfs_exc);
 
#define register_exception(c_name, py_name)                             \
    pygnomevfs_##c_name##_exc =                                   \
        PyErr_NewException ("gnome.vfs." py_name "Error",               \
                            pygnomevfs_exc, NULL);                \
    PyDict_SetItemString(d, py_name "Error", pygnomevfs_##c_name##_exc);
 
    register_exception(not_found,             "NotFound");
    register_exception(generic,               "Generic");
    register_exception(internal,              "Internal");
    register_exception(bad_parameters,        "BadParameters");
    register_exception(not_supported,         "NotSupported");
    register_exception(io,                    "IO");
    register_exception(corrupted_data,        "CorruptedData");
    register_exception(wrong_format,          "WrongFormat");
    register_exception(bad_file,              "BadFile");
    register_exception(too_big,               "TooBig");
    register_exception(no_space,              "NoSpace");
    register_exception(read_only,             "ReadOnly");    
    register_exception(invalid_uri,           "InvalidURI");
    register_exception(not_open,              "NotOpen");    
    register_exception(invalid_open_mode,     "InvalidOpenMode");
    register_exception(access_denied,         "AccessDenied");    
    register_exception(too_many_open_files,   "TooManyOpenFiles");
    register_exception(eof,                   "EOF");    
    register_exception(not_a_directory,       "NotADirectory");
    register_exception(in_progress,           "InProgress");
    register_exception(interrupted,           "Interrupted");
    register_exception(file_exists,           "FileExists");
    register_exception(loop,                  "Loop");
    register_exception(not_permitted,         "NotPermitted");
    register_exception(is_directory,          "IsDirectory");
    register_exception(no_memory,             "NoMemory");
    register_exception(host_not_found,        "HostNotFound");
    register_exception(invalid_host_name,     "InvalidHostName");
    register_exception(host_has_no_address,   "HostHasNoAddress");
    register_exception(login_failed,          "LoginFailed");
    register_exception(cancelled,             "Cancelled");
    register_exception(directory_busy,        "DirectoryBusy");
    register_exception(directory_not_empty,   "DirectoryNotEmpty");
    register_exception(too_many_links,        "TooManyLinks");
    register_exception(read_only_file_system, "ReadOnlyFileSystem");
    register_exception(not_same_file_system,  "NotSameFileSystem");
    register_exception(name_too_long,         "NameTooLong");
    register_exception(service_not_available, "ServiceNotAvailable");
    register_exception(service_obsolete,      "ServiceObsolete");
    register_exception(protocol_error,        "ProtocolError");
    register_exception(no_master_browser,     "NoMasterBrowser");
#if 0    
    register_exception(no_default,            "NoDefault");
    register_exception(no_handler,            "NoHandler");
    register_exception(parse,                 "Parse");
    register_exception(launch,                "Launch");
#endif    
#undef register_exception
}

struct _PyGnomeVFS_Functions pygnomevfs_api_functions = {
    pygnome_vfs_exception_check,
    pygnome_vfs_uri_new,
    &PyGnomeVFSURI_Type,
    pygnome_vfs_file_info_new,
    &PyGnomeVFSFileInfo_Type,
};

DL_EXPORT(void)
initvfs(void)
{
    PyObject *m, *d, *o;

    PyGnomeVFSURI_Type.ob_type = &PyType_Type;
    PyGnomeVFSFileInfo_Type.ob_type = &PyType_Type;
    PyGnomeVFSDirectoryHandle_Type.ob_type = &PyType_Type;
    PyGnomeVFSHandle_Type.ob_type = &PyType_Type;

    init_pygobject();
    if (!gnome_vfs_init()) {
	PyErr_SetString(PyExc_RuntimeError, "could not initialise gnome.vfs");
	return;
    }

    if (PyType_Ready(&PyGnomeVFSURI_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSFileInfo_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSDirectoryHandle_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSHandle_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSXferProgressInfo_Type) < 0)
	return;

    m = Py_InitModule("gnome.vfs", pygnomevfs_functions);
    d = PyModule_GetDict(m);

    register_constants(m);
    initialize_exceptions(d);
    PyDict_SetItemString(d, "Error", pygnomevfs_exc);

    PyDict_SetItemString(d, "URI", (PyObject *)&PyGnomeVFSURI_Type);
    PyDict_SetItemString(d, "FileInfo", (PyObject *)&PyGnomeVFSFileInfo_Type);
    PyDict_SetItemString(d, "DirectoryHandle",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "Handle", (PyObject *)&PyGnomeVFSHandle_Type);

    PyDict_SetItemString(d, "async", pygvfs_async_module_init ());

    PyDict_SetItemString(d, "open_directory",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "open", (PyObject *)&PyGnomeVFSHandle_Type);

    PyDict_SetItemString(d, "_PyGnomeVFS_API",
			 o=PyCObject_FromVoidPtr(&pygnomevfs_api_functions,NULL));
    Py_DECREF(o);

    init_pyorbit();

    monitor_hash = g_hash_table_new(g_direct_hash, g_direct_equal);
}
