/* update a spool file */


#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <syslog.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "server.h"
#include "hkp.h"
#include "lk.h"

/* An ASCII-armored key has at least this many lines. */

#define MIN_KEYLINES 5

/* return value: < 0 => error, > 0 => HTTP return code.  */

int pkspxy_update_cache (char *spoolfile, size_t spfl, const char *query)
{
  char _spoolfile[_POSIX_PATH_MAX];
  char tmpfname[_POSIX_PATH_MAX];
  char buff[1024];
  int fd = -1;
  int tmpfd = -1;
  FILE *fp = NULL, *sfp = NULL;
  struct stat sb;

  int code = -100;
  int hard = 0;
  int lines;
  struct pkspxy_strlist *p;

  char hostname[256];
  int port;
  char *s;

  if (!spoolfile)
  {
    spoolfile = _spoolfile;
    spfl = _POSIX_PATH_MAX;
  }

  pkspxy_spool_file (spoolfile, spfl, query);

  if ((fd = open (spoolfile, O_RDWR | O_CREAT, 0600)) == -1)
  {
    syslog (LOG_ERR, "Can't open %s, %m.", spoolfile);
    return -1;
  }

  if (lk_lock (fd, 1) == -1)
  {
    syslog (LOG_ERR, "Can't flock %s, %m.", spoolfile);
    goto bail;
  }

  if (fstat (fd, &sb) == -1)
  {
    syslog (LOG_ERR, "Can't fstat %s, %m.", spoolfile);
    goto bail;
  }

  if (sb.st_size == 0 || ((time (NULL) - sb.st_mtime) > RecheckInterval))
  {
    syslog (LOG_DEBUG, "updating spool file %s.", spoolfile);
    syslog (LOG_INFO, "query: `%s'", query);
    for (p = Keyservers; p ; p = p->next)
    {
      snprintf (tmpfname, sizeof (tmpfname), "/tmp/%s.XXXXXX", Progname);
      if ((tmpfd = mkstemp (tmpfname)) == -1)
      {
	syslog (LOG_ERR, "mkstemp failed. %m.");
	goto bail;
      }

      unlink (tmpfname);
      
      if ((fp = fdopen (tmpfd, "w+")) == NULL)
      {
	syslog (LOG_ERR, "fdopen failed. %m.");
	goto bail;
      }
      
      strncpy (hostname, p->str, sizeof (hostname));
      hostname[sizeof (hostname)-1] = '\0';

      port = 11371;

      if ((s = strchr (hostname, ':')))
      {
	*s = '\0';
	port = atoi (s + 1);
      }

      /* lines >= MIN_KEYLINES is a heuristic in use since key servers don't give
       * us proper error replies.
       */

      syslog (LOG_DEBUG, "Querying server %s:%d.", hostname, port);
      
      if ((code = hkp_doit (hostname, port, query, 0, fp, &lines)) == 200 && 
	  lines >= MIN_KEYLINES)
      {
	alarm (0); /* please don't disturb us _now_ */

	syslog (LOG_INFO, "Got key material from server %s:%d.", hostname, port);
	
	if ((sfp = fdopen (fd, "w+")) == NULL)
	{
	  code = -1;
	  goto bail;
	}
	
	rewind (fp);
	while (fgets (buff, sizeof (buff), fp))
	  fputs (buff, sfp);

	fflush (sfp);

	fclose (fp);
	fp = NULL;
	
	break;
      }
      else if (code == -1)
	hard = 1;

      fclose (fp);
      fp = NULL;
    }
  }
  else
    code = 204;

  if (!hard && code == 200 && lines < MIN_KEYLINES)
  {
    syslog (LOG_INFO, "Got no key material. Caching negative reply.");

    if ((sfp = fdopen (fd, "w+")) == NULL)
    {
      code = -1;
      goto bail;
    }

    fputs ("<head><title>key not found</title></head>\n", sfp);
    fputs ("<body>The key you asked for seems not to exist.\n", sfp);
    fputs ("</body>\n", sfp);

    fflush (sfp);
  }

  bail:

  if (fd >= 0)
    lk_unlock (fd);

  if (sfp)
    fclose (sfp);
  else if (fd >= 0)
    close (fd);


  if (fp)
    fclose (fp);

  return code;
}

