/*
** Copyright (C) 2000 Idan Shoham <idan@m-tech.ab.ca>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>
#include <iostream>

using namespace std;

#include "project.h"

int Project::ResourceIsUsedForTask( RESOURCE *r, TASK *t, int dayNo )
{
    for( ResourceTimeBlockIterator tb = r->begin_booked() ; 
	 tb != r->end_booked() ; tb++ )
    {
	if ( ( dayNo >= tb->start() ) 
	     && ( dayNo <= tb->finish() ) 
	     && ( tb->task() == t ) )
	{
	    Debug("ResourceIsUsedForTask(%s,%s,%d) - yes",
		  r->id(), t->id(), dayNo);
	    return 1;
	}
    }

    Debug("ResourceIsUsedForTask(%s,%s,%d) - no", r->id(), t->id(), dayNo);
    return 0;
}


int Project::FirstFreeTime(TASK *task, RESOURCE *r, int earliest, int duration)
{
    Debug("FirstFreeTime(%s,%d/%s,%d)",
		r->id(),earliest,days[earliest].s,duration);

    for ( int i = 0; earliest+i+duration<MAX_TIME; ++i )
    {
	if ( r->IsAvailable( earliest+i, earliest+i+duration-1 ) )
	    return earliest+i;
    }

    Error("Resource %s not available to do %s at time >= %s",
	  r->id(), task->id(), days[earliest].s);

    // never reached
    return -1;
}


void Project::BookResource(TASK *t, RESOURCE *parent, RESOURCE *r,
			   int tstart, int tfinish, TimeBlock::Type type)
{
  Debug("BookResource(%s,%s,%s,%d,%d)", t->id(),
	parent==NULL?"NULL":parent->id(),
	r==NULL?"NULL":r->id(),
	tstart, tfinish);
  
  if ( ! r->IsAvailable(tstart, tfinish) )
    {
	bool only_vac = true;
	Warning("Double booking! %s / %s / %s-%s", r->id(), t->id(),
		days[tstart].s, days[tfinish].s);
	for ( ResourceTimeBlockIterator tb = r->begin_booked() ; 
	      tb != r->end_booked() ; 
	      tb++ )
	{
	    if ( tb->overlap(tstart, tfinish) )
	    {
		Warning("Overlap: %s %s (%s-%s) / %s %s %s-%s",
			tb->task()->id(),
			tb->task()->assigned() == NULL ? "(unassigned)" :
			tb->task()->assigned()->id(),
			days[tb->task()->start()].s, days[tb->task()->finish()].s,
			t->id(),
			t->assigned()==NULL?"(unassigned)":t->assigned()->id(),
			days[tb->start()].s, days[tb->finish()].s);
		if ( ! tb->task()->isVacation() || ! t->isVacation() )
		    only_vac = false;
	    }
	}
	if ( only_vac )
	    Warning("You have double-booked vacations for %s.  Ignoring.",
		    r->id());
	else
	    Error("You cannot double-book tasks.");
    }

//    AddTimeBlock(t, &(t->when), tstart, tfinish, type);
    t->addTimeBlock(r, tstart, tfinish, type);
    r->addTimeBlock(t, tstart, tfinish, type);

    if ( r->is_group )
    {
	for ( RPLCI rl = r->contains.begin() ; rl != r->contains.end() ; rl++ )
	{
	    if ( ( *rl != parent ) && ( *rl != r ) )
		(*rl)->addTimeBlock(t, tstart, tfinish, type);
	}
    }
    else
    {
	for ( RPLCI rl = r->belongs_to.begin() ; rl != r->belongs_to.end() ; rl++ )
	{
	    if ( ( *rl != parent ) && ( *rl != r ) )
		(*rl)->addTimeBlock(t, tstart, tfinish, type);
	}
    }
}


void Project::ReverseBookResource(TASK *t, RESOURCE *parent, RESOURCE *r,
				  int tstart, int tfinish, TimeBlock::Type type)
{
    Debug("ReverseBookResource(%s,%s,%s,%d,%d)", t->id(),
	  parent==NULL?"NULL":parent->id(),
	  r==NULL?"NULL":r->id(),
	  tstart, tfinish);

    if ( ! r->reverseIsAvailable(tstart, tfinish) )
      {
	bool only_vac = true;
	Warning("Reverse Double booking! %s / %s / %s-%s", r->id(), t->id(),
		days[tstart].s, days[tfinish].s);
	for ( ResourceTimeBlockIterator tb = r->begin_rbooked() ; 
	      tb != r->end_rbooked() ; 
	      tb++ )
	{
	    if ( tb->overlap(tstart, tfinish) )
	    {
		Warning("Overlap: %s %s (%s-%s) / %s %s %s-%s",
			tb->task()->id(),
			tb->task()->assigned() == NULL ? "(unassigned)" :
			tb->task()->assigned()->id(),
			days[tb->task()->start()].s, days[tb->task()->finish()].s,
			t->id(),
			t->assigned()==NULL?"(unassigned)":t->assigned()->id(),
			days[tb->start()].s, days[tb->finish()].s);
		if ( ! tb->task()->isVacation() || ! t->isVacation() )
		    only_vac = false;
	    }
	}
	if ( only_vac )
	    Warning("ReverseBookResource: You have double-booked vacations for %s.  Ignoring.",
		    r->id());
	else
	    Error("ReverseBookResource: You cannot double-book tasks.");
    }

//    AddTimeBlock(t, &(t->rwhen), tstart, tfinish, type);
    t->addReverseTimeBlock(r, tstart, tfinish, type);

    r->addReverseTimeBlock(t, tstart, tfinish, type);

    if ( r->is_group )
    {
	for ( RPLCI rl = r->contains.begin() ; rl != r->contains.end() ; rl++ )
	{
	    if ( ( *rl != parent ) && ( *rl != r ) )
		(*rl)->addReverseTimeBlock(t, tstart, tfinish, type);
	}
    }
    else
    {
	for ( RPLCI rl = r->belongs_to.begin() ; rl != r->belongs_to.end() ; rl++ )
	{
	    if ( ( *rl != parent ) && ( *rl != r ) )
		(*rl)->addReverseTimeBlock(t, tstart, tfinish, type);
	}
    }
}


void Project::AssignResource(TASK *t, RESOURCE *r, int tstart)
{
  int real_remaining;

    Debug("AssignResource %s/%s start=%d", t->id(), r->id(), tstart);

    Debug("AssignResource(%s,%s,%d)", t->id(), r->id(), tstart);

    if ( t->scheduled )
	Error("Trying to double-book (%s,%s,%d)", t->id(), r->id(), tstart);

    if ( t->assigned() == NULL )
    {
	if ( t->remaining == 0.0 )
	    Error("AssignResource: t->remaining == 0 but no assigned resource");
	t->setAssigned( r) ;
    }
    else
    {
	if ( t->assigned() != r )
	    Error("AssignResource trying to assign second resource to task %s",
		  t->id() );
    }
    /* t->when = NULL; */

    if ( t->remaining == -1 )
    {
	// fullduration is now calculated in PredictOverruns()
	t->remaining = t->fullduration();
    }
    else
    {
      Debug("AssignResource(%s,%s): remaining pre-set to %d",
	    t->id(), r->id(), t->remaining);
    }

    //  time remaining for resource to do the task
    // 100% efficient during vacation !!
    if ( !t->isVacation() )
	{
	  real_remaining = int(t->remaining/r->efficiency()+.5);
	  Debug("%s %d %f %d",
		t->id(), t->remaining, r->efficiency(), real_remaining);
	}
    else
	{
	  real_remaining = t->remaining;
	}

    if ( t->block() )
    {
	while ( r->IsAvailable(tstart, tstart+real_remaining-1) == 0 )
	    ++tstart;
	BookResource(t, NULL, r, tstart, tstart+real_remaining-1,
		     TimeBlock::AUTO_SCHEDULE);
	if ( ( t->start() == INVALIDDAYNO )  || ( tstart < t->start() ) )
	    t->setStart( tstart );
	tstart += real_remaining;
	t->remaining = 0;
	t->scheduled = 1;
    }
    else
    {
        // book individual days
      while ( real_remaining > 0 )
	{
            // still some days to book
	    while ( ! r->IsAvailable(tstart, tstart) )
		++tstart;

	    if ( ( t->start() == INVALIDDAYNO )  || ( tstart < t->start() ) )
		t->setStart( tstart );

	    for ( int i = real_remaining ; i > 0 ; --i )
	    {
		if ( r->IsAvailable(tstart, tstart + i - 1) )
		{
		    Debug("AddTimeBlock(%s,%s,%d,%d)", t->id(), r->id(),
			  tstart, tstart + i - 1);
		    BookResource(t, NULL, r, tstart, tstart + i - 1,
				 TimeBlock::AUTO_SCHEDULE);
		    real_remaining -= i;
		    tstart += i;
		    break;
		}
	    }
	} // still some days to book
      t->remaining = 0;
    } // book individual days

    --tstart;
    Debug("%s finish = %d", t->id(), tstart);
    t->setFinish( tstart );
    t->scheduled = 1;
}


void Project::ReverseAssignResource(TASK *t, RESOURCE *r, int tfinish)
{
    int i;
    int real_rremaining;

    Debug("ReverseAssignResource(%s,%s,%d)", t->id(), r->id(), tfinish);

    if ( t->rscheduled )
    {
	Error("Trying to double-book (%s,%s,%d)", t->id(), r->id(), tfinish);
	return;
    }

    if ( t->assigned() == NULL )
	Error("ReverseAssignResource: t->assigned == NULL");
    if ( t->assigned() != r )
	Error("ReverseAssignResource trying to assign second resource to task %s",
	      t->id());

    if ( t->rremaining == -1 )
	t->rremaining = t->fullduration();

    Debug("ReverseAssignResource t->rremaining = %d", t->rremaining);

#if 0

    t->lfinish = tfinish;
    tfinish -= t->rremaining;
    t->rremaining = 0;

#else

    //  time remaining for resource to do the task
    // 100% efficient during vacation !!
    if ( !t->isVacation() )
	{
	  real_rremaining = int(t->rremaining/r->efficiency()+.5);
	}
    else
        {
	  real_rremaining = t->rremaining;
	}
    if ( t->block() )
    {
	while ( ! r->reverseIsAvailable(tfinish-real_rremaining+1, tfinish) )
	    --tfinish;
	ReverseBookResource(t, NULL, r, tfinish-real_rremaining+1, tfinish,
			    TimeBlock::AUTO_SCHEDULE);
	if ( ( t->lfinish() == INVALIDDAYNO )  || ( tfinish > t->lfinish() ) )
	    t->setLfinish( tfinish );
	tfinish -= real_rremaining;	/* to calc start later */
	t->rremaining = 0;
    }
    else
    {
	while (real_rremaining > 0 )
	{

	  Debug("ReverseAssignResource tfinish=%d", tfinish);

	  // book individual days
	  
	  // still some days to book
	  while ( ! r->reverseIsAvailable(tfinish, tfinish) )
	    --tfinish;

	  Debug("ReverseAssignResource got tfinish=%d", tfinish);


	  if ( ( t->lfinish() == INVALIDDAYNO )  || ( tfinish > t->lfinish() ) )
	    t->setLfinish( tfinish );

	  for ( i = real_rremaining ; i > 0 ; --i )
	    {
		if ( r->reverseIsAvailable(tfinish - i + 1, tfinish) )
		{
		    Debug("Reverse AddTimeBlock(%s,%s,%d,%d)",
			  t->id(), r->id(),
			  tfinish -i + 1, tfinish);
		    ReverseBookResource(t, NULL, r, tfinish - i + 1, tfinish,
					TimeBlock::AUTO_SCHEDULE);
		    real_rremaining -= i;
		    tfinish -= i;
		    break;
		}
	    }
	} // still some days to book
	t->rremaining = 0;
    } // book individual days

#endif
    
    ++tfinish ;
    Debug("Reverse %s start = %d", t->id(), tfinish);
    t->setLstart( tfinish );
    t->rscheduled = 1;
}


int EarliestByDependency(TASK *t)
{
    int t_earliest;

    if ( t->start() == INVALIDDAYNO )
	t_earliest = 0;
    else
	t_earliest = t->start();

    for ( TaskTimeBlockIterator tb = t->begin_when(); tb != t->end_when(); tb++ )
    {
	if ( tb->finish() + 1 > t_earliest )
	{
	    t_earliest = tb->finish() + 1;
	    //QQQ
	}
    }
    
    for ( TASK::PTRLIST::const_iterator pt = t->begin_depends(); 
	  pt != t->end_depends(); 
	  pt++ )
    {
	if ( (*pt)->finish() + 1 > t_earliest )
	{
	    t_earliest = (*pt)->finish() + 1;
	}
    }
    return t_earliest;
}

RESOURCE *Project::FindEarliestResource(TASK *t, int t_earliest)
{
    RESOURCE *r = NULL;
    RESOURCE *r2 = NULL;
    int tmin = MAX_TIME, tr;
    int real_remaining;

    for ( RPLCI rl = t->begin_cando(); rl != t->end_cando(); rl++ )
    {
        r2 = *rl;
        real_remaining = int(t->remaining/r2->efficiency()+.5);
	tr = FirstFreeTime(t, *rl, t_earliest, t->block() ? real_remaining : 1 );
	if( tmin > tr )
	{
	    tmin = tr;
	    r = *rl;
	}
    }

    return r;
}


void Project::AssignTask(TASK *t)
{
    Debug("AssignTask(%s)", t->id());

    if ( t->scheduled )
    {
      Debug("AssignTask(%s) -- already scheduled.", t->id());
      return;				/* well actually never reached */
    }

    // Trivial assignment case:
    if ( ( t->assigned() == NULL ) && ( t->numCandidates() == 1 ) )
	t->setAssigned( *(t->begin_cando()) );
    
    // If no-one assigned, find the best resource:
    if ( t->assigned() == NULL )
    {
	RESOURCE *r;
	int t_earliest = EarliestByDependency(t);

	if ( t->start() != INVALIDDAYNO )
	{
	    if ( t_earliest <= t->start() )
		t_earliest = t->start();
	    else
		Warning("Cannot start task %s at %s - moving to %s",
			t->id(), days[t->start()].s, days[t_earliest].s);
	}

	r = FindEarliestResource(t, t_earliest);
	Debug("AssignTask(1)");
	AssignResource(t, r, t_earliest );
    }
    else // someone already assigned
    {
	if ( t->start() == INVALIDDAYNO )
	  {
	    int t_earliest = EarliestByDependency(t);
	    Debug("AssignTask(2)");
	    AssignResource(t, t->assigned(), t_earliest );
	  }
	else
	  {
	    Debug("AssignTask(3)");
	    int t_earliest = t->start();
	    for ( TaskTimeBlockIterator tb=t->begin_when(); tb != t->end_when(); tb++ )
	    {
		if ( tb->finish() + 1 > t_earliest )
		{
		    t_earliest = tb->finish() + 1;
		}
	    }
	    AssignResource(t, t->assigned(), t_earliest);
	}
    }
    if ( t->start() == INVALIDDAYNO )
	Error("t->start == INVALIDDAYNO for task id %s", t->id());
}


int EligibleTask(TASK *t)
{
    // if already scheduled, it is not eligible
    if ( t->scheduled )
	return 0;

    // this clause says a task is not eligible if all depends have resources
    // assigned. But the "past" command lets us assign a resource for only
    // part of the time so this test is not good enough!
    // for ( tl = t->depends; tl != NULL; tl = tl->next )
    //    if ( tl->task->assigned == NULL ) 
    //  	  return 0;
    // Instead a task is not eligible until all depends have been scheduled
    for ( TASK::PTRLIST::const_iterator pt = t->begin_depends(); 
	  pt != t->end_depends(); 
	  pt++ )
	if ( !(*pt)->scheduled ) 
	    return 0;	

    return 1;
}


void Project::ReverseAssignTask(TASK *t)
{
    int t_latest;

    Debug("ReverseAssignTask(%s)", t->id());

    if ( t->rscheduled )
    {
	Debug("ReverseAssignTask(%s) -- already scheduled.", t->id());
	return;				/* well actually never reached */
    }

    // assume that the same resource will be used on the reverse schedule
    // as the forward, hence a resource must be assigned
    if ( t->assigned() == NULL )
	Error("ReverseAssignTask(%s) -- no resource assigned.", t->id());

    if ( t->lfinish() == INVALIDDAYNO )
    {
	t_latest = finishDay;
	for ( TASK::PTRLIST::const_iterator pt = t->begin_follows(); 
	      pt != t->end_follows(); 
	      pt++ )
	    if ( (*pt)->lstart() <= t_latest )
		t_latest = (*pt)->lstart() - 1;
	ReverseAssignResource(t, t->assigned(), t_latest );
    }
    else
    {
	t_latest = t->lfinish();
	for ( TaskTimeBlockIterator tb = t->begin_when(); tb != t->end_when(); tb++ )
	    if ( tb->start() - 1 < t_latest )
		t_latest = tb->start() - 1;
	ReverseAssignResource(t, t->assigned(), t_latest);
    }

    if ( t->lfinish() == INVALIDDAYNO )
	Error("t->lfinish == INVALIDDAYNO for task id %s", t->id());
}


int ReverseEligibleTask(TASK *t)
{
    // if already scheduled, it is not eligible
    if ( t->rscheduled )
	return 0;

    for ( TASK::PTRLIST::const_iterator pt = t->begin_follows(); 
	  pt != t->end_follows(); 
	  pt++ )
	if ( !(*pt)->rscheduled ) 
	    return 0;	

    return 1;
}


void Project::PredictOverruns()
{
    int duration;
    char buf[1000];

    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
    {
	TASK * t = *pt;
	float relative_days;

	/* copy the original full duration so that we can use it later
	   when generating the slippage reports */
	t->setOrigfullduration();

	if ( t->percent_complete() == 0 )
	    continue;			/* no work done yet */

	if ( ( t->begin_when() == t->end_when() ) && ( t->nDays() == 0 ) )
	    continue;			/* no info on when work done */

	if ( t->begin_when() != t->end_when() )
	{
	  relative_days = 0.;
	    for ( TaskTimeBlockIterator tb = t->begin_when() ; tb != t->end_when(); tb++ )
	    {
		if ( tb->type() == TimeBlock::WORK_DONE )
		{
		   relative_days += (tb->finish() - tb->start() + 1 )*tb->resource()->efficiency();
		}
	    }
	t->setDaysDone ( (int)(relative_days+0.5));
	}

	// Calculate the new duration	
	duration = (int)((double)t->nDays() / (0.01 * t->percent_complete()) + 0.5);

	// Apply the new duration to the task:
	if ( duration != t->fullduration() )
        {
	    sprintf(buf,
		    "%s changed from %d to %d days, based on completion of %g%% in %d day%s.",
		    t->id(), t->fullduration(), duration, t->percent_complete(),
		    t->nDays(), t->nDays()>1?"s":"");
	    t->setFullduration( duration );
	    //QQQ
	    if ( t->begin_when() != t->end_when() )
		t->remaining = duration - t->nDays();
	    if ( t->remaining > 0 )
		t->scheduled = 0;
	    t->setOverrun(buf);
        }
    }
}


void Project::ReverseScheduleTasks()
{
    TASK *preferred;
    int parents;
    int schedFinish;
    int finish;

    /* if the last scheduled finish is after the manually assigned finish
       then extend the finish date */
    schedFinish = finishDay;
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
    {
	if ( (*pt)->isVacation() )
	    continue;
	if ( ( schedFinish == INVALIDDAYNO )
	     || ( (*pt)->finish() > schedFinish ) )
	    schedFinish = (*pt)->finish();
    }
    if ( finishDay == INVALIDDAYNO )
    {
	finishDay = schedFinish;
    }
    else if ( schedFinish > finishDay )
    {
	Warning("Scheduled finish day (%s) is after assigned finish day (%s), ignoring latter",
		days[schedFinish].s, days[finishDay].s);
	finishDay = schedFinish;
    }

    /* fprintf(stderr, "finishDay = %d\n", finishDay); */

    /* any task that is complete or partially complete must retain the
       same schedule as forward schedule */
    /* similiarly all vacations must have same "schedule" */
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	if ( ( (*pt)->percent_complete() > 0.0 ) || (*pt)->isVacation() )
	{
	    (*pt)->rscheduled = 1;
	    (*pt)->setLstart( (*pt)->start() );
	    (*pt)->setLfinish( (*pt)->finish() );
	    (*pt)->copyWhenToReverseWhen();
	    for (TaskTimeBlockIterator tb = (*pt)->begin_rwhen(); 
		 tb != (*pt)->end_rwhen() ; tb++ )
		(*pt)->assigned()->addReverseTimeBlock((*pt), 
						       tb->start(), 
						       tb->finish(), 
						       tb->type());
	}
    
    do
    {
	// find the task with the latest finish
	finish = -1;
	parents = -1;
	preferred = NULL;
	for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	{
	    if ( !ReverseEligibleTask(*pt) )
		continue;
	    else
	    {
		if( (*pt)->finish() > finish )
		{
		    finish = (*pt)->finish();
		    preferred = (*pt);
		}
	    }
	}

	if ( preferred != NULL )
	    ReverseAssignTask(preferred);
	
    }
    while ( preferred != NULL );

    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	if( !(*pt)->rscheduled )
	    Error("Task %s not reverse scheduled - do you have a dependency loop?",
		  (*pt)->id());

    /* calculate slack */
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	(*pt)->setSlack( (*pt)->lstart() - (*pt)->start() );

#if 0
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	if ( !(*pt)->isVacation() )
	    fprintf(stderr, "task %s lstart %d lfinish %d slack %d\n",
		    (*pt)->id,
		    (*pt)->lstart,
		    (*pt)->lfinish,
		    (*pt)->slack);
#endif
}


void Project::ScheduleMilestones()
{
    /* assume task scheduling has been done */
    for ( MPLCI ml = beginMilestoneList() ; ml != endMilestoneList() ; ml++ )
    {
	MILESTONE *m = *ml;
	int day = INVALIDDAYNO;
	TASK *ct = NULL;

	for ( TPLCI tl = m->begin_depends() ; tl != m->end_depends() ; tl++ )
	{
	    TASK *t = *tl;
	    if ( ( day == INVALIDDAYNO ) || ( t->finish() > day ) )
	    {
		day = t->finish();
		ct = t;
	    }
	}

	m->setDay( day );
	m->setCritical( ct );
    }
}


void Project::ScheduleTasks()
{
    TASK *preferred;
    int children;

    Debug("Scheduling tasks");

    do
    {
	children = -1;
	preferred = NULL;
	for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	{
	    if ( !EligibleTask(*pt) )
		continue;

	    if( (*pt)->children() > children )
	    {
		children = (*pt)->children();
		preferred = (*pt);
	    }
	}

	if ( preferred != NULL )
	    AssignTask(preferred);
	
    }
    while ( preferred != NULL );

    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	if( !(*pt)->scheduled )
	    Error("Task %s not scheduled - do you have a dependency loop?",
		  (*pt)->id());
}


void Project::DoScheduling()
{
    Debug("Doing scheduling");

    LoadDays();

    PredictOverruns();

    ScheduleTasks();

    ReverseScheduleTasks();

    ScheduleMilestones();
}


void Project::WorkBlock( TASK *t, RESOURCE * r, int d1, int d2, 
			 TimeBlock::Type type )
{
    if ( t->nDays() != 0 )
	Error("Can't use past/future and done at same time(2)");

    if ( t->assigned() == NULL )
    {
	t->setAssigned(r);
    }
    else
    {
	if ( t->assigned() != r )
	    Error("Two resources assigned to same task %s", t->id());
    }

    if ( t->start() == INVALIDDAYNO )
	t->setStart( d1 );

    if ( d1 < t->start() )
	Error("Out of order assignment to task %s", t->id());

    BookResource( t, NULL, r, d1, d2, static_cast<TimeBlock::Type>(type) );

    if ( t->remaining == -1 )
    {
      t->remaining = t->duration();
      t->setFullduration( t->remaining );
    }
    
    t->remaining -= (int)((d2-d1+1)*r->efficiency()+0.5);
    Debug("%d %d",(d2-d1+1), (int)((d2-d1+1)*r->efficiency()+0.5));
    if ( t->remaining <= 0 )
    {
	t->setFinish( d2 );
	t->scheduled = 1;
	if ( !t->isVacation() )
	  Warning("Completed task %s",t->id());
    }
}



void Project::StartTask( TASK *t, int d, int type )
{
   switch(type)
   {
       case TNORM:
	   t->setStart(d);
	   
	   cerr << "StartTask: Invoking garbage - tread with care!\n";

           // printf("Setting %s->start to %d\n",t->id,d);
	   if(  t->finish() == INVALIDDAYNO )
	   {
	      t-> setFinish( t->start() + t->duration() );
	       // printf("Setting %s->finish to %d\n",t->id,t->finish);
	   }
	   else
	       t->setDuration( t->finish() - t->start() );

	   // single candidate?  just do it!
	   if ( t->numCandidates() == 1 )
	   {
	       t->setAssigned( *(t->begin_cando()) );
	       BookResource( t, NULL, t->assigned(), t->start(), t->start(), 
			     TimeBlock::AUTO_SCHEDULE );
	       t->setDuration( t->duration() - 1 );
	   }

	   // fix the start of the reverse schedule too!
	   t->setLstart(d);
	   
	   break;
	   
       case TACTUAL:
	   t->setAstart(d);
           // as for finish we just record the actual at this time
	   // (no auto setting of the finish
	   break;
	   
       case TBASE:
	   t->setBstart( d );
           // as for finish we just record the actual at this time
	   // (no auto setting of the finish
	   break;
	   
       default:
	   Error("Internal Error, unknown start-finish type; check that patchs are complete");
	   break;
   }
}



void Project::FinishTask( TASK *t, int d, int type )
{
    switch(type)
    {
	case TNORM:
	    t->setFinish( d );

	    cerr << "TASK::FinishTask: Invoking garbage - tread with care!\n";
	    
	    if ( t->start() == INVALIDDAYNO )
	    {
		t->setStart( t->finish() - t->duration() );
		if ( t->start() < 0 )
		    Error("Finish task date generates too early start date");
	    }
	    else
	    {
		t->setDuration( t->finish() - t->start() );
		if ( t->duration() <= 0 )
		    Error("Finish task date generates duration <= 0");
	    }
  
	    if ( t->assigned() != NULL )
	    {
		BookResource( t, NULL, t->assigned(), t->start(), t->finish(), 
			      TimeBlock::AUTO_SCHEDULE );
	    }
	    break;

	case TACTUAL:
	    // for right now all we do is record the time,
	    // later on we can work with actual duration
	    t->setAfinish( d );
	    break;

	case TBASE:
	    t->setBfinish( d );
	    break;

	default:
	    Error("Internal Error, unknown start-finish type; check that patchs are complete");
	    break;
    }
}


