/*
 * Copyright (c) 1997 Carnegie Mellon University. All Rights Reserved.
 *  
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is hereby granted (including for commercial or
 * for-profit use), provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative
 * works, or modified versions, and any portions thereof, and that
 * both notices appear in supporting documentation, and that credit
 * is given to Carnegie Mellon University in all publications reporting
 * on direct or indirect use of this code or its derivatives.
 *
 * THIS SOFTWARE IS EXPERIMENTAL AND IS KNOWN TO HAVE BUGS, SOME OF
 * WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON PROVIDES THIS
 * SOFTWARE IN ITS ``AS IS'' CONDITION, AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Carnegie Mellon encourages (but does not require) users of this
 * software to return any improvements or extensions that they make,
 * and to grant Carnegie Mellon the rights to redistribute these
 * changes without encumbrance.
 */


#include <stdio.h>

#ifdef BSD_KERNEL
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#else
#include <stdlib.h>
#endif

/* #define HFSC_DEBUG  */

#include "hfsc_err.h"
#include "hfsc_def.h"
#include "hfsc_decl.h"
#include "hfsc_fun.h"

/****************************************************************
 *
 *  NAME: newSC
 *
 *  DESCRIPTION:
 *    Allocate memory for a (two-pice linear) service curve data 
 *    structure and initialize it
 *
 *  PARAMETERS:
 *    m1 - the slope of the first segment
 *    d  - the length of the first segment projections on the x-axis, i.e.,
 *         x + d represents the x-coordinate where the first segment ends 
 *         and the second one starts
 *    m2 - the slope of the second segment
 *
 *  RETURN:
 *    - service curve item
 *
 *  NOTE:
 *    If m1 > m2 then the service curve is convex; otherwise
 *    the service curve is concave
 *
 ************************************************************************/

ServiceCurve *newSC(m1, d, m2)
  int m1;
  int d;
  int m2;
{
  ServiceCurve *psc;
  
#ifdef BSD_KERNEL
  if (!(psc = (ServiceCurve *)malloc(sizeof(ServiceCurve), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(psc, sizeof(ServiceCurve));
#else
  if ((psc = (ServiceCurve *)calloc(1, sizeof(ServiceCurve))) == NULL)
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

  psc->m1 = m1;
  psc->d  = d;
  psc->m2 = m2;

  return psc;
}


/****************************************************************
 *
 *  NAME: newRTSC
 *
 *  DESCRIPTION:
 *    Allocate memory for a (two-pice linear) run-time 
 *    service curve data structure and initialize it
 *
 *  PARAMETERS:
 *    x  - the x-coordinate where the first segment starts
 *    y  - the y-coordinate where the first segment starts
 *    m1 - the slope of the first segment
 *    d  - the length of the first segment projections on the x-axis, i.e.,
 *         x + d represents the x-coordinate where the first segment ends 
 *         and the second one starts
 *    m2 - the slope of the second segment
 *
 *  RETURN:
 *    - run-time service curve item
 *
 *  NOTE:
 *    If m1 > m2 then the service curve is convex; otherwise
 *    the service curve is concave
 *
 ************************************************************************/

RTServiceCurve *newRTSC(x, y, m1, d, m2)
  int x;
  int y;
  int m1;
  int d;
  int m2;
{
  RTServiceCurve *prtsc;
  
#ifdef BSD_KERNEL
  if (!(prtsc = 
    (RTServiceCurve *)malloc(sizeof(RTServiceCurve), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(prtsc, sizeof(RTServiceCurve));
#else
  if ((prtsc = 
    (RTServiceCurve *)calloc(1, sizeof(RTServiceCurve))) == NULL)
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

  prtsc->y  = y;
  prtsc->x  = x;
  prtsc->m1 = m1;
  prtsc->d  = d;
  prtsc->m2 = m2;

  return prtsc;
}


/****************************************************************
 *
 *  NAME: minRTSC
 *
 *  DESCRIPTION:
 *    Compute the minimum between a run-time service curve m
 *    and a service curve s translated by (x, y)
 * 
 *    m(t - x) = min(m(t), S(t - x) + y), for any t >= x
 * 
 *  The new run-time service curve is will start in (x, y)
 *  (it will be defined only for t >= x)
 * 
 *  PARAMETERS:
 *    prtsc  - the run-time service curve (m)
 *    psc    - the service curve (s)
 *    x, y   - the coordinates from where the service curve starts
 *
 *  NOTE:
 *    The slopes of the service curves m and s are identical.
 *    In this case, it can be shown that the resulting service curve 
 *    will have the same service slope, as well. 
 *
 ************************************************************************/

void minRTSC(prtsc, psc, x, y)
  ServiceCurve   *psc;
  RTServiceCurve *prtsc;
  int           x;
  int           y;
{
  int xp, yp;

  if (cmp_eq(prtsc->m1, 0) && cmp_eq(prtsc->m2, 0)) {
    /* this is the first packet ever received by the corresponding session */
    prtsc->x = x;
    prtsc->y = y;
    prtsc->m1 = psc->m1;   
    prtsc->d  = psc->d; 
    prtsc->m2 = psc->m2;
    return;   
  }

  if (cmp_g(psc->m1, psc->m2)) {
    /* both service curves are concave */  

    /* yc: service curve was not computed correctly in minRTSC().  
     * getSegmentIntersection(x, y, psc->d, psc->m1,
     * prtsc->x, prtsc->y, x + psc->d - prtsc->x, prtsc->m2, &xp, &yp);
     */
    getSegmentIntersection(x, y, psc->d, psc->m1,
			   prtsc->x + prtsc->d, prtsc->y + prtsc->d*prtsc->m1, 
			   0x7FFFFFFF,  /* extends to infinity */
			   prtsc->m2, &xp, &yp);

    if (xp || yp) {
      prtsc->m1 = psc->m1;
      prtsc->m2 = psc->m2;
      prtsc->d = xp - x;
    } else {
      if (prtsc->x > x) {
        return;
      }
      prtsc->m1 = psc->m1;
      prtsc->m2 = psc->m2;
      prtsc->d = psc->d;
    }      
  } else {
    /* service curve is convex */
    prtsc->m1 = psc->m1;
    prtsc->m2 = psc->m2;
    prtsc->d  = psc->d;
    if (cmp_leq(prtsc->x - x, prtsc->d)) 
      prtsc->x = x;
    prtsc->y = y;
    return;
  }


  prtsc->x = max(x, prtsc->x);
  prtsc->y = y;

}  
      

/****************************************************************
 *
 *  NAME: getLineIntersection
 *
 *  DESCRIPTION:
 *    Given two segments, compute the intersection point between 
 *    the two lines defined by these segments
 * 
 *  PARAMETERS:
 *    x1, y1 - the left-end coordinates of the first segment
 *    d1, m1 - the length of the first segment's x-projection, and
 *             its slope 
 *    x2, y2 - the left-end coordinates of the second segment
 *    d2, m2 - the length of the second segment x-projection, and
 *             its slope 
 *
 *  RETURN:
 *    The point of intersection, if any, and NULL otherwise
 *
 ************************************************************************/

void getLineIntersection(x1, y1, d1, m1, x2, y2, d2, m2, xp, yp)
  int x1, y1, d1, m1, x2, y2, d2, m2, *xp, *yp;
{
  /* the two segments are parallel, or define the same line */
  if (m1 == m2)
    return;

  x2 = x2-x1;
  y2 = y2-y1;

  /* compute the intersection between the lines defined by the two segments */
  /* n1 = y1 - m1*x1; n1 = 0 */
  /* y' = m1*x'   , where x' = x-x1, y' = y-y1
   * y' = m2*x' + y2-m2*x2
   * x' = (y2-m2*x2)/(m1-m2)
   * y' = m1*x'
   */
  *xp = (y2 - m2*x2)/(m1 - m2);
  *yp = m1*(*xp) + y1; 
  *xp = *xp + x1;
}


/****************************************************************
 *
 *  NAME: getSegmentIntersection
 *
 *  DESCRIPTION:
 *    Compute the intersection point between two segments, if any
 * 
 *  PARAMETERS:
 *    x1, y1 - the left-end coordinates of the first segment
 *    d1, m1 - the length of the first segment's x-projection, and
 *             its slope 
 *    x2, y2 - the left-end coordinates of the second segment
 *    d2, m2 - the length of the second segment x-projection, and
 *             its slope 
 *
 *  RETURN:
 *    The point of intersection, if any, and NULL otherwise
 *
 ************************************************************************/

void getSegmentIntersection(x1, y1, d1, m1, x2, y2, d2, m2, xp, yp)
  int x1, y1, d1, m1, x2, y2, d2, m2, *xp, *yp;
{
  if (m1 == m2)
    return;

  /* compute the intersection between the lines defined by the two segments */
  getLineIntersection(x1, y1, d1, m1, x2, y2, d2, m2, xp, yp);

  if (!xp && !yp)
    return;

  /* check whether this point belongs to both segments */
  if (cmp_leq(x1, *xp) && cmp_leq(*xp, x1 + d1) && 
    cmp_leq(x2, *xp) && cmp_leq(*xp, x2 + d2))
    return;
  
  *xp = *yp = 0;
}


/****************************************************************
 *
 *  NAME: getSC_y
 *
 *  DESCRIPTION:
 *    Compute the value of a two piece-linear service curve m for x, 
 *    i.e., m(x)
 *
 *  PARAMETERS:
 *    psc - the service curve (m)
 *    x     
 *  RETURN:
 *    m(x)
 *
 ************************************************************************/
    
int getSC_y(psc, x)
  ServiceCurve *psc;
  int x;
{
  if (cmp_l(x, 0))
    hfsc_panic(ErrMsg[ ERR_INVALID_XVAL]);

  if (cmp_leq(x, psc->d))
    /* y belongs to the first segment */  
    return x*psc->m1;
  else
    return psc->d*psc->m1 + (x - psc->d)*psc->m2;
}
   

/****************************************************************
 *
 *  NAME: getRTSC_y
 *
 *  DESCRIPTION:
 *    Compute the value of the run-time service curve m for x, 
 *    i.e., m(x)
 *
 *  PARAMETERS:
 *    prtsc - the run-time service curve (m)
 *    x     
 *  RETURN:
 *    m(x)
 *
 ************************************************************************/
    
int getRTSC_y(prtsc, x)
  RTServiceCurve *prtsc;
  int x;
{
  if (cmp_l(x, prtsc->x))
    return -1;

  if (cmp_leq(x, prtsc->x + prtsc->d))
    /* y belongs to the first segment */  
    return prtsc->y + (x - prtsc->x)*prtsc->m1;
  else
    return prtsc->y + prtsc->d*prtsc->m1 + (x - prtsc->x - prtsc->d)*prtsc->m2;
}
   

/****************************************************************
 *
 *  NAME: getRTSC_x
 *
 *  DESCRIPTION:
 *    Compute the inverse of the run-time service curve m for y, 
 *    i.e., y = m(x)
 *
 *  PARAMETERS:
 *    prtsc - the run-time service curve (m)
 *    y     
 *  RETURN:
 *    x
 *
 ************************************************************************/
    
int getRTSC_x(prtsc, y)
  RTServiceCurve *prtsc;
  int y;
{
  if (cmp_l(y, prtsc->y)) {
    printf("!! y=%u, prtsc->y = %u\n", y, prtsc->y);
    printf("!! x=%u, y=%u, m1=%u, d=%u, m2=%u\n", prtsc->x, prtsc->y, 
	   prtsc->m1, prtsc->d, prtsc->m2);
    hfsc_panic("getRTSC_x \n");
    return -10;
  }
  if (cmp_eq(y, prtsc->y))
    return prtsc->x;

  if (cmp_leq(y, prtsc->y + prtsc->d*prtsc->m1))
    /* x belongs to the first segment */  
    return (prtsc->m1 ? prtsc->x + (y - prtsc->y)/prtsc->m1 : INFINITY);
  else
    return (prtsc->m2 ? 
      prtsc->x + prtsc->d + (y - prtsc->y - prtsc->d*prtsc->m1)/prtsc->m2 :
      INFINITY);
}
   
/****************************************************************
 *
 *  NAME: cutRTSC
 *
 *  DESCRIPTION:
 *    Remove the portion (prtsc->x, x) from the run-time service 
 *    curve prtsc, where x = prtsc^{-1}(y).    And set prtsc->y to be 0
 *
 *  PARAMETERS:
 *    prtsc - the run-time service curve (m)
 *    y     
 *
 *  RETURN:
 *    null
 *
 ************************************************************************/
    
void cutRTSC(prtsc, y)
  RTServiceCurve *prtsc;
  int          y;
{
  int x = getRTSC_x(prtsc, y);

  /* [yhchu] bug shows up when you want to delete HFSCClass in kernel using
   * ./reset
   * cutRTSC
   * changeSessionStatus in hfsc_hierarchy.c
   * deleteHFSCClass in interface.c
   * classifierDeleteClass in classifier_structure.c
   * classifier_ioctl
   *
   * for now, remove hfsc_panic, return the same sc, and cross our fingers
   */
  if (cmp_l(x,  prtsc->x)) {
    /* hfsc_panic(ErrMsg[ERR_INVALID_YVAL]); */
    printf("[yhchu]: a possible bug here at hfsc_sc.c\n");
    printf("x=%d, y=%d\n", x, y);
    return;
  }

  if (cmp_geq(x, prtsc->x + prtsc->d)) {
    prtsc->x = x;
    prtsc->d = 0;
    prtsc->m1 = prtsc->m2;
  } else {
    prtsc->d -= x - prtsc->x;
    prtsc->x = x;
  }
  prtsc->y = 0;

} 


/****************************************************************
 *
 *  NAME: cutRTSC_X
 *
 *  DESCRIPTION:
 *    Remove the portion (prtsc->y, y) from the run-time service 
 *    curve prtsc.    
 *
 *  PARAMETERS:
 *    prtsc - the run-time service curve (m)
 *    x     
 *
 *  RETURN:
 *    y
 *
 ************************************************************************/
    
int cutRTSC_X(prtsc, x)
  RTServiceCurve *prtsc;
  int             x;
{
  int y;

  if (x < prtsc->x) {
    prtsc->x -= x;
    return 0;
  }

  y = getRTSC_y(prtsc, x);

  if (cmp_geq(x, prtsc->x + prtsc->d)) {
    prtsc->x = 0;
    prtsc->d = 0;
    prtsc->m1 = prtsc->m2;
  } else {
    prtsc->d -= x - prtsc->x;
    prtsc->x = 0;
  }
  prtsc->y = 0;

  return y;
} 


/****************************************************************
 *
 *  NAME: newGSC
 *
 *  DESCRIPTION:
 *    Allocate memory for a generalize service curve data 
 *    structure
 *
 *  PARAMETERS:
 *
 *  RETURN:
 *    - generalize service curve item
 *
 *
 ************************************************************************/

GenServiceCurve *newGSC()
{
  GenServiceCurve *pgsc;
  
#ifdef BSD_KERNEL
  if (!(pgsc = 
    (GenServiceCurve *)malloc(sizeof(GenServiceCurve), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(pgsc, sizeof(GenServiceCurve));
#else
  if ((pgsc = (GenServiceCurve *)calloc(1, sizeof(GenServiceCurve))) == NULL)
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

  return pgsc;
}


/****************************************************************
 *
 *  NAME: addSegToGSC
 *
 *  DESCRIPTION:
 *    add a new segment to a generalized service curve
 *
 *  PARAMETERS:
 *    psgc  - generalized service curve
 *    x, y  - the coordinates of the left-end of the segment
 *    d     - the length of the segment projection on x-axis 
 *    m     - the segment's slope
 *
 *  RETURN:
 *
 *  NOTE:
 *    In this implementation we consider only segments with d = INFINITY
 *
 ************************************************************************/
  
void addSegToGSC(pgsc, x, y, d, m)
  GenServiceCurve *pgsc;
  int x;
  int y;
  int d;
  int m;
{
  Segment *pseg, *ptemp;

  if (d != INFINITY)
    hfsc_panic(ErrMsg[ERR_INVALID_SEG1]);

#ifdef BSD_KERNEL
  if (!(pseg = (Segment *)malloc(sizeof(Segment), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(pseg, sizeof(Segment));
#else
  if (!(pseg = (Segment *)calloc(1, sizeof(Segment)))) 
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

  if (!pgsc->first) {
    pseg->x = x;
    pseg->y = y;
    pseg->d = d;
    pseg->m = m;
    pgsc->first = pseg;
    return;
  }

  if (cmp_l(x, pgsc->first->x)) {
    pseg->x = x;
    pseg->y = y;
    pseg->d = INFINITY;
    pseg->m = m;
    
    pseg->next = pgsc->first;
    pgsc->first = pseg; 

  } else {

    for (ptemp = pgsc->first; ptemp->next && x > ptemp->next->x; 
      ptemp = ptemp->next);

    if (!ptemp->next && cmp_g(x, ptemp->x + ptemp->d))
      hfsc_panic(ErrMsg[ERR_INVALID_SEG]);

    if (cmp_eq(ptemp->x, x)) {
#ifdef BSD_KERNEL
      free((caddr_t)pseg, M_TEMP);
#else
      free(pseg);
#endif
      ptemp->y += y;
      ptemp->m += m;
      pseg = ptemp;
   
    } else if (!ptemp->next || cmp_l(x, ptemp->next->x)) {

      pseg->x = x;
      pseg->y = ptemp->y + (x - ptemp->x)*ptemp->m + y;
      pseg->m = ptemp->m + m;
      pseg->d = INFINITY;
      pseg->next = ptemp->next;
      ptemp->next = pseg;

    } else { /* x == ptemp->next->x */
      
#ifdef BSD_KERNEL
      free((caddr_t)pseg, M_TEMP);
#else
      free(pseg);
#endif
      ptemp->next->y += y;
      pseg = ptemp;      
    }
  }

  for (ptemp = pseg->next; ptemp;  ptemp = ptemp->next) {
    ptemp->m += m;
    ptemp->y += y + (ptemp->x - x)*m;
  }

  compressGSC(pgsc);
}      



/****************************************************************
 *
 *  NAME: IsGSCLessThanSC
 *
 *  DESCRIPTION:
 *    Check whether all points of a general service curve have 
 *    their y-coordinates no larger than a given two-piece linear service
 *    curve, i.e., y < gs(x).
 *
 *  PARAMETERS:
 *    psc  - service curve
 *    psgc - generalize service curve
 *
 *  RETURN:
 *
 ************************************************************************/
    
int IsGSCLessThanSC(pgsc, psc)
  GenServiceCurve *pgsc;
  ServiceCurve *psc;
{
  Segment *pseg;

  for (pseg = pgsc->first; pseg; pseg = pseg->next) {
    if (cmp_g(pseg->y, getSC_y(psc, pseg->x))) {
#ifdef HFSC_DEBUG
      printf("\n IsGSCLessThanSC: FALSE!\n");
#endif
      return FALSE;
    }
    if (!pseg->next){
      if (((pseg->x < psc->d) && cmp_g(pseg->m, psc->m1)) || 
          cmp_g(pseg->m, psc->m2)) {
#ifdef HFSC_DEBUG
	printf("\n IsGSCLessThanSC: FALSE!\n");
#endif
        return FALSE;
      }
    }
  }

#ifdef HFSC_DEBUG
  printf("\n IsGSCLessThanGSC: TRUE!\n");
#endif
  return TRUE;
}


/****************************************************************
 *
 *  NAME: addSCtoGSC
 *
 *  DESCRIPTION:
 *    add a new service curve to a generalized service curve
 *
 *  PARAMETERS:
 *    pgsc  - generalized service curve
 *    psc   - service curve to be added
 *
 *  RETURN:
 *
 ************************************************************************/
  
void addSCtoGSC(pgsc, psc)
  GenServiceCurve *pgsc;
  ServiceCurve    *psc;
{
  addSegToGSC(pgsc, 0, 0, INFINITY, psc->m1);
  addSegToGSC(pgsc, psc->d, 0, INFINITY, psc->m2 - psc->m1);
}


/****************************************************************
 *
 *  NAME: subSCfromGSC
 *
 *  DESCRIPTION:
 *    substracted a new service curve to a generalized service curve
 *
 *  PARAMETERS:
 *    pgsc  - generalized service curve
 *    psc   - service curve to be substracted
 *
 *  RETURN:
 *
 ************************************************************************/
  
void subSCfromGSC(pgsc, psc)
  GenServiceCurve *pgsc;
  ServiceCurve    *psc;
{
  addSegToGSC(pgsc, 0, 0, INFINITY, -psc->m1);
  addSegToGSC(pgsc, psc->d, 0, INFINITY, psc->m1 - psc->m2);
}


/****************************************************************
 *
 *  NAME: compressGSC
 *
 *  DESCRIPTION:
 *    Replace all the adjacent segements with the same slope with
 *    one segment
 *
 *  PARAMETERS:
 *    pgsc - generalized service curve
 *
 *  RETURN:
 *
 ************************************************************************/
  
void compressGSC(pgsc)
  GenServiceCurve *pgsc;
{
  Segment *pseg = pgsc->first, *ptemp;

  while (pseg->next) {

    /* the next step is required to ensure consistency */
    pseg->d = pseg->next->x - pseg->x;

    if ((cmp_eq(pseg->x, pseg->next->x) ||
      (cmp_eq(pseg->m, pseg->next->m) && 
      cmp_eq(pseg->y + (pseg->next->x - pseg->x)*pseg->m, pseg->next->y)))) {
      ptemp    = pseg->next;
      pseg->next = pseg->next->next;
#ifdef BSD_KERNEL
      free((caddr_t)ptemp, M_TEMP);
#else
      free(ptemp);
#endif
    } else
      pseg = pseg->next;
  }
  
  if (pseg)
    pseg->d = INFINITY;
}
    

/****************************************************************
 *
 *  NAME: freeGSC
 *
 *  DESCRIPTION:
 *    Free memory alocated to the general service curve data structure
 *
 *  PARAMETERS:
 *    pgsc - the general service curve
 *
 *  RETURN:
 *
 ************************************************************************/
  
void freeGSC(pgsc)
  GenServiceCurve *pgsc;
{
  Segment *pseg = pgsc->first, *ptemp;

  while (pseg->next) {
    ptemp    = pseg->next;
    pseg->next = pseg->next->next;
#ifdef BSD_KERNEL
      free((caddr_t)pseg, M_TEMP);
#else
      free(pseg);
#endif
  } 

#ifdef BSD_KERNEL
      free((caddr_t)pgsc->first, M_TEMP);
      free((caddr_t)pgsc, M_TEMP);
#else
      free(pgsc->first);
      free(pgsc);
#endif
}

#ifdef HFSC_DEBUG    

/****************************************************************
 *
 *  NAME: printGSC
 *
 *  DESCRIPTION:
 *    Print a general service curve
 *
 *  PARAMETERS:
 *    psgc - the service curve
 *
 *  RETURN:
 *
 ************************************************************************/
    
void printGSC(text, pgsc)
  char            *text;
  GenServiceCurve *pgsc;
{
  Segment *pseg;
  
  printf("%s", text);
  for (pseg = pgsc->first; pseg; pseg = pseg->next) 
    printf(" (%d, %d, %d, %d),", pseg->x, pseg->y, pseg->m, pseg->d);

  printf("\n");
}

/****************************************************************
 *
 *  NAME: printSC
 *
 *  DESCRIPTION:
 *    Print a two-piece linear service curve
 *
 *  PARAMETERS:
 *    psc - the service curve
 *
 *  RETURN:
 *
 ************************************************************************/
    
void printSC(text, psc)
  char         *text;
  ServiceCurve *psc;
{
  printf("%s (%d, %d, %d)\n", text, psc->m1, psc->d, psc->m2);
}

/****************************************************************
 *
 *  NAME: printRTSC
 *
 *  DESCRIPTION:
 *    Print a real-time service curve
 *
 *  PARAMETERS:
 *    prtsc - the real-time service curve
 *
 *  RETURN:
 *
 ************************************************************************/
    
void printRTSC(text, prtsc)
  char           *text;
  RTServiceCurve *prtsc;
{
  printf("%s (%d, %d, %d, %d, %d)\n", 
    text, prtsc->x, prtsc->y, prtsc->m1, prtsc->d, prtsc->m2);
}

#endif
