/*
 * peel.cc --
 *
 *      A peel effect
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/effects/peel.cc,v 1.1 2002/02/07 04:18:19 chema Exp $ */


#include <dvmbasic.h>
#include <dvmbytegeom.h>

#include "peel.h"



PeelEffect::PeelEffect(int decimation) 
	: EffectModule(2, decimation)
{
	pfactor_.value = 0.0;
	pfactor_.from = 0.0;
	pfactor_.to = 1.0;
	pfactor_.incr = 1;

	// create the temporal frame
	temp_ = new Uncompressed;

	return;
}



int PeelEffect::command(int argc, const char*const* argv)
{
	return (EffectModule::command(argc, argv));
}



void PeelEffect::trigger_effect (Uncompressed* input)
{
	//printf ("w(%dx%d) ", input->w_, input->h_); fflush (stdout); // XXX

	// calculate the p factor
	if (pfactor_.incr == 1) {
		pfactor_.value = pfactor_.value + 0.05;
	} else {
		pfactor_.value = pfactor_.value - 0.05;
	}
	if (pfactor_.value > pfactor_.to) {
		pfactor_.value = pfactor_.to;
		pfactor_.incr = 0;
	} else if (pfactor_.value < pfactor_.from) {
		pfactor_.value = pfactor_.from;
		pfactor_.incr = 1;
	}

	// step 1: create the diagonal masks using a temporal ByteImage
	int d = (buffer_[0]->w_ > buffer_[0]->h_) ? buffer_[0]->w_ : buffer_[0]->h_;
	BitImage *diag_bit_mask = BitNew(d, d);
	BitImage *inv_diag_bit_mask = BitNew(d, d);
	// TBD: this shouldn't be done always!!!!!!!!!!!!!!
	ByteImage *tmp = ByteNew(d, d);
	for (int i = 0; i < d; i++) {
		for (int j = 0; j < d; j++) {
			if (i + j < d) {
				// BytePoke (tmp, i, j, 0x40);
			  tmp->firstByte[j*tmp->width+i] = 0x40;
			} else {
				// BytePoke (tmp, i, j, 0x80);
			  tmp->firstByte[j*tmp->width+i] = 0x80;
			}
		}
	}
	BitMakeFromKey (tmp, 60, 70, diag_bit_mask);
	BitMakeFromKey (tmp, 120, 140, inv_diag_bit_mask);
	ByteFree (tmp);

	// step 2: create the c_diagonal masks using a temporal ByteImage
	int hs = buffer_[0]->css_h_;
	int vs = buffer_[0]->css_v_;
	BitImage *c_diag_bit_mask = BitNew(d/hs, d/vs);
	BitImage *c_inv_diag_bit_mask = BitNew(d/hs, d/vs);
	// TBD: this shouldn't be done always!!!!!!!!!!!!!!
	tmp = ByteNew(d/hs, d/vs);
	for (int i = 0; i < d/hs; i++) {
		for (int j = 0; j < d/vs; j++) {
			if ((i * hs) + (j * vs) < d) {
				// BytePoke (tmp, i, j, 0x40);
			  tmp->firstByte[j*tmp->width+i] = 0x40;
			} else {
				// BytePoke (tmp, i, j, 0x80);
			  tmp->firstByte[j*tmp->width+i] = 0x80;
			}
		}
	}
	BitMakeFromKey (tmp, 60, 70, c_diag_bit_mask);
	BitMakeFromKey (tmp, 120, 140, c_inv_diag_bit_mask);
	ByteFree (tmp);


	// get the pixel position and length
	int pix_len = buffer_[0]->w_ + buffer_[0]->h_;
	int pix_pos_ = (int)(pfactor_.value * pix_len);
	if (pix_pos_%2 == 1) {
		pix_pos_++;
	}
	int x2_ = (pix_pos_ >= buffer_[0]->w_)? (buffer_[0]->w_ - 1) : (pix_pos_);
	int y2_ = (pix_pos_ >= buffer_[0]->h_)? (buffer_[0]->h_ - 1) : (pix_pos_);
	int x1_ = (pix_pos_ - buffer_[0]->h_ < 0)? (0) : (pix_pos_ - buffer_[0]->h_);
	int y1_ = (pix_pos_ - buffer_[0]->w_ < 0)? (0) : (pix_pos_ - buffer_[0]->w_);

	// initialize the temporal frame
	temp_->init(x2_ - x1_ + 1, y2_ - y1_ + 1, hs, vs, hs, vs, 
			x2_ - x1_ + 1, y2_ - y1_ + 1, 0, 0, 0, 0, 0, VIDREP_ALLOC_MEMORY);

	// get the affine transformation parameters
	float la = 0.0;
	float lb = -1.0;
	float lc = pix_pos_ - x1_;
	float ld = -1.0;
	float le = 0; 
	float lf = pix_pos_ - y1_;
	float ca = 0.0;
	float cb = -1.0 * ((float)vs/(float)hs);
	float cc = (float)(pix_pos_ - x1_) / (float)hs;
	float cd = -1.0 * ((float)hs/(float)vs);
	float ce = 0.0;
	float cf = (float)(pix_pos_ - y1_) / (float)vs;

	ByteImage *in1_lum = input->lum_;
	ByteImage *in2_lum = buffer_[0]->lum_;
	ByteImage *out_lum = output_->lum_;
	ByteImage *temp_lum = temp_->lum_;

	ByteImage *in1_cr = input->cr_;
	ByteImage *in2_cr = buffer_[0]->cr_;
	ByteImage *out_cr = output_->cr_;
	ByteImage *temp_cr = temp_->cr_;

	ByteImage *in1_cb = input->cb_;
	ByteImage *in2_cb = buffer_[0]->cb_;
	ByteImage *out_cb = output_->cb_;
	ByteImage *temp_cb = temp_->cb_;

	// Rotate, scale and fill temp buffer with affine transform of the input 1
	ByteAffine  (in1_lum,  temp_lum, la, lb, lc, ld, le, lf);
	ByteAffine  (in1_cr,  temp_cr, ca, cb, cc, cd, ce, cf);
	ByteAffine  (in1_cb,  temp_cb, ca, cb, cc, cd, ce, cf);

	ByteImage *tclip = ByteClip (in1_lum, 0, 0, 1, 1);
	ByteImage *oclip = ByteClip (out_lum, 0, 0, 1, 1);

	if (x1_ > 0) {
		ByteReclip (in2_lum, 0, 0, x1_, buffer_[0]->h_, tclip);
		ByteReclip (out_lum, 0, 0, x1_, buffer_[0]->h_, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in2_cr, 0, 0, x1_/hs, buffer_[0]->h_/vs, tclip);
		ByteReclip (out_cr, 0, 0, x1_ / hs, buffer_[0]->h_ / vs, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in2_cb, 0, 0, x1_ / hs, buffer_[0]->h_ / vs, tclip);
		ByteReclip (out_cb, 0, 0, x1_ / hs, buffer_[0]->h_ / vs, oclip);
		ByteCopy (tclip, oclip);
	}

	if (y1_ > 0) {
		ByteReclip (in2_lum, x1_, 0, x2_ - x1_ + 1, y1_, tclip);
		ByteReclip (out_lum, x1_, 0, x2_ - x1_ + 1, y1_, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in2_cr, x1_/hs, 0, (x2_-x1_+1)/hs, y1_/vs, tclip);
		ByteReclip (out_cr, x1_/hs, 0, (x2_-x1_+1)/hs, y1_/vs, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in2_cb, x1_/hs, 0, (x2_-x1_+1)/hs, y1_/vs, tclip);
		ByteReclip (out_cb, x1_/hs, 0, (x2_-x1_+1)/hs, y1_/vs, oclip);
		ByteCopy (tclip, oclip);
	}

	if (x2_ < buffer_[0]->w_ - 1) {
		ByteReclip (in1_lum, x2_+1, 0, buffer_[0]->w_-x2_-1, buffer_[0]->h_, tclip);
		ByteReclip (out_lum, x2_+1, 0, buffer_[0]->w_-x2_-1, buffer_[0]->h_, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in1_cr, (x2_+1)/hs, 0, 
				(buffer_[0]->w_-x2_-1)/hs, buffer_[0]->h_/vs, 
				tclip);
		ByteReclip (out_cr, (x2_+1)/hs, 0, 
				(buffer_[0]->w_-x2_-1)/hs, buffer_[0]->h_/vs, 
				oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in1_cb, (x2_+1)/hs, 0, 
				(buffer_[0]->w_-x2_-1)/hs, buffer_[0]->h_/vs, 
				tclip);
		ByteReclip (out_cb, (x2_+1)/hs, 0, 
				(buffer_[0]->w_-x2_-1)/hs, buffer_[0]->h_/vs, 
				oclip);
		ByteCopy (tclip, oclip);
	}

	if (y2_ < buffer_[0]->h_ - 1) {
		ByteReclip (in1_lum, x1_, y2_+1, x2_-x1_+1, buffer_[0]->h_-y2_-1, tclip);
		ByteReclip (out_lum, x1_, y2_+1, x2_-x1_+1, buffer_[0]->h_-y2_-1, oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in1_cr, x1_/hs, (y2_+1)/vs, 
				(x2_-x1_+1)/hs, (buffer_[0]->h_-y2_-1)/vs, 
				tclip);
		ByteReclip (out_cr, x1_/hs, (y2_+1)/vs,
				(x2_-x1_+1)/hs, (buffer_[0]->h_-y2_-1)/vs, 
				oclip);
		ByteCopy (tclip, oclip);

		ByteReclip (in1_cb, x1_/hs, (y2_+1)/vs, 
				(x2_-x1_+1)/hs, (buffer_[0]->h_-y2_-1)/vs, 
				tclip);
		ByteReclip (out_cb, x1_/hs, (y2_+1)/vs, 
				(x2_-x1_+1)/hs, (buffer_[0]->h_-y2_-1)/vs, 
				oclip);
		ByteCopy (tclip, oclip);
	}

	BitImage *diag_bit_clip = BitClip (diag_bit_mask, 
			(diag_bit_mask->unitWidth - (x2_-x1_+1)) / 2, 
			(diag_bit_mask->height - (y2_-y1_+1)) / 2,
			x2_-x1_+1, y2_-y1_+1);
	BitImage *inv_diag_bit_clip  = BitClip (inv_diag_bit_mask, 
			(diag_bit_mask->unitWidth - (x2_-x1_+1)) / 2,
			(diag_bit_mask->height - (y2_-y1_+1)) / 2, 
			x2_-x1_+1, y2_-y1_+1);

	ByteReclip (in2_lum, x1_, y1_, x2_-x1_+1, y2_-y1_+1, tclip);
	ByteReclip (out_lum, x1_, y1_, x2_-x1_+1, y2_-y1_+1, oclip);

	ByteCopyWithMask (tclip, diag_bit_clip, oclip);
	ByteCopyWithMask (temp_lum, inv_diag_bit_clip, oclip);

	BitReclip (c_diag_bit_mask,  
			(c_diag_bit_mask->unitWidth - ((x2_-x1_+1) / hs)) / 2,
			(c_diag_bit_mask->height - ((y2_-y1_+1) / vs)) / 2,
			(x2_-x1_+1) / hs, (y2_-y1_+1) / vs, diag_bit_clip);
	BitReclip (c_inv_diag_bit_mask, 
			(c_diag_bit_mask->unitWidth - ((x2_-x1_+1) / hs)) / 2,
			(c_diag_bit_mask->height - ((y2_-y1_+1) / vs)) / 2,
			(x2_-x1_+1) / hs, (y2_-y1_+1) / vs, inv_diag_bit_clip);

	// ByteReclip (out_cr, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, oclip);
	// ByteSet (oclip, 128);

	// ByteReclip (out_cb, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, oclip);
	// ByteSet (oclip, 128);

	ByteReclip (in2_cr, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, tclip);
	ByteReclip (out_cr, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, oclip);

	ByteCopyWithMask (tclip, diag_bit_clip, oclip);
	ByteCopyWithMask (temp_cr, inv_diag_bit_clip, oclip);

	ByteReclip (in2_cb, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, tclip);
	ByteReclip (out_cb, x1_/hs, y1_/vs, (x2_-x1_+1)/hs, (y2_-y1_+1)/vs, oclip);

	ByteCopyWithMask (tclip, diag_bit_clip, oclip);
	ByteCopyWithMask (temp_cb, inv_diag_bit_clip, oclip);

	BitFree (diag_bit_clip);
	BitFree (inv_diag_bit_clip);
	ByteFree (tclip);
	ByteFree (oclip);

	// deliver the output frame
	deliver_frame();

	return;
}

