#! /usr/bin/perl -a 
# -a option automatically splits input records into fields

# Build web pages for the matcompat database.
#
# The database is composed of categories of functions, with 
# categories broken out into sections.  Each category has
# a name and a description.  Each section has a description.
# Each function has a status (in octave, in matcompat, not
# in matcompat but available elsewhere or workaround available,
# or not available in octave), a name, a description and notes.
# See the database in README for the format of the database.
#
# Each category is introduced by category(name,description) and
# closed by endcategory().  $incategory is true between 
# category() and endcategory() calls. Similarly, each section is 
# introduced by section(description), closed by endsection() and 
# flagged by $insection.  Functions are introduced by an 
# entry(status,name,description), followed by any number of 
# note(text) calls and ended by endnote().  If there is no 
# entry(), then $inentry is false and note() refers to the  
# section() if $insection, category() if $incatgory or the
# entire database if neither.  
#
# Since the database is too large to fit on one web page (even 
# without markup), the main index.html points to a separate
# file cat-name.html for each category.  Within each category,
# each section is in a separate table.  At present there are no
# section, category or database notes, so no special handling
# is done.  The layout should be easy to change by modifying the
# appropriate start/end functions.
#
# In addition to generating the detailed index, this program 
# generates summary.html listing which functions are present 
# in octave, matcompat, elsewhere or nowhere.  It isn't clear
# if these should be in separate files or not.
#
# Entries are dumped into the file ENTRY, the main index in MAIN 
# and the summary entries into SUMMARY.

$target="HTML/";

# remove old files
unlink glob "index.html ".$target."summary.html ".$target."cat-*.html";

# find .m and .cc files in the tree so that we can link directly to
# them from the function name.
use File::Find ();
sub buildtree {
    if ( exists $tree{$_} ) {
	print STDOUT "Duplicate names: $File::Find::name and $tree{$_}.\n";
    } elsif ( /[.](m|cc)$/ ) {
	$tree{$_} = $File::Find::name;
    }
}
&File::Find::find(\&buildtree, '.');

# color coding for entry types (foreground and background)
%bg = ('*', 'white', 'x', 'white', '+', 'black', '?', 'black');
%fg = ('*', 'green', 'x', 'red', '+', 'yellow', '?', 'orange');

# Entry/note processing.  Currently notes are dumped into the 
# same box as the function description, but this may change.
$inentry = 0;
$innote = 0;
sub endnote
{
    if ($innote) {
#	print ENTRY "</pre>";
	$innote = 0;
    }
    $inentry = 0;
}
sub note($)
{
    my ( $note ) = @_;
    while ( $note =~ /(@([^@]*)@)/ ) {
	if ( exists $abbrev{$1} ) {
	    $rep = $abbrev{$1};
	    $note =~ s/$1/$rep/;
	} else {
	    $note =~ s/$1/$2/;
	}
    }
#    if (!$innote) { print ENTRY "<pre>"; }
    if (!$innote) { print ENTRY "<br>"; }
    else { print ENTRY "<br>"; }
    print ENTRY $note;
    $innote = 1;
}
sub entry($$$)
{
    my ($code, $name, $description) = @_;
    &endnote;
    $inentry = 1;
    push @inoctave, $name if $code eq '*';
    push @inmatcompat, $name if $code eq '+';
    push @inworkaround, $name if $code eq '?';
    push @inmissing, $name if $code eq 'x';
    $ref = "<font color=$fg{$code}>$name</font>";
    if ( exists $tree{$name.'.cc'} ) {
	$ref = "<a href=\"../$tree{$name.'.cc'}\">$ref</a>";
    } elsif ( exists $tree{$name.'.m'} ) {
	$ref = "<a href=\"../$tree{$name.'.m'}\">$ref</a>";
    }
    print ENTRY "<tr><td valign=top>$code<td width=80 valign=top bgcolor=$bg{$code}>$ref<td valign=top>$description\n";
}

# Section processing.  Each section is in a separate table, with nothing
# special connecting the sections.
$insection = 0;
$suppress_summary = 0; 
sub sectionsummary($@) {
    my ($code, @list) = @_;
    if ( $#list+1 ) {
	print SUMMARY "<tr><td valign=top>$code";
	print SUMMARY "<td bgcolor=$bg{$code}><font color=$fg{$code}>\n";
	print SUMMARY "@list</font>\n";
    }
}
sub endsection {
    &endnote;
    if ( $insection ) {
	print ENTRY "</table>\n";
	if ( ! $suppress_summary ) {
            # stupid hack: since I use a fake section table as
	    # documentation, I need to suppress this in the summary
	    print SUMMARY "<dd><table>\n";
	    sectionsummary('*',@inoctave);
	    sectionsummary('+',@inmatcompat);
	    sectionsummary('?',@inworkaround);
	    sectionsummary('x',@inmissing);
	    print SUMMARY "</table>\n";
	}
	$insection = 0;
    }
}
sub section($)
{
    my ( $name ) = @_;
    &endsection;
    print ENTRY "<p>$name\n";
    print ENTRY "<table border=1 cellspacing=2 cellpadding=1>\n";
    print SUMMARY "<dt>$name\n" if $incategory;


    @inoctave = ();
    @inmatcompat = ();
    @inworkaround = ();
    @inmissing = ();

    $insection = 1;
}

# Category processing.  Each category is in a separate file, with a link
# from the main index.
$incategory = 0;
sub endcategory {
    if ($incategory) {
	&endsection;
	print ENTRY "</body></html>\n";
	close ENTRY;

	print SUMMARY "</dl>\n";
	$incategory = 0;
    }
}
sub category($$)
{
    my ( $name, $description ) = @_;
    $file = "cat-$name.html";

    &endcategory;

    print MAIN "<li><a href=\"$target$file\">$description</a>\n";
    print SUMMARY "<li><a href=\"$file\">$description</a><dl>\n";

    # dump fake section with explanation of codes
    # but don't let it show up in the summary
    $suppress_summary = 1; 
    open(ENTRY, ">$target$file");
    print ENTRY "<html><head><title>Matcompat: $description</title><body>";
    print ENTRY "<H2>$description</H2>\n";
    section("Explanation of tables");
    entry('*', $fg{'*'}.' ', 'Function exists in octave 2.0.x');
    note('Noted incompatibilities and missing features');
    entry('+', $fg{'+'}.' ', 'Function exists in matcompat');
    note('Author and home for the function.');
    entry('?', $fg{'?'}.' ', 'Function doesn\'t exist');
    note('Explanation of what you can do instead.');
    entry('x', $fg{'x'}.' ', 'Function doesn\'t exist, and no work-around');
    &endsection;
    $suppress_summary = 0;

    $incategory = 1;
}

# set up html document header for index.html
open(MAIN, ">index.html") or die ("could not create index.html");
use POSIX "strftime";
$DATE = strftime("%Y.%m.%d", localtime);
print MAIN <<"EOF";
<html>
<!-- Automatically generated file: to modify, edit makeweb.pl and rebuild -->
<head><title>Octave <-> Matlab Compatibility Database</title></head>
<body>
<h1>Octave <-> Matlab Compatibility Database</h1>
Download the complete set from
<a href="http://users.powernet.co.uk/kienzle/octave/matcompat-$DATE.tar.gz">
http://users.powernet.co.uk/kienzle/octave/matcompat-$DATE.tar.gz</a>.<p>

This is free software.  See the file <a href="licenses">licenses</a> for
details.<p>

These functions were gathered from various sources on the web.  As
much as possible, these sources are noted in the file compat.dat, and
linked into these pages.  The functions included herein may have been
modified against the originals.  They may or may not have accepted the
changes, and they may have made subsequent changes, so be careful when
you go back to the original sources for updates.<p>

There may be functions in the distribution which are not listed on
these pages (e.g., extensions, demos and tests which are not part of
Matlab).  Check the <a href="scripts">individual directories</a> for a
complete list.<p>

A number of functions depend on third party libraries, and have not
been included here.  Look to
<a href="http://users.powernet.co.uk/kienzle/octave/index.html">
http://users.powernet.co.uk/kienzle/octave/index.html</a> for those.<p>

Some compatibility issues can only be resolved by modifying Octave itself.
A number of <a href="patches">patches</a> have been included in this
distribution.  Many of the patches will have already been applied if you
are using bleeding-edge Octave.  Most will not apply if you are using the
2.0 series.  Chances are that the patches will not apply cleanly to your
version and may even conflict with each other.  Do yourself a favour and
a get a recent version of Octave if you need the functionality.<p>

<h2><a href="HTML/summary.html">Summary</a></h2>

<h2>Detailed listings</h2>
<ol>
EOF

# set up html document header for summary.html
open(SUMMARY, ">".$target."summary.html") or die ("could not create summary.html");
print SUMMARY <<"EOF";
<html>
<!-- Automatically generated file: do not edit -->
<head><title>Octave <-> Matlab Compatibility Summary</title>
<body>
<h1>Octave <-> Matlab Compatibility Summary</h1>
<ol>
EOF

# process database    
open(FILE, "compat.dat") or die ("could not find README");
while(<FILE>) {
    next if /^#/; # skip comments
    next if /^[ \t.]*$/; # skip empty lines
    if ( /^(@.*@)\s*(\S.*\S)\s*$/ ) {
	$abbrev{$1} = $2;
    } elsif ( /^[.]\s*(\S.*\S)\s*>>\s*(\S.*|)$/ ) {
	category($1,$2);
    } elsif ( /^[.]\s*(\w.*|)$/ ) {
	section($1);
    } elsif ( /^:[?+](.*)$/ ) {
	note($1); 
    } elsif ( /^([ +*?x])\s*([a-zA-Z_0-9.]*)[- \t]*(\S.*|)$/ ) {
	entry($1,$2,$3) if /^([ +*?x])\s*([a-zA-Z_0-9.]*)[- \t]*(\S.*|)$/;
    } else {
	print "ignoring: $_";
    }
}
&endcategory;
close FILE;

# close off summary list
print MAIN "</ol>";
print SUMMARY <<"EOF";
</ol></body></html>
EOF
close SUMMARY;

# Add author list to main
print MAIN "<h2>Authors</h2>\nMatcompat collects the labours of:\n<ul>\n";
open AUTHORS, "sed -n 's/<.*>//p' licenses|";
while (<AUTHORS>) { print MAIN "<li>$_" }
close AUTHORS;
print MAIN "</ul>\nMy apologies for missing and misspeled names.\n";

# Add editor list to main
print MAIN "<hr>\nPaul Kienzle (<a href=\"mailto:pkienzle\@kienzle.powernet.co.uk\">pkienzle\@kienzle.powernet.co.uk</a>)<br>\n";
$now = localtime(time);
print MAIN $now;
print MAIN "\n</body></html>";
close MAIN;
