
#include "config.h"

#ifdef __SOLARIS__

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include <math.h>
#include "data.h"
#include <kstat.h>

/* Solaris */
#include <kstat.h>
//#define set_bit(x,y) ((*(y))|=(1L<<(x-1)))
//#define clear_bit(x,y) ((*(y))&=(~(1L<<(x-1))))
//#define change_bit(x,y) ((*(y))^=(1L<<(x-1)))
//#define test_bit(x,y) (((*(y))&(1<<(x-1)))?!0L:0L)
//#define MAX(x,y) (((x)<(y))?(y):(x))

/* size of buffer to copy /proc/net/xxx to */
#define BUFSIZE 2048

/********** Global variables **********/
int type = 0;			/* What kind of data is gathered */
count_t average = {(float)0,(float)0};   /* average count */
count_t max = {(float)0,(float)0};       /* maximum count */
count_t total = {(float)0,(float)0};     /* total count   */

/********** Static variables **********/
static char *iface_name;   /* Name of the interface to be queried. */
static count_t last = {(float)0,(float)0};	/* Previously read values. */
static float *inarray;     /* Array for receive counts. */
static float *outarray;	   /* Array for transmit counts. */
static int numavg;         /* number of samples to average */

static kstat_ctl_t *kc = NULL;
static kstat_t *if_ksp = NULL;  

/********** Function definitions **********/

/* return values for read_*  */
#define READ_ALLOC_ERR    -5	/* Not enough memory on the stack */
#define READ_FOPEN_ERR    -4	/* Can't open file */
#define READ_FREAD_ERR    -3	/* Can't read file */
#define READ_IFACE_ERR    -2	/* Interface not found */
#define READ_SCAN_ERR     -1	/* Unknown file layout */
#define READ_BYTES         1	/* Function reads byte counts */
#define READ_PACKETS       2	/* Function reads packet counts */

/* Read the byte or packet count for the network interface  
   named in `iface' from /proc/net/dev, store it in the  
   variable pointed to by `pcnt', if `pcnt' is not NULL. */
static int read_dev(count_t * pcnt, char *iface);

/********** Function implementations **********/

void report_error(char *msg)
{
  /* cleanup for ip_acct */
  cleanup();
  /* Write the message to stderr. */
  fprintf(stderr, "error: %s.\n", msg);
  /* Terminate the program. */
  exit(1);
}

int initialize(char *iface, int num_avg/*  , int kb */)
{
  int r;
  
  /* Initialize global variables. */
  iface_name = iface;
  numavg = num_avg;
  average.in = (float)0;
  average.out = (float)0;
  total.in  = (float)0;
  total.out = (float)0;

  inarray = (float*)malloc(numavg*sizeof(float));
  outarray = (float*)malloc(numavg*sizeof(float));
  if (inarray == 0 || outarray == 0) {
    report_error("Memory allocation failed");
  }
  for (r = 0; r < numavg; r++) {
    inarray[r] = (float)0;
    outarray[r] = (float)0;
  }

  r = read_dev(&last, iface);
  switch (r) {
  case READ_FOPEN_ERR:
    report_error("Could not open kstat");
    type = ERROR_TYPE;
    break;
  case READ_IFACE_ERR:
    report_error("Interface not found in kstat");
    type = ERROR_TYPE;
    break;
  case READ_BYTES:
    type = BYTES_TYPE;
    break;
  case READ_PACKETS:
    type = PACKETS_TYPE;
    break;
  default:
    report_error("Unknown return value from read_dev");
    type = ERROR_TYPE;
  }    
  return type;
}  

int cleanup(void)
{
  free(inarray);
  free(outarray);
  return (0);
}

void update_avg(int seconds)
{
  count_t current = {0.0,0.0};
  count_t diff;
  int i;
  static int index;

  /* Quit if invalid number of seconds is given */
  if (seconds <= 0)
    return;
  /* Read the data. */
  i = read_dev(&current, iface_name);
  switch (i) {
  case READ_FOPEN_ERR:
    report_error("Could not open /proc/net/dev");
    break;
  case READ_IFACE_ERR:
    //exit(0);
    break;
  case READ_SCAN_ERR:
    report_error("Error scanning /proc/net/dev");
    break;
  }

  /* Try to detect counter overrun. current and last are floating point
   * values, but current is filled from a %u, and so capped to UINT_MAX */
  if (current.in < last.in) {
    diff.in = current.in + (UINT_MAX - last.in);
    printf("xnetload warning: incoming counter overrun.\n");
  } else {
    diff.in = current.in - last.in;
  }
  if (current.out < last.out) {
    diff.out = current.out + (UINT_MAX - last.out);
    printf("xnetload warning: outgoing counter overrun.\n");
  } else {
    diff.out = current.out - last.out;
  }

  /* Add that to the total */
  total.in  += diff.in;
  total.out += diff.out;

  /* Calculate the difference per update */
  diff.in /= seconds;
  diff.out /= seconds;
  /* Update the arrays. */
  inarray[index] = diff.in;
  outarray[index++] = diff.out;
  if (index == numavg) {
    index = 0;
  }

  /* Calculate the average */
  average.in = average.out = (float)0;
  for (i = 0; i < numavg; i++) {
    average.in += inarray[i];
    average.out += outarray[i];
  }
  average.in /= numavg;
  average.out /= numavg;

  /* Update the maximum. */
  if (average.in > max.in) max.in = average.in;
  if (average.out > max.out) max.out = average.out;

  /* Store current count for further reference. */
  memcpy(&last, &current, sizeof(count_t));
}

int read_dev(count_t * pcnt, char *iface)
{
  char buf[BUFSIZE];
  int num, retval;
  char *pch;
  unsigned long int values[16];
  kstat_t *ksp;
  kstat_named_t *data;

  if (!kc) kc = kstat_open();
  if (!kc) {
    /* Unable to open kstat. */
    return READ_FOPEN_ERR;
  } 
  /* Read the file into a buffer. */
    /* grab all active devices, adding them as we go */ 
  for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
    if (strcmp(ksp->ks_class, "net") == 0) {
      kstat_read(kc, ksp, NULL); 
      if (kstat_data_lookup(ksp, "ipackets") && 
          kstat_data_lookup(ksp, "opackets") &&
          kstat_data_lookup(ksp, "rbytes") &&
          kstat_data_lookup(ksp, "obytes")) {
          if (strstr(ksp->ks_name, iface)){ 
            if_ksp = ksp;
            break;
          }
      }
    }
  }

  /* Seek the interface we're looking for. */
  if (!if_ksp){
    /* Interface not found. */
    return READ_IFACE_ERR;
  }
  retval = READ_PACKETS;
  data = (kstat_named_t *)kstat_data_lookup(if_ksp, "rbytes");
  pcnt->in = (float)data->value.ui32;
  data = (kstat_named_t *)kstat_data_lookup(if_ksp, "obytes");
  pcnt->out = data->value.ui32;
  return retval;
}

#endif /* __SOLARIS__ */
