/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <Winsock2.h>
#else
#include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "net_irc_client.h"

#define DEFAULT_NICK      "nonick"
#define DEFAULT_REAL_NAME "not set"



struct _MIRCCallbacks {
  void (*notify_connected)(MNetIRCClient *sender);
  void (*notify_join)(MNetIRCClient *sender,
                      const char *channel, MIRCUser *user);
  void (*notify_name_list_ready)(MNetIRCClient *sender, const char *channel);
  void (*notify_leave)(MNetIRCClient *sender,
                       const char *channel, MIRCUser *user);
  void (*notify_message)(MNetIRCClient *sender,
                         MIRCMessageType type,
                         const char *channel, MIRCUser *user,
                         const char *msg);
  void (*notify_topic)(MNetIRCClient *sender,
                      const char *channel, const char *topic);
  void (*notify_kick)(MNetIRCClient *sender,
                      MIRCUser *kicker, MIRCUser *kicked,
                      const char *channel, const char *comment);
  void (*notify_invite)(MNetIRCClient *sender,
                        const char *inviter, const char *invitee,
                        const char *channel);
  void (*notify_quit)(MNetIRCClient *sender, const char *user,
                      const char *comment);
  void (*notify_rename)(MNetIRCClient *sender, 
                        const char *old_nick, const char *new_nick);
  
  int (*notify_error)(MNetIRCClient *sender,
                      int error); /* non-fatal errors */
};



#define SEND(cli, data) mnet_send_line(cli->client, data), g_free(data);

#define LIST_ADD(list, lsize, item) do {\
  lsize++;\
  list= g_realloc(list, sizeof(item)*lsize);\
  list[lsize-1]= item;\
} while (0)
#define LIST_DEL(list, lsize, index, size) do { if (index+1 < lsize) memmove(list+index, list+index+1, (size)*(lsize-index)); lsize--; } while (0)

#define SKIP_QUOTE(s) (*s==':' ? s+1 : s)


static int irc_debug=2;

static int handle_incoming(MNetIRCClient *client, const char *status,
                           char *line);

static int handle_who_list(MNetIRCClient *client, char **args);

static int handle_name_list(MNetIRCClient *client, const char *channel,
                            const char *names);

static int handle_channel_topic(MNetIRCClient *client,
                                const char *channel,
                                const char *topic);

static char *get_nick_part(const char *str)
{
  char *res= g_strdup(str);
  char *end= g_utf8_strchr(res, strlen(res), '!');
  if (end)
    *end= 0;

  return res;
}


static MIRCUser *create_user()
{
  MIRCUser *user= g_malloc0(sizeof(MIRCUser));
  user->refcount= 1;
  return user;
}


static MIRCUser *break_down_user_info(MNetIRCClient *client, const char *user)
{
  MIRCUser *info= create_user();
  const char *beg= user;
  const char *ptr= strchr(beg, '!');

  if (!ptr)
    goto error;

  info->nick= g_strndup(beg, ptr-beg);
  beg= ptr+1;
  ptr= strchr(beg,'@');
  if (!ptr)
  {
    g_free(info->nick);
    goto error;
  }
  info->user_name= g_strndup(beg, ptr-beg);
  
  if (nick_equal(client->my_nick, info->nick))
    info->flags |= MIUIsMyself;

  return info;
  
error:
  g_free(info);
  g_warning("error in user specification: %s", user);
  return NULL;
}


int channel_equal(const char *nick1, const char *nick2)
{
  gunichar c1, c2;
  
  while (*nick1 && *nick2)
  {
    c1= g_unichar_tolower(g_utf8_get_char(nick1));
    c2= g_unichar_tolower(g_utf8_get_char(nick2));
    if (c1 == '[') c1= '{'; /* wierd scand collation */
    else if (c1 == ']') c1= '}'; /* wierd scand collation */
    else if (c1 == '\\') c1= '|'; /* wierd scand collation */
    if (c2 == '[') c2= '{'; /* wierd scand collation */
    else if (c2 == ']') c2= '}'; /* wierd scand collation */
    else if (c2 == '\\') c2= '|'; /* wierd scand collation */
    if (c1 != c2)
      return 0;

    nick1= g_utf8_next_char(nick1);
    nick2= g_utf8_next_char(nick2);
  }

  return *nick1 == *nick2;
}


int nick_equal(const char *nick1, const char *nick2)
{  
  if (*nick1 == '@' || *nick1 == '&') nick1++;
  if (*nick2 == '@' || *nick2 == '&') nick2++;

  return channel_equal(nick1, nick2);
}




MNetIRCClient *mnet_init_irc(int use_ssl)
{
  MNetIRCClient *irc= g_malloc0(sizeof(MNetIRCClient));

  irc->callbacks= g_malloc0(sizeof(MIRCCallbacks));
  
  if (use_ssl)
    irc->client= mnet_init_ssl_client(64);
  else
    irc->client= mnet_init_client(64);
  irc->my_nick= g_strdup(DEFAULT_NICK);

  irc->my_user_name= g_strdup(g_get_user_name());
  
  irc->my_real_name= g_strdup(DEFAULT_REAL_NAME);

  {
    char host[128];
    char domain[256];

    if (gethostname(host, sizeof(host)) < 0)
    {
      host[0]= 0;
    }
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
    domain[0]= 0;
#else
    if (getdomainname(domain, sizeof(domain)) < 0)
    {
      domain[0]= 0;
    }
#endif

    if (*host && *domain)
    {
      irc->my_net_name= g_strdup_printf("%s.%s", host, domain);
    }
  }

  irc->state= MISDisconnected;
  
  return irc;
}


int mnet_irc_get_fd(MNetIRCClient *cli)
{
  return cli->client->socket;
}


void mnet_irc_set_my_name(MNetIRCClient *cli, const char *nick, const char *username, const char *realname)
{
  g_return_if_fail(cli->state < MISConnected);

  g_free(cli->my_nick);
  cli->my_nick= g_strdup(nick);

  g_free(cli->my_real_name);
  cli->my_real_name= g_strdup(realname);

  g_free(cli->my_user_name);
  cli->my_real_name= g_strdup(username);
}


void mnet_irc_set_my_name_password(MNetIRCClient *cli, const char *nick, const char *username, const char *realname, const char *password)
{
  mnet_irc_set_my_name(cli, nick, username, realname);

  cli->my_password= g_strdup(password);
}


int mnet_irc_connect(MNetIRCClient *cli, const char *host, int port)
{
  char *str;

  g_return_val_if_fail(cli->my_real_name && cli->my_nick, -1);

  if (mnet_client_connect(cli->client, host, port) < 0)
    return -1;

  if (cli->my_password)
    str= g_strdup_printf("USER %s foo bar %s :%s", cli->my_user_name,
                         cli->my_password, cli->my_real_name);
  else
    str= g_strdup_printf("USER %s foo bar :%s", cli->my_real_name,
                         cli->my_real_name);
  mnet_send_line(cli->client, str);
  g_free(str);

  str= g_strdup_printf("NICK %s", cli->my_nick);
  mnet_send_line(cli->client, str);
  g_free(str);

  cli->state= MISRegistering;
  return 0;
}


int mnet_irc_process(MNetIRCClient *cli)
{
  char status[64];
  char *line;
  int ready;
  
  g_return_val_if_fail(cli->state >= MISRegistering, -1);

  if (mnet_flush_data(cli->client) < 0)
  {
    if (irc_debug)
      g_warning("error flushing irc data");
    return -1;
  }

  ready= mnet_wait_ready(cli->client, 1);
  if (ready < 0)
  {
    if (irc_debug)
      g_warning("error waiting for irc data");
    return -1;
  } 
  else if (ready & MWReadOK)
  {
    if (mnet_read_data(cli->client) < 0)
    {
      if (irc_debug)
        g_warning("error reading irc data");
      return -1;
    }
  }
  while ((line= mnet_get_line(cli->client)))
  {
    if (irc_debug > 1)
      g_message("srv==> %s", line);

    if (strncmp(line, "PING ", 5)==0)
    {
      line[1]= 'O';
      mnet_send_line(cli->client, line);

      if (irc_debug)
        g_message("PONGING server");
      
      g_free(line);
      continue;
    }

    {
      char *p, *pp;
      p= strchr(line, ' ');
      if (p)
      {
        p++;
        pp= strchr(p, ' ');
        if (!pp || pp-p > (int)sizeof(status)-1)
        {
          g_warning("bad data from server: %s", line);
        }
        else
        {
          strncpy(status, p, pp-p);
          status[pp-p]= 0;
        }
      }
      else
      {
        g_warning("bad data from server: %s", line);
      }
    }

    switch (cli->state)
    {
    case MISRegistering:
      if (strcmp(status, "376")==0) // end of MOTD
      {
        cli->state= MISConnected;

        (*cli->callbacks->notify_connected)(cli);
      } 
      else if (*status == '4')
      {
        /* error */
        if ((*cli->callbacks->notify_error)(cli, atoi(status)) < 0)
        {
          g_free(line);
          return -1;
        }
      }
      break;
    case MISConnected:
      if (*status == '4')
      {
        /* error */
        if ((*cli->callbacks->notify_error)(cli, atoi(status)) < 0)
        {
          g_free(line);
          return -1;
        }
      }
      else if (isdigit(*status))
      {
        int st= atoi(status);
        switch (st)
        {
        case 332: // topic
          {
            gchar **tokens= g_strsplit(line, " ", 5);
            if (!tokens)
              return -1;
            if (handle_channel_topic(cli, tokens[3], SKIP_QUOTE(tokens[4])) < 0)
            {
              g_free(line);
              g_strfreev(tokens);
              return -1;
            }
            g_strfreev(tokens);
          }
          break;
        case 352: // who list
          {
            gchar **tokens= g_strsplit(line, " ", 11);
            if (!tokens)
              return -1;
            if (handle_who_list(cli, tokens+3) < 0)
            {
              g_strfreev(tokens);
              g_free(line);
              return -1;
            }
            g_strfreev(tokens);
          }
          break;
        case 353: // name list
          {
            gchar **tokens= g_strsplit(line, " ", 6);
            if (!tokens)
              return -1;
            if (handle_name_list(cli, tokens[4],
                                 SKIP_QUOTE(tokens[5])) < 0)
            {
              g_strfreev(tokens);
              g_free(line);
              return -1;
            }
            g_strfreev(tokens);
          }
          break;
        case 366: // end of name list
          {
            gchar **tokens= g_strsplit(line, " ", 5);
            if (!tokens)
              return -1;
            (*cli->callbacks->notify_name_list_ready)(cli, tokens[3]);
            g_strfreev(tokens);
          }
          break;
        }
      }
      else
        if (handle_incoming(cli, status, line) < 0)
        {
          g_free(line);
          return -1;
        }
      break;

    case MISDisconnected:
      g_free(line);
      return -1;
      break;
    }
    
    g_free(line);
  }

  if (mnet_flush_data(cli->client) < 0)
  {
    if (irc_debug)
      g_warning("error flushing irc data");
    return -1;
  }

  return 0;
}


int mnet_irc_set_callbacks(MNetIRCClient *client,
                           void (*notify_connected)(MNetIRCClient *sender),
                           void (*notify_join)(MNetIRCClient *sender, const char *channel, MIRCUser *user),
                           void (*notify_name_list_ready)(MNetIRCClient *sender, const char *channel),
                           void (*notify_leave)(MNetIRCClient *sender, const char *channel, MIRCUser *user),
                           void (*notify_message)(MNetIRCClient *sender, MIRCMessageType message_type, const char *channel, MIRCUser *user, const char *msg),
                           void (*notify_topic)(MNetIRCClient *sender, const char *channel, const char *topic),
                           void (*notify_kick)(MNetIRCClient *sender, MIRCUser *kicker, MIRCUser *kicked, const char *channel, const char *comment),
                           void (*notify_invite)(MNetIRCClient *sender, const char *inviter, const char *invitee, const char *channel),
                           void (*notify_quit)(MNetIRCClient *sender, const char *user, const char *comment),
                           void (*notify_rename)(MNetIRCClient *sender, const char *old_nick, const char *new_nick),
                           int (*notify_error)(MNetIRCClient *sender, int error))
{
  client->callbacks->notify_connected= notify_connected;
  client->callbacks->notify_join= notify_join;
  client->callbacks->notify_name_list_ready= notify_name_list_ready;
  client->callbacks->notify_leave= notify_leave;
  client->callbacks->notify_message= notify_message;
  client->callbacks->notify_topic= notify_topic;
  client->callbacks->notify_kick= notify_kick;
  client->callbacks->notify_invite= notify_invite;
  client->callbacks->notify_quit= notify_quit;
  client->callbacks->notify_rename= notify_rename;
  client->callbacks->notify_error= notify_error;
  
  return 0;
}






void mnet_irc_user_retain(MIRCUser *user)
{
  user->refcount++;
}


void mnet_irc_user_release(MIRCUser *user)
{
  user->refcount--;
  if (user->refcount <= 0)
  {
    g_free(user->nick);
    g_free(user->real_name);
    g_free(user->user_name);
    g_free(user);
  }
}




static MIRCChannel *find_channel(MIRCChannel **channels, int count,
                                 const char *name)
{
  int i;
  for (i= 0; i < count; i++)
  {
    if (channel_equal(name, channels[i]->name))
    {
      return channels[i];
    }
  }
  return NULL;
}


static MIRCUser *find_user(MIRCUser **users, int count, const char *nick)
{
  int i;
  for (i= 0; i < count; i++)
  {
    if (nick_equal(nick, users[i]->nick))
    {
      return users[i];
    }
  }
  return NULL;
}


MIRCUser *mnet_irc_find_user(MNetIRCClient *cli, const char *channel, const char *nick)
{
  MIRCChannel *chan= find_channel(cli->channels, cli->channels_num, channel);
  
  if (chan)
  {
    return find_user(chan->users, chan->users_num, nick);
  }
  return NULL;
}


static void free_channel(MNetIRCClient *client, MIRCChannel *channel)
{
  unsigned int i;

  for (i= 0; i < client->channels_num; i++)
  {
    if (client->channels[i] == channel)
    {
      LIST_DEL(client->channels, client->channels_num, i, sizeof(MIRCChannel*));
      break;
    }
  }
  
  for (i= 0; i < channel->users_num; i++)
    mnet_irc_user_release(channel->users[i]);
  g_free(channel->name);
  g_free(channel->topic);
  g_free(channel->users);
  g_free(channel);
}


static MIRCUser *channel_add_user(MNetIRCClient *client,
                            MIRCChannel *channel,
                            const char *sender)
{
  unsigned int i;
  char *nick= g_strdup(get_nick_part(sender));

  /* add to channel's user list */
  for (i= 0; i < channel->users_num; i++)
  {
    if (nick_equal(channel->users[i]->nick, nick))
      break;
  }
  if (i == channel->users_num) /* not yet in list, append */
  {
    channel->users_num++;
    channel->users= g_realloc(channel->users, sizeof(MIRCUser*)*channel->users_num);
    channel->users[i]= break_down_user_info(client, sender);
  }
  g_free(nick);

  return channel->users[i];
}


static int handle_who_list(MNetIRCClient *client, char **args)
{
  MIRCChannel *chan;

  /* <channel> <user> <host> <server> <nick> <H|G>[*][@|+] :<hopcount> <real name> */

  chan= find_channel(client->channels, client->channels_num, args[0]);
  if (!chan)
  {
    if (irc_debug)
      g_warning("ignoring WHO response to unknown channel %s", args[0]);
    return 0;
  }

  {
    MIRCUser *user= find_user(chan->users, chan->users_num, args[4]);

    if (!user)
      return 0;

    g_free(user->user_name);
    user->user_name= g_strdup(args[1]);
    g_free(user->real_name);
    user->real_name= g_strdup(args[7]);

    user->flags= 0;
    if (nick_equal(client->my_nick, user->nick))
      user->flags |= MIUIsMyself;
    if (strchr(args[5], '@'))
      user->flags |= MIUOperator;
    if (strchr(args[5], '+'))
      user->flags |= MIUVoice;
  }
  return 0;
}


static int handle_name_list(MNetIRCClient *client, const char *channel,
                            const char *names)
{
  int i, j;
  MIRCChannel *chan;
  gchar **namel;

  chan= find_channel(client->channels, client->channels_num, channel);
  if (!chan)
  {
    g_warning("got name list for unknown channel %s", channel);
    return 0;
  }
  namel= g_strsplit(names, " ", -1);
  if (!namel)
    return -1;

  j= chan->users_num;
  for (i= 0; namel[i] && *namel[i]; i++)
    chan->users_num++;
  chan->users= g_realloc(chan->users, sizeof(MIRCUser*)*chan->users_num);
  for (i= 0; namel[i] && *namel[i]; i++)
  {
    chan->users[j+i]= create_user();
    chan->users[j+i]->nick= namel[i];
    if (nick_equal(client->my_nick, namel[i]))
      chan->users[j+i]->flags |= MIUIsMyself;
  }
  g_free(namel);

  return 0;
}


static int handle_join(MNetIRCClient *client, 
                       const char *sender,
                       const char *channel)
{
  char *nick= get_nick_part(sender);
  MIRCChannel *chan;

  g_message("%s joins %s", sender, channel);

  /* check whether channel already known */
  chan= find_channel(client->channels, client->channels_num, channel);
  if (chan)
  {
    if (nick_equal(nick, client->my_nick))
    {
      g_warning("got join message for an already joined channel");
    }
    else
    {
      MIRCUser *user;
      /* new user joining in a channel we're in */
      if ((user=channel_add_user(client, chan, sender))==0)
      {
        g_free(nick);
        return -1;
      }
      (*client->callbacks->notify_join)(client, chan->name, user);

      /* send a WHO to that user */
      SEND(client, g_strdup_printf("WHO %s", nick));
    }
  }
  else
  {
    if (nick_equal(nick, client->my_nick))
    {
      MIRCUser *user;

      /* means we have joined the channel */
      chan= g_malloc0(sizeof(MIRCChannel));
      chan->name= g_strdup(channel);

      LIST_ADD(client->channels, client->channels_num, chan);

      user= break_down_user_info(client, sender);

      (*client->callbacks->notify_join)(client, chan->name, user);

      mnet_irc_user_release(user);
      
      /* send a WHO to the channel */
      SEND(client, g_strdup_printf("WHO %s", channel));
    }
    else
    {
      /* someone else joined an unknown channel */
      g_warning("got join message for unknown channel %s", channel);
    }
  }
  g_free(nick);
  return 0;
}


static int remove_user_from_channel(MNetIRCClient *client, MIRCUser *user,
                                    const char *channel)
{
  MIRCChannel *chan= NULL;

  /* locate the channel */
  chan= find_channel(client->channels, client->channels_num, channel);
  if (!chan)
  {
    /* unknown channel!? */
    g_message("got leave for an unjoined channel");
    return 0;
  }

  /* check if ourself */
  if (nick_equal(user->nick, client->my_nick))
  {
    /* free whole channel */
    free_channel(client, chan);
  }
  else
  {
    unsigned int j;
    int ok=0;

    /* locate user */
    for (j= 0; j < chan->users_num; j++)
    {
      if (user == chan->users[j])
      {
        mnet_irc_user_release(chan->users[j]);
        LIST_DEL(chan->users, chan->users_num, j, sizeof(MIRCUser*));
        ok= 1;
        break;
      }
    }
    if (!ok)
    {
      g_warning("error removing user %s from channel %s",
                user->nick, chan->name);
    }
  }
  return 0;
}


static int handle_leave(MNetIRCClient *client, 
                       const char *sender,
                       const char *channel)
{
  char *nick= get_nick_part(sender);
  MIRCChannel *chan;
  MIRCUser *user;
  int res;

  g_message("%s leaves %s", sender, channel);

  chan= find_channel(client->channels, client->channels_num, channel);
  if (!chan)
  {
    g_free(nick);
    g_warning("leave from unknown channel?");
    return -1;
  }
  user= find_user(chan->users, chan->users_num, nick);
  
  mnet_irc_user_retain(user);

  res= remove_user_from_channel(client, user, channel);

  (*client->callbacks->notify_leave)(client, channel, user);

  g_free(nick);

  mnet_irc_user_release(user);
  
  return res;
}


static int handle_channel_topic(MNetIRCClient *client,
                                const char *channel,
                                const char *topic)
{
  MIRCChannel *chan;
  
  chan= find_channel(client->channels, client->channels_num, channel);
  if (!chan)
  {
    g_warning("got topic change for unknown channel %s", channel);
    return 0;
  }
  
  if (chan->topic)
    g_free(chan->topic);
  chan->topic= g_strdup(topic);

  (*client->callbacks->notify_topic)(client, channel, topic);

  return 0;
}


static int handle_message(MNetIRCClient *client,
                          const char *event,
                          const char *sender, 
                          const char *target,
                          const char *message)
{
  MIRCMessageType type;
  MIRCUser *user= NULL;
  
  if (strcmp(event, "NOTICE")==0)
    type= MIMNotice;
  else
  {
    if (*target == '#' || *target == '&')
    {
      char *nick= get_nick_part(sender);
      MIRCChannel *chan= find_channel(client->channels, client->channels_num, target);
      if (!chan)
      {
        g_warning("message to bad channel %s", target);
        return -1;
      }
      type= MIMChannel;
      user= find_user(chan->users, chan->users_num, nick);
      mnet_irc_user_retain(user);
      g_free(nick);
    }
    else
    {
      user= break_down_user_info(client, sender);
      if (!user)
        return 0;

      type= MIMPrivate;
    }
  }

  (*client->callbacks->notify_message)(client, type,
                                       type == MIMChannel ? target : NULL,
                                       user, message);
  if (user)
    mnet_irc_user_release(user);

  return 0;
}


static int handle_kick(MNetIRCClient *client,
                       const char *sender,
                       const char *channel,
                       const char *target,
                       const char *comment)
{
  MIRCUser *kicker;
  MIRCUser *kicked;
  MIRCChannel *chan= find_channel(client->channels, client->channels_num,
                                  channel);
  if (!chan)
  {
    /* kick in an unknown channel */
    return 0;
  }

  kicker= find_user(chan->users, chan->users_num, sender);
  kicked= find_user(chan->users, chan->users_num, target);
  
  if (!kicker || !kicked) /* !? */
    return 0;
  
  (*client->callbacks->notify_kick)(client, kicker, kicked,
                                    channel, comment);

  return remove_user_from_channel(client, kicked, channel);
}


static int handle_quit(MNetIRCClient *client,
                       const char *sender, const char *comment)
{
  char *nick= get_nick_part(sender);
  MIRCChannel *channel;
  MIRCUser *user;
  unsigned int i;

  /* remove user from all channels */
  for (i= 0; i < client->channels_num; i++)
  {
    channel= client->channels[i];
    user= find_user(channel->users, channel->users_num, nick);
    if (user)
    {
      remove_user_from_channel(client, user, channel->name);
    }
  }

  (*client->callbacks->notify_quit)(client, nick, comment);

  g_free(nick);

  return 0;
}


static int handle_invite(MNetIRCClient *client,
                         const char *sender,
                         const char *target,
                         const char *channel)
{
  char *snick= get_nick_part(sender);
  char *tnick= get_nick_part(target);
  int res= 0;

  if (!snick || !tnick)
  {
    if (snick) g_free(snick);
    return -1;
  }

  if (res==0)
    (*client->callbacks->notify_invite)(client, snick, tnick, channel);
  g_free(snick);
  g_free(tnick);

  return res;
}


static int handle_nick_change(MNetIRCClient *client,
                              const char *old_nick, const char *new_nick)
{
  int me= nick_equal(old_nick, client->my_nick);
  /* change users name in all channels */
  //XXX change name 

  if (me)
  {
    g_free(client->my_nick);
    client->my_nick= g_strdup(new_nick);

    (*client->callbacks->notify_rename)(client, NULL, new_nick);
  }
  else
  {
    (*client->callbacks->notify_rename)(client, old_nick, new_nick);
  }

  return 0;
}





static struct {
  char *name;
  int tokens;
} events[]= {
  {"JOIN", 3},
  {"PART", 3},
  {"TOPIC", 4},
  {"PRIVMSG", 4},
  {"NOTICE", 4},
  {"KICK", 5},
  {"INVITE", 4},
  {"NICK", 3},
  {"QUIT", 3},
  {"MODE", 4},
  {0,0}
};

static int handle_incoming(MNetIRCClient *client, const char *status, char *line)
{
  gchar **tokens= NULL;
  int i;
  int res= -1;

  
  for (i= 0; events[i].name; i++)
  {
    if (strcmp(events[i].name, status)==0)
    {
      int c= 0;
      tokens= g_strsplit(line, " ", events[i].tokens);
      while (tokens[c]) c++;
      if (c != events[i].tokens)
      {
        g_strfreev(tokens);
        tokens= NULL;
        g_warning("got bad %s message from server: %s", status, line);
      }
      break;
    }
  }
  if (!tokens || !events[i].name)
  {
    if (irc_debug)
      g_warning("unknown incoming message: %s", status);
    return -1;
  }

  if (strcmp(status, "JOIN")==0)
  {
    res= handle_join(client, SKIP_QUOTE(tokens[0]), SKIP_QUOTE(tokens[2]));
  }
  else if (strcmp(status, "PART")==0)
  {
    res= handle_leave(client, SKIP_QUOTE(tokens[0]), SKIP_QUOTE(tokens[2]));
  }
  else if (strcmp(status, "NICK")==0)
  {
    res= handle_nick_change(client, tokens[0], tokens[2]);
  }
  else if (strcmp(status, "QUIT")==0)
  {
    res= handle_quit(client, SKIP_QUOTE(tokens[0]), SKIP_QUOTE(tokens[2]));
  }
  else if (strcmp(status, "TOPIC")==0)
  {
    res= handle_channel_topic(client, tokens[2], SKIP_QUOTE(tokens[3]));
  }
  else if (strcmp(status, "PRIVMSG")==0 ||
           strcmp(status, "NOTICE")==0)
  {
    res= handle_message(client, status, 
                        SKIP_QUOTE(tokens[0]), SKIP_QUOTE(tokens[2]),
                        SKIP_QUOTE(tokens[3]));
  }
  else if (strcmp(status, "KICK")==0)
  {
    res= handle_kick(client, 
                     SKIP_QUOTE(tokens[0]), tokens[2], tokens[3],
                     SKIP_QUOTE(tokens[4]));
  }
  else if (strcmp(status, "INVITE")==0)
  {
    res= handle_invite(client, 
                       SKIP_QUOTE(tokens[0]), tokens[2], 
                       SKIP_QUOTE(tokens[3]));
  } 
  else if (strcmp(status, "MODE")==0)
  {
    //XXX
    res= 0;
  }
    
  
  if (tokens)
    g_strfreev(tokens);
  
  return res;
}



