/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "MRI.h"

double
fmedian5 (double a, double b, double c, double d, double e)
{
	double t;

	/* Sort ab */
	if (a > b) { t = b; b = a; a = t; }
	/* Sort abc */
	if (b > c) {
		t = c;
		c = b;
		if (a > t) {
			b = a;
			a = t;
		}
		else {
			b = t;
		}
	}
	/* Sort abcd */
	if (c > d) {
		t = d;
		d = c;
		if (b <= t) {
			c = t;
		}
		else {
			c = b;
			if (a <= t) {
				b = t;
			}
			else {
				b = a;
				a = t;
			}
		}
	}
	if (e >= c) /* Order is abcde or abced */
		return c;
	else if (e >= b) /* Order is abecd */
		return e;
	else /* Order is aebcd or eabcd */
		return b;
}

/******************************************************************************/

struct FMedianFilterData {
	struct link *next;
	int	width;
	int	height;
	int	freedata;
	int	mask;
	struct MRI_ScanLine *buf[5];
};

static void
StartFMedianFilter (void *private, int width, int height, int freedata)
{
	struct FMedianFilterData *wd = private;

	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, width, height, TRUE);
}

static void
FMedianFilterRow (void *private, void *data)
{
	struct FMedianFilterData *wd = private;
	int i, x;

	wd->buf[4] = (struct MRI_ScanLine *)data;
	if (wd->buf[0] != (struct MRI_ScanLine *)0) {
		struct MRI_ScanLine *out = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
		double *R[6], *G[6], *B[6];

		for (x = 0; x < 5; x++) {
			R[x] = wd->buf[x]->R;
			G[x] = wd->buf[x]->G;
			B[x] = wd->buf[x]->B;
		}
		R[5] = out->R;
		G[5] = out->G;
		B[5] = out->B;
		for (x = 0; x < 2; x++) {
			R[5][x] = R[2][x];
			G[5][x] = G[2][x];
			B[5][x] = B[2][x];
		}
		for (x = 2; x < wd->width-2; x++) {
			if (wd->mask & TONE_R)
			R[5][x] = fmedian5 (R[2][x],
fmedian5(R[0][x-2], R[1][x-1], R[2][x], R[3][x+1], R[4][x+2]),
fmedian5(R[0][x], R[1][x], R[2][x], R[3][x], R[4][x]),
fmedian5(R[2][x-2], R[2][x-1], R[2][x], R[2][x+1], R[2][x+2]),
fmedian5(R[0][x+2], R[1][x+1], R[2][x], R[3][x-1], R[4][x-2]));
			else R[5][x] = R[2][x];
			if (wd->mask & TONE_G)
			G[5][x] = fmedian5 (G[2][x],
fmedian5(G[0][x-2], G[1][x-1], G[2][x], G[3][x+1], G[4][x+2]),
fmedian5(G[0][x], G[1][x], G[2][x], G[3][x], G[4][x]),
fmedian5(G[2][x-2], G[2][x-1], G[2][x], G[2][x+1], G[2][x+2]),
fmedian5(G[0][x+2], G[1][x+1], G[2][x], G[3][x-1], G[4][x-2]));
			else G[5][x] = G[2][x];
			if (wd->mask & TONE_B)
			B[5][x] = fmedian5 (B[2][x],
fmedian5(B[0][x-2], B[1][x-1], B[2][x], B[3][x+1], B[4][x+2]),
fmedian5(B[0][x], B[1][x], B[2][x], B[3][x], B[4][x]),
fmedian5(B[2][x-2], B[2][x-1], B[2][x], B[2][x+1], B[2][x+2]),
fmedian5(B[0][x+2], B[1][x+1], B[2][x], B[3][x-1], B[4][x-2]));
			else B[5][x] = B[2][x];
		}
		for (x = wd->width-2; x < wd->width; x++) {
			R[5][x] = R[2][x];
			G[5][x] = G[2][x];
			B[5][x] = B[2][x];
		}
		(*wd->next->row) (wd->next->private, out);
		if (wd->freedata)
			free (wd->buf[0]);
	}
	else if (wd->buf[2] != (struct MRI_ScanLine *)0) {
		struct MRI_ScanLine *out = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
		double *R[2], *G[2], *B[2];
		R[0] = out->R;
		G[0] = out->G;
		B[0] = out->B;
		R[1] = wd->buf[2]->R;
		G[1] = wd->buf[2]->G;
		B[1] = wd->buf[2]->B;
		for (x = 0; x < wd->width; x++) {
			R[0][x] = R[1][x];
			G[0][x] = G[1][x];
			B[0][x] = B[1][x];
		}
		(*wd->next->row) (wd->next->private, out);
	}

	for (i = 0; i < 4; i++)
		wd->buf[i] = wd->buf[i+1];
}

static void
CloseFMedianFilter (void *private)
{
	struct FMedianFilterData *wd = private;
	int i, x;
	struct MRI_ScanLine *out;

	for (i = 2; i < 4; i++) {
		double *R[2], *G[2], *B[2];
		out = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
		R[0] = out->R;
		G[0] = out->G;
		B[0] = out->B;
		R[1] = wd->buf[i]->R;
		G[1] = wd->buf[i]->G;
		B[1] = wd->buf[i]->B;
		for (x = 0; x < wd->width; x++) {
			R[0][x] = R[1][x];
			G[0][x] = G[1][x];
			B[0][x] = B[1][x];
		}
		(*wd->next->row) (wd->next->private, out);
	}
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}


struct link *
GenFMedianFilter (struct link *next, int mask)
{
	int i;
	struct FMedianFilterData *wd = malloc (sizeof (struct FMedianFilterData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct FMedianFilterData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartFMedianFilter;
	ep->row = FMedianFilterRow;
	ep->close = CloseFMedianFilter;
	ep->private = wd;
	wd->mask = mask;
	wd->next = next;

	for (i = 0; i < 5; i++)
		wd->buf[i] = (struct MRI_ScanLine *)0;

	return ep;
}
