/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>

#include "MRI.h"
#include "vmedian.h"

struct MyWBData {
	int	width;
	int	height;
	int	freedata;
	int	n;
	fpixel	*data;
	double	*weights;
	MRI_balance *res;
};

void
MyWBStart (void *private, int width, int height, int freedata)
{
	struct MyWBData *wd = private;
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	wd->n = 0;
	wd->data = (fpixel *)malloc(sizeof(fpixel) * width * height);
	wd->weights = (double *)malloc(sizeof(double) * width * height);
	if (wd->data == NULL || wd->weights == NULL) {
		fprintf (stderr, "MedGrayWorld: Unable to allocate memory\n");
	}
}

void
MyWBRow (void *private, void *data)
{
	struct MyWBData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;
	int x;

	if (wd->weights != NULL && wd->data != NULL)
	for (x = 0; x < wd->width; x++) {
		if (G[x] > 0 && R[x] > 0 && B[x] > 0) {
			double max, mag;
			mag = sqrt ((double)G[x] * (double)G[x] +
			            (double)R[x] * (double)R[x] +
			            (double)B[x] * (double)B[x]);
			wd->weights[wd->n] = sqrt (mag);  /* Non-linear scale */
#ifdef DEBUG
			fprintf (stderr, "i=%d R=%d G=%d B=%d mag=%g, wt=%g\n", wd->n, R[x], G[x], B[x], mag, wd->weights[wd->n]);
#endif
			max = G[x];
			if (R[x] > max) max = R[x];
			if (B[x] > max) max = B[x];
			wd->data[wd->n].r = max / R[x];
			wd->data[wd->n].g = max / G[x];
			wd->data[wd->n].b = max / B[x];
			wd->n++;
		}
	}
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

void
MyWBClose (void *private)
{
	struct MyWBData *wd = private;
	fpixel res;

	if (wd->data != NULL && wd->weights != NULL) {
		res = VectorMedianW (wd->n, wd->data, wd->weights, 1.0e-5);
		wd->res->rgain = wd->res->ggain/res.g * res.r;
		wd->res->bgain = wd->res->ggain/res.g * res.b;
	}
	if (wd->data) free (wd->data);
	if (wd->weights) free (wd->weights);
	free (wd);
}

struct link *
GenMyWB (MRI_balance *bp)
{
	struct MyWBData *wd = malloc (sizeof (struct MyWBData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MyWBData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	wd->res = bp;
	ep->start = MyWBStart;
	ep->row = MyWBRow;
	ep->close = MyWBClose;
	ep->private = wd;
	return ep;
}

struct GrayWorldData {
	int	width;
	int	height;
	int	freedata;
	unsigned short R, G, B;
	MRI_balance *res;
};

void
GrayWorldStart (void *private, int width, int height, int freedata)
{
	struct GrayWorldData *wd = private;
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	if (width != 1 || height != 1)
		fprintf (stderr, "GrayWorld: too big width=%d, height=%d\n", width, height);
}

void
GrayWorldRow (void *private, void *data)
{
	struct GrayWorldData *wd = private;
	struct MRI_ScanLine *sl = data;

	wd->R = ((unsigned short *)(sl->R))[0];
	wd->G = ((unsigned short *)(sl->G))[0];
	wd->B = ((unsigned short *)(sl->B))[0];
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

void
GrayWorldClose (void *private)
{
	struct GrayWorldData *wd = private;

	wd->res->rgain = wd->R == 0 ? 256 : (256.0 * wd->G)/wd->R;
	wd->res->ggain = 256;
	wd->res->bgain = wd->B == 0 ? 256 : (256.0 * wd->G)/wd->B;
#ifdef DEBUG
	fprintf (stderr, "Gray world: R=%d G=%d B=%d\n", wd->R, wd->G, wd->B);
	fprintf (stderr, "Gray world: RG=%d GG=%d BB=%d\n", wd->res->rgain, wd->res->ggain, wd->res->bgain);
#endif
	free (wd);
}

struct link *
GenGrayWorld (MRI_balance *bp)
{
	struct GrayWorldData *wd = malloc (sizeof (struct GrayWorldData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct GrayWorldData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	wd->res = bp;
	ep->start = GrayWorldStart;
	ep->row = GrayWorldRow;
	ep->close = GrayWorldClose;
	ep->private = wd;
	return ep;
}

MRI_balance
MRI_EstimateWB (MRI *mri, MRI_Region *region)
{
	MRI_balance res;
	struct link *head;
	int	dx, dy;

	/* Determine base luminosity from camera's green value. */
	MRI_GetBalance (&res, mri, "camera");

	/* Set subsampling parameters. */
	if (region) {
	  dx = region->width/8;
	  dy = region->height/8;
	}
	else {
	  dx = MRI_GetWidth(mri)/8;
	  dy = MRI_GetHeight(mri)/8;
	}
	if (dx < 2) dx = 2; else if (dx & 1) dx--;
	if (dy < 2) dy = 2; else if (dy & 1) dy--;

	/* Create processing filter sequence starting from the rear.
	 * The last link in the pipeline outputs scanlines to the output device.
	 */
	head = GenMyWB (&res);
	head = MRI_GenSubsampler (dx, dy, head);

	if (region)
          MRI_ProcessImageRegion (head, mri, 0, region);
	else
          MRI_ProcessImage (head, mri, 0);
        MRI_FlushPipeline (head);

	return res;
}

MRI_balance
MRI_GrayWorldWB (MRI *mri, MRI_Region *region)
{
	MRI_balance res;
	
	MRI_GetBalance (&res, mri, "camera");
	if (region) {
	  struct link *head;

	  /* Create processing filter sequence starting from the rear.
	   * The last link in the pipeline outputs scanlines to the output device.
	   */
	  head = GenGrayWorld (&res);
	  head = MRI_GenSubsampler (region->width, region->height, head);

          MRI_ProcessImageRegion (head, mri, 0, region);
          MRI_FlushPipeline (head);
	}
	else {
	  res.rgain = res.ggain * (mri->Ravg < 1.0e-5 ? 1.0 : mri->Gavg / mri->Ravg);
	  res.bgain = res.ggain * (mri->Bavg < 1.0e-5 ? 1.0 : mri->Gavg / mri->Bavg);
	}
	return res;
}

struct BinGrayWorldData {
	int	width;
	int	height;
	int	freedata;
	unsigned char *bitset;
	double	R, G, B;
	MRI_balance *res;
};

#define	BINS	128		/* In each color channel. */
#define	LSHIFT	7
#define SHIFT	9		/* # of bits to shift to get bin number. */

void
BinGrayWorldStart (void *private, int width, int height, int freedata)
{
	struct BinGrayWorldData *wd = private;

	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	wd->R = wd->G = wd->B = 0.0;
	wd->bitset = malloc (BINS * BINS * BINS / 8);
	if (wd->bitset == NULL) {
		fprintf (stderr, "BinGrayWorld: unable to allocate memory\n");
	}
	else
		memset (wd->bitset, 0, BINS * BINS * BINS / 8);
}

void
BinGrayWorldRow (void *private, void *data)
{
	struct BinGrayWorldData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;
	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;

	if (wd->bitset) {
	for (x = 0; x < wd->width; x++) {
		int idx, r, g, b;
		r = R[x] >> (SHIFT+3);
		g = G[x] >> SHIFT;
		b = B[x] >> SHIFT;
		idx = (((r << LSHIFT) | g) << LSHIFT) | b;
		if (((wd->bitset[idx] >> (r & 0x7)) & 1) == 0) {
			wd->bitset[idx] |= 1 << (r & 0x7);
			wd->R += R[x] & 0xFE00;
			wd->G += G[x] & 0xFE00;
			wd->B += B[x] & 0xFE00;
		}
	}
	}
	if (wd->freedata) {
		MRI_FreeScanLine (sl);
	}
}

void
BinGrayWorldClose (void *private)
{
	struct BinGrayWorldData *wd = private;

	wd->res->rgain = wd->res->ggain * (wd->R == 0 ? 1.0 : (1.0*wd->G)/wd->R);
	wd->res->bgain = wd->res->ggain * (wd->B == 0 ? 1.0 : (1.0*wd->G)/wd->B);
	if (wd->bitset) free (wd->bitset);
	free (wd);
}

struct link *
GenBinGrayWorld (MRI_balance *bp)
{
	struct BinGrayWorldData *wd = malloc (sizeof (struct BinGrayWorldData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct BinGrayWorldData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	wd->res = bp;
	ep->start = BinGrayWorldStart;
	ep->row = BinGrayWorldRow;
	ep->close = BinGrayWorldClose;
	ep->private = wd;
	return ep;
}

MRI_balance
MRI_BinGrayWorldWB (MRI *mri, MRI_Region *region)
{
	MRI_balance res;
	
	MRI_GetBalance (&res, mri, "camera");
	struct link *head;

	/* Create processing filter sequence starting from the rear.
	 * The last link in the pipeline outputs scanlines to the output device.
	 */
	head = GenBinGrayWorld (&res);
	head = MRI_GenSubsampler (4, 4, head);

	if (region)
          MRI_ProcessImageRegion (head, mri, 0, region);
	else
          MRI_ProcessImage (head, mri, 0);
        MRI_FlushPipeline (head);

	return res;
}
