/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: pictures_month.c,v 1.19 2004/08/27 18:41:38 ostborn Exp $
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"

#ifdef HAVE_LIBGD
#ifdef HAVE_GD_GD_H
#include <gd/gd.h>
#include <gd/gdfonts.h>
#else
#include <gd.h>
#include <gdfonts.h>
#endif
#endif

#include "mconfig.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mlocale.h"
#include "misc.h"

#include "pictures.h"
#include "generate.h"

/* calculating the box
**
** +------------------------------+
** |+---------------++-----------+|
** ||               ||_-_    (2) ||
** ||#              ||   ~-_     ||
** ||##  #    (1)   |+-----------+|
** ||## ##          |+-----------+|
** ||#####          ||       (3) ||
** ||#####          ||           ||
** |+---------------++-----------+|
** +------------------------------+
**
*/

#define IM_BARS	 12

#define IM_CELLSPACING	  4

#define IM_BOX1_HEIGHT	200
#define IM_BOX2_HEIGHT	((IM_BOX1_HEIGHT - 3 * IM_CELLSPACING) / 2)
#define IM_BOX3_HEIGHT	IM_BOX2_HEIGHT

/* gdSmallFont */
#define IM_FONT_HEIGHT	 13
#define IM_FONT_WIDTH	  6

#define IM_BOX1_BAR_WIDTH	 10
#define IM_BOX1_BAR_SHIFT	  2
#define IM_BOX1_BAR_SPACE	 10

#define IM_BOX2_BAR_WIDTH	  8
#define IM_BOX2_BAR_SPACE	  4

#define IM_BOX3_BAR_WIDTH	 IM_BOX2_BAR_WIDTH
#define IM_BOX3_BAR_SPACE	 IM_BOX2_BAR_SPACE

#define IM_BOX1_X1	IM_CELLSPACING * 2 + IM_FONT_HEIGHT
#define IM_BOX1_Y1	IM_CELLSPACING * 2 + IM_FONT_HEIGHT
#define IM_BOX1_X2	IM_BOX1_X1 + IM_BARS * (IM_BOX1_BAR_WIDTH + IM_BOX1_BAR_SPACE)
#define IM_BOX1_Y2	IM_BOX1_Y1 + IM_BOX1_HEIGHT

#define IM_BOX2_X1	IM_BOX1_X2 + IM_CELLSPACING * 3
#define IM_BOX2_Y1	IM_BOX1_Y1
#define IM_BOX2_X2	IM_BOX2_X1 + IM_BARS * (IM_BOX2_BAR_WIDTH + IM_BOX2_BAR_SPACE)
#define IM_BOX2_Y2	IM_BOX2_Y1 + IM_BOX2_HEIGHT

#define IM_BOX3_X1	IM_BOX2_X1
#define IM_BOX3_Y1	IM_BOX2_Y2 + 3 * IM_CELLSPACING
#define IM_BOX3_X2	IM_BOX2_X2
#define IM_BOX3_Y2	IM_BOX3_Y1 + IM_BOX3_HEIGHT

#define IM_WIDTH	IM_BOX2_X2 + IM_CELLSPACING * 2 + IM_FONT_HEIGHT
#define IM_HEIGHT	IM_BOX3_Y2 + IM_CELLSPACING * 2 + IM_FONT_HEIGHT

#define IM_LAB1_Y1	IM_HEIGHT - IM_FONT_HEIGHT - IM_CELLSPACING


#define IM_FILENAME	"monthly_usage.png"

char *create_pic_12_month(mconfig *ext_conf, mlist *history, const char *subpath) {
#ifdef HAVE_LIBGD
	mlist *l = history, *last;
	data_WebHistory max;
	int i = IM_BARS;
	gdImage *im;
	FILE *f;
	char filename[255];
	rgb_tripple rgb;
	char int2str[20]; /* 20 is a guess only !! */
	int first_month = 0;
	int x1, y1, x2, y2;
	config_output *conf = ext_conf->plugin_conf;

	static char href[255] = "";

	int col_border, col_backgnd,
		col_pages, col_files,
		col_visits, col_border_sh,
		col_hits, col_xfer;

	max.hits	= 0;
	max.files	= 0;
	max.pages	= 0;
	max.visits	= 0;
	max.hosts	= 0;
	max.xfersize	= 0;

	/* go to the last element */
	while (l->next) l = l->next;

	last = l;
#define	SET_MAX(x) if (data->data.hist->data.web.x > max.x) max.x = data->data.hist->data.web.x

	while (i && l) {
		mdata *data = l->data;

		if (data) {
			SET_MAX(hits);
			SET_MAX(files);
			SET_MAX(pages);
			SET_MAX(visits);
			SET_MAX(hosts);
			SET_MAX(xfersize);

			if (l == last)
				first_month = data->data.hist->month;
		}

		i--;
		l = l->prev;
	}
#undef SET_MAX

	l = last;

	i = IM_BARS - i - 1;


	im = gdImageCreate( IM_WIDTH + 1,IM_HEIGHT + 1 );

	col_border	= gdImageColorAllocate(im, 0x00, 0x00, 0x00);

	html3torgb3(conf->col_shadow, &rgb);
	col_border_sh	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);

	html3torgb3(conf->col_backgnd, &rgb);
	col_backgnd	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_hits, &rgb);
	col_hits	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_files, &rgb);
	col_files	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_pages, &rgb);
	col_pages	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_visits, &rgb);
	col_visits	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_xfer, &rgb);
	col_xfer	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);

/* background */
	gdImageFilledRectangle(im, 0, 0, IM_WIDTH-1, IM_HEIGHT-1, col_backgnd);
/* surrounding border */
	gdImageRectangle( im, 1, 1, IM_WIDTH-1, IM_HEIGHT-1, col_border );
	gdImageRectangle( im, 0, 0, IM_WIDTH, IM_HEIGHT, col_border_sh );

/* left border - max hits */
	sprintf(int2str, "%li", max.hits);
	x1 = IM_CELLSPACING;
	y1 = IM_BOX1_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);
/* right border - max visits */
	sprintf(int2str, "%li", max.visits);
	x1 = IM_WIDTH - IM_CELLSPACING - IM_FONT_HEIGHT;
	y1 = IM_BOX2_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);
/* right border - max kbytes */
	sprintf(int2str, "%.0f", max.xfersize / 1024);
	x1 = IM_WIDTH - IM_CELLSPACING - IM_FONT_HEIGHT;
	y1 = IM_BOX3_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);

/* right border - 'pages/files/hits' */
	x1 = IM_CELLSPACING;
	y1 = IM_BOX1_Y2;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Pages"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Pages"), col_pages);

	y1 -= strlen(_("Pages")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, "/", col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, "/", col_border);

	y1 -= IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Files"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Files"), col_files);

	y1 -= strlen(_("Files")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, "/", col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, "/", col_border);

	y1 -= IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Hits"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Hits"), col_hits);


/* right border - 'visits' */
	x1 = (IM_BOX2_X2) - (IM_CELLSPACING) - strlen(_("Visits")) * IM_FONT_WIDTH;
	y1 = IM_CELLSPACING;
	gdImageString(im, gdFontSmall, x1+1, y1+1, _("Visits"), col_border_sh);
	gdImageString(im, gdFontSmall, x1, y1, _("Visits"), col_visits);
/* right border - 'kbytes' */
	x1 = (IM_BOX3_X2) - (IM_CELLSPACING) - strlen(_("KBytes")) * IM_FONT_WIDTH;
	y1 = IM_LAB1_Y1;
	gdImageString(im, gdFontSmall, x1+1, y1+1, _("KBytes"), col_border_sh);
	gdImageString(im, gdFontSmall, x1, y1, _("KBytes"), col_xfer);


/* header */
	x1 = IM_BOX1_X1;
	y1 = IM_CELLSPACING;
	gdImageString(im, gdFontSmall, x1, y1, _("Usage summary for "), col_border);

	x1 += strlen(_("Usage summary for ")) * IM_FONT_WIDTH;
	gdImageString(im, gdFontSmall, x1, y1, conf->hostname, col_border);


/* border for BOX1 */
	gdImageRectangle(im,
		IM_BOX1_X1 - IM_CELLSPACING,
		IM_BOX1_Y1 - IM_CELLSPACING,
		IM_BOX1_X2 + IM_CELLSPACING,
		IM_BOX1_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX1_X1 - IM_CELLSPACING + 1,
		IM_BOX1_Y1 - IM_CELLSPACING + 1,
		IM_BOX1_X2 + IM_CELLSPACING + 1,
		IM_BOX1_Y2 + IM_CELLSPACING + 1,
		col_border_sh);

/* border for BOX2 */
	gdImageRectangle(im,
		IM_BOX2_X1 - IM_CELLSPACING,
		IM_BOX2_Y1 - IM_CELLSPACING,
		IM_BOX2_X2 + IM_CELLSPACING,
		IM_BOX2_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX2_X1 - IM_CELLSPACING + 1,
		IM_BOX2_Y1 - IM_CELLSPACING + 1,
		IM_BOX2_X2 + IM_CELLSPACING + 1,
		IM_BOX2_Y2 + IM_CELLSPACING + 1,
		col_border_sh);

/* border for BOX3 */
	gdImageRectangle(im,
		IM_BOX3_X1 - IM_CELLSPACING,
		IM_BOX3_Y1 - IM_CELLSPACING,
		IM_BOX3_X2 + IM_CELLSPACING,
		IM_BOX3_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX3_X1 - IM_CELLSPACING + 1,
		IM_BOX3_Y1 - IM_CELLSPACING + 1,
		IM_BOX3_X2 + IM_CELLSPACING + 1,
		IM_BOX3_Y2 + IM_CELLSPACING + 1,
		col_border_sh);


	while (i >= 0  && l) {
		mdata *data = l->data;

		if (data) {
			/* Box 1 */

			if (max.hits) {
				int y2b = IM_BOX1_Y2;
				int y1b = IM_BOX1_Y1;
				x1 = IM_BOX1_X1 + i * (IM_BOX1_BAR_SPACE + IM_BOX1_BAR_WIDTH);
				x2 = x1 + IM_BOX1_BAR_WIDTH;

				y2 = y2b;
				y1 = y2 - ((double)data->data.hist->data.web.hits / max.hits) * (y2b - y1b - 1);

				if (y1 != y2) {
					gdImageFilledRectangle(im, x1, y1, x2, y2, col_hits);
					gdImageRectangle(im, x1, y1, x2, y2, col_border);
				}

				x1 += IM_BOX1_BAR_SHIFT;
				x2 += IM_BOX1_BAR_SHIFT;

				y1 = y2 - ((double)data->data.hist->data.web.files / max.hits) * (y2b - y1b - 1);

				if (y1 != y2) {
					gdImageFilledRectangle(im, x1, y1, x2, y2, col_files);
					gdImageRectangle(im, x1, y1, x2, y2, col_border);
				}

				x1 += IM_BOX1_BAR_SHIFT;
				x2 += IM_BOX1_BAR_SHIFT;

				y1 = y2 - ((double)data->data.hist->data.web.pages / max.hits) * (y2b - y1b - 1);

				if (y1 != y2) {
					gdImageFilledRectangle(im, x1, y1, x2, y2, col_pages);
					gdImageRectangle(im, x1, y1, x2, y2, col_border);
				}
			}

			/* Box 2 */

			if (max.visits) {
				int y2b = IM_BOX2_Y2;
				int y1b = IM_BOX2_Y1;

				x1 = IM_BOX2_X1 + i * (IM_BOX2_BAR_SPACE + IM_BOX2_BAR_WIDTH);
				x2 = x1 + IM_BOX2_BAR_WIDTH;

				y2 = y2b;
				y1 = y2 - ((double)data->data.hist->data.web.visits / max.visits) * (y2b - y1b - 1);

				if (y1 != y2) {
					gdImageFilledRectangle(im, x1, y1, x2, y2, col_visits);
					gdImageRectangle(im, x1, y1, x2, y2, col_border);
				}
			}

			/* Box 3 */

			if (max.xfersize) {
				double percent = (double)data->data.hist->data.web.xfersize / max.xfersize;
				int y2b = IM_BOX3_Y2;
				int y1b = IM_BOX3_Y1;

				x1 = IM_BOX3_X1 + i * (IM_BOX3_BAR_SPACE + IM_BOX3_BAR_WIDTH);
				x2 = x1 + IM_BOX3_BAR_WIDTH;

				y2 = y2b;
				y1 = y2 - (double)( percent * (y2b - y1b - 1 ));
				if (y1 != y2) {
					gdImageFilledRectangle(im, x1, y1, x2, y2, col_xfer);
					gdImageRectangle(im, x1, y1, x2, y2, col_border);
				}
			}
		}

		x1 = IM_BOX1_X1 + i * (IM_BOX1_BAR_SPACE + IM_BOX1_BAR_WIDTH);
		y1 = IM_LAB1_Y1;

		gdImageString(im, gdFontSmall, x1, y1, get_month_string((first_month-- + IM_BARS) % IM_BARS,1), col_border);

		i--;
		l = l->prev;
	}

	sprintf(filename, "%s%s%s/%s",
		conf->outputdir ? conf->outputdir : ".",
		subpath ? "/" : "",
		subpath ? subpath : "",
		IM_FILENAME);

	if ((f = fopen (filename, "wb"))) {
		gdImagePng(im, f);

		fclose(f);
	}

	gdImageDestroy(im);

	sprintf(href, "<center><img src=\"%s\" alt=\"%s\" width=%i height=%i></center>\n",
		IM_FILENAME, _("Monthly Overview"), IM_WIDTH+1, IM_HEIGHT+1);

	return href;
#else
	return NULL;
#endif
}

