/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: generate.c,v 1.100 2004/08/27 18:41:37 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>


#include "mconfig.h"
#include "misc.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "datatypes/count/datatype.h"
#include "datatypes/visited/datatype.h"
#include "mplugins.h"

#include "pictures.h"
#include "plugin_config.h"

#include "generate.h"

#define HIGHLIGHT	1
#define GROUPING	2
#define VISITS		4
#define INDEX		8
#define BROKEN_LINK	16
#define PERCENT		32
#define RESOLVE_TLD     64
#define VISITS_TRAFFIC  128
#define SORT_BY_KEY     256
#define VIEW_DURATION   512

#define BOTTOM_THRESHOLD	16

#ifdef ENABLE_INDENT_HTML
static int html_indent_depth = 0;
#endif

/* Returns a pointer to an allocated space containing a text label,
 * eventually shortened to cut chars, and encoded.
 * Note conf->dont_cut_urls and conf->dont_escape_entities applies.
 * if cut == 0 it doesn't shorten the text.
 * if text length > cut then text is shortened and '...' is added
 * to it.
 */
static char *url_label(config_output *conf, char *url, int cut)
{
	int url_len = strlen(url);
	char *tmp_url;

	if (cut && !conf->dont_cut_urls && url_len > cut) {
		tmp_url = malloc(cut + 3 + 1);
		if (!tmp_url) return NULL;
		memcpy(tmp_url, url, cut);
		tmp_url[cut++] = '.';
		tmp_url[cut++] = '.';
		tmp_url[cut++] = '.';
		tmp_url[cut] = '\0';
	} else {
		tmp_url = strdup(url);
		if (!tmp_url) return NULL;
	}

	if (conf->dont_escape_entities) {
		return tmp_url;
	} else {
		char *enc_url = html_encode(tmp_url);
		free(tmp_url);
		return enc_url;
	}
}


static void print_url_label(config_output *conf, FILE *f, char *url, int cut)
{
	char *enc_url = url_label(conf, url, cut);

	if (enc_url) {
		fprintf(f, "%s", enc_url);
		free(enc_url);
	}
}

static int show_mhash (mconfig *ext_conf, FILE *f, mhash *h, int count, int opt) {
	int i;
	config_output *conf = ext_conf->plugin_conf;
	long sum = 0;
	mdata **md;

	if (!h) return 0;

	sum = mhash_sumup(h);

	if (opt & SORT_BY_KEY) {
		md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);
	} else {
		md = mhash_sorted_to_marray(h, M_SORTBY_COUNT, M_SORTDIR_DESC);
	}

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			char *enc_url;
			unsigned int c = mdata_get_count(data);
			
			/* VIEW_DURATION uses the VISITED-datatype differently !!
			 * 
			 * .count : DURATION in seconds
			 * .vcount: is_hit ?
			 */
			if (c == 0 && !(opt & VIEW_DURATION)) {
				fprintf(stderr, "%s.%d: Kick Jan ! mdata_get_count() returns 0 (%s, %d)!\n", 
					__FILE__, __LINE__,
					data->key,
					data->type);
				continue;
			}

			enc_url = url_label(conf, data->key, 40);

			indent(f, html_indent_depth++);
			fprintf(f,"<tr>\n");

			if (opt & INDEX) {
				indent(f, html_indent_depth);
				fprintf(f,"<td align=\"right\">%d</td>\n", i+1);
			}

			indent(f, html_indent_depth);
			if (opt & VIEW_DURATION) {
				fprintf(f,"<td align=\"right\">%s</td>\n", seconds_to_string((double)c, 1));
			} else {
				fprintf(f,"<td align=\"right\">%d</td>\n", c);
			}

			if (opt & PERCENT && sum) {
				indent(f, html_indent_depth);
				fprintf(f,"<td align=\"right\">%.2f</td>\n", c * 100.0 / sum);
			}

			if (opt & VISITS && data->type == M_DATA_TYPE_VISITED ) {
				/* if the vcount is used as 'traffic' */
				indent(f, html_indent_depth);
				if (opt & VISITS_TRAFFIC) {
					fprintf(f,"<td align=\"right\">%s</td>\n", bytes_to_string(mdata_get_vcount(data)));
				} else {
					fprintf(f,"<td align=\"right\">%.0f</td>\n", mdata_get_vcount(data));
				}
			}

			if (opt & VIEW_DURATION && data->type == M_DATA_TYPE_VISITED ) {
				/* use count as duration and vcount as hits per page view */
				indent(f, html_indent_depth);
				fprintf(f,"<td align=\"right\">%.0f</td>\n", mdata_get_vcount(data));
				fprintf(f,"<td align=\"right\">%s</td>\n", mdata_get_vcount(data) ?
						(char *)seconds_to_string((double)(mdata_get_count(data) / mdata_get_vcount(data)), 1) :
						"--");
			}

			if ((opt & GROUPING) && mdata_is_grouped(data)) {
				indent(f, html_indent_depth);
				fprintf(f,"<td class=\"grouping\">%s</td>\n", enc_url);
			} else {
				if (opt & HIGHLIGHT) {
					indent(f, html_indent_depth);
					if (conf->assumedprotocol == NULL || strstr(data->key, "://")) {
						fprintf(f,"<td><a href=\"%s\">%s</a></td>\n", data->key, enc_url);
					} else {
						fprintf(f,"<td><a href=\"%s://%s%s%s\">%s</a></td>\n", conf->assumedprotocol, (conf->vhost_name->used ? conf->vhost_name->ptr : conf->hostname), *data->key == '/' ? "" : "/", data->key, enc_url);
					}
				} else {
					/* added option to resolve tlds on the fly */
					indent(f, html_indent_depth);
					if (opt & RESOLVE_TLD) {
						char *c = url_label(conf, misoname(data->key), 40);
						fprintf(f,"<td>%s</td>\n", c);
						free(c);
					} else {
						fprintf(f,"<td>%s</td>\n", enc_url);
					}
				}
			}

			if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
				struct tm *_tm;
				char timebuf[32] = "";

				indent(f, html_indent_depth);

				if (data->data.brokenlink.referrer) {
					if (strcmp(data->data.brokenlink.referrer, "-")) {
						free(enc_url);
						enc_url = url_label(conf, data->data.brokenlink.referrer, 40);
						fprintf(f,"<td><a href=\"%s\">%s</a></td>\n", data->data.brokenlink.referrer, enc_url);
					} else {
						fprintf(f,"<td>%s</td>\n", data->data.brokenlink.referrer);
					}
				} else {
					fprintf(f,"<td>%s</td>\n", "-");
				}

				_tm = localtime(&(data->data.brokenlink.timestamp));
				if (strftime(timebuf, sizeof(timebuf)-1, "%x", _tm) == 0) {
					fprintf(stderr, "output::modlogan.show_mhash: strftime failed\n");
				}
				indent(f, html_indent_depth);
				fprintf(f,"<td>%s</td>\n", timebuf);
			}
			indent(f, --html_indent_depth);
			fprintf(f,"</tr>\n");

			free(enc_url);
		}
	}

	free(md);

	return 0;
}

mlist * get_next_element(mhash *h) {
	mlist *ret = NULL;
	int max = 0, i;

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l; l = l->next) {
			mdata *data = l->data;

			if (!l->data) continue;

			if (mdata_get_count(data) > max) {
				max = mdata_get_count(data);
				ret = l;
			}
		}
	}
	/* invert the value */
	if (ret) {
		mdata_set_count(ret->data, -mdata_get_count(ret->data));
	}

	return ret;
}

static int cleanup_elements(mhash *h) {
	int i;
	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			mdata *data = l->data;

			if (!l->data) continue;


			if (mdata_get_count(data) > 0) {
				/* fprintf(stderr, "%s.%d: count (%d) > 0 in cleanup\n",
				 __FILE__, __LINE__,
				 mdata_get_count(data));*/
			} else {
				mdata_set_count(data, -mdata_get_count(data));
			}
		}
	}

	return 0;
}

static int show_visit_path (mconfig *ext_conf, FILE *f, mhash *h, int count, int opt) {
	int i = 0;
	config_output *conf = ext_conf->plugin_conf;
	mlist *l;
	long sum;

	if (!h) return 0;

	sum = mhash_sumup(h);

	while ((l = get_next_element(h)) && i < count) {

		indent(f, html_indent_depth++);
		fprintf(f,"<tr>\n");

		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			int c;
			char *lastkey = NULL;
			int count = 1;

			c = -data->data.sublist.count;

			i++;
			indent(f, html_indent_depth);
			fprintf(f,"<td align=\"right\">%d</td>\n", i);

			fprintf(f,"<td align=\"right\">%d</td>\n", c);
			fprintf(f,"<td align=\"right\">%.2f</td>\n", c * 100.0 / sum);

			fprintf(f,"<td align=\"left\">");

			for (;sl && sl->data;sl = sl->next) {
				mdata *sld = sl->data;
				
				if (lastkey) {
					if (!strcmp(lastkey, sld->key)) {
						count++;
						continue;
					} else {
						fprintf(f, "%dx&nbsp;", count);
						print_url_label(conf, f, lastkey, 40);
						fprintf(f, "<br />");
					}
				}
				
				count = 1;
				if (lastkey) free(lastkey);
				lastkey = strdup(sld->key);
			}

			if (lastkey) {
				fprintf(f, "%dx&nbsp;", count);
				print_url_label(conf, f, lastkey, 40);
				fprintf(f, "<br />");
				free(lastkey);
			}

			fprintf(f,"</td>");

		}

		indent(f, --html_indent_depth);
		fprintf(f,"</tr>\n");
	}

	cleanup_elements(h);

	return 0;
}

mhash *get_entry_pages(mconfig *ext_conf, mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l; l = l->next) {
			mdata *data = l->data;
			mlist *sl;

			if (!l->data) continue;

			sl = l->data->data.sublist.sublist;

			if (data->type != M_DATA_TYPE_SUBLIST) {
				M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "datatype not a sublist: %d - %s\n", data->type, data->key);
				return NULL;
			}

			if (sl && sl->data) {
				mdata *insd;
				const char *key = splaytree_insert(ext_conf->strings, sl->data->key);

				insd = mdata_Count_create(key, l->data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

mhash *get_exit_pages(mconfig *ext_conf, mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			mlist *sl;

			if (!l->data || !l->data->data.sublist.sublist) continue;

			/* go to the last element */
			for (sl = l->data->data.sublist.sublist; sl->next; sl = sl->next);

			if (sl->data) {
				mdata *sld = sl->data;
				mdata *insd;
				const char *key = splaytree_insert(ext_conf->strings, sld->key);
				
				insd = mdata_Count_create(key, l->data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

mhash *get_visit_path_length(mconfig *ext_conf, mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list;l; l = l->next) {
			mdata *data = l->data;
			mlist *sl;
			long c;
			char str[255];
			mdata *insd;
			const char *key;

			if (!l->data|| !l->data->data.sublist.sublist) continue;

			/* go to the last element */
			for (sl = data->data.sublist.sublist, c = 0; sl; sl = sl->next, c++);

			snprintf(str, sizeof(str)-1, "%5ld", c);
			
			key = splaytree_insert(ext_conf->strings, str);

			insd = mdata_Count_create(key, data->data.sublist.count, M_DATA_STATE_PLAIN);
			mhash_insert_sorted(ret, insd);
		}
	}

	return ret;
}

mhash *get_visit_duration(mconfig *ext_conf, mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		for (l = h->data[i]->list; l; l = l->next) {
			mdata *data;
			mlist *sl;

			/* go to the last element */
			char str[255];
			mdata *insd;
			mdata *sld;

			if (!l->data || !l->data->data.sublist.sublist) continue;

			data = l->data;
			sl = data->data.sublist.sublist;
			sld = sl->data;

			if (sld) {
				time_t t1 = sld->data.brokenlink.timestamp, t2;
				const char *key;

				for (;sl->next; sl = sl->next);

				t2 = sl->data->data.brokenlink.timestamp;

				if ((t2 - t1) >= 60) {
					snprintf(str, sizeof(str)-1, "%5ld %s", (t2 - t1) / 60, _("min"));
				} else {
					snprintf(str, sizeof(str)-1, " < 1 %s", _("min"));
				}
				
				key = splaytree_insert(ext_conf->strings, str);

				insd = mdata_Count_create(key, data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

double get_pages_per_visit(mstate_web *staweb) {
	double d = 0, v = 0;
	mhash *h;
	int i;

	/* count pages of finished visits */
	h = staweb->visits;

	if (!h) return 0;

	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			if (!l->data) continue;

			if (l->data->type != M_DATA_TYPE_SUBLIST) {
				fprintf(stderr, "%s.%d\n", __FILE__, __LINE__);
				return -1;
			}
			d += mlist_count(l->data->data.sublist.sublist) * l->data->data.sublist.count;
			v += l->data->data.sublist.count;
		}
	}


	/* count pages of unfinished visits */
	h = staweb->visit_hash;

	if (!h) return d / v;

	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			if (!l->data) continue;

			if (l->data->type != M_DATA_TYPE_VISIT) {
				fprintf(stderr, "%s.%d: \n", __FILE__, __LINE__);
				return -1;
			}
			d += l->data->data.visit->count - 1;
			v++;
		}
	}

	return d / v;
}

double get_visit_full_duration(mhash *h) {
	double d = 0;
	int i;

	if (!h) return 0;

	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			mdata *data = l->data;
			mlist *sl;

			if (!l->data) continue;

			if ((sl = data->data.sublist.sublist)) {
				/* go to the last element */
				mdata *sld = sl->data;

				if (sld) {
					time_t t1 = sld->data.brokenlink.timestamp, t2;

					/* go to the last element */
					for (;sl->next; sl = sl->next);

					t2 = sl->data->data.brokenlink.timestamp;

					/* last timestamp - firsttimestamp == duration of the visit */
					d += (t2 - t1);
				}
			}
		}
	}

	return d;
}

mhash *get_path_length(mconfig *ext_conf, mhash *h) {
	mhash *ret;
	mlist *l;

	if (!h) return 0;

	ret = mhash_init( 32 );

	while ((l = get_next_element(h))) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			long c = 0;
			char str[255];
			mdata *insd;
			const char *key;

			if (!sl) continue;
			
			/* go to the last element */
				
			for (; sl; c++, sl = sl->next);

			snprintf(str, sizeof(str)-1, "%5ld", c);
			
			key = splaytree_insert(ext_conf->strings, str);

			insd = mdata_Count_create(key, 1, M_DATA_STATE_PLAIN);
			mhash_insert_sorted(ret, insd);
		}
	}

	cleanup_elements(h);

	return ret;
}

static int show_status_mhash (FILE *f, mhash *h, int count) {
	int i;
	mdata **md;
	long sum;

	if (!h) return 0;

	sum = mhash_sumup(h);

	md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			indent(f, html_indent_depth);
			
			fprintf(f,"<tr><td align=\"right\">%i</td><td align=\"right\">%.02f</td><td>%s - %s</td></tr>\n",
				data->data.count.count, 100*((double)data->data.count.count/(double)sum), data->key,
				mhttpcodes(strtol(data->key, NULL, 10)));
		}
	}

	free(md);

	return 0;
}

static void table_start(FILE *f, char *str, int colspan) {
	if (f == NULL) return;

	indent(f, html_indent_depth);
	fprintf(f, "<p />\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<center>\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<table border=\"1\" %s cellspacing=\"1\" cellpadding=\"3\">\n", colspan < 0 ? "width=\"100%\"" : "");

	indent(f, html_indent_depth);
	fprintf(f, "<tr><th colspan=\"%d\">%s</th></tr>\n", colspan < 0 ? -colspan : colspan, str);
}

static void table_end(FILE *f) {
	indent(f, --html_indent_depth);
	fprintf(f, "</table>\n");

	indent(f, --html_indent_depth);
	fprintf(f, "</center>\n");

	indent(f, html_indent_depth);
	fprintf(f, "<p />\n");
}

static void file_start(FILE *f, mconfig *ext_conf, time_t timestamp) {
	char buf[255];
	struct tm *_tm;
	time_t t;
	config_output *conf = ext_conf->plugin_conf;

	if (include_file(f, conf->html_header, "page header")) {
		char *tmp;
		fprintf(f, "<?xml version=\"1.0\" encoding=\"%s\"?>\n"
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"DTD/xhtml1-transitional.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%s\" lang=\"%s\">\n\n"
			"<head>\n"
			" <title>%s</title>\n"
			" <link rel=\"stylesheet\" href=\"modlogan.css\" type=\"text/css\" />\n"
			" <meta http-equiv=\"Content-Type\" content=\"text/html; charset=%s\" />\n"
			" <meta http-equiv=\"Content-Language\" content=\"%s\" />\n"
			" <meta name=\"ROBOTS\" content=\"NOINDEX, NOARCHIVE, NOFOLLOW\" />\n"
			"</head>\n"
			"<body>\n",
			conf->cont_charset,
			conf->cont_language,
			conf->cont_language,
			_("Statistics"),
			conf->cont_charset,
			conf->cont_language);

		if(conf->vhost_name->used) {
		    tmp = malloc( strlen( _("Statistics for %1$s") ) - 3 + conf->vhost_name->used - 1 );
		    sprintf( tmp, _("Statistics for %1$s"), conf->vhost_name->ptr );
		} else {
		    tmp = malloc( strlen( _("Statistics for %1$s") ) - 3 + strlen( conf->hostname ) );
		    sprintf( tmp, _("Statistics for %1$s"), conf->hostname );
		}
		fprintf(f, "<h1>%s</h1>\n", tmp );
		free(tmp);
		if (timestamp != 0) {
			t = timestamp;
			_tm = localtime(&t);
			strftime(buf, sizeof(buf), "%X %x", _tm);
			fprintf(f, "<b>%s: </b>%s<br />\n", _("Last record"), buf);
		}
		t = time(NULL);
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		fprintf(f, "<b>%s: </b>%s<br />\n", _("Generated at"), buf);
		fprintf(f, "<hr /><br />\n");
	}

	indent(f, html_indent_depth++);
	fprintf(f, "<table width=\"100%%\" cellpadding=\"4\" cellspacing=\"0\">\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<tr valign=\"top\">\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<td class=\"skeleton\">\n");
}

static void file_end(FILE *f, mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;

	indent(f, --html_indent_depth);
	fprintf(f, "</td>\n");

	indent(f, --html_indent_depth);
	fprintf(f, "</tr>\n");

	indent(f, --html_indent_depth);
	fprintf(f, "</table>\n");

	if (include_file(f, conf->html_footer, "page footer")) {
		fprintf(f, "<hr />");

		if (conf->show_validation_links) {
/* Tell the users that this is valid XHTML 1.0  :) */
			fprintf(f, "<a href=\"http://validator.w3.org/check/referer\">"
				" <img border=\"0\" src=\"http://www.w3.org/Icons/valid-xhtml10\" "
				"alt=\"Valid XHTML 1.0!\" height=\"31\" width=\"88\" align=\"right\" />\n"
				"</a>\n"
				"<a href=\"http://jigsaw.w3.org/css-validator/check/referer/\">\n"
				" <img border=\"0\" width=\"88\" height=\"31\" "
				"src=\"http://jigsaw.w3.org/css-validator/images/vcss.gif\" "
				"alt=\"Valid CSS!\" align=\"right\" />\n"
				"</a>");
		}

		fprintf(f, "%s <a href=\"%s\">%s %s</a>\n", _("Output generated by"),"http://www.modlogan.org/", PACKAGE, VERSION);
		fprintf(f, "</body></html>\n");
	}
}

static void file_start_index(FILE *f, mconfig *ext_conf, time_t timestamp) {
	char buf[255];
	struct tm *_tm;
	time_t t;
	config_output *conf = ext_conf->plugin_conf;

	if (include_file(f, conf->html_header, "page header")) {
		char *tmp;
		fprintf(f, "<?xml version=\"1.0\" encoding=\"%s\"?>\n"
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"DTD/xhtml1-transitional.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"%s\" lang=\"%s\">\n\n"
			"<head>\n"
			" <title>%s</title>\n"
			" <link rel=\"stylesheet\" href=\"modlogan.css\" type=\"text/css\" />\n"
			" <meta http-equiv=\"Content-Type\" content=\"text/html; charset=%s\" />\n"
			" <meta http-equiv=\"Content-Language\" content=\"%s\" />\n"
			" <meta name=\"ROBOTS\" content=\"NOINDEX, NOARCHIVE, NOFOLLOW\" />\n"
			"</head>\n"
			"<body>\n",
			conf->cont_charset,
			conf->cont_language,
			conf->cont_language,
			_("Statistics"),
			conf->cont_charset,
			conf->cont_language);

		if(conf->vhost_name->used) {
		    tmp = malloc( strlen( _("Statistics for %1$s") ) - 3 + conf->vhost_name->used - 1 );
		    sprintf( tmp, _("Statistics for %1$s"), conf->vhost_name->ptr );
		} else {
		    tmp = malloc( strlen( _("Statistics for %1$s") ) - 3 + strlen( conf->hostname ) );
		    sprintf( tmp, _("Statistics for %1$s"), conf->hostname );
		}
		fprintf(f, "<h1>%s</h1>\n", tmp );
		free(tmp);
		if (timestamp != 0) {
			t = timestamp;
			_tm = localtime(&t);
			strftime(buf, sizeof(buf), "%X %x", _tm);
			fprintf(f, "<b>%s: </b>%s<br />\n", _("Last record"), buf);
		}
		t = time(NULL);
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		fprintf(f, "<b>%s: </b>%s<br />\n", _("Generated at"), buf);
		fprintf(f, "<hr /><br />\n");
	}

	indent(f, html_indent_depth++);
	fprintf(f, "<table width=\"100%%\" cellpadding=\"4\" cellspacing=\"0\">\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<tr valign=\"top\">\n");

	indent(f, html_indent_depth++);
	fprintf(f, "<td class=\"skeleton\">\n");
}

static void file_end_index(FILE *f, mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;

	indent(f, --html_indent_depth);
	fprintf(f, "</td>\n");

	indent(f, --html_indent_depth);
	fprintf(f, "</tr>\n");

	indent(f, --html_indent_depth);
	fprintf(f, "</table>\n");

	if (include_file(f, conf->html_footer, "page footer")) {
/* Tell the users that this is valid HTML 4.0  :) */
		fprintf(f, "<hr />");
		if (conf->show_validation_links) {
			fprintf(f, "<a href=\"http://validator.w3.org/check/referer\">"
				" <img border=\"0\" src=\"http://www.w3.org/Icons/valid-xhtml10\" "
				"alt=\"Valid XHTML 1.0!\" height=\"31\" width=\"88\" align=\"right\" />\n"
				"</a>\n"
				"<a href=\"http://jigsaw.w3.org/css-validator/check/referer/\">\n"
				" <img border=\"0\" width=\"88\" height=\"31\" "
				"src=\"http://jigsaw.w3.org/css-validator/images/vcss.gif\" "
				"alt=\"Valid CSS!\" align=\"right\" />\n"
				"</a>");
		}
		fprintf(f, "%s <a href=\"%s\">%s %s</a>\n", _("Output generated by"),"http://www.modlogan.org/", PACKAGE, VERSION);
		fprintf(f, "</body></html>\n");
	}
}

//                     maximum   act_count  product
char *table_header(int maxcount, int count, const char *str) {
	static char trans_buf[255];
	// <maxcount> of <count> <str>
	// %1$d - maxcount
	// %2$d - count
	// %3$s - str
	// Order is replaceable..
	snprintf( trans_buf, sizeof( trans_buf ) - 1, _("%1$d of %2$d %3$s" ), 
						(maxcount > count) || (maxcount < 0) ? count : maxcount,
					  count, str);

	return trans_buf;
}

#define	M_TYPE_REPORT	1
#define M_TYPE_PAGE	2

typedef enum { M_REPORT_UNSET, M_REPORT_REQ_URL, M_REPORT_REF_URL,
		M_REPORT_OS, M_REPORT_HOSTS, M_REPORT_ENTRY_PAGES,
		M_REPORT_EXIT_PAGES, M_REPORT_USERAGENT, M_REPORT_INDEXED,
		M_REPORT_REQ_PROT, M_REPORT_REQ_METH, M_REPORT_STATUS_CODES,
		M_REPORT_ROBOTS, M_REPORT_BOOKMARKS, M_REPORT_BROKEN_LINKS,
		M_REPORT_INTERNAL_ERROR, M_REPORT_SEARCH_ENGINE,
		M_REPORT_SEARCH_STRINGS, M_REPORT_COUNTRIES, M_REPORT_SUMMARY,
		M_REPORT_HOURLY, M_REPORT_DAILY, M_REPORT_EXTENSION,
		M_REPORT_VISIT_PATH, M_REPORT_VISIT_DURATION,
		M_REPORT_VISIT_PATH_LENGTH, M_REPORT_VIEW_DURATION,
		M_REPORT_VHOSTS,

		M_PAGE_INDEX = 128, M_PAGE_001, M_PAGE_002, M_PAGE_003,
		M_PAGE_004, M_PAGE_000
} mreport_type;

static int get_menu_items (mconfig *ext_conf, mstate *state, mlist *l) {
	mdata *data;
	config_output *conf = ext_conf->plugin_conf;
	int i = 0;
	mstate_web *staweb = state->ext;
	const char *key;
	
	
	key = splaytree_insert(ext_conf->strings, "/000");
	data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_INDEX);
	mlist_insert(l, data);
	
	/* page 0 */
	key = splaytree_insert(ext_conf->strings, "/000/000");
	data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_000);
	mlist_insert(l, data);

	key = splaytree_insert(ext_conf->strings, "/000/000/000");
	data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_SUMMARY);
	mlist_insert(l, data);

	key = splaytree_insert(ext_conf->strings, "/000/000/001");
	data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_DAILY);
	mlist_insert(l, data);

	key = splaytree_insert(ext_conf->strings, "/000/000/002");
	data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_HOURLY);
	mlist_insert(l, data);

/* page 1 */
	i = 0;
	if (conf->max_req_urls && mhash_count(staweb->req_url_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/001/001");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_REQ_URL);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_ref_urls && mhash_count(staweb->ref_url_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/001/002");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_REF_URL);
		mlist_insert(l, data);
		i = 1;
	}

	if (mhash_count(staweb->visits)) {
		if (conf->max_entry_pages) {
			key = splaytree_insert(ext_conf->strings, "/000/001/003");
			data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_ENTRY_PAGES);
			mlist_insert(l, data);
			i = 1;
		}

		if (conf->max_exit_pages) {
			key = splaytree_insert(ext_conf->strings, "/000/001/004");
			data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_EXIT_PAGES);
			mlist_insert(l, data);
			i = 1;
		}

		if (conf->max_visit_paths) {
			key = splaytree_insert(ext_conf->strings, "/000/001/005");
			data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_VISIT_PATH);
			mlist_insert(l, data);
			i = 1;
		}

		if (conf->max_visit_path_lengths) {
			key = splaytree_insert(ext_conf->strings, "/000/001/006");
			data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_VISIT_PATH_LENGTH);
			mlist_insert(l, data);
			i = 1;
		}

		if (conf->max_visit_durations) {
			key = splaytree_insert(ext_conf->strings, "/000/001/007");
			data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_VISIT_DURATION);
			mlist_insert(l, data);
			i = 1;
		}
	}

	if (conf->max_view_durations && mhash_count(staweb->views)) {
		key = splaytree_insert(ext_conf->strings, "/000/001/008");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_VIEW_DURATION);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_extensions && mhash_count(staweb->extension)) {
		key = splaytree_insert(ext_conf->strings, "/000/001/009");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_EXTENSION);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_vhosts && mhash_count(staweb->vhost_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/001/010");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_VHOSTS);
		mlist_insert(l, data);
		i = 1;
	}

	if (i) {
		key = splaytree_insert(ext_conf->strings, "/000/001");
		data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_001);
		mlist_insert(l, data);
	}

/* page 2 */
	i = 0;
	if (conf->max_os && mhash_count(staweb->os_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/002/001");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_OS);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_hosts && mhash_count(staweb->host_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/002/002");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_HOSTS);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_ua && mhash_count(staweb->ua_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/002/003");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_USERAGENT);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_countries && mhash_count(staweb->country_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/002/004");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_COUNTRIES);
		mlist_insert(l, data);
		i = 1;
	}

	if (i) {
		key = splaytree_insert(ext_conf->strings, "/000/002");
		data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_002);
		mlist_insert(l, data);
	}
/* page 3 */

	i = 0;
	if (conf->max_indexed_pages && mhash_count(staweb->indexed_pages)) {
		key = splaytree_insert(ext_conf->strings, "/000/003/001");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_INDEXED);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_robots && mhash_count(staweb->robots)) {
		key = splaytree_insert(ext_conf->strings, "/000/003/002");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_ROBOTS);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_bookmarks && mhash_count(staweb->bookmarks)) {
		key = splaytree_insert(ext_conf->strings, "/000/003/003");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_BOOKMARKS);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_search_engines && mhash_count(staweb->searchsite)) {
		key = splaytree_insert(ext_conf->strings, "/000/003/004");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_SEARCH_ENGINE);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_search_strings && mhash_count(staweb->searchstring)) {
		key = splaytree_insert(ext_conf->strings, "/000/003/005");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_SEARCH_STRINGS);
		mlist_insert(l, data);
		i = 1;
	}

	if (i) {
		key = splaytree_insert(ext_conf->strings, "/000/003");
		data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_003);
		mlist_insert(l, data);
	}

/* page 4 */
	i = 0;
	if (conf->max_req_prot && mhash_count(staweb->req_prot_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/004/001");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_REQ_PROT);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_req_meth && mhash_count(staweb->req_meth_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/004/002");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_REQ_METH);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_status_codes && mhash_count(staweb->status_hash)) {
		key = splaytree_insert(ext_conf->strings, "/000/004/003");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_STATUS_CODES);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_broken_links && mhash_count(staweb->status_missing_file)) {
		key = splaytree_insert(ext_conf->strings, "/000/004/004");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_BROKEN_LINKS);
		mlist_insert(l, data);
		i = 1;
	}

	if (conf->max_internal_errors && mhash_count(staweb->status_internal_error)) {
		key = splaytree_insert(ext_conf->strings, "/000/004/005");
		data = mdata_Count_create(key, M_TYPE_REPORT, M_REPORT_INTERNAL_ERROR);
		mlist_insert(l, data);
		i = 1;
	}

	if (i) {
		key = splaytree_insert(ext_conf->strings, "/000/004");
		data = mdata_Count_create(key, M_TYPE_PAGE, M_PAGE_004);
		mlist_insert(l, data);
	}

	return 0;
}

char * get_menu_item(int type) {
	switch(type) {
	case M_REPORT_REQ_URL:
		return _("Requested URL's");
	case M_REPORT_REF_URL:
		return _("Referrers");
	case M_REPORT_OS:
		return _("Operating system");
	case M_REPORT_HOSTS:
		return _("Hosts");
	case M_REPORT_ENTRY_PAGES:
		return _("Entry Pages");
	case M_REPORT_EXIT_PAGES:
		return _("Exit Pages");
	case M_REPORT_USERAGENT:
		return _("Browsers");
	case M_REPORT_INDEXED:
		return _("Indexed Pages");
	case M_REPORT_REQ_PROT:
		return _("Request Protocol");
	case M_REPORT_REQ_METH:
		return _("Request Method");
	case M_REPORT_STATUS_CODES:
		return _("Status Code");
	case M_REPORT_ROBOTS:
		return _("Robots");
	case M_REPORT_BOOKMARKS:
		return _("Bookmarked Pages");
	case M_REPORT_BROKEN_LINKS:
		return _("Broken Links");
	case M_REPORT_INTERNAL_ERROR:
		return _("Internal Errors");
	case M_REPORT_SEARCH_ENGINE:
		return _("SearchEngines");
	case M_REPORT_SEARCH_STRINGS:
		return _("SearchStrings");
	case M_REPORT_COUNTRIES:
		return _("Countries");
	case M_REPORT_SUMMARY:
		return _("Summary");
	case M_REPORT_HOURLY:
		return _("Hourly Statistics");
	case M_REPORT_DAILY:
		return _("Daily Statistics");
	case M_REPORT_EXTENSION:
		return _("Extensions");
	case M_REPORT_VISIT_PATH:
		return _("Visit Path");
	case M_REPORT_VISIT_PATH_LENGTH:
		return _("Path Length");
	case M_REPORT_VISIT_DURATION:
		return _("Visit Duration");
	case M_REPORT_VIEW_DURATION:
		return _("View Duration");
	case M_REPORT_VHOSTS:
		return _("Vhosts");
	case M_PAGE_INDEX:
		return _("Index");
	case M_PAGE_000:
		return _("Overview");
	case M_PAGE_001:
		return _("URLs");
	case M_PAGE_002:
		return _("User");
	case M_PAGE_003:
		return _("Searchengines");
	case M_PAGE_004:
		return _("Server Internals");
	default:
		return "(null)";
	}
}

char *get_url(mconfig *ext_conf, int year, int month, char *sub, char *report) {
	static char filename[255];
	config_output *conf = ext_conf->plugin_conf;

	/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
	** courtesy of Jan Kneschke
	*/
	/* suffix for the generated statistics page files */
	if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
		snprintf(filename, 255, "m_usage_%04d%02d.html%s%.3s%s",
			year, month,
			sub ? "#" : "",
			sub ? sub : "",
			report ? report : "");
	} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
		snprintf(filename, 255, "m_usage_%04d%02d_%.3s_%s.%s",
			year, month,
			sub ? sub : "",
			report ? report : "",
			conf->pages_suffix );
	} else {
		snprintf(filename, 255, "m_usage_%04d%02d_%.3s.html%s%s",
			year, month, sub, report ? "#" : "",
			report ? report : "");
	}

	return filename;
}

static int write_menu_page(mconfig *ext_conf, mstate *state,FILE *f, int type, char *sub, char *report) {
	indent(f, html_indent_depth);
	fprintf(f, "<tr><td class=\"menu\">&nbsp;&nbsp;<a href=\"%s\">[%s]</a></td></tr>\n",
		get_url(ext_conf, state->year, state->month, sub, report), get_menu_item(type));

	return 0;
}

static int write_menu_report(mconfig *ext_conf, mstate *state,FILE *f, int type, char *sub, char *report, int highlight) {
	indent(f, html_indent_depth);
	fprintf(f, "<tr><td class=\"%s\">&nbsp;&nbsp;&nbsp;&nbsp;<a href=\"%s\">[%s]</a></td></tr>\n",
		highlight ? "menu_highlight" : "menu",
		get_url(ext_conf, state->year, state->month, sub, report),
		get_menu_item(type));

	return 0;
}


static int write_menu (mconfig *ext_conf, mstate *state,FILE *f, mlist *l, char *sub, int cur_item) {
	config_output *conf = ext_conf->plugin_conf;

	indent(f, html_indent_depth++);
	fprintf(f, "<table width=\"150\">\n");
	if (conf->show_month_in_menu) {
		indent(f, html_indent_depth);
		fprintf(f, "<tr><td class=\"menu\" align=\"center\"><b>%s %04d</b></td></tr>\n",
			get_month_string(state->month,0), state->year);
	}

	for (;l; l = l->next) {
		mdata *data = l->data;
		char *sep_main, *sep_sub, *sep_report;
		
	/* seperate menu string */
		sep_main = strchr(data->key, '/');
		sep_sub = strchr(sep_main+1, '/');

		sep_main++;
		if (!sep_sub) {
			indent(f, html_indent_depth);
			/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
			** courtesy of Jan Kneschke
			*/
			/* suffix for the generated statistics page files */
			fprintf(f, "<tr><td class=\"menu\"><a href=\"index.%s\">[%s]</a></td></tr>\n",
			  conf->pages_suffix,
				get_menu_item(data->data.count.grouped));
		} else {
			sep_sub++;
			sep_report = strchr(sep_sub, '/');
			if (!sep_report) {
				if (conf->page_style && (!strcasecmp(conf->page_style, "seppage") ||
							 !strcasecmp(conf->page_style, "onepage"))
				    ) {
					if (l->next) {
						mdata *_data = l->next->data;
						char *_sep_main, *_sep_sub, *_sep_report;
						
				/* seperate menu string */
						_sep_main = strchr(_data->key, '/');
						_sep_sub = strchr(_sep_main+1, '/');

						_sep_main++;
						if (_sep_sub) {
							_sep_sub++;
							_sep_report = strchr(_sep_sub, '/');
							if (_sep_report) {
								_sep_report++;
								write_menu_page(ext_conf, state, f, data->data.count.grouped, sep_sub, _sep_report);
							}
						}
					}
				} else {
					write_menu_page(ext_conf, state, f, data->data.count.grouped, sep_sub, NULL);
				}
			} else {
				sep_report++;

				if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
					write_menu_report(ext_conf, state, f, data->data.count.grouped, sep_sub, sep_report, cur_item == data->data.count.grouped);
				} else if (!strncmp(sub, sep_sub,3))
					write_menu_report(ext_conf, state, f, data->data.count.grouped, sep_sub, sep_report, cur_item == data->data.count.grouped);
			}
		}
	}

	indent(f, --html_indent_depth);
	fprintf(f, "</table>\n");

	return 0;
}

static int write_css(mconfig *ext_conf) {
	char *filename;
	int ret;
	FILE *f2;
	config_output *conf = ext_conf->plugin_conf;

	/* open the css file, search includepath as well */
	if ((f2 = mfopen (ext_conf, conf->cssfile, "r")) == NULL) {
		fprintf(stderr, "%s.%d: can't open %s: %s\n",
			__FILE__, __LINE__,
			conf->cssfile,
			strerror(errno));
		return -1;
	}

	/* build destination filename */
	filename = (char *)malloc(strlen(conf->outputdir) + strlen("/modlogan.css") + 1);
	if (!filename) return -1;
	sprintf(filename, "%s/modlogan.css", conf->outputdir);

	(void)unlink(filename);
	ret = symlink(conf->cssfile, filename);
	free(filename);

	if (ret != 0) {
		fprintf(stderr, "writing CSS-definition for %s failed: %s\n",
			conf->outputdir,
			strerror(errno));
	}
	/* Close the css source file */
	fclose(f2);

	return 0;
}

static int write_report_header(mconfig *ext_conf, FILE *f, char *sub, char *report) {
	config_output *conf = ext_conf->plugin_conf;

	if (f == NULL) return -1;

	indent(f, html_indent_depth);
	if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
		fprintf(f, "<center><a name=\"%.3s%s\"></a><a href=\"#000\">[top]</a></center>", sub, report);
	} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
		fprintf(f, "<center><a name=\"%s\"></a></center>", report);
	} else {
		fprintf(f, "<center><a name=\"%s\"></a><a href=\"#000\">[top]</a></center>", report);
	}

	return 0;
}


/**
 * data container for the coloums of the tables
 *
 */

typedef struct {
	const char *name;  /*< visible name of the coloumn */
	const char *class; /*< CSS class, may be NULL */
} fields_def;

/**
 * data container for generic reports
 *
 */

typedef struct {
	mreport_type type; /*< type of report */
	mhash *data;       /*< data source (staweb->...) */
	int max;           /*< maximum fields to show */
	const char *title; /*< visible title of the table */
	int options;       /*< options for show_mhash */
	int show_graph;    /*< for the graph ? (conf->show...) */
	char * (*draw_graph)(mconfig * ext_conf, mstate * state);
	                   /*< graph function (may be NULL) */
	fields_def fields[5];
	                   /*< coloumns */
} reports_def;

static int generate_monthly_output(mconfig * ext_conf, mstate * state) {
	unsigned int i, s_200, s_304;
	double d = 0;
	FILE *f = NULL;
	config_output *conf = ext_conf->plugin_conf;
	char *outputdir = conf->outputdir ? conf->outputdir : ".";
	char *filename;
	data_WebHistory sumdat, maxdat;
	int last_day = 1;
	mlist *menu_items = mlist_init(), *l;
	char last_sub[4] = { 0, 0, 0, 0 };
	char last_report[4] = { 0, 0, 0, 0 };
	mstate_web *staweb = NULL;

	/* each 'generic' report is defined in this structure
	 *
	 * NOTE: the data field has to be NULL in the declaration as staweb isn't
	 *       checked yet. The real assignment is done later
	 */

	reports_def reports[] = {
		/* 0 */
		  /* id             data  max. entry to show  title */
		{ M_REPORT_REQ_URL, NULL, conf->max_req_urls, _("from Requested URL's"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 1 */
		{ M_REPORT_REF_URL, NULL, conf->max_ref_urls, _("from Referring URL's"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Referrer"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 2 */
		{ M_REPORT_VIEW_DURATION, NULL, conf->max_req_urls, _("from View Durations"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 3 */
		{ M_REPORT_OS, NULL, conf->max_os, _("from Used Operating Systems"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }

			}
		},
		/* 4 */
		{ M_REPORT_HOSTS, NULL, conf->max_hosts, _("from Hosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 5 */
		{ M_REPORT_ENTRY_PAGES, NULL, conf->max_entry_pages, _("from Entry Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Visits"), "visits" },
				{ _("Entry Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 6 */
		{ M_REPORT_EXIT_PAGES, NULL, conf->max_exit_pages, _("from Exit Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Visits"), "visits" },
				{ _("Exit Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 7 */
		{ M_REPORT_INDEXED, NULL, conf->max_indexed_pages, _("from Indexed Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Indexed Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 8 */
		{ M_REPORT_USERAGENT, NULL, conf->max_ua, _("from Used Browsers"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 9 */
		{ M_REPORT_REQ_PROT, NULL, conf->max_req_prot, _("from Used Request Protocol"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Protocol"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 10 */
		{ M_REPORT_REQ_METH, NULL, conf->max_req_meth, _("from Used Request Method"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Method"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 11 */
		{ M_REPORT_ROBOTS, NULL, conf->max_robots, _("from Robots"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Robot"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 12 */
		{ M_REPORT_BOOKMARKS, NULL, conf->max_bookmarks, _("from Bookmarked Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Bookmarked Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 13 */
		{ M_REPORT_BROKEN_LINKS, NULL, conf->max_broken_links, _("from Missing File / Broken Link"),
			GROUPING | INDEX | BROKEN_LINK | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 14 */
		{ M_REPORT_INTERNAL_ERROR, NULL, conf->max_internal_errors, _("from Internal Errors"),
			INDEX | BROKEN_LINK,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 15 */
		{ M_REPORT_SEARCH_STRINGS, NULL, conf->max_search_strings, _("from SearchStrings"),
			INDEX | PERCENT | GROUPING,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Search String"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 16 */
		{ M_REPORT_SEARCH_ENGINE, NULL, conf->max_search_engines, _("from SearchEngines"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,

			{
				{ _("Hits"), "hits" },
				{ _("Search Engine"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 17 */
		{ M_REPORT_EXTENSION, NULL, conf->max_extensions, _("from Extensions"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				1,
				mplugin_modlogan_create_pic_ext,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Extensions"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 18 */
		{ M_REPORT_VISIT_DURATION, NULL, conf->max_visit_durations, _("from Time Per Visit"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				mplugin_modlogan_create_pic_vd,
			{
				{ _("Visits"), "visits" },
				{ _("Time per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 19 */
		{ M_REPORT_VISIT_PATH_LENGTH, NULL, conf->max_visit_path_lengths, _("from Visit Path Length"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				mplugin_modlogan_create_pic_vpl,

			{
				{ _("Visits"), "visits" },
				{ _("Pages per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 20 */
		{ M_REPORT_COUNTRIES, NULL, conf->max_countries, _("from Countries"),
			VISITS | INDEX | PERCENT | RESOLVE_TLD,
				conf->show_country_graph,
				mplugin_modlogan_create_pic_countries,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Country"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 21 */
		{ M_REPORT_VHOSTS, NULL, conf->max_vhosts, _("from Vhosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				conf->show_vhost_graph,
				mplugin_modlogan_create_pic_vhost,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Vhost"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		{0, NULL, 0, NULL, 0, 0, NULL,
			{
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		}
	};

	if (state == NULL)
		return -1;

	if (state->ext == NULL)
		return -1;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return -1;

	staweb = state->ext;

	/*
	 * set the data
	 */
	reports[0].data = staweb->req_url_hash;
	reports[1].data = staweb->ref_url_hash;
	reports[2].data = staweb->views;
	reports[3].data = staweb->os_hash;
	reports[4].data = staweb->host_hash;
	reports[5].data = get_entry_pages(ext_conf, staweb->visits); /* free me */
	reports[6].data = get_exit_pages(ext_conf, staweb->visits); /* free me */
	reports[7].data = staweb->indexed_pages;
	reports[8].data = staweb->ua_hash;
	reports[9].data = staweb->req_prot_hash;
	reports[10].data = staweb->req_meth_hash;
	reports[11].data = staweb->robots;
	reports[12].data = staweb->bookmarks;
	reports[13].data = staweb->status_missing_file;
	reports[14].data = staweb->status_internal_error;
	reports[15].data = staweb->searchstring;
	reports[16].data = staweb->searchsite;
	reports[17].data = staweb->extension;
	reports[18].data = get_visit_duration(ext_conf, staweb->visits); /* free me */
	reports[19].data = get_visit_path_length(ext_conf, staweb->visits); /* free me */
	reports[20].data = staweb->country_hash;
	reports[21].data = staweb->vhost_hash;
/*	reports[22].data = ;
	reports[23].data = ;
	reports[24].data = ;
	reports[25].data = ;
	reports[26].data = ;*/

	if (write_css(ext_conf)) {
		return -1;
	}

	get_menu_items(ext_conf, state, menu_items);

	for (l = menu_items; l && l->data; l = l->next) {
/* sort the list */
		mlist *min_l = NULL;
		mlist *hl;
		char *min = "";
		char *last;
		
		last = l->data->key;

		for (hl = l->next; hl && hl->data; hl = hl->next) {
			if ( strcmp(hl->data->key, min) > 0 &&
			     strcmp(hl->data->key, last) < 0) {
				min = hl->data->key;
				min_l = hl;
			}
		}

		if (min_l) {
			mdata *d;

			d = l->data;
			l->data = min_l->data;
			min_l->data = d;
		}
	}

	/* write the menu */

	for (l = menu_items; l && l->data; l = l->next) {
		mdata *data = l->data;
		char *sep_main, *sep_sub, *sep_report = NULL;
		
	/* seperate menu string */
		sep_main = strchr(data->key, '/');
		sep_main++;

		sep_sub = strchr(sep_main, '/');
		if (sep_sub) {
			sep_sub++;
			sep_report = strchr(sep_sub, '/');
			if (sep_report) {
				sep_report++;
			}
		}
	/* open file */

		if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
			if (sep_main && strncmp(last_sub, sep_main, 3)) {
				strncpy(last_sub, sep_main, 3);

				if (f) {
					file_end(f, ext_conf);
					fclose(f);
				}

				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				filename = (char *)malloc(strlen(outputdir) + strlen("/m_usage_") + 4 + 2 + 1 + strlen(conf->pages_suffix) + 1);
				if (!filename) return -1;
				sprintf(filename, "%s/m_usage_%04d%02d.%s",
					conf->outputdir ? conf->outputdir : ".",
					state->year, state->month, conf->pages_suffix );

				f = fopen(filename, "w");
				if (!f) fprintf(stderr, "%s.%d: %s %s\n", __FILE__, __LINE__, _("Can't open file"), filename);
				free(filename);
				if (!f) return -1;

				/* create menu */
				file_start(f,ext_conf,state->timestamp);

				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);

				indent(f, --html_indent_depth);
				fprintf(f, "</td>\n");

				indent(f, html_indent_depth++);
				fprintf(f, "<td class=\"skeleton\">\n");
			}
		} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
			if (sep_sub && sep_report &&
				(strncmp(last_sub, sep_sub, 3) != 0 ||
				strncmp(last_report, sep_report, 3) != 0)) {

				strncpy(last_report, sep_report, 3);
				strncpy(last_sub, sep_sub, 3);

				if (f) {
					file_end(f,ext_conf);
					fclose(f);
				}

				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				filename = (char *)malloc(strlen(outputdir) + strlen("/m_usage_") + 4 + 2 + 1
						          + (sep_sub ? strlen(sep_sub): 0) + 1
							  + (sep_report ? strlen(sep_report): 0)
							  + strlen(conf->pages_suffix) + 1);
				if (!filename) return -1;
				sprintf(filename, "%s/m_usage_%04d%02d_%.3s_%s.%s",
					conf->outputdir ? conf->outputdir : ".",
					state->year, state->month,
					sep_sub ? sep_sub : "",
					sep_report ? sep_report : "",
					conf->pages_suffix);

				f = fopen(filename, "w");
				if (!f) fprintf(stderr, "%s.%d: %s %s\n", __FILE__, __LINE__, _("Can't open file"), filename);
				free(filename);
				if (!f) return -1;

				/* create menu */
				file_start(f,ext_conf, state->timestamp);

				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);

				indent(f, --html_indent_depth);
				fprintf(f, "</td>\n");

				indent(f, html_indent_depth++);
				fprintf(f, "<td class=\"skeleton\">\n");
			}
		} else {
			if (sep_sub && strncmp(last_sub, sep_sub, 3)) {
				strncpy(last_sub, sep_sub, 3);

				if (f) {
					file_end(f,ext_conf);
					fclose(f);
				}

				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				filename = (char *)malloc(strlen(outputdir) + strlen("/m_usage_") + 4 + 2 + 1
						          + strlen(last_sub) + 1
							  + strlen(conf->pages_suffix) + 1);
				if (!filename) return -1;
				sprintf(filename, "%s/m_usage_%04d%02d_%s.%s",
					conf->outputdir,
					state->year, state->month, last_sub, conf->pages_suffix);

				f = fopen(filename, "w");
				if (!f) fprintf(stderr, "%s.%d: %s %s\n", __FILE__, __LINE__, _("Can't open file"), filename);
				free(filename);
				if (!f) return -1;

				/* create menu */
				file_start(f,ext_conf, state->timestamp);

				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);

				indent(f, --html_indent_depth);
				fprintf(f, "</td>\n");

				indent(f, html_indent_depth++);
				fprintf(f, "<td class=\"skeleton\">\n");
			}
		}

		if (f == NULL) {
#if 0
			fprintf(stderr, "%s.%d: no filedescriptor opened !!!\n", __FILE__, __LINE__);
#endif
			continue;
		}

	/* write report */

		/* data.count.grouped is used in a differend fashion here */
		switch (data->data.count.grouped) {
		case M_REPORT_VISIT_PATH:
			{
				write_report_header(ext_conf, f, sep_sub, sep_report);

				table_start(f,
					    table_header(conf->max_visit_paths,
							 mhash_count(staweb->visits),
							 _("from Visit Path")),
					    4);

				indent(f, html_indent_depth);
				fprintf(f,
					"<tr><th>#</th><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
					"visits", _("Visits"),
					_("Visit Path")
					);

				show_visit_path(ext_conf, f, staweb->visits,
						conf->max_visit_paths,
						HIGHLIGHT | GROUPING | INDEX | PERCENT);

				if (conf->max_visit_paths > BOTTOM_THRESHOLD) {
					indent(f, html_indent_depth);
					fprintf(f,
						"<tr><th>#</th><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
						"visits", _("Visits"),
						_("Visit Path")
						);
				}
				table_end(f);
				break;
			}
		case M_REPORT_STATUS_CODES:
			{
				char *ref;

				ref = mplugin_modlogan_create_pic_status(ext_conf, state);

				if (ref && *ref) {
					fprintf(f, "%s", ref);
				}

				write_report_header(ext_conf, f, sep_sub, sep_report);
				table_start(f, table_header(conf->max_status_codes, mhash_count(staweb->status_hash),_("from Status Code")), 3);
				indent(f, html_indent_depth);
				fprintf(f,"<tr><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
					"hits",	_("Hits"),
					_("Status Code")
					);
				show_status_mhash(f,staweb->status_hash, conf->max_status_codes);
				if (conf->max_status_codes > BOTTOM_THRESHOLD) {
					indent(f, html_indent_depth);
					fprintf(f,"<tr><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
						"hits",	_("Hits"),
						_("Status Code")
						);
				}
				table_end(f);
				break;
			}
		case M_REPORT_SUMMARY:
			{

				write_report_header(ext_conf, f, sep_sub, sep_report);
				sumdat.files	= maxdat.files		= 0;
				sumdat.xfersize	= maxdat.xfersize	= 0;
				sumdat.hits	= maxdat.hits		= 0;
				sumdat.hosts	= maxdat.hosts		= 0;
				sumdat.pages	= maxdat.pages		= 0;
				sumdat.visits	= maxdat.visits		= 0;

				/* count the values */
				for ( i = 0; i < 31; i++) {
					if (staweb->days[i].hits) last_day = i+1;
					sumdat.files	+= staweb->days[i].files;
					sumdat.xfersize	+= staweb->days[i].xfersize;
					sumdat.hits	+= staweb->days[i].hits;
					sumdat.hosts	+= staweb->days[i].hosts;
					sumdat.pages	+= staweb->days[i].pages;
					sumdat.visits	+= staweb->days[i].visits;

					if (maxdat.files < staweb->days[i].files)
						maxdat.files	= staweb->days[i].files;
					if (maxdat.hits < staweb->days[i].hits)
						maxdat.hits	= staweb->days[i].hits;
					if (maxdat.hosts < staweb->days[i].hosts)
						maxdat.hosts	= staweb->days[i].hosts;
					if (maxdat.pages < staweb->days[i].pages)
						maxdat.pages	= staweb->days[i].pages;
					if (maxdat.visits < staweb->days[i].visits)
						maxdat.visits	= staweb->days[i].visits;
					if (maxdat.xfersize < staweb->days[i].xfersize)
						maxdat.xfersize	= staweb->days[i].xfersize;
				}



				maxdat.hosts = sumdat.hosts = mhash_count(staweb->host_hash);

				table_start(f, _("Summary"), 4);

				/* Totals */
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", "hits", _("Total Hits"), sumdat.hits);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", "files", _("Total Files"), sumdat.files);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", "pages", _("Total Pages"), sumdat.pages);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", "visits", _("Total Visits"), sumdat.visits);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%.0f</td></tr>\n", "traffic", _("Transfered KBytes"), sumdat.xfersize / 1024);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", "hosts",  _("Total Hosts"), sumdat.hosts);

				fprintf(f,"<tr><th colspan=\"2\">&nbsp;</th><th>%s</th><th>%s</th></tr>\n", _("avg"), _("max"));

				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", "hits",
					_("Hits per Day"), sumdat.hits / last_day, maxdat.hits);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", "files",
					_("Files per Day"), sumdat.files / last_day, maxdat.files);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", "pages",
					_("Pages per Day"), sumdat.pages / last_day, maxdat.pages);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", "visits",
					_("Visits per Day"), sumdat.visits / last_day, maxdat.visits);
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%.0f</td><td align=\"right\">%.0f</td></tr>\n", "traffic",
					_("Transfered Kbytes per Day"), (sumdat.xfersize / 1024) / last_day, maxdat.xfersize / 1024);
#if 0
				/* this value is fake, as me are not counting the host per day */
				fprintf(f,"<tr><td class=\"%s\">&nbsp;</td><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", "hosts",
					_("Hosts per Day"), sumdat.hosts / last_day, maxdat.hosts);
#endif

				
				fprintf(f,"<tr><td>&nbsp;</td><td>%s</td><td align=\"right\">", _("Time per visit"));
				if (sumdat.visits) {
					double allvisitduration = get_visit_full_duration(staweb->visits);
					d = allvisitduration / (double)sumdat.visits;
					fprintf(f,"%s",seconds_to_string(d, 0));
				} else {
					fprintf(f,"---");
				}
				fprintf(f,"</td><td align=\"right\">---</td></tr>\n");

				if (sumdat.visits) {
					d = sumdat.pages / (double)sumdat.visits;
#if 0
					fprintf(stderr, "%s.%d: %.2f %.2f\n",
						__FILE__, __LINE__,
						sumdat.pages / (double)sumdat.visits,
						get_pages_per_visit(staweb)
					       );
#endif
				} else {
					d = 0;
				}
				fprintf(f,"<tr><td>&nbsp;</td><td>%s</td><td align=\"right\">%.2f</td><td align=\"right\">%s</td></tr>\n",
					_("Pages per visit"), d, "---");

				if (sumdat.pages && staweb->views) {
					double allviewduration = mhash_sumup(staweb->views);
					d = allviewduration / (double)sumdat.pages;
				} else {
					d = 0;
				}
				fprintf(f,"<tr><td>&nbsp;</td><td>%s</td><td align=\"right\">%s</td><td align=\"right\">%s</td></tr>\n",
					_("Duration per page"), seconds_to_string(d, 0), "---");

				s_200 = mhash_get_value(staweb->status_hash, "200");
				s_304 = mhash_get_value(staweb->status_hash, "304");

				d = ((double)s_304/(s_200+s_304)) * 100;
				fprintf(f,"<tr><td>&nbsp;</td><td>%s</td><td align=\"right\">%.2f%%</td><td align=\"right\">%s</td></tr>\n",
					_("Cache Hit ratio"), d, "---");

				table_end(f);
				break;
			}
		case M_REPORT_DAILY:
			{
				char *ref;

				write_report_header(ext_conf, f, sep_sub, sep_report);
				if (conf->show_daily_graph) {

					ref = mplugin_modlogan_create_pic_31_day(ext_conf, state);

					if (ref && *ref) {
						fprintf(f, "%s", ref);
					}
				}

				table_start(f, _("Daily Statistics"), 6);
				indent(f, html_indent_depth);
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Day"),
					"hits",	 _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);

				for ( i = 0; i < last_day; i++) {
					fprintf(f,"<tr><td>%d</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%.0f</td></tr>\n",
						i+1,
						staweb->days[i].hits,
						staweb->days[i].files,
						staweb->days[i].pages,
						staweb->days[i].visits,
						staweb->days[i].xfersize / 1024
						);
				}

				if (last_day > BOTTOM_THRESHOLD) {
					indent(f, html_indent_depth);
					fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
						_("Day"),
						"hits", _("Hits"),
						"files", _("Files"),
						"pages", _("Pages"),
						"visits", _("Visits"),
						"traffic", _("KBytes")
						);
				}
				table_end(f);

				break;
			}
		case M_REPORT_HOURLY:
			{
				char *ref;

				write_report_header(ext_conf, f, sep_sub, sep_report);
				if (conf->show_hourly_graph) {
					ref = mplugin_modlogan_create_pic_24_hour(ext_conf, state);

					if (ref && *ref) {
						fprintf(f, "%s", ref);
					}
				}

				table_start(f, _("Hourly Statistics"), 6);
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Hour"),
					"hits", _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);
				for ( i = 0; i < 24; i++) {
					fprintf(f,"<tr><td>%d</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%.0f</td></tr>\n",
						i,
						staweb->hours[i].hits,
						staweb->hours[i].files,
						staweb->hours[i].pages,
						staweb->hours[i].visits,
						staweb->hours[i].xfersize / 1024
						);
				}
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Hour"),
					"hits", _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);
				table_end(f);
				break;
			}
		default:
			{
				int i = 0;

				/* don't try the generate a HTML-page for the master-pages */
				if (data->data.count.grouped > 128)
					break;

				for (i = 0; reports[i].data != NULL; i++) {
					if (reports[i].type == data->data.count.grouped) {
						int w = 0, j;

						write_report_header(ext_conf, f, sep_sub, sep_report);

						if (reports[i].show_graph && reports[i].draw_graph) {
							char *ref = reports[i].draw_graph(ext_conf, state);

							if (ref && *ref) {
								fprintf(f, "%s", ref);
							}
						}

						/* calculate the number coloumns */
						for (j = 0; reports[i].fields[j].name != NULL; j++, w++);
						if (reports[i].options & INDEX) w++;
						if (reports[i].options & PERCENT) w++;

						/* write header */
						table_start(f,
							    table_header(reports[i].max, mhash_count(reports[i].data), reports[i].title),
							    w);

						indent(f, html_indent_depth);
						fprintf(f, "<tr>");

						if (reports[i].options & INDEX) {
							fprintf(f, "<th>#</th>");
						}

						for (j = 0; reports[i].fields[j].name != NULL; j++) {
							if (reports[i].fields[j].class) {
								fprintf(f, "<th class=\"%s\">%s</th>",
									reports[i].fields[j].class,
									reports[i].fields[j].name);
							} else {
								fprintf(f, "<th>%s</th>",
									reports[i].fields[j].name);
							}

							if (j == 0 && reports[i].options & PERCENT) {
								fprintf(f, "<th>%%</th>");
							}
						}

						fprintf(f, "</tr>\n");
#if 0
						fprintf(stderr, "%s.%d: %d (%s - %d)\n", __FILE__, __LINE__, i, reports[i].title, reports[i].max);
#endif
						show_mhash(ext_conf, f, reports[i].data, reports[i].max, reports[i].options);

						if (reports[i].max > BOTTOM_THRESHOLD) {
							indent(f, html_indent_depth);

							fprintf(f, "<tr>");
							if (reports[i].options & INDEX) {
								fprintf(f, "<th>#</th>");
							}

							for (j = 0; reports[i].fields[j].name != NULL; j++) {
								if (reports[i].fields[j].class) {
									fprintf(f, "<th class=\"%s\">%s</th>",
										reports[i].fields[j].class,
										reports[i].fields[j].name);
								} else {
									fprintf(f, "<th>%s</th>",
										reports[i].fields[j].name);
								}

								if (j == 0 && reports[i].options & PERCENT) {
									fprintf(f, "<th>%%</th>");
								}
							}

							fprintf(f, "</tr>\n");
						}

						table_end(f);

						break;
					}
				}

				if (reports[i].data == NULL) {
					fprintf(stderr,
						"%s.%d: don't know how to generate a report for %d\n",
						__FILE__, __LINE__,
						data->data.count.grouped);
					return -1;
				}
			}
			break;
		}
	}

	mhash_free(reports[5].data);
	mhash_free(reports[6].data);
	mhash_free(reports[18].data);
	mhash_free(reports[19].data);

	mlist_free(menu_items);

	if (f) {
		file_end(f,ext_conf);

		fclose (f);
	}

	return 0;
}

static int generate_history_output(mconfig *ext_conf, mlist *history) {
	mlist *l = history;
	FILE *f;
	char *filename;
	config_output *conf = ext_conf->plugin_conf;
	data_History hist, yearly;
	long max_mean_hits = -1;
	long max_mean_files = -1;
	long max_mean_pages = -1;
	long max_mean_visits = -1;
	double max_mean_xfersize = -1;
	long max_hits = -1;
	long max_files = -1;
	long max_pages = -1;
	long max_visits = -1;
	double max_xfersize = -1;
 
	if (history == NULL) return -1;

	/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
	** courtesy of Jan Kneschke
	*/
	/* suffix for the generated statistics page files */

	filename = (char *)malloc(strlen(conf->outputdir) + strlen("/index.") + strlen(conf->pages_suffix) + 1);
	if (!filename) return -1;
	sprintf(filename, "%s/index.%s",
		conf->outputdir,
		conf->pages_suffix );
	f = fopen(filename, "w");
	free(filename);
	if (!f) return -1;

	file_start_index(f,ext_conf,0);

	if (conf->show_monthly_graph) {
		char * ref = mplugin_modlogan_create_pic_X_month(ext_conf, history);

		if (ref && *ref) {
			fprintf(f, "%s", ref);
		}
	}

	table_start(f, _("History"), -11);

	fprintf(f,"<tr><th>&nbsp;</th><th colspan=\"5\">%s</th><th colspan=\"5\">%s</th></tr>",
		_("Average/day"),
		_("Totals")
		);
	fprintf(f,"<tr><th>%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th>"
		"</tr>\n",
		_("Month"),
		"hits", _("Hits"),
		"files", _("Files"),
		"pages", _("Pages"),
		"visits", _("Visits"),
		"traffic", _("KBytes"),
		"hits", _("Hits"),
		"files", _("Files"),
		"pages", _("Pages"),
		"visits", _("Visits"),
		"traffic", _("KBytes")
		);


#define HIST(x) \
	hist.data.web.x = 0;

	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
#undef HIST
	hist.days_passed = 0;

#define HIST(x) \
	yearly.data.web.x = 0;

	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
#undef HIST
	yearly.year = 0;
	yearly.days_passed = 0;

	if (conf->show_monthly_maxima) {
		/* go to the last element */
		for (l = history; l->next; l = l->next);

		/* walk the list backwards */
		for (;l; l = l->prev) {
			mdata *data = l->data;

			if (!data) break;
	
			if (data->data.hist->days_passed != 0) {
				long mean_hits, mean_files, mean_pages, mean_visits;
				double mean_xfersize;
				long hits, files, pages, visits;
				double xfersize;
#define max(x,y) ((x) > (y) ? (x) : (y))
			
				mean_hits = data->data.hist->data.web.hits / data->data.hist->days_passed;
				mean_files = data->data.hist->data.web.files / data->data.hist->days_passed;
				mean_pages = data->data.hist->data.web.pages / data->data.hist->days_passed;
				mean_visits = data->data.hist->data.web.visits / data->data.hist->days_passed;
				mean_xfersize = data->data.hist->data.web.xfersize / 1024 / data->data.hist->days_passed;

				max_mean_hits = max(mean_hits, max_mean_hits);
				max_mean_files = max(mean_files, max_mean_files);
				max_mean_pages = max(mean_pages, max_mean_pages);
				max_mean_visits = max(mean_visits, max_mean_visits);
				max_mean_xfersize = max(mean_xfersize, max_mean_xfersize);
			
				hits = data->data.hist->data.web.hits;
				files = data->data.hist->data.web.files;
				pages = data->data.hist->data.web.pages;
				visits = data->data.hist->data.web.visits;
				xfersize = data->data.hist->data.web.xfersize / 1024;

				max_hits = max(hits, max_hits);
				max_files = max(files, max_files);
				max_pages = max(pages, max_pages);
				max_visits = max(visits, max_visits);
				max_xfersize = max(xfersize, max_xfersize);
#undef max
			}
		}
	}
	
	
	/* go to the last element */
	for (l = history; l->next; l = l->next);

	/* walk the list backwards */
	for (;l; l = l->prev) {
		mdata *data = l->data;

		if (!data) break;

		if (data->data.hist->days_passed != 0) {
			if (yearly.year > data->data.hist->year) {
				/* add yearly summary */
				fprintf(f,"<tr><th>%04d</th>"
					"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
					"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
					"<th class=\"%s\">%.0f</th>"
					"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
					"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
					"<th class=\"%s\">%.0f</th>"
					"</tr>\n",
					yearly.year,

					"hits", yearly.data.web.hits / yearly.days_passed,
					"files", yearly.data.web.files / yearly.days_passed,
					"pages", yearly.data.web.pages / yearly.days_passed,
					"visits", yearly.data.web.visits / yearly.days_passed,
					"traffic", yearly.data.web.xfersize / 1024 / yearly.days_passed,
					"hits", yearly.data.web.hits,
					"files", yearly.data.web.files,
					"pages", yearly.data.web.pages,
					"visits", yearly.data.web.visits,
					"traffic", yearly.data.web.xfersize / 1024
					);
			}
			{
				long mean_hits, mean_files, mean_pages, mean_visits;
				double mean_xfersize;
				long hits, files, pages, visits;
				double xfersize;
				
				mean_hits = data->data.hist->data.web.hits / data->data.hist->days_passed;
				mean_files = data->data.hist->data.web.files / data->data.hist->days_passed;
				mean_pages = data->data.hist->data.web.pages / data->data.hist->days_passed;
				mean_visits = data->data.hist->data.web.visits / data->data.hist->days_passed;
				mean_xfersize = data->data.hist->data.web.xfersize / 1024 / data->data.hist->days_passed;
			
				hits = data->data.hist->data.web.hits;
				files = data->data.hist->data.web.files;
				pages = data->data.hist->data.web.pages;
				visits = data->data.hist->data.web.visits;
				xfersize = data->data.hist->data.web.xfersize / 1024;

				fprintf(f,"<tr><td class=\"centerb\"><a href=\"%s\">%s&nbsp;%04i</a></td>",
					get_url(ext_conf, data->data.hist->year, data->data.hist->month, "000", "000"),
					get_month_string(data->data.hist->month,1),
					data->data.hist->year);

			
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_mean_hits > mean_hits)?'r':'m', mean_hits);
			       	fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_mean_files > mean_files)?'r':'m', mean_files);
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_mean_pages > mean_pages)?'r':'m', mean_pages);
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_mean_visits > mean_visits)?'r':'m', mean_visits);
				fprintf(f,"<td class=\"tiny%c\">%.0f</td>", (max_mean_xfersize > mean_xfersize)?'r':'m', mean_xfersize);
			
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_hits > hits)?'r':'m', hits);
			       	fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_files > files)?'r':'m', files);
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_pages > pages)?'r':'m', pages);
				fprintf(f,"<td class=\"tiny%c\">%ld</td>", (max_visits > visits)?'r':'m', visits);
				fprintf(f,"<td class=\"tiny%c\">%.0f</td>", (max_xfersize > xfersize)?'r':'m', xfersize);

				fprintf(f,"</tr>\n");
			}
					
			if (yearly.year > data->data.hist->year) {
				yearly.year = data->data.hist->year;
				yearly.days_passed = data->data.hist->days_passed;

#define HIST(x) \
	yearly.data.web.x = data->data.hist->data.web.x;
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
#undef HIST

			} else {
#define HIST(x) \
	yearly.data.web.x += data->data.hist->data.web.x
				yearly.year = data->data.hist->year;
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);

				yearly.days_passed += data->data.hist->days_passed;
#undef HIST
			}
#define HIST(x) \
	hist.data.web.x += data->data.hist->data.web.x

			HIST(hits);
			HIST(files);
			HIST(pages);
			HIST(visits);
			HIST(xfersize);
			hist.days_passed += data->data.hist->days_passed;
#undef HIST
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_OUTPUT, M_DEBUG_LEVEL_WARNINGS,
				 "count == 0, is this ok ?? splitby for '%s' without an entry ??\n",
				 data->key);
		}
	}

	if (yearly.year && yearly.days_passed) {
		/* add the last yearly summary */
		fprintf(f,"<tr><th>%04d</th>"
			"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
			"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
			"<th class=\"%s\">%.0f</th>"
			"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
			"<th class=\"%s\">%ld</th><th class=\"%s\">%ld</th>"
			"<th class=\"%s\">%.0f</th>"
			"</tr>\n",
			yearly.year,

			"hits", yearly.data.web.hits / yearly.days_passed,
			"files", yearly.data.web.files / yearly.days_passed,
			"pages", yearly.data.web.pages / yearly.days_passed,
			"visits", yearly.data.web.visits / yearly.days_passed,
			"traffic", yearly.data.web.xfersize / 1024 / yearly.days_passed,
			"hits", yearly.data.web.hits,
			"files", yearly.data.web.files,
			"pages", yearly.data.web.pages,
			"visits", yearly.data.web.visits,
			"traffic", yearly.data.web.xfersize / 1024
			);
	}

	if (hist.days_passed) {
		/* overall summary */
		fprintf(f,"<tr><th>%s</th>"
			"<th>%ld</th><th>%ld</th>"
			"<th>%ld</th><th>%ld</th>"
			"<th>%.0f</th>"
			"<th>%ld</th><th>%ld</th>"
			"<th>%ld</th><th>%ld</th>"
			"<th>%.0f</th>"
 			"</tr>\n",

			_("total"),
			hist.data.web.hits / hist.days_passed,
			hist.data.web.files / hist.days_passed,
			hist.data.web.pages / hist.days_passed,
			hist.data.web.visits / hist.days_passed,
			hist.data.web.xfersize / 1024 / hist.days_passed,
			hist.data.web.hits,
			hist.data.web.files,
			hist.data.web.pages,
			hist.data.web.visits,
			hist.data.web.xfersize / 1024
			);
	}

	table_end(f);

	file_end_index(f,ext_conf);

	fclose(f);

	return 0;
}


int mplugins_output_modlogan_generate_monthly_output(mconfig * ext_conf, mstate * state, const char *subpath) {
	int ret;
	if (!mplugins_output_modlogan_patch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not patch config\n");
		return -1;
	}

	if (subpath) {
		char *s;
		config_output *conf = ext_conf->plugin_conf;
		/* create the subdir if neccesary */

		s = malloc(strlen(subpath) + strlen(conf->outputdir) + strlen("//") + 1);
		sprintf(s, "%s/%s/",
			conf->outputdir,
			subpath);
		if (-1 == mkdir(s, 0755)) {
			if (errno != EEXIST) {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "mkdir failed: %s\n", strerror(errno));
				return -1;
			}
		}

		buffer_copy_string(conf->vhost_name, subpath);
		/* modify output directory */
		free(conf->outputdir);
		conf->outputdir = s;

		fprintf(stderr, "generating output in %s\n", conf->outputdir);
	}

	if ((ret = generate_monthly_output(ext_conf, state))) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generate_monthly_output failed\n");
	}

	if (!mplugins_output_modlogan_unpatch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not un-patch config\n");
		return -1;
	}
	return 0;
}

int mplugins_output_modlogan_generate_history_output(mconfig *ext_conf, mlist *history, const char *subpath) {
	int ret;
	if (!mplugins_output_modlogan_patch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not patch config\n");
		return -1;
	}

	if (subpath) {
		char *s;
		config_output *conf = ext_conf->plugin_conf;
		/* create the subdir if neccesary */

		s = malloc(strlen(subpath) + strlen(conf->outputdir) + strlen("//") + 1);
		sprintf(s, "%s/%s/",
			conf->outputdir,
			subpath);

		if (-1 == mkdir(s, 0755)) {
			if (errno != EEXIST) {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "mkdir failed: %s\n", strerror(errno));
				return -1;
			}
		}

		/* modify output directory */
		free(conf->outputdir);
		conf->outputdir = s;

		fprintf(stderr, "generating history in %s\n", conf->outputdir);
	}

	ret = generate_history_output(ext_conf, history);

	if (!mplugins_output_modlogan_unpatch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not un-patch config\n");
		return -1;
	}
	return 0;
}

