#ifndef PDUS_H
#define PDUS_H

#include <cassert>

#include "SDLTools.h"
#include "ObjectFwd.h"
class StaticDecorationBase;


//----------------------------------------------------------------------------
#define DECLARE_PDU(C) \
  public: \
    ~C(); \
    static Uint16 sm_pduId

//----------------------------------------------------------------------------
#define DECLARE_OBJECT_PDU(O) \
  public: \
    O##PDU(); \
    O##PDU(const O *o); \
    ~O##PDU(); \
    void update(const O *o); \
    static Uint16 sm_createPDUId; \
    static Uint16 sm_updatePDUId



//----------------------------------------------------------------------------
class PDUVisitor;
class PDUConstVisitor;

//----------------------------------------------------------------------------
#define DECLARE_PDU_VISITOR_API() \
    void do_accept(PDUVisitor &v); \
    void do_accept(PDUConstVisitor &v) const

//----------------------------------------------------------------------------
#define DECLARE_PDU_VISITOR_API_BODY(C) \
void C::do_accept(PDUVisitor &v) { v.visit(this); } \
void C::do_accept(PDUConstVisitor &v) const { v.visit(this); }



//----------------------------------------------------------------------------
class PDUBase
{
  public:
    //------------------------------------------------------------------------
    enum Ids
    {
        HEADER_PDUID                  = 0x0000,

        CREATE_BARRIER_PDUID          = 0x0001,
        CREATE_BLACKHOLE_PDUID        = 0x0002,
        CREATE_CRATE_PDUID            = 0x0003,
        CREATE_GRENADE_PDUID          = 0x0004,
        CREATE_GRINDER_PDUID          = 0x0005,
        CREATE_MAGNET_PDUID           = 0x0006,
        CREATE_MISSILE_PDUID          = 0x0007,
        CREATE_MORTAR_PDUID           = 0x0008,
        CREATE_PARTICLEFOUNTAIN_PDUID = 0x0009,
        CREATE_PLATFORM_PDUID         = 0x000a,
        CREATE_PROJECTILE_PDUID       = 0x000b,
        CREATE_SAMBATTERY_PDUID       = 0x000c,
        CREATE_SHIP_PDUID             = 0x000d,
        CREATE_SWITCH_PDUID           = 0x000e,
        CREATE_TANK_PDUID             = 0x000f,
        CREATE_THORN_PDUID            = 0x0010,
        CREATE_TURRET_PDUID           = 0x0011,
        
        UPDATE_BARRIER_PDUID          = 0x0101,
        UPDATE_BLACKHOLE_PDUID        = 0x0102,
        UPDATE_CRATE_PDUID            = 0x0103,
        UPDATE_GRENADE_PDUID          = 0x0104,
        UPDATE_GRINDER_PDUID          = 0x0105,
        UPDATE_MAGNET_PDUID           = 0x0106,
        UPDATE_MISSILE_PDUID          = 0x0107,
        UPDATE_MORTAR_PDUID           = 0x0108,
        UPDATE_PARTICLEFOUNTAIN_PDUID = 0x0109,
        UPDATE_PLATFORM_PDUID         = 0x010a,
        UPDATE_PROJECTILE_PDUID       = 0x010b,
        UPDATE_SAMBATTERY_PDUID       = 0x010c,
        UPDATE_SHIP_PDUID             = 0x010d,
        UPDATE_SWITCH_PDUID           = 0x010e,
        UPDATE_TANK_PDUID             = 0x010f,
        UPDATE_THORN_PDUID            = 0x0110,
        UPDATE_TURRET_PDUID           = 0x0111,

        DELETE_OBJECT_PDUID           = 0x0200
    };

  protected:
    //------------------------------------------------------------------------
    PDUBase();

  public:
    //------------------------------------------------------------------------
    virtual ~PDUBase() = 0;

    //------------------------------------------------------------------------
    inline void accept(PDUVisitor &v) { do_accept(v); }
    inline void accept(PDUConstVisitor &v) const { do_accept(v); }

  private:
    //------------------------------------------------------------------------
    virtual void do_accept(PDUVisitor &v) = 0;
    virtual void do_accept(PDUConstVisitor &v) const = 0;
};


//----------------------------------------------------------------------------
class ObjectPDUBase : public PDUBase
{
  protected:
    //------------------------------------------------------------------------
    ObjectPDUBase();
    ObjectPDUBase(const ObjectBase *o);

  public:
    //------------------------------------------------------------------------
    virtual ~ObjectPDUBase() = 0;

    //------------------------------------------------------------------------
    bool m_modified;
    bool m_visited;

    Uint16 m_id;

  protected:
    //------------------------------------------------------------------------
    template<class T>
    void do_update(T &m, const T newValue)
    {
        if (m != newValue)
        {
            m = newValue;
            m_modified = true;
        }
    }
};


//----------------------------------------------------------------------------
class StaticDecorationObjectPDUBase : public ObjectPDUBase
{
  protected:
    //------------------------------------------------------------------------
    StaticDecorationObjectPDUBase();
    StaticDecorationObjectPDUBase(const StaticDecorationBase *o);

  public:
    //------------------------------------------------------------------------
    virtual ~StaticDecorationObjectPDUBase() = 0;

    //------------------------------------------------------------------------
    Uint16 m_x;
    Uint16 m_y;
};



//----------------------------------------------------------------------------
class BarrierPDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(Barrier);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;
    Uint8 m_wh;
    Uint8 m_activated;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class BlackHolePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(BlackHole);

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class CratePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(Crate);

  public:
    //------------------------------------------------------------------------
    Uint8 m_type;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class GrenadePDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(Grenade);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class GrinderPDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(Grinder);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;
    Uint8 m_frame;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class MagnetBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(MagnetBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;
    Uint8 m_wh;
    Uint8 m_frame;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class MissilePDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(Missile);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;
    Uint16 m_angle;
    Uint8 m_thrust;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class MortarBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(MortarBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class ParticleFountainBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(ParticleFountainBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class PlatformPDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(Platform);

  public:
    //------------------------------------------------------------------------
    Uint8 m_w;
    Uint8 m_leftEdge;
    Uint8 m_rightEdge;
    Uint8 m_frame;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class ProjectileBasePDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(ProjectileBase);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class SAMBatteryBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(SAMBatteryBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class ShipPDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(Ship);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;
    Uint8 m_type;
    Uint16 m_angle;
    Uint8 m_thrust;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class SwitchBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(SwitchBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class TankPDU : public ObjectPDUBase
{
    DECLARE_OBJECT_PDU(Tank);

  public:
    //------------------------------------------------------------------------
    Sint16 m_x;
    Sint16 m_y;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class ThornPDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(Thorn);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;
    Uint16 m_size;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class TurretBasePDU : public StaticDecorationObjectPDUBase
{
    DECLARE_OBJECT_PDU(TurretBase);

  public:
    //------------------------------------------------------------------------
    Uint8 m_orientation;
    Uint8 m_type;

  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};


//----------------------------------------------------------------------------
class HeaderPDU : public PDUBase
{
    DECLARE_PDU(HeaderPDU);

  public:
    //------------------------------------------------------------------------
    HeaderPDU();

    //------------------------------------------------------------------------
    Uint16 m_version;
    Uint16 m_length;
    
  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};



//----------------------------------------------------------------------------
class DeleteObjectPDU : public PDUBase
{
    DECLARE_PDU(DeleteObjectPDU);

  public:
    //------------------------------------------------------------------------
    DeleteObjectPDU(const Uint16 id) { m_id = id; }

    //------------------------------------------------------------------------
    Uint16 m_id;
    
  private:
    //------------------------------------------------------------------------
    DECLARE_PDU_VISITOR_API();
};



//----------------------------------------------------------------------------
class PDUVisitor
{
  protected:
    //------------------------------------------------------------------------
    PDUVisitor();

  public:
    //------------------------------------------------------------------------
    virtual ~PDUVisitor() = 0;

    //------------------------------------------------------------------------
    inline void visit(BarrierPDU *pdu) { do_visit(pdu); }
    inline void visit(BlackHolePDU *pdu) { do_visit(pdu); }
    inline void visit(CratePDU *pdu) { do_visit(pdu); }
    inline void visit(GrenadePDU *pdu) { do_visit(pdu); }
    inline void visit(GrinderPDU *pdu) { do_visit(pdu); }
    inline void visit(MagnetBasePDU *pdu) { do_visit(pdu); }
    inline void visit(MissilePDU *pdu) { do_visit(pdu); }
    inline void visit(MortarBasePDU *pdu) { do_visit(pdu); }
    inline void visit(ParticleFountainBasePDU *pdu) { do_visit(pdu); }
    inline void visit(PlatformPDU *pdu) { do_visit(pdu); }
    inline void visit(ProjectileBasePDU *pdu) { do_visit(pdu); }
    inline void visit(SAMBatteryBasePDU *pdu) { do_visit(pdu); }
    inline void visit(ShipPDU *pdu) { do_visit(pdu); }
    inline void visit(SwitchBasePDU *pdu) { do_visit(pdu); }
    inline void visit(TankPDU *pdu) { do_visit(pdu); }
    inline void visit(ThornPDU *pdu) { do_visit(pdu); }
    inline void visit(TurretBasePDU *pdu) { do_visit(pdu); }

    //------------------------------------------------------------------------
    inline void visit(HeaderPDU *pdu) { do_visit(pdu); }
    inline void visit(DeleteObjectPDU *pdu) { do_visit(pdu); }

  private:
    //------------------------------------------------------------------------
    virtual void do_visit(BarrierPDU *pdu) = 0;
    virtual void do_visit(BlackHolePDU *pdu) = 0;
    virtual void do_visit(CratePDU *pdu) = 0;
    virtual void do_visit(GrenadePDU *pdu) = 0;
    virtual void do_visit(GrinderPDU *pdu) = 0;
    virtual void do_visit(MagnetBasePDU *pdu) = 0;
    virtual void do_visit(MissilePDU *pdu) = 0;
    virtual void do_visit(MortarBasePDU *pdu) = 0;
    virtual void do_visit(ParticleFountainBasePDU *pdu) = 0;
    virtual void do_visit(PlatformPDU *pdu) = 0;
    virtual void do_visit(ProjectileBasePDU *pdu) = 0;
    virtual void do_visit(SAMBatteryBasePDU *pdu) = 0;
    virtual void do_visit(ShipPDU *pdu) = 0;
    virtual void do_visit(SwitchBasePDU *pdu) = 0;
    virtual void do_visit(TankPDU *pdu) = 0;
    virtual void do_visit(ThornPDU *pdu) = 0;
    virtual void do_visit(TurretBasePDU *pdu) = 0;

    //------------------------------------------------------------------------
    virtual void do_visit(HeaderPDU *pdu) { assert(false); }
    virtual void do_visit(DeleteObjectPDU *pdu) { assert(false); }
};


//----------------------------------------------------------------------------
class PDUConstVisitor
{
  protected:
    //------------------------------------------------------------------------
    PDUConstVisitor();

  public:
    //------------------------------------------------------------------------
    virtual ~PDUConstVisitor() = 0;

    //------------------------------------------------------------------------
    inline void visit(const BarrierPDU *pdu) { do_visit(pdu); }
    inline void visit(const BlackHolePDU *pdu) { do_visit(pdu); }
    inline void visit(const CratePDU *pdu) { do_visit(pdu); }
    inline void visit(const GrenadePDU *pdu) { do_visit(pdu); }
    inline void visit(const GrinderPDU *pdu) { do_visit(pdu); }
    inline void visit(const MagnetBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const MissilePDU *pdu) { do_visit(pdu); }
    inline void visit(const MortarBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const ParticleFountainBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const PlatformPDU *pdu) { do_visit(pdu); }
    inline void visit(const ProjectileBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const SAMBatteryBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const ShipPDU *pdu) { do_visit(pdu); }
    inline void visit(const SwitchBasePDU *pdu) { do_visit(pdu); }
    inline void visit(const TankPDU *pdu) { do_visit(pdu); }
    inline void visit(const ThornPDU *pdu) { do_visit(pdu); }
    inline void visit(const TurretBasePDU *pdu) { do_visit(pdu); }

    //------------------------------------------------------------------------
    inline void visit(const HeaderPDU *pdu) { do_visit(pdu); }
    inline void visit(const DeleteObjectPDU *pdu) { do_visit(pdu); }

  private:
    //------------------------------------------------------------------------
    virtual void do_visit(const BarrierPDU *pdu) = 0;
    virtual void do_visit(const BlackHolePDU *pdu) = 0;
    virtual void do_visit(const CratePDU *pdu) = 0;
    virtual void do_visit(const GrenadePDU *pdu) = 0;
    virtual void do_visit(const GrinderPDU *pdu) = 0;
    virtual void do_visit(const MagnetBasePDU *pdu) = 0;
    virtual void do_visit(const MissilePDU *pdu) = 0;
    virtual void do_visit(const MortarBasePDU *pdu) = 0;
    virtual void do_visit(const ParticleFountainBasePDU *pdu) = 0;
    virtual void do_visit(const PlatformPDU *pdu) = 0;
    virtual void do_visit(const ProjectileBasePDU *pdu) = 0;
    virtual void do_visit(const SAMBatteryBasePDU *pdu) = 0;
    virtual void do_visit(const ShipPDU *pdu) = 0;
    virtual void do_visit(const SwitchBasePDU *pdu) = 0;
    virtual void do_visit(const TankPDU *pdu) = 0;
    virtual void do_visit(const ThornPDU *pdu) = 0;
    virtual void do_visit(const TurretBasePDU *pdu) = 0;

    //------------------------------------------------------------------------
    virtual void do_visit(const HeaderPDU *pdu) { assert(false); }
    virtual void do_visit(const DeleteObjectPDU *pdu) { assert(false); }
};

#endif //PDUS_H
