/* ======================================================================
 * vector3d.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-02 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <string.h>
#include <math.h>

// -- Local Includes
#include "vector3d.hh"

using namespace MeshIO;

// -- Code Segment

Vector3D::Vector3D(void)
{
    Clear();
}

Vector3D::Vector3D(float x, float y, float z, float w)
{
    coord.x = x; coord.y = y; coord.z = z; coord.w = w;
}

Vector3D::Vector3D(float _fCoord[3])
{
    for (int i = 0; i < 3; i++) {
	fCoord[i] = _fCoord[i];
    }
}

Vector3D &Vector3D::operator += (const Vector3D vVec)
{
    for (int i = 0; i < 3; i++) {
        fCoord[i] += vVec.fCoord[i];
    }
    return *this;
}

Vector3D &Vector3D::operator -= (const Vector3D vVec)
{    
    for (int i = 0; i < 3; i++) {
        fCoord[i] -= vVec.fCoord[i];
    }
    return *this;
}

Vector3D Vector3D::operator + (const Vector3D vVec) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
        vRes.fCoord[i] = fCoord[i] + vVec.fCoord[i];
    }
    return vRes;
}

Vector3D Vector3D::operator - (const Vector3D vVec) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] - vVec.fCoord[i];
    }
    return vRes;
}

Vector3D Vector3D::operator / (const Vector3D vVec) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] / vVec.fCoord[i];
    }
    return vRes;
}

Vector3D Vector3D::operator * (const Vector3D vVec) const
{
    return Vector3D(fCoord[1] * vVec.fCoord[2] - fCoord[2] * vVec.fCoord[1],
		    fCoord[2] * vVec.fCoord[0] - fCoord[0] * vVec.fCoord[2],
		    fCoord[0] * vVec.fCoord[1] - fCoord[1] * vVec.fCoord[0]);
}

Vector3D &Vector3D::operator += (const float fVal)
{
    for (int i = 0; i < 3; i++) {
        fCoord[i] += fVal;
    }
    return *this;
}

Vector3D &Vector3D::operator -= (const float fVal)
{
    for (int i = 0; i < 3; i++) {
	fCoord[i] -= fVal;
    }
    return *this;
}

Vector3D &Vector3D::operator /= (const float fVal)
{
    for (int i = 0; i < 3; i++) {
	fCoord[i] /= fVal;
    }
    return *this;
}

Vector3D &Vector3D::operator *= (const float fVal)
{
    for (int i = 0; i < 3; i++) {
	fCoord[i] *= fVal;
    }
    return *this;
}

Vector3D Vector3D::operator + (const float fVal) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] + fVal;
    }
    return vRes;
}

Vector3D Vector3D::operator - (const float fVal) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] - fVal;
    }
    return vRes;
}

Vector3D Vector3D::operator / (const float fVal) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] / fVal;
    }
    return vRes;
}

Vector3D Vector3D::operator * (const float fVal) const
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes.fCoord[i] = fCoord[i] * fVal;
    }
    return vRes;
}

Vector3D Vector3D::operator - (void) const
{
    return Vector3D(-fCoord[0], -fCoord[1], -fCoord[2]);
}

Vector3D &Vector3D::operator = (const Vector3D vVec)
{
    for (int i = 0; i < 3; i++) {
	fCoord[i] = vVec.fCoord[i];
    }
    return *this;
}

float Vector3D::Magnitude(void)
{
    return (float) sqrt(fCoord[0] * fCoord[0] + 
			fCoord[1] * fCoord[1] + 
			fCoord[2] * fCoord[2]);
}

void Vector3D::Normalize(void)
{
    // Compute magnitude of vector
    float fMagnitude = Magnitude();
    
    // Sanity check
    if (fMagnitude == 0.0f) return;
    
    // Normalize all components with magnitude
    for (int i = 0; i < 3; i++) {
	fCoord[i] /= fMagnitude;
    }
}

void Vector3D::Clear(void)
{
    memset(fCoord, 0, sizeof(float) * 3); coord.w = 1.0f;
}

float Dot(Vector3D vVec1, Vector3D vVec2)
{
    float fRes = 0.0f;    
    for (int i = 0; i < 3; i++) {
	fRes += vVec1[i] * vVec2[i];
    }
    return fRes;
}

Vector3D operator * (float fVal, Vector3D vVec)
{
    Vector3D vRes;
    for (int i = 0; i < 3; i++) {
	vRes[i] = fVal * vVec[i];
    }
    return vRes;
}



