/* mg-work-core.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-base.h"
#include "mg-server.h"
#include "mg-work-core.h"
#include "mg-query.h"
#include "mg-context.h"
#include "mg-target.h"
#include "mg-entity.h"
#include "mg-field.h"
#include "mg-qf-field.h"
#include "mg-qf-value.h"
#include "mg-qf-all.h"
#include "mg-parameter.h"
#include "mg-db-field.h"
#include "mg-join.h"
#include "mg-condition.h"
#include "mg-db-constraint.h"
#include "mg-database.h"
#include "mg-db-table.h"
#include "mg-graphviz.h"
#include "mg-renderer.h"
#include "mg-result-set.h"
#include "mg-referer.h"

static void mg_work_core_class_init (MgWorkCoreClass *class);
static void mg_work_core_init (MgWorkCore *obj);
static void mg_work_core_dispose (GObject *object);
static void mg_work_core_initialize (MgWorkCore *core);

struct _MgWorkCorePriv {
	/* copy of query_select to which various fields have been added */
        MgQuery           *query_select_hybrid;

	/* objects replacements corresponding to queries being copied */
        GHashTable        *replacements_query_select; /* from the 'query' arg of _new() to 'query_select' */
        GHashTable        *replacements_query_select_hybrid; /* from the 'query_select' to 'query_select_hybrid' */

	GHashTable        *target_to_ref_pk_field; /* */

	/* list of MgQfValue fields added to 'query_select_hybrid' to express a condition in the modif queries */
	GSList            *for_cond_value_qfields;

	/* list of ('query_select_hybrid') fields which have been added when using FK constraints */
	GSList            *ref_integ_fields;

	/* performances improvement */
	GHashTable        *nodes_hash; /* key=node->param, value= MgWorkCoreNode node */
};

/* 
 * this structure represents a dependency resulting from a join between 
 * 'this->dependant->target' (FK) and 'this->target' (Ref'ed PK),
 * the MgJoin for this dependency being 'this->join'
 */
typedef struct _TargetDep{
	MgTarget          *target;
	GSList            *depend_on; /* list of TargetDep structures */

	struct _TargetDep *dependant;
	MgJoin            *join;      /* join between 'this->dependant->target' and 'this->target' */ 
} TargetDep;
#define TARGET_DEP(x) ((TargetDep *) x)

static void nullified_query_cb (MgQuery *query, MgWorkCore *core);
static void nullified_target_cb (MgTarget *target, MgWorkCore *core);
static void nullified_context_cb (MgContext *context, MgWorkCore *core);

MgContext *make_work_context_target (MgWorkCore *core);
MgContext *make_work_context_no_target (MgWorkCore *core);
TargetDep *make_target_deps_recurs (MgWorkCore *core, MgTarget *on_target, 
				    const GSList *joins, GHashTable *joinsdep);
TargetDep *make_target_deps (MgWorkCore *core);
void       make_target_deps_free (TargetDep *dep);
gboolean   make_target_select_queries_improved (MgWorkCore *core, TargetDep *dep, GHashTable *target_query_h, 
						GHashTable *replacements, GError **error);
gboolean   modif_target_depends_on (MgWorkCore *core,  TargetDep *modif_target_dep, MgTarget *target);

static void make_query_update (MgWorkCore *core);
static void make_query_delete (MgWorkCore *core);
static void make_query_insert (MgWorkCore *core);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
mg_work_core_get_type (void)
{
        static GType type = 0;

        if (!type) {
                static const GTypeInfo info = {
                        sizeof (MgWorkCoreClass),
                        (GBaseInitFunc) NULL,
                        (GBaseFinalizeFunc) NULL,
                        (GClassInitFunc) mg_work_core_class_init,
                        NULL,
                        NULL,
                        sizeof (MgWorkCore),
                        0,
                        (GInstanceInitFunc) mg_work_core_init
                };

                type = g_type_register_static (MG_BASE_TYPE, "MgWorkCore", &info, 0);
        }

        return type;
}

static void
mg_work_core_class_init (MgWorkCoreClass * class)
{
        GObjectClass   *object_class = G_OBJECT_CLASS (class);

        parent_class = g_type_class_peek_parent (class);
        object_class->dispose = mg_work_core_dispose;
}

static void
mg_work_core_init (MgWorkCore * core)
{
	core->args_context = NULL;
	core->work_context = NULL;
	core->modif_target = NULL;

	core->query_select = NULL;
	core->query_update = NULL;
	core->query_delete = NULL;
	core->query_insert = NULL;
	core->data_rs = NULL;

	core->nodes = NULL;
	core->params_in_data_rs = NULL;
	core->params_not_in_data_rs = NULL;

	core->params_modif_queries_value_providers = NULL;

	core->no_show_params = NULL;

	core->priv = g_new0 (MgWorkCorePriv, 1);
	core->priv->query_select_hybrid = NULL;
	core->priv->replacements_query_select = NULL;
	core->priv->replacements_query_select_hybrid = NULL;
	core->priv->target_to_ref_pk_field = g_hash_table_new (NULL, NULL);
	core->priv->for_cond_value_qfields = NULL;
	core->priv->ref_integ_fields = NULL;
	core->priv->nodes_hash = NULL;
}

/**
 * mg_work_core_new
 * @query: a #MgQuery object
 * @modified: a #MgTarget object, or %NULL
 *
 * Creates a new #MgWorkCore object and fills it with appropriate data
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #MgDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * Returns: the new object
 */
GObject *
mg_work_core_new (MgQuery *query, MgTarget *modified)
{
	GObject *obj;
	MgWorkCore *core;
	GHashTable *replacements;

	g_return_val_if_fail (query && IS_MG_QUERY (query), NULL);
	g_return_val_if_fail (mg_query_get_query_type (query) == MG_QUERY_TYPE_SELECT, NULL);
	
	if (modified) {
		g_return_val_if_fail (IS_MG_TARGET (modified), NULL);
		g_return_val_if_fail (mg_target_get_query (modified) == query, NULL);
		g_return_val_if_fail (mg_entity_is_writable (mg_target_get_represented_entity (modified)), NULL);
	}

	obj = g_object_new (MG_WORK_CORE_TYPE, "conf", mg_base_get_conf (MG_BASE (query)), NULL);
	core = MG_WORK_CORE (obj);

	replacements = g_hash_table_new (NULL, NULL);
	core->query_select = MG_QUERY (mg_query_new_copy (query, replacements));
	mg_query_order_fields_using_join_conds (core->query_select);
	g_signal_connect (G_OBJECT (core->query_select), "nullified",
			  G_CALLBACK (nullified_query_cb), core);

	if (modified) {
		core->modif_target = g_hash_table_lookup (replacements, modified);
		g_signal_connect (G_OBJECT (core->modif_target), "nullified",
				  G_CALLBACK (nullified_target_cb), core);
	}

	core->priv->replacements_query_select = replacements;

	mg_work_core_initialize (core);

	return obj;
}

static void
nullified_query_cb (MgQuery *query, MgWorkCore *core)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (nullified_query_cb), core);

	if (query == core->query_select) 
		core->query_select = NULL;

	if (core->priv && (query == core->priv->query_select_hybrid)) 
		core->priv->query_select_hybrid = NULL;

	if (query == core->query_update) 
		core->query_update = NULL;
	
	if (query == core->query_delete) 
		core->query_delete = NULL;

	if (query == core->query_insert) 
		core->query_insert = NULL;
	
	g_object_unref (G_OBJECT (query));
}

static void
nullified_target_cb (MgTarget *target, MgWorkCore *core)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (target),
					      G_CALLBACK (nullified_target_cb), core);
	core->modif_target = NULL;
}

static void
nullified_context_cb (MgContext *context, MgWorkCore *core)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (context),
					      G_CALLBACK (nullified_context_cb), core);
       
	if (context == core->args_context) 
		core->args_context = NULL;

	if (context == core->work_context) 
		core->work_context = NULL;

	g_object_unref (G_OBJECT (context));
}


static void
mg_work_core_dispose (GObject *object)
{
        MgWorkCore *core;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_MG_WORK_CORE (object));
        core = MG_WORK_CORE (object);

	/* data */
	if (core->data_rs) {
		g_object_unref (G_OBJECT (core->data_rs));
		core->data_rs = NULL;
	}

	/* hash tables */
	if (core->priv) {
		if (core->priv->query_select_hybrid)
			nullified_query_cb (core->priv->query_select_hybrid, core);

		if (core->priv->replacements_query_select) {
			g_hash_table_destroy (core->priv->replacements_query_select);
			core->priv->replacements_query_select = NULL;
		}
		
		if (core->priv->replacements_query_select_hybrid) {
			g_hash_table_destroy (core->priv->replacements_query_select_hybrid);
			core->priv->replacements_query_select_hybrid = NULL;
		}

		if (core->priv->target_to_ref_pk_field) {
			g_hash_table_destroy (core->priv->target_to_ref_pk_field);
			core->priv->target_to_ref_pk_field = NULL;
		}

		if (core->priv->for_cond_value_qfields) {
			g_slist_free (core->priv->for_cond_value_qfields);
			core->priv->for_cond_value_qfields = NULL;
		}

		if (core->priv->ref_integ_fields) {
			g_slist_free (core->priv->ref_integ_fields);
			core->priv->ref_integ_fields = NULL;
		}

		if (core->priv->nodes_hash) {
			g_hash_table_destroy (core->priv->nodes_hash);
			core->priv->nodes_hash = NULL;
		}

		g_free (core->priv);
		core->priv = NULL;
	}
	
	/* nodes */
	if (core->nodes) {
		GSList *list = core->nodes;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (core->nodes);
		core->nodes = NULL;
	}

	if (core->params_in_data_rs) {
		g_slist_free (core->params_in_data_rs);
		core->params_in_data_rs = NULL;
	}

	if (core->params_not_in_data_rs) {
		g_slist_free (core->params_not_in_data_rs);
		core->params_not_in_data_rs = NULL;
	}

	if (core->params_modif_queries_value_providers) {
		g_slist_free (core->params_modif_queries_value_providers);
		core->params_modif_queries_value_providers = NULL;
	}

	if (core->no_show_params) {
		g_slist_free (core->no_show_params);
		core->no_show_params = NULL;
	}
	
	/* contexts */
	if (core->args_context)
		nullified_context_cb (core->args_context, core);
	
	if (core->work_context) 
		nullified_context_cb (core->work_context, core);
	
	/* queries */
	if (core->query_select)
		nullified_query_cb (core->query_select, core);
	if (core->query_update)
		nullified_query_cb (core->query_update, core);
	if (core->query_delete)
		nullified_query_cb (core->query_delete, core);
	if (core->query_insert)
		nullified_query_cb (core->query_insert, core);
	
	/* mofifiable target */
	if (core->modif_target)
		nullified_target_cb (core->modif_target, core);

	/* work_context_qf_position */
	if (core->work_context_qf_position) {
		g_hash_table_destroy (core->work_context_qf_position);
		core->work_context_qf_position = NULL;
	}
	
	/* for the parent class */
        parent_class->dispose (object);
}

/*
 * Real initialization
 */
static void 
mg_work_core_initialize (MgWorkCore *core)
{
	MgQuery *top_query = NULL;
	GSList *list;
	GSList *fields;

	/*
	 * Making modifications to the SELECT query to:
	 * -> expand any '*' into all the corresponding fields and order by those fields
	 * -> add PK fields for the entity which will be modified
	 */
	/* expand any '*' into all the corresponding fields */
	fields = mg_query_expand_all_field (core->query_select, NULL);
	list = fields;
	while (list) {
		mg_query_set_order_by_field (core->query_select, MG_QFIELD (list->data), G_MAXINT, TRUE);
		list = g_slist_next (list);
	}
	g_slist_free (fields);

	/* add PK fields for the entity which will be modified (and ordering on them) */
	if (core->modif_target) {
		GSList *pk_fields = mg_query_get_target_pkfields (core->query_select, core->modif_target);
		if (! pk_fields) {
			MgDbConstraint *pkcons = NULL;
			MgEntity *entity;

			entity = mg_target_get_represented_entity (core->modif_target);
			pkcons = mg_db_table_get_pk_constraint (MG_DB_TABLE (entity));
			if (pkcons) {
				GSList *fields, *list;
				MgQfield *pkf;

				fields = mg_db_constraint_pkey_get_fields (pkcons);
				list = fields;
				while (list) {
					if (! (pkf =  mg_query_get_any_field_by_ref_field (core->query_select, 
											   NULL, MG_FIELD (list->data)))) {
						pkf = MG_QFIELD (mg_qf_field_new_with_objects (core->query_select, 
											       core->modif_target,
											       MG_FIELD (list->data)));
						mg_entity_add_field (MG_ENTITY (core->query_select), MG_FIELD (pkf));
						mg_qfield_set_visible (MG_QFIELD (pkf), TRUE);
						mg_qfield_set_internal (MG_QFIELD (pkf), TRUE);
						mg_base_set_name (MG_BASE (pkf), "PK field");
						g_object_unref (G_OBJECT (pkf));
					}
					else {
						if (! mg_qfield_is_visible (MG_QFIELD (pkf))) {
							mg_qfield_set_visible (MG_QFIELD (pkf), TRUE);
							mg_qfield_set_internal (MG_QFIELD (pkf), TRUE);
						}
					}

					if (mg_query_get_order_by_field (core->query_select, pkf, NULL) < 0) 
						mg_query_set_order_by_field (core->query_select, pkf, G_MAXINT, TRUE);
					list = g_slist_next (list);
				}
				g_slist_free (fields);
			}
		}
		else {
			GSList *flist = pk_fields;
			while (flist) {
				g_assert (mg_qfield_is_visible (MG_QFIELD (flist->data)));

				if (mg_query_get_order_by_field (core->query_select, MG_QFIELD (flist->data), NULL) < 0) 
					mg_query_set_order_by_field (core->query_select, MG_QFIELD (flist->data), 
								     G_MAXINT, TRUE);

				flist = g_slist_next (flist);
			}
			g_slist_free (pk_fields);
		}
	}
 
	/*
	 * args context : parameters required for the 'select query' to be executed
	 */
	if (!core->args_context) {
		core->args_context = mg_entity_get_exec_context (MG_ENTITY (core->query_select));
		g_signal_connect (G_OBJECT (core->args_context), "nullified",
				  G_CALLBACK (nullified_context_cb), core);
	}

	/*
	 * work context : parameters required to execute dthe derived INSERT, UPDATE and DELETE queries
	 */
	if (core->modif_target) {
		core->work_context = make_work_context_target (core);
		top_query = core->priv->query_select_hybrid;
	}
	else 
		core->work_context = make_work_context_no_target (core);
	g_signal_connect (G_OBJECT (core->work_context), "nullified",
			  G_CALLBACK (nullified_context_cb), core);
	
	/*
	 * making modification queries
	 */
	if (core->modif_target) {
		make_query_update (core);
		make_query_delete (core);
		make_query_insert (core);

#ifdef debug_NO
		if (1) {
			/*
			 * Graph output
			 */

			MgGraphviz *graph;
			graph = MG_GRAPHVIZ (mg_graphviz_new (mg_base_get_conf (MG_BASE (core->query_select))));
			if (top_query)
				mg_graphviz_add_to_graph (graph, G_OBJECT (top_query));
			mg_graphviz_add_to_graph (graph, G_OBJECT (core->query_update));
			mg_graphviz_add_to_graph (graph, G_OBJECT (core->query_delete));
			mg_graphviz_add_to_graph (graph, G_OBJECT (core->query_insert));
			mg_graphviz_add_to_graph (graph, G_OBJECT (core->work_context));
			mg_graphviz_save_file (graph, "_work_context.dot", NULL);
			g_object_unref (G_OBJECT (graph));
		}
#endif
	}

#ifdef debug_NO
	g_print ("---------------------------------- >> ARGS CONTEXT ----------------------------------\n");
	mg_base_dump (MG_BASE (core->args_context), 0);
	g_print ("---------------------------------- << ARGS CONTEXT ----------------------------------\n");
	g_print ("---------------------------------- >> WORK CONTEXT ----------------------------------\n");
	mg_base_dump (MG_BASE (core->work_context), 0);
	g_print ("---------------------------------- << WORK CONTEXT ----------------------------------\n");
	g_print ("---------------------------------- >> QUERY SELECT ----------------------------------\n");
	mg_base_dump (MG_BASE (core->query_select), 0);
	g_print ("---------------------------------- << QUERY SELECT ----------------------------------\n");
	g_print ("---------------------------------- >> QUERY SELECT HYBRID ----------------------------------\n");
	mg_base_dump (MG_BASE (core->priv->query_select_hybrid), 0);
	g_print ("---------------------------------- << QUERY SELECT HYBRID ----------------------------------\n");
#endif
	
	/* 
	 * Preparing the list of parameters for which no entry will have to be displayed in the
	 * widgets using 'core'
	 * computing the core->no_show_params list 
	 */
	if (core->priv->query_select_hybrid) {
		GSList *tmplist;

		/* alias parameters */
		list = core->work_context->parameters;
		while (list) {
			MgParameter *tmp;

			g_object_get (G_OBJECT (list->data), "full_bind", &tmp, NULL);
			if (tmp) {
#ifdef debug_NO
				g_print ("NO SHOW param %p (%s) because is alias\n", list->data, mg_base_get_name (list->data));
#endif
				core->no_show_params = g_slist_append (core->no_show_params, list->data);
			}
			list = g_slist_next (list);
		}

		/* condition parameters */
		g_object_get (G_OBJECT (core->priv->query_select_hybrid), 
			      "really_all_fields", &list, NULL);
		while (list) {
			if (g_slist_find (core->priv->for_cond_value_qfields, list->data)) {
				MgParameter *param;

				param = mg_context_find_parameter_for_field (core->work_context, 
									     MG_QFIELD (list->data));
				g_assert (param);
				core->no_show_params = g_slist_append (core->no_show_params, param);
#ifdef debug_NO
				g_print ("NO SHOW param %p (%s) because for cond\n", param, mg_base_get_name (param));
#endif
			}
		       
			list = g_slist_next (list);
		}

		/* params for internal fields */
		tmplist = mg_entity_get_visible_fields (MG_ENTITY (core->priv->query_select_hybrid));
		list = tmplist;
		while (list) {
			if (mg_qfield_is_internal (MG_QFIELD (list->data)) &&
			    !g_slist_find (core->priv->ref_integ_fields, list->data)) {
				MgParameter *param;
				MgQfield *value_prov;

				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
				if (value_prov) {
					param = mg_context_find_parameter_for_field (core->work_context, value_prov);
					if (param) {
						core->no_show_params = g_slist_append (core->no_show_params, param);
#ifdef debug_NO
						g_print ("NO SHOW param %p (%s) because internal\n", param, mg_base_get_name (param));
#endif
					}
				}
			}
			list = g_slist_next (list);
		}
		g_slist_free (tmplist);
	}

	/*
	 * computing work_context_qf_position 
	 */
	core->work_context_qf_position = g_hash_table_new (NULL, NULL);
	list = core->work_context->nodes;
	while (list) {
		MgContextNode *context_node = MG_CONTEXT_NODE (list->data);

		if (context_node->query) {
			GSList *fields;
			GSList *list2;
			MgField *field;
			gint pos;
			
			fields = mg_entity_get_visible_fields (MG_ENTITY (context_node->query));
			list2 = fields;
			while (list2) {
				if (IS_MG_QF_FIELD (list2->data)) {
					MgQfield *query_select_field;
					
					field = mg_qf_field_get_ref_field (MG_QF_FIELD (list2->data));
					query_select_field = 
						mg_query_get_visible_field_by_ref_field (core->query_select,
											 NULL, field);
					if (query_select_field) {
						pos = mg_entity_get_field_index (MG_ENTITY (core->query_select), 
										 MG_FIELD (query_select_field));
						g_assert (pos >= 0); /* by construction of the queries */
						g_hash_table_insert (core->work_context_qf_position, list2->data, 
								     GINT_TO_POINTER (pos));
					}
				}
				list2 = g_slist_next (list2);
			}
			g_slist_free (fields);
		}
		list = g_slist_next (list);
	}

	/*
	 * computing priv->nodes_hash
	 */
	core->priv->nodes_hash = g_hash_table_new (NULL, NULL);
	list = core->nodes;
	while (list) {
		g_hash_table_insert (core->priv->nodes_hash, MG_WORK_CORE_NODE (list->data)->param, list->data);
		list = g_slist_next (list);
	}

	/* trying to store some 'proposed' values or parameters's values to be used by the parameters
	 * in work_context. For example is the select query contains "WHERE conf_id=8" then the
	 * work_context's parameter for conf_id should have a 'proposed' value of the work_context's
	 * parameter corresponding to the '8' value */
	list = mg_query_get_main_conditions (core->query_select);
	if (list && core->modif_target) {
		GSList *ptr = list;
		MgCondition *cond;
		
		while (ptr) {
			cond = MG_CONDITION (ptr->data);
			if (mg_condition_get_cond_type (cond) == MG_CONDITION_LEAF_EQUAL) {
				MgQfield *field1, *field2;

				field1 = mg_condition_leaf_get_operator (cond, MG_CONDITION_OP_LEFT);
				field2 = mg_condition_leaf_get_operator (cond, MG_CONDITION_OP_RIGHT);

				if (IS_MG_QF_FIELD (field2) && IS_MG_QF_VALUE (field1)) {
					MgQfield *tmp = field1;
					field1 = field2;
					field2 = tmp;
				}

				if (IS_MG_QF_FIELD (field1) && IS_MG_QF_VALUE (field2)) {
					MgQfield *rfield1, *rfield2, *value_prov;	

					rfield1 = g_hash_table_lookup (core->priv->replacements_query_select_hybrid, field1);
					rfield2 = g_hash_table_lookup (core->priv->replacements_query_select_hybrid, field2);
					g_object_get (G_OBJECT (rfield1), "value_provider", &value_prov, NULL);

					if (value_prov && rfield2) {
						MgParameter *param1, *param2;

						param1 = mg_context_find_parameter_for_field (core->work_context, value_prov);
						param2 = mg_context_find_parameter_for_field (core->work_context, rfield2);

						if (param1 && param2) 
							mg_context_set_param_default_alias (core->work_context, param1, param2);
#ifdef debug_NO
						g_print ("Context DEFAULT ALIAS Param1: %p (%s), Param2: %p (%s)\n", 
							 param1, mg_base_get_name (param1), 
							 param2, mg_base_get_name (param2));
#endif
					}
				}
			}
			ptr = g_slist_next (ptr);
		}
		g_slist_free (list);
	}
}

static void
make_query_update (MgWorkCore *core)
{
	GSList *list;
	MgTarget *target;
	MgQuery *query;
	MgConf *conf = mg_base_get_conf (MG_BASE (core->query_select));
	MgCondition *cond = NULL;
	GHashTable *replacements;
	MgQuery *top_query = core->priv->query_select_hybrid;

	/* Query object */
	query = MG_QUERY (mg_query_new (conf));
	mg_query_set_query_type (query, MG_QUERY_TYPE_UPDATE);
	target = MG_TARGET (mg_target_new_with_entity (query, mg_target_get_represented_entity (core->modif_target)));
	mg_query_add_target (query, target, NULL);
	replacements = g_hash_table_new (NULL, NULL);

	/* Fields */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_MG_QF_FIELD (list->data)) {
			if (mg_qf_field_get_target (MG_QF_FIELD (list->data)) == 
			    g_object_get_data (G_OBJECT (top_query), "for_target")) {
				MgField *field, *value_prov, *new_value_prov;
				MgParameter *param;
				
				field = MG_FIELD (mg_qf_field_new_with_objects (query, target,
						  mg_qf_field_get_ref_field (MG_QF_FIELD (list->data))));
				mg_base_set_name (MG_BASE (field), mg_base_get_name (MG_BASE (list->data)));
				mg_entity_add_field (MG_ENTITY (query), field);
				if (!mg_qfield_is_visible (MG_QFIELD (list->data)))
					mg_qfield_set_visible (MG_QFIELD (field), FALSE);

				g_hash_table_insert (replacements, list->data, field);
				g_object_unref (G_OBJECT (field));
				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);

				if (value_prov) {
					g_assert (IS_MG_QF_VALUE (value_prov));

					param = mg_context_find_parameter_for_field (core->work_context, 
										     MG_QFIELD (value_prov));
					new_value_prov = MG_FIELD (mg_qf_value_new (query, 
							 mg_qf_value_get_server_data_type (MG_QF_VALUE (value_prov))));
					mg_qf_value_set_is_parameter (MG_QF_VALUE (new_value_prov), TRUE);
					mg_base_set_name (MG_BASE (new_value_prov), 
							  mg_base_get_name (MG_BASE (value_prov)));
					mg_entity_add_field (MG_ENTITY (query), new_value_prov);
					mg_qfield_set_visible (MG_QFIELD (new_value_prov), FALSE);
					g_object_unref (G_OBJECT (new_value_prov));
					g_object_set (G_OBJECT (field), "value_provider", new_value_prov, NULL);
				
					mg_parameter_add_dest_field (param, MG_QFIELD (new_value_prov));
				}
			}
		}

		list = g_slist_next (list);
	}

	/* Condition */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_MG_QF_VALUE (list->data)) {
			if (g_slist_find (core->priv->for_cond_value_qfields, list->data)) {
				MgQfield *field, *value;
				MgCondition *subcond;
				MgParameter *param;

				if (!cond) {
					cond = MG_CONDITION (mg_condition_new (query, MG_CONDITION_NODE_AND));
					mg_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}

				field = g_hash_table_lookup (replacements, 
							     g_object_get_data (G_OBJECT (list->data), "for_cond"));
				g_assert (field);
				g_assert (MG_QF_FIELD (field));
				
				param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (list->data));
				g_assert (param);
				value = MG_QFIELD (mg_qf_value_new (query, 
						   mg_field_get_data_type (mg_qf_field_get_ref_field (MG_QF_FIELD (field)))));
				mg_qf_value_set_is_parameter (MG_QF_VALUE (value), TRUE);
				mg_base_set_name (MG_BASE (value), mg_base_get_name (MG_BASE (list->data)));
				mg_entity_add_field (MG_ENTITY (query), MG_FIELD (value));
				mg_qfield_set_visible (MG_QFIELD (value), FALSE);
				g_object_unref (G_OBJECT (value));
				mg_parameter_add_dest_field (param, MG_QFIELD (value));

				subcond = MG_CONDITION (mg_condition_new (query, MG_CONDITION_LEAF_EQUAL));
				mg_condition_leaf_set_operator (subcond, MG_CONDITION_OP_LEFT, field);
				mg_condition_leaf_set_operator (subcond, MG_CONDITION_OP_RIGHT, value);
				mg_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

	
	core->query_update = query;
	g_signal_connect (G_OBJECT (core->query_update), "nullified",
			  G_CALLBACK (nullified_query_cb), core);

#ifdef debug_NO
	{
		gchar *sql;
		sql = mg_renderer_render_as_sql (MG_RENDERER (query), NULL, 0, NULL);
		g_print ("WorkCore UPDATE: %s\n", sql);
		g_free (sql);
	}
#endif

	/* Free memory */
	g_hash_table_destroy (replacements);
}

static void
make_query_delete (MgWorkCore *core)
{
	GSList *list;
	MgTarget *target;
	MgQuery *query;
	MgConf *conf = mg_base_get_conf (MG_BASE (core->query_select));
	MgCondition *cond = NULL;
	MgQuery *top_query = core->priv->query_select_hybrid;

	/* Query object */
	query = MG_QUERY (mg_query_new (conf));
	mg_query_set_query_type (query, MG_QUERY_TYPE_DELETE);
	target = MG_TARGET (mg_target_new_with_entity (query, mg_target_get_represented_entity (core->modif_target)));
	mg_query_add_target (query, target, NULL);
	
	/* Condition */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_MG_QF_VALUE (list->data)) {
			if (g_slist_find (core->priv->for_cond_value_qfields, list->data)) {
				MgQfield *fieldv, *fieldf;
				MgCondition *subcond;
				MgParameter *param;

				if (!cond) {
					cond = MG_CONDITION (mg_condition_new (query, MG_CONDITION_NODE_AND));
					mg_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}

				fieldf = g_object_get_data (G_OBJECT (list->data), "for_cond");
				fieldf = MG_QFIELD (mg_qf_field_new_with_objects (query, target,
										 mg_qf_field_get_ref_field (MG_QF_FIELD (fieldf))));
				mg_entity_add_field (MG_ENTITY (query), MG_FIELD (fieldf));
				g_object_unref (G_OBJECT (fieldf));
				mg_qfield_set_visible (fieldf, FALSE);

				param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (list->data));
				g_assert (param);
				fieldv = MG_QFIELD (mg_qf_value_new (query, 
							   mg_field_get_data_type (mg_qf_field_get_ref_field (MG_QF_FIELD (fieldf)))));
				mg_qf_value_set_is_parameter (MG_QF_VALUE (fieldv), TRUE);
				mg_entity_add_field (MG_ENTITY (query), MG_FIELD (fieldv));
				mg_qfield_set_visible (MG_QFIELD (fieldv), FALSE);
				g_object_unref (G_OBJECT (fieldv));
				mg_parameter_add_dest_field (param, MG_QFIELD (fieldv));

				subcond = MG_CONDITION (mg_condition_new (query, MG_CONDITION_LEAF_EQUAL));
				mg_condition_leaf_set_operator (subcond, MG_CONDITION_OP_LEFT, fieldf);
				mg_condition_leaf_set_operator (subcond, MG_CONDITION_OP_RIGHT, fieldv);
				mg_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

#ifdef debug_NO
	{
		gchar *sql;
		sql = mg_renderer_render_as_sql (MG_RENDERER (query), NULL, 0, NULL);
		g_print ("WorkCore DELETE: %s\n", sql);
		g_free (sql);
	}
#endif
	
	core->query_delete = query;
	g_signal_connect (G_OBJECT (core->query_delete), "nullified",
			  G_CALLBACK (nullified_query_cb), core);
}

static void
make_query_insert (MgWorkCore *core)
{
	GSList *list;
	MgTarget *target;
	MgQuery *query;
	MgConf *conf = mg_base_get_conf (MG_BASE (core->query_select));
	MgQuery *top_query = core->priv->query_select_hybrid;

	/* Query object */
	query = MG_QUERY (mg_query_new (conf));
	mg_query_set_query_type (query, MG_QUERY_TYPE_INSERT);
	target = MG_TARGET (mg_target_new_with_entity (query, mg_target_get_represented_entity (core->modif_target)));
	mg_query_add_target (query, target, NULL);

	/* Fields */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_MG_QF_FIELD (list->data)) {
			if (mg_qf_field_get_target (MG_QF_FIELD (list->data)) == 
			    g_object_get_data (G_OBJECT (top_query), "for_target")) {
				MgField *field, *value_prov;
				MgParameter *param;
				
				field = MG_FIELD (mg_qf_field_new_with_objects (query, target,
						  mg_qf_field_get_ref_field (MG_QF_FIELD (list->data))));
				mg_entity_add_field (MG_ENTITY (query), field);
				if (!mg_qfield_is_visible (MG_QFIELD (list->data)))
					mg_qfield_set_visible (MG_QFIELD (field), FALSE);

				g_object_unref (G_OBJECT (field));
				g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
				if (value_prov) {
					g_assert (IS_MG_QF_VALUE (value_prov));

					param = mg_context_find_parameter_for_field (core->work_context, 
										     MG_QFIELD (value_prov));
					g_assert (param);
					value_prov = MG_FIELD (mg_qf_value_new (query, 
							       mg_qf_value_get_server_data_type (MG_QF_VALUE (value_prov))));
					mg_qf_value_set_is_parameter (MG_QF_VALUE (value_prov), TRUE);
					mg_entity_add_field (MG_ENTITY (query), value_prov);
					mg_qfield_set_visible (MG_QFIELD (value_prov), FALSE);
					g_object_unref (G_OBJECT (value_prov));
					g_object_set (G_OBJECT (field), "value_provider", value_prov, NULL);
				
					mg_parameter_add_dest_field (param, MG_QFIELD (value_prov));
				}
			}
		}

		list = g_slist_next (list);
	}

#ifdef debug_NO
	{
		gchar *sql;
		sql = mg_renderer_render_as_sql (MG_RENDERER (query), NULL, 0, NULL);
		g_print ("WorkCore INSERT: %s\n", sql);
		g_free (sql);
	}
#endif

	core->query_insert = query;
	g_signal_connect (G_OBJECT (core->query_insert), "nullified",
			  G_CALLBACK (nullified_query_cb), core);
}




/*
 * Computing the MgContext and the WorkCoreNodes
 * when core->modif_target is NULL
 */
MgContext *
make_work_context_no_target (MgWorkCore *core)
{
	GSList *fields, *list;
	GSList *params = NULL;
	MgContext *context;

        fields = mg_entity_get_visible_fields (MG_ENTITY (core->query_select));
        list = fields;
        while (list) {
                MgQfield *field = MG_QFIELD (list->data);
		
		if (! IS_MG_QF_ALL (field)) {
			MgWorkCoreNode *node = g_new0 (MgWorkCoreNode, 1);

			node->param = MG_PARAMETER (mg_parameter_new_with_dest_field (field, 
										      mg_field_get_data_type (MG_FIELD (field))));
			node->position = mg_entity_get_field_index (MG_ENTITY (core->query_select), MG_FIELD (field));
			core->nodes = g_slist_append (core->nodes, node);
			params = g_slist_append (params, node->param);
			core->params_in_data_rs = g_slist_append (core->params_in_data_rs, node->param);
			
			/* set the parameter properties */
			mg_base_set_name (MG_BASE (node->param), mg_base_get_name (MG_BASE (field)));
			mg_base_set_description (MG_BASE (node->param), mg_base_get_description (MG_BASE (field)));
			if (G_OBJECT_TYPE (field) == MG_QF_FIELD_TYPE) {
				MgField *realfield = mg_qf_field_get_ref_field (MG_QF_FIELD (field));
				if (G_OBJECT_TYPE (realfield) == MG_DB_FIELD_TYPE)
					mg_parameter_set_not_null (node->param,
								   !mg_db_field_is_null_allowed (MG_DB_FIELD (realfield)));
			}
		}

                list = g_slist_next (list);
        }

	context = MG_CONTEXT (mg_context_new (mg_base_get_conf (MG_BASE (core->query_select)), params));

	/* get rid of the params list since we don't use them anymore */
	list = params;
        while (list) { 
                g_object_unref (G_OBJECT (list->data));
                list = g_slist_next (list);
        }
        g_slist_free (params);

	return context;
}


/*
 * Computing the MgContext and the MgWorkCoreNodes
 * when core->modif_target is NOT NULL
 */
MgContext *
make_work_context_target (MgWorkCore *core)
{
	GHashTable *targets_h; /* (key, value) = (target, SELECT query for that target) */
	TargetDep *targets_d;
	GSList *fields, *list;
	GSList *sel_queries = NULL;
	MgQuery *top_query;
	MgContext *context;
	GHashTable *replacements; /* objects replacements in the copy of 'core->query_select' */
	GSList *cond_fields;
	GSList *main_conds;

	g_return_val_if_fail (core->query_select, NULL);


	/*
	 * Create a tree of all the target dependencies
	 */
	targets_d = make_target_deps (core);
	targets_h = g_hash_table_new (NULL, NULL);

	/*
	 * make a copy of 'core->query_select' into 'top_query'
	 */
	replacements = g_hash_table_new (NULL, NULL);
	top_query = MG_QUERY (mg_query_new_copy (core->query_select, replacements));
	g_object_set_data (G_OBJECT (top_query), "for_target", g_hash_table_lookup (replacements, core->modif_target));
	sel_queries = g_slist_append (sel_queries, top_query);
	g_hash_table_insert (targets_h, core->modif_target, top_query);
	list = core->priv->ref_integ_fields;
	fields = NULL;
	while (list) {
		gpointer field = g_hash_table_lookup (replacements, list->data);
		if (field)
			fields = g_slist_prepend (fields, field);
		list = g_slist_next (list);
	}
	core->priv->ref_integ_fields = g_slist_concat (core->priv->ref_integ_fields, fields);

	/*
	 * POSSIBLE IMPROVEMENT:
	 * Parse top_query's joins which have one target corresponding to the modif target,
	 * and if one of the fields used by the join condition is not visible, then move it
	 * close to the one which is visible.
	 *
	 * The problem is that the parameter for "fk_field" of such a join is set close to the "fk_field" and usually
	 * we want it at the same place as what gets displayed in the SELECT (which is the "ref_pk_field").
	 */

	/*
	 * For each MgQfValue in 'top_query', we need to remove the "value_provider" property if it exists because
	 * these parameters are only aliases for parameters we have in the 'core->args_context' and we don't 
	 * want to have an entry (MgEntryCombo) created for them.
	 */
	g_object_get (G_OBJECT (top_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_MG_QF_VALUE (list->data))
			g_object_set (G_OBJECT (list->data), "value_provider", NULL, NULL);
		list = g_slist_next (list);
	}

	/*
	 * make a list ('sel_queries') of SELECT queries: one for each target in the 'core->query_select'
	 * query. Only the MgQfield of type MG_QF_FIELD are inserted into the new queries.
	 *
	 * For the 'core->modif_target', we don't start from scratch, but from the 'top_query' query created just above
	 */
	fields = mg_entity_get_visible_fields (MG_ENTITY (core->query_select));
	list = fields;
	while (list) {
		if (IS_MG_QF_FIELD (list->data)) {
			MgTarget *target = mg_qf_field_get_target (MG_QF_FIELD (list->data));
			MgQuery *query;
			MgField *realfield;
			MgTarget *newtarget;

			if (target != core->modif_target) {
				query = g_hash_table_lookup (targets_h, target);
				if (query) {
					/* add a field */
					realfield = mg_qf_field_get_ref_field (MG_QF_FIELD (list->data));
					newtarget = g_object_get_data (G_OBJECT (query), "for_target");
					realfield = MG_FIELD (mg_qf_field_new_with_objects (query, newtarget, realfield));
					mg_entity_add_field (MG_ENTITY (query), realfield);
					g_object_unref (G_OBJECT (realfield));
					
					/* set it to appear in the ORDER BY list */
					mg_query_set_order_by_field (query, MG_QFIELD (realfield), G_MAXINT, TRUE);
				}
				else {
					/* create a new query and insert in into targets_h, and add a field */
					query = MG_QUERY (mg_query_new (mg_base_get_conf (MG_BASE (core->query_select))));
					mg_query_set_query_type (query, MG_QUERY_TYPE_SELECT);
					
					mg_base_set_name (MG_BASE (query), mg_base_get_name (MG_BASE (list->data)));
					mg_base_set_description (MG_BASE (query), 
								 mg_base_get_description (MG_BASE (list->data)));
					
					realfield = mg_qf_field_get_ref_field (MG_QF_FIELD (list->data));
					newtarget = MG_TARGET (mg_target_new_with_entity (query, 
											  mg_field_get_entity (realfield)));
					g_assert (mg_query_add_target (query, newtarget, NULL));
					g_object_set_data (G_OBJECT (query), "for_target", newtarget);
					
					realfield = MG_FIELD (mg_qf_field_new_with_objects (query, newtarget, realfield));
					mg_entity_add_field (MG_ENTITY (query), realfield);
					g_object_unref (G_OBJECT (newtarget));
					g_object_unref (G_OBJECT (realfield));
					
					g_hash_table_insert (targets_h, target, query);
					sel_queries = g_slist_append (sel_queries, query);
				}
				mg_base_set_name (MG_BASE (realfield), mg_base_get_name (MG_BASE (list->data)));
				mg_base_set_description (MG_BASE (realfield), 
							 mg_base_get_description (MG_BASE (list->data)));
			}
			else {
				/* Add a MgQfValue for that field */
				MgQfValue *vfield;
				MgField *reffield;
				const gchar *plugin;
				MgDataHandler *default_dh = NULL;

				query = top_query;
				realfield = g_hash_table_lookup (replacements, list->data);

				vfield = MG_QF_VALUE (mg_qf_value_new (query, mg_field_get_data_type (realfield)));
				mg_entity_add_field_before (MG_ENTITY (query), MG_FIELD (vfield), MG_FIELD (realfield));
				mg_base_set_name (MG_BASE (vfield), mg_base_get_name (MG_BASE (realfield)));
				mg_base_set_description (MG_BASE (vfield), mg_base_get_description (MG_BASE (realfield)));
				g_object_unref (G_OBJECT (vfield));
				mg_qf_value_set_is_parameter (vfield, TRUE);
				mg_qfield_set_visible (MG_QFIELD (vfield), FALSE);
				g_object_set (G_OBJECT (realfield), "value_provider", vfield, NULL);
				mg_qfield_set_internal (MG_QFIELD (vfield), TRUE);
				
				/* MgQfValue properties */
				if (IS_MG_QF_FIELD (realfield) && 
				    IS_MG_DB_FIELD (mg_qf_field_get_ref_field (MG_QF_FIELD (realfield)))) {
					MgDbField *dbfield = MG_DB_FIELD (mg_qf_field_get_ref_field (MG_QF_FIELD (realfield)));
					MgServer *srv;
					
					mg_qf_value_set_not_null (vfield, !mg_db_field_is_null_allowed (dbfield));
					if (mg_db_field_get_default_value (dbfield))
						mg_qf_value_set_default_value (vfield, 
									       mg_db_field_get_default_value (dbfield));

					srv = mg_conf_get_server (mg_base_get_conf (MG_BASE (core->query_select)));
					default_dh = mg_server_get_object_handler (srv, G_OBJECT (dbfield));
				}
				else {
					TO_IMPLEMENT;
				}
				
				/* Entry plugin if available */
				g_object_get (G_OBJECT (list->data), "handler_plugin", &plugin, NULL);
				if (plugin) 
					g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
				else {
					if (default_dh) {
						plugin = mg_base_get_name (MG_BASE (default_dh));
						g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
					}
				}
								
				/* keep a reference to the field in 'core->query_select' which we are now treating */
				g_object_set_data (G_OBJECT (vfield), "position", list->data);
			}
		}
		list = g_slist_next (list);
	}

	g_slist_free (fields);

	/* For each target dependency, from the corresponding join, add the missing fields participating in the
	 * join to the SELECT queries associated with each target;
	 *
	 * And create some MgParameter objects on the 'FK' side and make them have a value provider on the 'PK' side.
	 */
	make_target_select_queries_improved (core, targets_d, targets_h, replacements, NULL);
	top_query = g_hash_table_lookup (targets_h, core->modif_target);

	/* 
	 * Analyse all the main conditions (= the conditions from the query_select whih are always aveluated)
	 * and copy some of them to the other SELECT queries created in the previous step. The idea is to
	 * copy some of the restrictions that are in the original query into the other SELECT queries from which
	 * some values must be chosen.
	 *
	 * Example: if we have the main query as 'SELECT a.name, b.name FROM a INNER JOIN b WHERE b.id=12'
	 * then we want the a.fkfield to be selected from 'SELECT b.pkfield, b.name FROM b WHERE b.id=12' and
	 * not from 'SELECT b.pkfield, b.name FROM'.
	 *
	 * FIXME: this implementation is not correct, better do it _after_ the work_context has been created
	 * so we don't add some useless parameters in the select queries.
	 *
	 * At the moment only conditions whith non parameter values are accepted.
	 */
	main_conds = mg_query_get_main_conditions (core->query_select);
	list = main_conds;
	while (list) {
		gboolean fields_ok = TRUE;
		MgTarget *sel_target = NULL;

		/* see if this condition is OK to be processed further */
		cond_fields = mg_condition_get_ref_objects_all (MG_CONDITION (list->data));
		fields = cond_fields;
		while (fields && fields_ok) {
			if (!IS_MG_QF_FIELD (fields->data) &&
			    !IS_MG_QF_VALUE (fields->data))
				fields_ok = FALSE;

			if (IS_MG_QF_FIELD (fields->data)) {
				if (!sel_target) 
					sel_target = mg_qf_field_get_target (MG_QF_FIELD (fields->data));
				else {
					if (sel_target != mg_qf_field_get_target (MG_QF_FIELD (fields->data)))
						fields_ok = FALSE;
				}
			}

			if (IS_MG_QF_VALUE (fields->data) && mg_qf_value_is_parameter (MG_QF_VALUE (fields->data)))
				fields_ok = FALSE;

			fields = g_slist_next (fields);
		}
		
		/* process that condition if fields_ok && sel_target */
		if (fields_ok && sel_target && (sel_target != core->modif_target)) {
			GHashTable *tmprepl;
			MgQuery *sel_query = g_hash_table_lookup (targets_h, sel_target);
			MgCondition *cond = MG_CONDITION (list->data), *newcond, *andcond;
			MgQfield *cfield;
			
			/* initialize a temporary hash table to be used by for replacements in condition */
			tmprepl = g_hash_table_new (NULL, NULL);
			g_hash_table_insert (tmprepl, core->query_select, sel_query);

			/* copy the fields taking part of the condition if they don't already exist */
			fields = cond_fields;
			while (fields) {
				gboolean field_done = FALSE;

				if (IS_MG_QF_FIELD (fields->data)) {
					MgField *ref_field = mg_qf_field_get_ref_field (MG_QF_FIELD (fields->data));
					MgField *realfield = 
						(MgField*) mg_query_get_any_field_by_ref_field (sel_query, 
												NULL, ref_field);

					if (!realfield) {
						gchar *str;
						MgTarget *newtarget = g_object_get_data (G_OBJECT (sel_query), 
											 "for_target");
						cfield = MG_QFIELD (mg_qf_field_new_with_objects (sel_query, newtarget, 
												  ref_field));
						mg_entity_add_field (MG_ENTITY (sel_query), MG_FIELD (cfield));
						str = g_strdup_printf ("_%s", mg_base_get_name (MG_BASE (fields->data)));
						mg_base_set_name (MG_BASE (cfield), str);
						mg_qfield_set_visible (cfield, FALSE);
						g_free (str);
						g_hash_table_insert (tmprepl, fields->data, cfield);
						g_object_unref (G_OBJECT (cfield));
					}
					else
						g_hash_table_insert (tmprepl, fields->data, realfield);
					field_done = TRUE;
				}
				
				if (!field_done && IS_MG_QF_VALUE (fields->data)) {
					/* make a copy of that field since MgQfValue fields are not copied above */
					cfield = MG_QFIELD (mg_qfield_new_copy (MG_QFIELD (fields->data)));
					g_object_set (G_OBJECT (cfield), "query", sel_query, NULL);
					mg_entity_add_field (MG_ENTITY (sel_query), MG_FIELD (cfield));
					mg_base_set_name (MG_BASE (cfield), "_main cond copy");
					mg_qfield_set_visible (cfield, FALSE);
					g_hash_table_insert (tmprepl, fields->data, cfield);

					mg_qf_value_set_is_parameter (MG_QF_VALUE (cfield), 
								      mg_qf_value_is_parameter (MG_QF_VALUE (fields->data)));
					g_object_unref (G_OBJECT (cfield));
					field_done = TRUE;
				}
				g_assert (field_done);

				fields = g_slist_next (fields);
			}

			/* copy of the condition */
			andcond = mg_query_get_condition (sel_query);
			if (!andcond) {
				andcond = MG_CONDITION (mg_condition_new (sel_query, MG_CONDITION_NODE_AND));
				mg_query_set_condition (sel_query, andcond);
				mg_base_set_name (MG_BASE (andcond), "_: AND");
				g_object_unref (G_OBJECT (andcond));
			}
			else {
				if (mg_condition_is_leaf (andcond)) {
					g_object_ref (G_OBJECT (andcond));
					newcond = MG_CONDITION (mg_condition_new (sel_query, MG_CONDITION_NODE_AND));
					mg_base_set_name (MG_BASE (newcond), "_: AND");
					mg_query_set_condition (sel_query, newcond);
					g_assert (mg_condition_node_add_child (newcond, andcond, NULL));
					g_object_unref (G_OBJECT (andcond));
					g_object_unref (G_OBJECT (newcond));
					andcond = newcond;
				}
			}
			
			newcond = MG_CONDITION (mg_condition_new_copy (cond, tmprepl));
			mg_referer_replace_refs (MG_REFERER (newcond), tmprepl);
			g_hash_table_destroy (tmprepl);
			g_assert (mg_condition_node_add_child (andcond, newcond, NULL));
			g_object_unref (G_OBJECT (newcond));
		}

		g_slist_free (cond_fields);

		list = g_slist_next (list);
	}
	g_slist_free (main_conds);


	/* Preparing the WHERE condition values for modification queries (UPDATE and DELETE) */
	cond_fields = mg_query_get_target_pkfields (core->query_select, core->modif_target);
	if (!cond_fields) {
		/* we don't have a primary key in the list of fields, so we just add all the fields
		 * and hope we don't modify too many tuples in the database.
		 * (there is no JOIN in the UPDATE and DELETE, so we only have fields visible
		 * from the original SELECT query to make criteria of).
		 */
		GSList *tmplist = mg_entity_get_visible_fields (MG_ENTITY (core->query_select));
		list = tmplist;
		while (list) {
			if (IS_MG_QF_FIELD (list->data) && 
			    (mg_qf_field_get_target (MG_QF_FIELD (list->data)) == core->modif_target)) {
				cond_fields = g_slist_append (cond_fields, list->data);
			}
			list = g_slist_next (list);
		}
		g_slist_free (tmplist);
	}
	list = cond_fields;
	while (list) {
		MgQfValue *vfield;
		
		vfield = MG_QF_VALUE (mg_qf_value_new (top_query, mg_field_get_data_type (MG_FIELD (list->data))));
		mg_base_set_name (MG_BASE (vfield), "_Condition Value");
		mg_entity_add_field (MG_ENTITY (top_query), MG_FIELD (vfield));
		g_object_unref (G_OBJECT (vfield));
		mg_qf_value_set_is_parameter (vfield, TRUE);
		mg_qfield_set_visible (MG_QFIELD (vfield), FALSE);
		mg_qfield_set_internal (MG_QFIELD (vfield), TRUE);
		
		/* MgQfValue Properties */
		if (IS_MG_QF_FIELD (list->data) && 
		    IS_MG_DB_FIELD (mg_qf_field_get_ref_field (MG_QF_FIELD (list->data)))) {
			MgDbField *dbfield = MG_DB_FIELD (mg_qf_field_get_ref_field (MG_QF_FIELD (list->data)));
			
			mg_qf_value_set_not_null (vfield, !mg_db_field_is_null_allowed (dbfield));
			if (mg_db_field_get_default_value (dbfield))
				mg_qf_value_set_default_value (vfield, mg_db_field_get_default_value (dbfield));
		}
		else {
			TO_IMPLEMENT;
		}

		/* keep a reference to the field in 'core->query_select' which we are now treating */
		g_object_set_data (G_OBJECT (vfield), "position", list->data);
		g_object_set_data (G_OBJECT (vfield), "for_cond", g_hash_table_lookup (replacements, list->data));
		core->priv->for_cond_value_qfields = g_slist_append (core->priv->for_cond_value_qfields, vfield);
		
		list = g_slist_next (list);
	}
	g_slist_free (cond_fields);

	/*
	 * create context, and for each parameter, see if it is an alias for a parameter in 'core->args_context'
	 */
	context = mg_entity_get_exec_context (MG_ENTITY (top_query));
	if (core->args_context)
		list = core->args_context->parameters;
	else
		list = NULL;
	while (list) {
		GSList *ff = mg_parameter_get_dest_fields (MG_PARAMETER (list->data));
		MgQfield *alias_qf = NULL;

		while (ff && !alias_qf) {
			alias_qf = g_hash_table_lookup (replacements, ff->data);
			ff = g_slist_next (ff);
		}

		if (alias_qf) { /* find the param. in 'context' which is for 'alias_qf' */
			MgParameter *alias = mg_context_find_parameter_for_field (context, alias_qf);
			
			if (alias) 
				g_object_set (G_OBJECT (alias), "full_bind", list->data, NULL);
		}
			
		list = g_slist_next (list);
	}
	core->priv->replacements_query_select_hybrid = replacements;

	/*
	 * create MgWorkCoreNode nodes 
	 */
	list = context->parameters;
	while (list) {
		gint pos = -1;
		GSList *dest = mg_parameter_get_dest_fields (MG_PARAMETER (list->data));
		MgField *field;

		while (dest && (pos==-1)) {
			field = g_object_get_data (G_OBJECT (dest->data), "position");
			if (field)
				pos = mg_entity_get_field_index (MG_ENTITY (core->query_select), field);
			dest = g_slist_next (dest);
		}

		if (pos > -1) {
			MgWorkCoreNode *node = g_new0 (MgWorkCoreNode, 1);
			node->param = MG_PARAMETER (list->data);
			node->position = pos;
			core->nodes = g_slist_append (core->nodes, node);
			core->params_in_data_rs = g_slist_append (core->params_in_data_rs, node->param);
		}
		else {
			core->params_not_in_data_rs = g_slist_append (core->params_not_in_data_rs, list->data);
#ifdef debug_NO
			g_print ("Don't create a code node for param %p\n", list->data);
			dest = mg_parameter_get_dest_fields (MG_PARAMETER (list->data));
			g_print ("For %d field(s):\n", g_slist_length (dest));
			while (dest) {
				mg_base_dump (dest->data, 10);
				dest = g_slist_next (dest);
			}
#endif
		}
		
		list = g_slist_next (list);
	}

	/*
	 * Free allocated memory
	 */
#ifdef debug_NO
	list = sel_queries;
	while (list) {
		gchar *gname = g_strdup_printf ("_sel_query%d.dot", g_slist_position (sel_queries, list));
		MgGraphviz *graph = MG_GRAPHVIZ (mg_graphviz_new (mg_base_get_conf (MG_BASE (core->query_select))));
		mg_graphviz_add_to_graph (graph, G_OBJECT (top_query));
		mg_graphviz_save_file (graph, gname, NULL);
		g_object_unref (G_OBJECT (graph));
		g_print ("Written file %s\n", gname);
		g_free (gname);

		g_print ("------------------------------------------- SEL QUERY ---------------------\n");
		mg_base_dump (list->data, 10);

		list = g_slist_next (list);
	}
#endif
	list = sel_queries;
	while (list) {
		if (list->data != top_query)
			g_object_unref (G_OBJECT (list->data));
		else {
			core->priv->query_select_hybrid = top_query;
			g_signal_connect (G_OBJECT (core->priv->query_select_hybrid), "nullified",
					  G_CALLBACK (nullified_query_cb), core);
		}
		list = g_slist_next (list);
	}
	g_slist_free (sel_queries);
	make_target_deps_free (targets_d);
	g_hash_table_destroy (targets_h);


	/*
	 * computes the core->params_modif_queries_value_providers list
	 */
	if (core->priv->query_select_hybrid) {
		g_object_get (G_OBJECT (core->priv->query_select_hybrid), "really_all_fields", &list, NULL);
		while (list) {
			if (IS_MG_QF_FIELD (list->data)) {
				if (mg_qf_field_get_target (MG_QF_FIELD (list->data)) == 
				    g_object_get_data (G_OBJECT (top_query), "for_target")) {
					MgField *value_prov;
					MgParameter *param;
					
					g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
					if (value_prov) {
						g_assert (IS_MG_QF_VALUE (value_prov));
						
						param = mg_context_find_parameter_for_field (context, MG_QFIELD (value_prov));
						g_assert (param);
						core->params_modif_queries_value_providers = 
							g_slist_append (core->params_modif_queries_value_providers, param);
					}
				}
			}
			
			list = g_slist_next (list);
		}
	}

	return context;
}

static void            improve_queries_with_fk_constraint (MgWorkCore *core, TargetDep *dep, 
							   MgQuery *query_fk, MgQuery *query_ref_pk, 
							   MgDbConstraint *fkcons,
							   GHashTable *replacements);
static void            improve_queries_with_join_condition (MgWorkCore *core, TargetDep *dep, 
							    MgQuery *query_fk, MgQuery *query_ref_pk, 
							    MgTarget *fk_target, MgTarget *ref_pk_target,
							    MgCondition *cond, GHashTable *replacements);


gboolean
make_target_select_queries_improved (MgWorkCore *core, TargetDep *dep, GHashTable *target_query_h,
				     GHashTable *replacements, GError **error)
{
	MgQuery *sel_query1, *sel_query2;
	MgJoin *join = dep->join;
	MgTarget *target1, *target2;
	MgEntity *ent1, *ent2;
	MgCondition *cond;
	GSList *list;

	/* recursive behaviour */
	list = dep->depend_on;
	while (list) {
		if (!make_target_select_queries_improved (core, TARGET_DEP (list->data), target_query_h, 
							  replacements, error)) {
			TO_IMPLEMENT; /* set error msg */
			return FALSE;
		}
		list = g_slist_next (list);
	}

	/* this TargetDep */
	if (!join)
		return TRUE;

	cond = mg_join_get_condition (join);

	target1 = dep->dependant->target;
	target2 = dep->target;
	ent1 = mg_target_get_represented_entity (target1);
	ent2 = mg_target_get_represented_entity (target2);

	sel_query1 = g_hash_table_lookup (target_query_h, target1); /* query with FK */
	sel_query2 = g_hash_table_lookup (target_query_h, target2); /* query with ref'ed PK */
	if (!sel_query1 || !sel_query2)
		return TRUE;

	if (cond) {
		MgTarget *t1, *t2;
		gboolean is_equi_join;
		if (mg_condition_represents_join (cond, &t1, &t2, &is_equi_join)) {
			if (is_equi_join &&
			    (((t1 == target1) && (t2 == target2)) ||
			     ((t1 == target2) && (t2 == target1)))) {
				/* here we are sure that we have a condition which is in the form 
				 * t1.a=t2.b AND t1.c=t2.d...
				 */
				improve_queries_with_join_condition (core, dep, sel_query1, sel_query2,
								     target1, target2,
								     cond, replacements);
			}
			else {
				TO_IMPLEMENT; /* set error msg */
				g_print ("Equi join: %d\n", is_equi_join );
				return FALSE;
			}
		}
		else {
			TO_IMPLEMENT; /* set error msg */
			return FALSE;
		}
	}
	else {
		GSList *fklist = mg_conf_get_entities_fk_constraints (mg_base_get_conf (MG_BASE (core)),
								      ent1, ent2, TRUE);
		if (fklist) {
			improve_queries_with_fk_constraint (core, dep, sel_query1, sel_query2, 
							    MG_DB_CONSTRAINT (fklist->data), replacements);
			g_slist_free (fklist);
		}
	}

	return TRUE;
}

static void
improve_queries_with_db_fields (MgWorkCore *core, TargetDep *dep, 
				MgQuery *query_fk, MgQuery *query_ref_pk,
				const gchar *fk_entity_name, const gchar *ref_pk_entity_name,
				MgField *fk_field, MgField *ref_pk_field,
				GHashTable *replacements)
{
	MgQfield *tmp;
	MgQfField *qfield, *qfield_pk, *query_sel_field;
	MgQfValue *vfield;
	MgCondition *cond, *newcond;
	gchar *str;

	/* adding PK fields if necessary to 'core->query_select'*/
	tmp = mg_query_get_any_field_by_ref_field (core->query_select, NULL, ref_pk_field);
	if (!tmp) {
		MgTarget *newtarget;
		newtarget = dep->target;
		qfield = MG_QF_FIELD (mg_qf_field_new_with_objects (core->query_select,
								    newtarget, ref_pk_field));
		query_sel_field = qfield;
		mg_entity_add_field (MG_ENTITY (core->query_select), MG_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		mg_qfield_set_internal (MG_QFIELD (query_sel_field), TRUE);
		mg_base_set_name (MG_BASE (query_sel_field), "_FKcons: PK field");
		core->priv->ref_integ_fields = g_slist_prepend (core->priv->ref_integ_fields, query_sel_field);
	}
	else 
		query_sel_field = MG_QF_FIELD (tmp);

	mg_qfield_set_visible (MG_QFIELD (query_sel_field), TRUE);
	mg_qfield_set_internal (MG_QFIELD (query_sel_field), TRUE);

	/* adding PK fields if necessary to query_ref_pk */
	tmp = mg_query_get_any_field_by_ref_field (query_ref_pk, NULL, ref_pk_field);
	if (!tmp) {
		MgTarget *newtarget;
		newtarget = g_object_get_data (G_OBJECT (query_ref_pk), "for_target");
		qfield = MG_QF_FIELD (mg_qf_field_new_with_objects (query_ref_pk,
								    newtarget, ref_pk_field));
		mg_entity_add_field (MG_ENTITY (query_ref_pk), MG_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		mg_qfield_set_internal (MG_QFIELD (qfield), TRUE);
		mg_base_set_name (MG_BASE (qfield), "_FKcons: PK field");
		core->priv->ref_integ_fields = g_slist_prepend (core->priv->ref_integ_fields, qfield);
	}
	else {
		qfield = MG_QF_FIELD (tmp);
		if (! mg_qfield_is_visible (MG_QFIELD (tmp))) {
			mg_qfield_set_visible (MG_QFIELD (tmp), TRUE);
			mg_qfield_set_internal (MG_QFIELD (tmp), TRUE);
		}
	}
	mg_qfield_set_visible (MG_QFIELD (qfield), TRUE);
	qfield_pk = qfield;

	/* adding FK fields if necessary to 'core->query_select'*/
	tmp = mg_query_get_any_field_by_ref_field (core->query_select, NULL, fk_field);
	if (!tmp) {
		MgTarget *newtarget;
		newtarget = dep->dependant->target;
		qfield = MG_QF_FIELD (mg_qf_field_new_with_objects (core->query_select,
								    newtarget, fk_field));
		query_sel_field = qfield;
		mg_entity_add_field (MG_ENTITY (core->query_select), MG_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		mg_qfield_set_internal (MG_QFIELD (query_sel_field), TRUE);
		mg_base_set_name (MG_BASE (query_sel_field), "_FKcons: FK field");
		core->priv->ref_integ_fields = g_slist_prepend (core->priv->ref_integ_fields, query_sel_field);
	}
	else
		query_sel_field = MG_QF_FIELD (tmp);
	mg_qfield_set_visible (MG_QFIELD (query_sel_field), TRUE);

	/* adding FK fields if necessary to query_fk */
	tmp = mg_query_get_any_field_by_ref_field (query_fk, NULL, fk_field);
	if (!tmp) {
		MgTarget *newtarget;
		newtarget = g_object_get_data (G_OBJECT (query_fk), "for_target");
		qfield = MG_QF_FIELD (mg_qf_field_new_with_objects (query_fk,
								    newtarget, fk_field));
		mg_entity_add_field (MG_ENTITY (query_fk), MG_FIELD (qfield));
		g_object_unref (G_OBJECT (qfield));
		mg_qfield_set_internal (MG_QFIELD (qfield), TRUE);
		mg_base_set_name (MG_BASE (qfield), "_FKcons: FK field");
		core->priv->ref_integ_fields = g_slist_prepend (core->priv->ref_integ_fields, qfield);
	}
	else {
		/* We may need to remove a MgQfValue value provider which has been attached to this field
		 * because otherwise we have an MgParameter in the work_context which corresponds to nothing */
		MgQfield *vfield;
		
		qfield = MG_QF_FIELD (tmp);
		g_object_get (G_OBJECT (qfield), "value_provider", &vfield, NULL);
		if (vfield) 
			mg_entity_remove_field (MG_ENTITY (query_fk), MG_FIELD (vfield));
		
	}
	mg_qfield_set_visible (MG_QFIELD (qfield), TRUE);
	
	if (replacements && !g_hash_table_lookup (replacements, query_sel_field))
		g_hash_table_insert (replacements, query_sel_field, qfield);

	/* add MgQfValue for this field */
	vfield = MG_QF_VALUE (mg_qf_value_new (query_fk, mg_field_get_data_type (fk_field)));
	mg_entity_add_field_before (MG_ENTITY (query_fk), MG_FIELD (vfield), MG_FIELD (qfield));
	g_object_unref (G_OBJECT (vfield));
	mg_qf_value_set_is_parameter (vfield, TRUE);
	mg_qfield_set_visible (MG_QFIELD (vfield), FALSE);
	mg_qfield_set_internal (MG_QFIELD (vfield), TRUE);
	g_object_set (G_OBJECT (vfield), "value_provider", qfield_pk, NULL); /* when PK changes, so does this MgQfValue */
	g_object_set (G_OBJECT (qfield), "value_provider", vfield, NULL); /* FK is linked to the MgQfValue */

	/* condition */
	cond = mg_query_get_condition (query_fk);
	if (!cond) {
		cond = MG_CONDITION (mg_condition_new (query_fk, MG_CONDITION_NODE_AND));
		mg_query_set_condition (query_fk, cond);
		mg_base_set_name (MG_BASE (cond), "_FKcons: AND");
		g_object_unref (G_OBJECT (cond));
	}
	else {
		if (mg_condition_is_leaf (cond)) {
			g_object_ref (G_OBJECT (cond));
			newcond = MG_CONDITION (mg_condition_new (query_fk, MG_CONDITION_NODE_AND));
			mg_base_set_name (MG_BASE (newcond), "_FKcons: AND");
			mg_query_set_condition (query_fk, newcond);
			g_assert (mg_condition_node_add_child (newcond, cond, NULL));
			g_object_unref (G_OBJECT (cond));
			g_object_unref (G_OBJECT (newcond));
			cond = newcond;
		}
	}
	
	newcond = MG_CONDITION (mg_condition_new (query_fk, MG_CONDITION_LEAF_EQUAL));
	g_assert (mg_condition_node_add_child (cond, newcond, NULL));
	mg_condition_leaf_set_operator (newcond, MG_CONDITION_OP_LEFT, MG_QFIELD (qfield));
	mg_condition_leaf_set_operator (newcond, MG_CONDITION_OP_RIGHT, MG_QFIELD (vfield));
	str = g_strdup_printf ("_Fkcons: %s->%s", fk_entity_name, ref_pk_entity_name);
	mg_base_set_name (MG_BASE (newcond), str);
	g_free (str);
	g_object_unref (G_OBJECT (newcond));

	/* MgQfValue properties */
	str = g_strdup_printf ("_FKcons: Val %s", mg_base_get_name (MG_BASE (fk_field)));
	mg_base_set_name (MG_BASE (vfield), str);
	g_free (str);
	mg_base_set_description (MG_BASE (vfield), mg_base_get_description (MG_BASE (fk_field)));
	mg_qf_value_set_not_null (vfield, !mg_db_field_is_null_allowed (MG_DB_FIELD (fk_field)));
	if (mg_db_field_get_default_value (MG_DB_FIELD (fk_field)))
		mg_qf_value_set_default_value (vfield, mg_db_field_get_default_value (MG_DB_FIELD (fk_field)));
	
	/* keep a reference to the field in 'core->query_select' which we 
	   are now treating */
	g_object_set_data (G_OBJECT (vfield), "position", query_sel_field);


	/* */
	g_hash_table_insert (core->priv->target_to_ref_pk_field, dep->target, vfield);
}

static void
improve_queries_with_fk_constraint (MgWorkCore *core, TargetDep *dep, MgQuery *query_fk, 
				    MgQuery *query_ref_pk, MgDbConstraint *fkcons,
				    GHashTable *replacements)
{
	GSList *pairs, *list;
	const GSList *clist;
	
	pairs = mg_db_constraint_fkey_get_fields (fkcons);
	list = pairs;
	while (list) {
		improve_queries_with_db_fields (core, dep, query_fk, query_ref_pk,
						mg_base_get_name (MG_BASE (mg_db_constraint_get_table (fkcons))),
						mg_base_get_name (MG_BASE (mg_db_constraint_fkey_get_ref_table (fkcons))),
						MG_FIELD (MG_DB_CONSTRAINT_FK_PAIR (list->data)->fkey),
						MG_FIELD (MG_DB_CONSTRAINT_FK_PAIR (list->data)->ref_pkey),
						replacements);
		
		list = g_slist_next (list);
	}
	g_slist_free (pairs);


	/* setting query_ref_pk to be a 'param_source' for query_fk */
	clist = mg_query_get_param_sources (query_fk);
	if (!g_slist_find (clist, query_ref_pk))
		mg_query_add_param_source (query_fk, query_ref_pk);
}


static void
improve_queries_with_join_condition (MgWorkCore *core, TargetDep *dep, 
				     MgQuery *query_fk, MgQuery *query_ref_pk, 
				     MgTarget *fk_target, MgTarget *ref_pk_target,
				     MgCondition *cond, GHashTable *replacements)
{
	const GSList *list, *mainconds;
	
	
	mainconds = mg_condition_get_main_conditions (cond);
	list = mainconds;
	while (list) {
		MgCondition *cond = MG_CONDITION (list->data);
		MgQfield *field;
		MgField *fk_field = NULL, *ref_pk_field = NULL;

		g_return_if_fail (mg_condition_get_cond_type (cond) == MG_CONDITION_LEAF_EQUAL);
		
		field = mg_condition_leaf_get_operator (cond, MG_CONDITION_OP_LEFT);
		g_return_if_fail (field && IS_MG_QF_FIELD (field));
		if (mg_qf_field_get_target (MG_QF_FIELD (field)) == fk_target)
			fk_field = mg_qf_field_get_ref_field (MG_QF_FIELD (field));
		else
			ref_pk_field = mg_qf_field_get_ref_field (MG_QF_FIELD (field));
		
		field = mg_condition_leaf_get_operator (cond, MG_CONDITION_OP_RIGHT);
		g_return_if_fail (field && IS_MG_QF_FIELD (field));
		if (mg_qf_field_get_target (MG_QF_FIELD (field)) == fk_target)
			fk_field = mg_qf_field_get_ref_field (MG_QF_FIELD (field));
		else
			ref_pk_field = mg_qf_field_get_ref_field (MG_QF_FIELD (field));
		
		g_return_if_fail (fk_field);
		g_return_if_fail (ref_pk_field);

		improve_queries_with_db_fields (core, dep, query_fk, query_ref_pk,
						mg_base_get_name (MG_BASE (mg_field_get_entity (fk_field))),
						mg_base_get_name (MG_BASE (mg_field_get_entity (ref_pk_field))),
						fk_field, ref_pk_field, 
						replacements);
		
		list = g_slist_next (list);
	}
	

	/* setting query_ref_pk to be a 'param_source' for query_fk */
	list = mg_query_get_param_sources (query_fk);
	if (!g_slist_find (list, query_ref_pk))
		mg_query_add_param_source (query_fk, query_ref_pk);
}

gboolean
modif_target_depends_on (MgWorkCore *core, TargetDep *modif_target_dep, MgTarget *target)
{
	GSList *list;

	if (target == core->modif_target)
		return TRUE;

	if (modif_target_dep->target == target)
		return TRUE;

	list = modif_target_dep->depend_on;
	while (list) {
		if (modif_target_depends_on (core, TARGET_DEP (list->data), target))
			return TRUE;
		list = g_slist_next (list);
	}

	return FALSE;
}

#ifdef debug
static void
target_dep_dump (TargetDep *dep, gint offset)
{
	gchar *str;
        guint i;
	GSList *list;

	/* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

	g_print ("%sDEP target=%p (%s) dependant:%p\n", str, dep->target, mg_base_get_name (MG_BASE (dep->target)), 
		 dep->dependant ? dep->dependant->target : NULL);
	list = dep->depend_on;
	while (list) {
		target_dep_dump (TARGET_DEP (list->data), offset+5);
		list = g_slist_next (list);
	}
	g_free (str);
}
#endif


/*
 * make all the required work to call make_target_deps_recurs
 */
TargetDep *
make_target_deps (MgWorkCore *core)
{
	TargetDep *dep;
	GSList *joins;
	GHashTable *hash;
	
	g_return_val_if_fail (core->query_select, NULL);
	g_return_val_if_fail (core->modif_target, NULL);

	joins = mg_query_get_joins (core->query_select);
	hash = g_hash_table_new (NULL, NULL);

	dep = make_target_deps_recurs (core, core->modif_target, joins, hash);

	g_slist_free (joins);
	g_hash_table_destroy (hash);

#ifdef debug_NO
	g_print ("###################################### target deps\n");
	target_dep_dump (dep, 0);
#endif

	return dep;
}

/*
 * Creates a TargetDep structure to hold all the MgTarget on which 'on_target' depends.
 */
TargetDep *
make_target_deps_recurs (MgWorkCore *core, MgTarget *on_target, const GSList *joins, GHashTable *joinsdep)
{
	TargetDep *dep = NULL;
	const GSList *list;

	dep = g_new0 (TargetDep, 1);
	dep->target = on_target;
	dep->depend_on = NULL;
	dep->join = NULL;

	list = joins;
	while (list) {
		if (!g_hash_table_lookup (joinsdep, list->data)) {
			/* This join has not yet been treaded */
			MgTarget *t1, *t2, *tmp = NULL;
			MgJoinType type;
			gboolean useit = FALSE;

			t1 = mg_join_get_target_1 (MG_JOIN (list->data));
			t2 = mg_join_get_target_2 (MG_JOIN (list->data));
			type = mg_join_get_join_type (MG_JOIN (list->data));
			switch (type) {
			case MG_JOIN_TYPE_INNER:
				if ((t1 == on_target) || (t2 == on_target)) {
					useit = TRUE;
					if (t1 == on_target)
						tmp = t2;
					else
						tmp = t1;
				}
				break;
			case MG_JOIN_TYPE_RIGHT_OUTER:
				tmp = t1;
				t1 = t2;
				t2 = tmp;
			case MG_JOIN_TYPE_LEFT_OUTER:
				if (t1 == on_target) {
					useit = TRUE;
					tmp = t2;
				}
			default:
				/* we can't do anything here */
				break;
			}
			
			if (useit) {
				TargetDep *tdep;

				g_hash_table_insert (joinsdep, list->data, tmp);
				tdep = make_target_deps_recurs (core, tmp, joins, joinsdep);
				dep->depend_on = g_slist_append (dep->depend_on, tdep);
				tdep->join = MG_JOIN (list->data);
				tdep->dependant = dep;
			}
		}
		list = g_slist_next (list);
	}

	return dep;
}

void
make_target_deps_free (TargetDep *dep)
{
	GSList *list = dep->depend_on;
	
	while (list) {
		make_target_deps_free (TARGET_DEP (list->data));
		list = g_slist_next (list);
	}

	if (dep->depend_on)
		g_slist_free (dep->depend_on);
	g_free (dep);
}



/**
 * mg_work_core_run_select_query
 * @core: a #MgWorkCore object
 * @error: location to store error, or %NULL
 *
 * Runs the @core->selec_query SELECT query and stores the resultset in @core->data_rs
 *
 * Returns: TRUE if not error occurred
 */
gboolean
mg_work_core_run_select_query (MgWorkCore *core, GError **error)
{
	g_return_val_if_fail (core && IS_MG_WORK_CORE (core), FALSE);

	/* get rid of old data */
	if (core->data_rs) {
		g_object_unref (G_OBJECT (core->data_rs));
		core->data_rs = NULL;
	}

	if (core->query_select) {
		gchar *sql;

		/* Actual running query */
		sql = mg_renderer_render_as_sql (MG_RENDERER (core->query_select), 
						 core->args_context, 0, error);
		if (sql) {
			MgConf *conf = mg_base_get_conf (MG_BASE (core->query_select));
			
#ifdef debug
			g_print ("SQL: %s\n", sql);
#endif
			core->data_rs = mg_server_do_query (mg_conf_get_server (conf), sql, 
							    MG_SERVER_QUERY_SQL, NULL);

			/* Setting columns names */
			if (core->data_rs) {
				GSList *list, *fields;
				gint i=0;

				fields = mg_entity_get_visible_fields (MG_ENTITY (core->query_select));
				list = fields;
				while (list) {
					mg_resultset_set_col_name (core->data_rs, i,
								   mg_base_get_name (MG_BASE (list->data)));
					list = g_slist_next (list);
					i++;
				}
				g_slist_free (fields);
			}
			g_free (sql);
		}
	}

	return core->data_rs ? TRUE : FALSE;
}

/**
 * mg_work_core_find_param
 * @core: a #MgWorkCore object
 * @field: a #MgQfield which belongs to the SELECT query given as 'query' argument to mg_work_core_new(), or
 *         to @core->query_select (which is a copy of the query given as 'query' argument to mg_work_core_new())
 * @in_exec_context: TRUE if to search in the exec context, or FALSE to search among the "data" context
 *
 * Finds the #MgParameter 'attached' to the values of @field. The @in_exec_context specifies if the 
 * parameter is to be searched among the required parameters to execute the SELECT query 
 * given as 'query' argument to mg_work_core_new(), or among the results of the execution of that query.
 *
 * Returns: the requested #MgParameter, or %NULL
 */
MgParameter *
mg_work_core_find_param (MgWorkCore *core, MgQfield *field, gboolean in_exec_context)
{
	gpointer q_sel_field;
	MgParameter *param = NULL;

	g_return_val_if_fail (core && IS_MG_WORK_CORE (core), NULL);
	g_return_val_if_fail (field && IS_MG_QFIELD (field), NULL);

	q_sel_field = g_hash_table_lookup (core->priv->replacements_query_select, field);
	if (!q_sel_field)
		/* @field already belongs to @core->query_select */
		q_sel_field = (gpointer) field;

	if (in_exec_context) {
		/* try in 'args_context' */
		param = mg_context_find_parameter_for_field (core->args_context, MG_QFIELD (q_sel_field));
		if (!param && (q_sel_field != (gpointer) field))
			param = mg_context_find_parameter_for_field (core->args_context, field);
	}
	else {
		/* try in 'work_context' */
		if (core->modif_target) {
			gpointer q_sel_impr_field;
			gpointer qf_value;
			
			q_sel_impr_field = g_hash_table_lookup (core->priv->replacements_query_select_hybrid, 
								q_sel_field);
			g_return_val_if_fail (q_sel_impr_field, NULL);
			
			param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (q_sel_field));
			if (!param) {
				g_object_get (G_OBJECT (q_sel_impr_field), "value_provider", &qf_value, NULL);
				if (qf_value)
					param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (qf_value));
			}
		}
		else 
			param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (q_sel_field));
	}
	
	return param;
}


/**
 * mg_work_core_find_core_node
 * @core: a #MgWorkCoreNode object
 * @param: a #MgParameter object
 *
 * Find the #MgWorkCoreNode node which handles @param.
 *
 * Returns: the corresponding #MgWorkCoreNode
 */
MgWorkCoreNode *
mg_work_core_find_core_node (MgWorkCore *core, MgParameter *param)
{
	g_return_val_if_fail (core && IS_MG_WORK_CORE (core), NULL);

	return g_hash_table_lookup (core->priv->nodes_hash, param);
}


/**
 * mg_work_core_find_context_node
 * @core: a #MgWorkCoreNode object
 * @field:a #MgQfield which belongs to the SELECT query given as 'query' argument to mg_work_core_new(), or
 *         to @core->query_select (which is a copy of the query given as 'query' argument to mg_work_core_new())
 *
 * Depending on if there is a target to be modified of not, the data for the @field field
 * will be presented to the user as a simple entry area, or as a combo box. This function makes it
 * possible to get the #MgContextNode where the data pointer by @field will be "used" (either through a 
 * #MgParameter for simple entry area, or in a SELECT query for combo boxes).
 *
 * Returns: the #MgContextNode, or %NULL if not found.
 */
MgContextNode *
mg_work_core_find_context_node (MgWorkCore *core, MgQfield *field)
{
	MgContextNode *cnode = NULL;
	MgParameter *param;

	g_return_val_if_fail (core && IS_MG_WORK_CORE (core), NULL);
	g_return_val_if_fail (core->priv, NULL);

	param = mg_work_core_find_param (core, field, FALSE);
	if (param)
		cnode = mg_context_find_node_for_param (core->work_context, param);

	if (cnode)
		return cnode;
	
	if (core->modif_target) {
		gpointer q_sel_field;

		q_sel_field = g_hash_table_lookup (core->priv->replacements_query_select, field);
		if (!q_sel_field)
			/* @field already belongs to @core->query_select */
			q_sel_field = (gpointer) field;
		
		
		if (IS_MG_QF_FIELD (q_sel_field)) {
			MgTarget *target = mg_qf_field_get_target (MG_QF_FIELD (q_sel_field));
			q_sel_field = g_hash_table_lookup (core->priv->target_to_ref_pk_field, target);
			if (q_sel_field) {
				param = mg_context_find_parameter_for_field (core->work_context, MG_QFIELD (q_sel_field));
				if (param)
					cnode = mg_context_find_node_for_param (core->work_context, param);
			}
		}
	}

	return cnode;
}
