/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
package com.sap.dbtech.jdbc.translators;

import java.sql.*;
import java.io.OutputStream;
import java.io.Writer;
import com.sap.dbtech.jdbc.*;
import com.sap.dbtech.jdbc.exceptions.*;
import com.sap.dbtech.jdbc.packet.*;
import com.sap.dbtech.util.*;
import com.sap.dbtech.vsp00.*;
import com.sap.dbtech.vsp001.*;
/**
 *
 */
public abstract class AbstractGetval
{
    private ConnectionSapDB connection;
    protected byte [] descriptor;
    protected StructuredMem streamBuffer;
    protected int itemsInBuffer;
    protected int itemSize;
    protected boolean atEnd;
    protected boolean firstChunk = true;
    protected long longPosition = 0;
    protected long longSize = -1;
    /**
     * creates a new AbstractGetval
     */
    public
    AbstractGetval (
        ConnectionSapDB connection,
        byte [] descriptor,
        StructuredMem dataPart,
        int itemSize)
    {
        this.connection = connection;
        this.atEnd = false;
        this.itemSize = itemSize;
        this.setupStreamBuffer (descriptor, dataPart);
    }
    /**
     *
     * @exception StreamIOException
     */
    protected boolean
    nextChunk ()
    throws StreamIOException
    {
        try {
            int valMode = this.descriptor [LongDesc.Valmode_O];
            if ((valMode == LongDesc.LastData_C) || (valMode == LongDesc.AllData_C)) {
                this.atEnd = true;
                return false;
            }
            this.firstChunk = false;
            RequestPacket requestPacket = this.connection.getRequestPacket ();
            ReplyPacket replyPacket;
            DataPart longpart = requestPacket.initGetval (this.connection.getAutoCommit());
            longpart.putInt1 (0, 0);
            longpart.putBytes (this.descriptor, 1);
            int maxval = Integer.MAX_VALUE - 1;
            longpart.putInt4 (maxval, 1 + LongDesc.Vallen_O);
            longpart.addArg (1, LongDesc.Size_C);
            longpart.close ();
            try {
                replyPacket = this.connection.execute (requestPacket, this, ConnectionSapDB.GC_DELAYED);
            }
            catch (SQLExceptionSapDB sqlExc) {
                throw new StreamIOException (sqlExc);
            }
            replyPacket.findPart (PartKind.Longdata_C);
            int dataPos = replyPacket.getPartDataPos ();
            this.descriptor = replyPacket.getDataBytes (dataPos, LongDesc.Size_C + 1);
            if(this.descriptor[LongDesc.Valmode_O] == LongDesc.StartposInvalid_C) {
            	throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_INVALID_STARTPOSITION));
            }
            this.setupStreamBuffer (this.descriptor,
                replyPacket.getPointer (dataPos));
            return true;
        }
        catch (SQLException exc) {
            throw new StreamIOException ((SQLExceptionSapDB) exc);
        }
        catch (PartNotFound notFoundExc) {
            throw new StreamIOException(new InternalJDBCError (
                MessageTranslator.translate(MessageKey.ERROR_LONGDATAEXPECTED)
             ));
        }
    }
    /**
     *
     */
    protected ReplyPacket
    executeGetval (
        byte [] descriptor)
    throws SQLException
    {
        RequestPacket requestPacket = this.connection.getRequestPacket ();
        ReplyPacket replyPacket;
        DataPart longpart = requestPacket.initGetval (this.connection.getAutoCommit());
        longpart.putInt1 (0, 0);
        longpart.putBytes (descriptor, 1);
        longpart.putInt4 (Integer.MAX_VALUE-1, 1 + LongDesc.Vallen_O);
        longpart.addArg (1, LongDesc.Size_C);
        longpart.close ();
        replyPacket = this.connection.execute (requestPacket, this, ConnectionSapDB.GC_DELAYED);
        return replyPacket;
    }
    /**
     *
     * @param descriptor byte[]
     * @param dataPart com.sap.dbtech.util.StructuredMem
     */
    private void setupStreamBuffer (byte [] descriptor, StructuredMem dataPart) {
        StructuredMem desc = new StructuredBytes (descriptor);
        int dataStart;

        dataStart = desc.getInt4 (LongDesc.Valpos_O) - 1;
        this.itemsInBuffer = desc.getInt4 (LongDesc.Vallen_O) / this.itemSize;
        //if (this.bytesInBuffer == 0) {
        //    // an empty part is really invalid
        //    desc.traceOn(System.out);
        //    throw new NegativeArraySizeException ();
        //}
        this.streamBuffer = dataPart.getPointer (dataStart);
        this.descriptor = descriptor;
        if(descriptor[LongDesc.InternPos_O]==0   &&
           descriptor[LongDesc.InternPos_O+1]==0 &&
		   descriptor[LongDesc.InternPos_O+2]==0 &&
		   descriptor[LongDesc.InternPos_O+3]==0) {
        	descriptor[LongDesc.InternPos_O+3] = 1;
        }
    }
    /**
     * Returns the number of bytes in the <code>BLOB</code> value
     * designated by this <code>Blob</code> object.
     * @return length of the <code>BLOB</code> in bytes
     * @exception SQLException if there is an error accessing the
     * length of the <code>BLOB</code>
     */
    public long
    lengthInBytes()
    throws SQLException
    {
        if (this.longSize > -1) {
            return this.longSize;
        }
        StructuredMem desc = new StructuredBytes (this.descriptor);
        this.longSize = desc.getInt4 (LongDesc.MaxLen_O);
        if ( this.longSize > 0){
            return this.longSize;
        }

        this.longSize = this.longSizeRequest ();
        return this.longSize;
    }

    /**
     * send a request to get the length of a LONG value.
     *
     * This is done by sending a getval where the valmode is set
     * to DataTrunc_C
     */
    protected long
    longSizeRequest ()
    throws SQLException
    {
        long result;
        byte [] requestDescriptor = new byte [this.descriptor.length];
        byte [] resultDescriptor;
        ReplyPacket replyPacket;

        // copy descriptor
        System.arraycopy (this.descriptor, 0, requestDescriptor, 0,
            this.descriptor.length);
        requestDescriptor [LongDesc.Valmode_O] = LongDesc.DataTrunc_C;
        replyPacket = this.executeGetval (requestDescriptor);
        // get descriptor and read intern_pos
        try {
            replyPacket.findPart (PartKind.Longdata_C);
        }
        catch (PartNotFound notFoundExc) {
            throw new InternalJDBCError (
                MessageTranslator.translate(MessageKey.ERROR_LONGDATAEXPECTED));
        }
        int dataPos = replyPacket.getPartDataPos ();
        resultDescriptor = replyPacket.getDataBytes (dataPos, LongDesc.Size_C + 1);
        StructuredBytes descBytes = new StructuredBytes (resultDescriptor);
        /*
         * The result is the Pascal index of the append position,
         *   so 1 has to be subtracted
         */
        result = descBytes.getInt4(LongDesc.MaxLen_O);
        return result;
    }
    abstract java.io.InputStream getAsciiStream ()
        throws java.sql.SQLException;

    abstract java.io.InputStream getBinaryStream ()
        throws java.sql.SQLException;

    abstract java.io.Reader getCharacterStream ()
        throws java.sql.SQLException;


   /**
     * Determines the character position at which the specified substring
     * <code>searchstr</code> appears in the <code>CLOB</code>.  The search
     * begins at position <code>start</code>.
     * @param searchstr the substring for which to search
     * @param start the position at which to begin searching; the first position
     *              is 1
     * @return the position at which the substring appears, else -1; the first
     *         position is 1
     * @exception SQLException if there is an error accessing the
     * <code>CLOB</code> value
     */
  final public long position(String searchstr, long start) throws SQLException
    {
        throw new NotImplemented (
            MessageTranslator.translate(MessageKey.ERROR_POSITION_NOTIMPLEMENTED)
        );
    }

  final public int setBytes(long pos, byte[] bytes) throws SQLException {
    /**@todo: Implement this java.sql.Blob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETBYTES_NOTIMPLEMENTED)
    );
  }
  final public int setBytes(long pos, byte[] bytes, int offset, int len) throws SQLException {
    /**@todo: Implement this java.sql.Blob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETBYTES_NOTIMPLEMENTED)
    );
  }

  final public OutputStream setBinaryStream(long pos) throws SQLException {
    /**@todo: Implement this java.sql.Blob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETBINARYSTREAM_NOTIMPLEMENTED)
    );
  }
  final public void truncate(long len) throws SQLException {
    /**@todo: Implement this java.sql.Blob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_TRUNCATE_NOTIMPLEMENTED)
    );
  }

  final public int setString(long pos, String str) throws SQLException {
    /**@todo: Implement this java.sql.Clob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETSTRING_NOTIMPLEMENTED)
    );
  }

  final public int setString(long pos, String str, int offset, int len) throws SQLException {
    /**@todo: Implement this java.sql.Clob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETSTRING_NOTIMPLEMENTED)
    );
  }

  final public OutputStream setAsciiStream(long pos) throws SQLException {
    /**@todo: Implement this java.sql.Clob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETASCIISTREAM_NOTIMPLEMENTED)
    );
  }

  final public Writer setCharacterStream(long pos) throws SQLException {
    /**@todo: Implement this java.sql.Clob method*/
    throw new java.lang.UnsupportedOperationException(
        MessageTranslator.translate(MessageKey.ERROR_SETCHARACTERSTREAM_NOTIMPLEMENTED)
    );
  }
  final public long position(java.sql.Clob lob, long start)
        throws SQLException
    {
        throw new NotImplemented (
            MessageTranslator.translate(MessageKey.ERROR_POSITION_NOTIMPLEMENTED)
        );
    }
}
