/*!
 * \file    OMS_VarOid.hpp
 * \author  IvanS, MarkusSi, PeterG
 * \brief   OID for variable sized persistent objects
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#ifndef __OMS_VAROID_HPP
#define __OMS_VAROID_HPP

#include "Oms/OMS_Oid.hpp"
#include "Oms/OMS_DbpError.hpp"

class OmsHandle;

/*--------------------------------------------------------------------*/
/// Defines an OID of a persistent object of variable length.
struct AFX_EXT_CLASS OmsVarOid : public OmsObjectId
{
public:
  /// Constructor
  inline OmsVarOid();
  inline OmsVarOid(OmsHandle&h, const OmsExternalOid&r);

  /*--------------------------------------------------------------------*/
  /// Defines the smaller than operator for OIDs of the type OmsVarOid. 
  /*!
  ** Defines the smaller than operator for OIDs of the type OmsVarOid.
  ** The statement that an OID is smaller than another must not be understood 
  ** as having semantic meaning. This operator has been provided merely 
  ** to enable sorting to be carried out according to OIDs. 
  */
  inline bool operator< ( const OmsVarOid&r ) const;

  /*--------------------------------------------------------------------*/
  /// Defines the smaller than or equal to operator for OIDs of the type OmsVarOid. 
  /*!
  ** Defines the smaller than or equal to operator for OIDs of the type 
  ** OmsVarOid. The statement that an OID is smaller than or equal to another 
  ** must not be understood as having semantic meaning. This operator has 
  ** been provided merely to enable sorting to be carried out according to OIDs.
  */
  inline bool operator<=( const OmsVarOid&r ) const;
  
  /*--------------------------------------------------------------------*/
  /// Defines the equality of OIDs of the type OmsVarOid.
  inline bool operator==( const OmsVarOid&r ) const;
  
  /*--------------------------------------------------------------------*/
  /// Defines the inequality of two OIDs of the type OmsVarOid.
  inline bool operator!=( const OmsVarOid&r ) const;
  
  /*--------------------------------------------------------------------*/
  /// Provides true for the NULL OID; otherwise, false;
  inline bool operator! () const; 
  
  /*--------------------------------------------------------------------*/
  /// Provides true if the OID differs from the NULL OID; otherwise, false.
  inline operator bool  () const; 

  static bool omsCheckOid ( OmsHandle&, const OmsExternalOid& r );
  bool omsCheckOid ( OmsHandle& ) const;
  const void* operator()(OmsHandle&) const;
  const void* omsDeRef(OmsHandle&) const;
};

inline OmsVarOid::OmsVarOid() : OmsObjectId() {};

inline OmsVarOid::OmsVarOid( OmsHandle&h, const OmsExternalOid&r )
{
  OmsObjectId::operator=( r );
  if (!omsCheckOid( h )) {
    throw DbpError(E_INCOMPATIBLE_OID);
  }
}

inline bool OmsVarOid::operator< ( const OmsVarOid&r ) const
{ 
  return OmsObjectId::operator<(r);
}

inline bool OmsVarOid::operator<=( const OmsVarOid&r ) const
{
  return OmsObjectId::operator<=(r);
}

inline bool OmsVarOid::operator==( const OmsVarOid&r ) const 
{ 
  return OmsObjectId::operator==(r);
}

inline bool OmsVarOid::operator!=( const OmsVarOid&r ) const
{
  return OmsObjectId::operator!=(r);
}

inline bool OmsVarOid::operator! () const 
{ 
  return (OmsObjectId::operator!()) ;
}  

inline OmsVarOid::operator bool () const 
{ 
  return (OmsObjectId::operator bool()) ;  
}


#endif // __OMS_VAROID_HPP
