// -*- c++ -*-
#ifndef INCLUDED_MATH3D_M4X4_H
#define INCLUDED_MATH3D_M4X4_H
/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: m4x4.h,v 1.10 2000/10/09 17:55:51 jeh Exp $
 */

#ifndef INCLUDED_MATH3D_MATH3DDEF_H
#include <math3d/math3ddef.h>
#endif

namespace Math3d {
  
  class M3d;
  class M4d;
  class MQuat;
  class MRot;
  class MRect;
  
/**
 * 4x4 matrix class.
 */
  class _CCMATH3D M4x4 {
  protected:
    double d_m[4][4];
    public:
      M4x4() {};
      M4x4(const M4x4& A);
      const M4x4& operator=(const M4x4& A);

      void zero();
      void identity();
      inline void set(int row, int col, double value);
      void copy(const M4x4& A);
      
      inline double* operator[](int index);
      inline operator double*();
      void setRow(int row, const M3d& A);
      void setRow(int row, const M4d& A);
      void setCol(int col, const M3d& A);
      void setCol(int col, const M4d& A);

      M4x4 operator+(const M4x4& A);
      M4x4 operator+();
      const M4x4& operator+=(const M4x4& A);
      M4x4 operator-();
      M4x4 operator-(const M4x4& A);
      const M4x4& operator-=(const M4x4& A);
      M4x4 operator*(const M4x4& A);
      const M4x4& operator*=(const M4x4& A);
      M4x4 operator*(double k);
      const M4x4& operator*=(double k);
      void neg();
      void abs();
      void transpose();
      void add(const M4x4& A, const M4x4& B); 
      void sub(const M4x4& A, const M4x4& B); 
      void mul(const M4x4& A, const M4x4& B);
      void scalar(double k);
      double det();
      void adjoint();
      bool inv();
      void reorthogonalize(int limit=10);
      
      void loadTranslate(double tx, double ty, double tz);
      void translate(double tx, double ty, double tz);
      void translate(const M3d& T);
      void translateNeg(const M3d& T);
      void loadScale(double sx, double sy, double sz);
      void scale(double sx, double sy, double sz);
      void scale(const M3d& S);
      void loadRotateX(double phi);
      void rotateX(double phi);
      void loadRotateY(double phi);
      void rotateY(double phi);
      void loadRotateZ(double phi);
      void rotateZ(double phi);
      void loadRotate(const MQuat& q);
      void loadRotate(const MRot& rot);
      void loadRotate(const M3d& axis, double phi);
      void rotate(const MQuat& q);
      void rotate(const MRot& rot);
      void rotate(const M3d& axis, double phi);
      void loadProj(double fov, double clipNear, double clipFar, double aspect);
      void proj(double fov, double clipNear, double clipFar, double aspect);
      void loadPick(const MRect& rect, const MRect& viewport);
      void pick(const MRect& rect, const MRect& viewport);
      
      inline const double* operator[](int index) const;
      inline operator const double*() const;
      M3d getRow(int row) const;
      M3d getCol(int col) const;

      M3d operator*(const M3d& C) const;
      M4d operator*(const M4d& C) const;
      void transform(M3d& C) const;
      void transform(M4d& C) const;

      bool operator==(const M4x4& A) const;
      bool operator!=(const M4x4& A) const;
      bool cmp(const M4x4& A, double epsilon=EPSILON) const;

      friend MQuat;
  };

  inline const double*
  M4x4::operator[](int index) const
  {
    ASSERT(index>=0 && index<4); 
    return((double*)&d_m[index]);
  }
  inline double*
  M4x4::operator[](int index) 
  {
    ASSERT(index>=0 && index<4); 
    return((double*)&d_m[index]);
  }
  
  inline M4x4::operator
  double*()
  {
  return(&d_m[0][0]);
  }
  inline M4x4::operator
  const double*() const 
  {
    return(&d_m[0][0]);
  }
  
  inline void
  M4x4::set(int row, int col, double value)
  {
    ASSERT(row>=0 && row<4 && col>=0 && col<4); 
    d_m[row][col]=value;
  }

  extern _CCMATH3D ostream& operator << (ostream& co, const M4x4& m);
  
}
#endif // INCLUDED_MATH3D_M4X4_H


