#ifndef TAGCOLL_IMPLICATIONS_H
#define TAGCOLL_IMPLICATIONS_H

/*
 * Collection of tag implications and a a TagcollFilter to apply or compress
 * them
 * 
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <tagcoll/TagcollConsumer.h>
#include <tagcoll/TagcollFilter.h>

#include <tagcoll/OpSet.h>

#include <map>

namespace Tagcoll
{

// List of explicit implications that can be used to expand a tagged collection
// while it's being parsed
template <class TAG>
class ImplicationList : public TagcollConsumer<TAG>
{
protected:
	// DAG arcs: child -> {parents}
	typedef std::map< TAG, OpSet<TAG> > impl_t;
	impl_t implications;

	// Get the set of all tags seen when walking through all parent lists
	OpSet<TAG> getDestinations(const TAG& tag, const OpSet<TAG>& seen = OpSet<TAG>()) const throw ();

	// Return true if tag1 can reach tag2 walking through some path in its parent list
	bool reaches(const TAG& tag1, const TAG& tag2, const OpSet<TAG>& seen = OpSet<TAG>()) const throw ();

public:
	virtual ~ImplicationList() throw () {}

	virtual void consume(const TAG& item) throw () {}
	virtual void consume(const TAG& item, const OpSet<TAG>& tags) throw ();
	
	// Expand a single tag
	OpSet<TAG> expand(const TAG& tag) throw ();

	// Expand a full tagset
	OpSet<TAG> expand(const OpSet<TAG>& tags) throw ();

	// Remove unnecessary arcs from the dag
	void pack() throw ();
	
	// Output the fully expanded implication dag to a TagcollConsumer
	void outputFull(TagcollConsumer<TAG>& consumer) throw ();

	// Output the implication dag to a TagcollConsumer
	void output(TagcollConsumer<TAG>& consumer) throw ();
};

template <class TAG>
class ApplyImplications : public TagcollFilter<TAG>
{
protected:
	ImplicationList<TAG>& implications;
	
public:
	ApplyImplications(ImplicationList<TAG>& implications) throw ()
		: implications(implications) {}

	virtual void consume(const TAG& item) throw ()
	{
		this->consumer->consume(item);
	}

	virtual void consume(const TAG& item, const OpSet<TAG>& tags) throw ()
	{
		this->consumer->consume(item, implications.expand(tags));
	}
};

// TagcollConsumer that just outputs the input to a given file handle, removing
// the tags that would be re-added if expanding with the given explicit
// implications
template <class TAG>
class CompressImplications : public TagcollFilter<TAG>
{
protected:
	ImplicationList<TAG>& implications;

public:
	CompressImplications(ImplicationList<TAG>& implications) throw ()
		: implications(implications) {}

	virtual void consume(const TAG& item) throw ()
	{
		this->consumer->consume(item);
	}
	
	virtual void consume(const TAG& item, const OpSet<TAG>& tags) throw ();
};

};

// vim:set ts=4 sw=4:
#endif
