/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  attributes.cc - PangoAttibute C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "attributes.h"
#include "font.h"
#include "../gdk/bitmap.h"

using namespace Inti;

Pango::Attribute::Attribute(PangoAttribute *attr, bool copy)
: attr_(copy ? pango_attribute_copy(attr) : attr)
{
}
	
Pango::Attribute::Attribute(const Attribute& src)
: attr_(pango_attribute_copy(src.pango_attribute()))
{
}

Pango::Attribute::~Attribute()
{
	if (attr_)
	{
		pango_attribute_destroy((PangoAttribute*)attr_);
		attr_ = 0;
	}		
}

Pango::Attribute& 
Pango::Attribute::operator=(const Attribute& src)
{
	if (src.attr_ == attr_)
		return *this;
		
	pango_attribute_destroy((PangoAttribute*)attr_);
	attr_ = pango_attribute_copy(src.pango_attribute());
	return *this;
}

const PangoAttrClass* 
Pango::Attribute::pango_attr_class() const 
{ 
	return pango_attribute()->klass; 
}
		
Pango::Attribute::operator PangoAttribute* () const 
{ 
	return this ? pango_attribute() : 0; 
}

Pango::AttrType
Pango::Attribute::type() const
{
	return (AttrType)pango_attr_class()->type;
}

bool
Pango::Attribute::is_language() const
{
	return type() == ATTR_LANGUAGE;
}

bool
Pango::Attribute::is_family() const
{
	return type() == ATTR_FAMILY;
}

bool
Pango::Attribute::is_foreground() const
{
	return type() == ATTR_FOREGROUND;
}

bool
Pango::Attribute::is_background() const
{
	return type() == ATTR_BACKGROUND;
}

bool
Pango::Attribute::is_size() const
{
	return type() == ATTR_SIZE;
}

bool
Pango::Attribute::is_style() const
{
	return type() == ATTR_STYLE;
}

bool
Pango::Attribute::is_weight() const
{
	return type() == ATTR_WEIGHT;
}

bool
Pango::Attribute::is_variant() const
{
	return type() == ATTR_VARIANT;
}

bool
Pango::Attribute::is_stretch() const
{
	return type() == ATTR_STRETCH;
}

bool
Pango::Attribute::is_underline() const
{
	return type() == ATTR_UNDERLINE;
}

bool
Pango::Attribute::is_strikethrough() const
{
	return type() == ATTR_STRIKETHROUGH;
}

bool
Pango::Attribute::is_rise() const
{
	return type() == ATTR_RISE;
}

bool
Pango::Attribute::is_font_description() const
{
	return type() == ATTR_FONT_DESC;
}

bool
Pango::Attribute::is_shape() const
{
	return type() == ATTR_SHAPE;
}

bool
Pango::Attribute::is_scale() const
{
	return type() == ATTR_SCALE;
}

unsigned int
Pango::Attribute::start_index() const
{
	return pango_attribute()->start_index;
}

unsigned int
Pango::Attribute::end_index() const
{
	return pango_attribute()->end_index;
}

bool
Pango::Attribute::equal(const Attribute& other) const
{
	return pango_attribute_equal(attr_, other.pango_attribute());
}

bool
Pango::Attribute::operator==(const Attribute& other) const

{
	return equal(other);
}

bool 
Pango::Attribute::operator!=(const Attribute& other) const
{
	return !equal(other);
}

void
Pango::Attribute::set_start_index(unsigned int start)
{
	attr_->start_index = start;
}

void 
Pango::Attribute::set_end_index(unsigned int end)
{
	attr_->end_index = end;
}

void 
Pango::Attribute::set_index(unsigned int start, unsigned int end)
{
	attr_->start_index = start;
	attr_->end_index = end;
}

/*  Pango::AttrString
 */

Pango::AttrString::AttrString(PangoAttribute *attribute, bool copy)
: Attribute(attribute, copy)
{
}

String
Pango::AttrString::get_string() const
{
	return pango_attr_string()->value;
}

void
Pango::AttrString::set_string(const String& font_family_or_language)
{
	g_free(pango_attr_string()->value);
	pango_attr_string()->value = g_strdup(font_family_or_language.c_str());
}

/*  Pango::AttrLanguage
 */

Pango::AttrLanguage::AttrLanguage(PangoLanguage *language)
: Attribute(pango_attr_language_new(language), false)
{
}

PangoLanguage*
Pango::AttrLanguage::get_language() const
{
	return pango_attr_language()->value;
}

void
Pango::AttrLanguage::set_language(PangoLanguage *language)
{
	pango_attr_language()->value = language;
}

/*  Pango::AttrFamily
 */

Pango::AttrFamily::AttrFamily(const String& font_family)
: AttrString(pango_attr_family_new(font_family.c_str()), false)
{
}		

/*  Pango::AttrColor
 */

Pango::AttrColor::AttrColor(PangoAttribute *attribute, bool copy)
: Attribute(attribute, copy)
{
}

unsigned short
Pango::AttrColor::red() const
{
	return pango_attr_color()->color.red;
}

unsigned short
Pango::AttrColor::green() const
{
	return pango_attr_color()->color.green;
}

unsigned short
Pango::AttrColor::blue() const
{
	return pango_attr_color()->color.blue;
}

void
Pango::AttrColor::set_color(unsigned short red, unsigned short green, unsigned short blue)
{
	pango_attr_color()->color.red = red;
	pango_attr_color()->color.green = green;
	pango_attr_color()->color.blue = blue;
}

void
Pango::AttrColor::set_grey(unsigned short value)
{
	set_color(value, value, value);
}

void
Pango::AttrColor::set_red(unsigned short value)
{
	pango_attr_color()->color.red = value;
}

void
Pango::AttrColor::set_green(unsigned short value)
{
	pango_attr_color()->color.green = value;
}

void
Pango::AttrColor::set_blue(unsigned short value)
{
	pango_attr_color()->color.blue = value;
}

/*  Pango::AttrForeground
 */

Pango::AttrForeground::AttrForeground(unsigned short red, unsigned short green, unsigned short blue)
: AttrColor(pango_attr_foreground_new(red, green, blue), false)
{
}

/*  Pango::AttrBackground
 */

Pango::AttrBackground::AttrBackground(unsigned short red, unsigned short green, unsigned short blue)
: AttrColor(pango_attr_background_new(red, green, blue), false)
{
}

/*  Pango::AttrInt
 */

Pango::AttrInt::AttrInt(PangoAttribute *attribute, bool copy)
: Attribute(attribute, copy)
{
}

int
Pango::AttrInt::value() const
{
	return pango_attr_int()->value;
}

void
Pango::AttrInt::set_value(int value)
{
	pango_attr_int()->value = value;
}

/*  Pango::AttrSize
 */

Pango::AttrSize::AttrSize(int size)
: AttrInt(pango_attr_size_new(size), false)
{
}

/*  Pango::AttrStyle
 */

Pango::AttrStyle::AttrStyle(Style style)
: AttrInt(pango_attr_style_new((PangoStyle)style), false)
{
}

Pango::Style
Pango::AttrStyle::style() const
{
	return (Style)value();
}

void
Pango::AttrStyle::set_style(Style style)
{
	set_value((int)style);
}

/*  Pango::AttrWeight
 */

Pango::AttrWeight::AttrWeight(Weight weight)
: AttrInt(pango_attr_weight_new((PangoWeight)weight), false)
{
}

Pango::Weight
Pango::AttrWeight::weight() const
{
	return (Weight)value();
}

void
Pango::AttrWeight::set_weight(Weight weight)
{
	set_value((int)weight);
}

/*  Pango::AttrVariant
 */

Pango::AttrVariant::AttrVariant(Variant variant)
: AttrInt(pango_attr_variant_new((PangoVariant)variant), false)
{
}

Pango::Variant
Pango::AttrVariant::variant() const
{
	return (Variant)value();
}

void
Pango::AttrVariant::set_variant(Variant variant)
{
	set_value((int)variant);
}

/*  Pango::AttrStretch
 */

Pango::AttrStretch::AttrStretch(Stretch stretch)
: AttrInt(pango_attr_stretch_new((PangoStretch)stretch), false)
{
}

Pango::Stretch
Pango::AttrStretch::stretch() const
{
	return (Stretch)value();
}

void
Pango::AttrStretch::set_stretch(Stretch stretch)
{
	set_value((int)stretch);
}

/*  Pango::AttrUnderline
 */

Pango::AttrUnderline::AttrUnderline(Underline underline)
: AttrInt(pango_attr_underline_new((PangoUnderline)underline), false)
{
}

Pango::Underline
Pango::AttrUnderline::underline() const
{
	return (Underline)value();
}

void
Pango::AttrUnderline::set_underline(Underline underline)
{
	set_value((int)underline);
}

/*  Pango::AttrStrikethrough
 */

Pango::AttrStrikethrough::AttrStrikethrough(bool strikethrough)
: AttrInt(pango_attr_strikethrough_new(strikethrough), false)
{
}

bool
Pango::AttrStrikethrough::strikethrough() const
{
	return value() != 0;
}

void
Pango::AttrStrikethrough::set_strikethrough(bool strikethrough)
{
	set_value((int)strikethrough);
}

/*  Pango::AttrRise
 */

Pango::AttrRise::AttrRise(int rise)
: AttrInt(pango_attr_rise_new(rise), false)
{
}		

/*  Pango::AttrFontDesc
 */

Pango::AttrFontDesc::AttrFontDesc(const FontDescription& desc)
: Attribute(pango_attr_font_desc_new(desc.pango_font_description()), false)
{
}

Pointer<Pango::FontDescription>
Pango::AttrFontDesc::font_description() const
{
	return G::Boxed::wrap<FontDescription>(PANGO_TYPE_FONT_DESCRIPTION, pango_attr_font_desc()->desc);
}

void
Pango::AttrFontDesc::set_font_description(const FontDescription& desc)
{
	pango_font_description_free(pango_attr_font_desc()->desc);
	pango_attr_font_desc()->desc = pango_font_description_copy(desc.pango_font_description());
}

/*  Pango::AttrShape
 */

Pango::AttrShape::AttrShape(const Rectangle& ink_rect, const Rectangle& logical_rect)
: Attribute(pango_attr_shape_new(ink_rect.pango_rectangle(), logical_rect.pango_rectangle()), false)
{
}

const Pango::Rectangle&
Pango::AttrShape::ink_rect() const
{
	return reinterpret_cast<Rectangle&>(pango_attr_shape()->ink_rect);
}

const Pango::Rectangle&
Pango::AttrShape::logical_rect() const
{
	return reinterpret_cast<Rectangle&>(pango_attr_shape()->logical_rect);
}

void
Pango::AttrShape::set_rects(const Rectangle& ink_rect, const Rectangle& logical_rect)
{
	pango_attr_shape()->ink_rect = *ink_rect.pango_rectangle();
	pango_attr_shape()->logical_rect = *logical_rect.pango_rectangle();
}

void
Pango::AttrShape::set_ink_rect(const Rectangle& rect)
{
	pango_attr_shape()->ink_rect = *rect.pango_rectangle();
}

void
Pango::AttrShape::set_ink_rect(int x, int y, int width, int height)
{
	pango_attr_shape()->ink_rect.x = x;
	pango_attr_shape()->ink_rect.y = y;
	pango_attr_shape()->ink_rect.width = width;
	pango_attr_shape()->ink_rect.height = height;
}

void
Pango::AttrShape::set_logical_rect(const Rectangle& rect)
{
	pango_attr_shape()->logical_rect = *rect.pango_rectangle();
}

void
Pango::AttrShape::set_logical_rect(int x, int y, int width, int height)
{
	pango_attr_shape()->logical_rect.x = x;
	pango_attr_shape()->logical_rect.y = y;
	pango_attr_shape()->logical_rect.width = width;
	pango_attr_shape()->logical_rect.height = height;
}

/*  Pango::AttrFloat
 */

Pango::AttrFloat::AttrFloat(PangoAttribute *attribute, bool copy)
: Attribute(attribute, copy)
{
}

double
Pango::AttrFloat::value() const
{
	return pango_attr_float()->value;
}

void
Pango::AttrFloat::set_value(double value)
{
	pango_attr_float()->value = value;
}

/*  Pango::AttrScale
 */

Pango::AttrScale::AttrScale(double scale_factor)
: AttrFloat(pango_attr_scale_new(scale_factor), false)
{
}		

/*  Pango::AttrStipple
 */

Pango::AttrStipple::AttrStipple(Gdk::Bitmap& stipple)
: Attribute(gdk_pango_attr_stipple_new(stipple.gdk_bitmap()), false)
{
}

Gdk::Bitmap*
Pango::AttrStipple::stipple() const
{
	return G::Object::wrap<Gdk::Bitmap>(gdk_pango_attr_stipple()->stipple);
}

/*  Pango::AttrEmbossed
 */

Pango::AttrEmbossed::AttrEmbossed(bool embossed)
: Attribute(gdk_pango_attr_embossed_new(embossed), false)
{
}

bool
Pango::AttrEmbossed::embossed() const
{
	return gdk_pango_attr_embossed()->embossed != 0;
}

/*  Pango::AttrIterator
 */

Pango::AttrIterator::AttrIterator(PangoAttrIterator *iterator)
: iterator_(iterator)
{
}

Pango::AttrIterator::AttrIterator(const AttrIterator& src)
: iterator_(pango_attr_iterator_copy(src.pango_attr_iterator()))
{
}

Pango::AttrIterator&
Pango::AttrIterator::operator=(const AttrIterator& src)
{
	if (src.iterator_ == iterator_)
		return *this;

	if (iterator_)
		pango_attr_iterator_destroy(iterator_);

	iterator_ = pango_attr_iterator_copy(src.iterator_);
	return *this;
}

Pango::AttrIterator::~AttrIterator()
{
	if (iterator_)
	{
		pango_attr_iterator_destroy(iterator_);
		iterator_ = 0;
	}
}

int
Pango::AttrIterator::start_index() const
{
	int start;
	pango_attr_iterator_range(iterator_, &start, 0);
	return start;
}

int
Pango::AttrIterator::end_index() const
{
	int end;
	pango_attr_iterator_range(iterator_, 0, &end);
	return end;
}

void
Pango::AttrIterator::range(int *start, int *end) const
{
	pango_attr_iterator_range(iterator_, start, end);
}

Pointer<Pango::Attribute>
Pango::AttrIterator::get(AttrType type)
{
	return new Attribute(pango_attr_iterator_get(iterator_, (PangoAttrType)type), true);
}

void
Pango::AttrIterator::get_font(FontDescription& desc, PangoLanguage **language, std::vector<Pointer<Attribute> > *nonfont_attributes) const
{
	GSList *extra_attrs = 0;
	pango_attr_iterator_get_font(pango_attr_iterator(), desc.pango_font_description(), language, &extra_attrs);
	if (nonfont_attributes)
	{
		GSList *next = extra_attrs;
		while (next != 0)
		{
			Pointer<Attribute> tmp_attribute(new Attribute((PangoAttribute*)next->data, true));
			nonfont_attributes->push_back(tmp_attribute);
			next = g_slist_next(next);
		}
		g_slist_free(extra_attrs);
	}
}

bool 
Pango::AttrIterator::get_attrs(std::vector<Pointer<Attribute> >& attrs) const
{
	g_return_val_if_fail(attrs.empty(), false);
	GSList *first = pango_attr_iterator_get_attrs(pango_attr_iterator());
	GSList *next = first;

	while (next != 0)
	{
		Pointer<Attribute> tmp_attr(new Attribute((PangoAttribute*)next->data, false));
		attrs.push_back(tmp_attr);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !attrs.empty();
}

bool
Pango::AttrIterator::next()
{
	return pango_attr_iterator_next(iterator_);
}

/*  Pango::AttrList
 */

Pango::AttrList::AttrList()
: G::Boxed(PANGO_TYPE_ATTR_LIST, pango_attr_list_new(), false)
{
}

Pango::AttrList::AttrList(PangoAttrList *list)
: G::Boxed(PANGO_TYPE_ATTR_LIST, list)
{
}

Pango::AttrList::AttrList(PangoAttrList *list, bool copy)
: G::Boxed(PANGO_TYPE_ATTR_LIST, list, copy)
{
}

Pango::AttrList::AttrList(const AttrList& src)
: G::Boxed(PANGO_TYPE_ATTR_LIST, src.g_boxed(), true)
{
}

Pango::AttrList::~AttrList()
{
}

Pango::AttrList&
Pango::AttrList::operator=(const AttrList& src)
{
	if (src.boxed_ != boxed_)
		copy(src);
	return *this;
}

Pango::AttrList::operator PangoAttrList* () const
{
	return this ? pango_attr_list() : 0;
}

Pointer<Pango::AttrIterator>
Pango::AttrList::get_iterator() const
{
	return new AttrIterator(pango_attr_list_get_iterator(pango_attr_list()));
}

void
Pango::AttrList::insert(Attribute& attr, bool before)
{
	before ? pango_attr_list_insert_before(pango_attr_list(), attr) : pango_attr_list_insert(pango_attr_list(), attr);
}

void
Pango::AttrList::change(Attribute& attr)
{
	pango_attr_list_change(pango_attr_list(), attr.pango_attribute());
}

void
Pango::AttrList::splice(AttrList& other, int pos, int length)
{
	pango_attr_list_splice(pango_attr_list(), other.pango_attr_list(), pos, length);
}

namespace { // attr_filter_slot_callback

gboolean attr_filter_slot_callback(PangoAttribute *attr, gpointer data)
{
	Pango::AttrList::AttrFilterSlot *slot = static_cast<Pango::AttrList::AttrFilterSlot*>(data);
	Pointer<Pango::Attribute> tmp_attr(new Pango::Attribute(attr, true));
	return slot->call(*tmp_attr);
}

} // attr_filter_slot_callback

Pointer<Pango::AttrList> 
Pango::AttrList::filter(const AttrFilterSlot *slot)
{
	PangoAttrList *attr_list = pango_attr_list_filter(pango_attr_list(), &attr_filter_slot_callback, (void*)slot);
	return G::Boxed::wrap<AttrList>(PANGO_TYPE_ATTR_LIST, attr_list, false);
}

