/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gconf/schema.h
//! @brief GConfSchema C++ wrapper interface.

#ifndef INTI_GCONF_SCHEMA_H
#define INTI_GCONF_SCHEMA_H

#ifndef GCONF_GCONF_SCHEMA_H
extern "C"
{
#include <gconf/gconf-schema.h>
}
#endif

#ifndef INTI_GCONF_VALUE_H
#include <inti/gconf/value.h>
#endif

namespace Inti {

namespace GConf {

//! @class Schema schema.h inti/gconf/schema.h
//! @brief A GConfSchema C++ wrapper class.
//!
//! A <EM>schema</EM> is a value that describes a key-value pair. It might include the
//! type of the pair, documentation describing the pair, the name of the application
//! creating the pair, etc. Its primary purpose is to provide documentation about the
//! key to system administrators manipulating the GConf database. Secondarily, schemas
//! contain a good default value for the key; GConf will automatically return this value
//! when a key is unset. Schemas also include the name of the application that created
//! a key (useful when trying to clean old junk out of the database, for example).
//!
//! Schemas are normally installed from special schema description files; the gconftool
//! program knows how to read these and install the schemas into the GConf database. 
//! Normally, schemas are not installed by application code, though the interface for
//! doing so is a public part of the GConf API.

class Schema
{
	GConfSchema *schema_;
	
public:
//! @name Constructors
//! @{

	Schema();
	//! Construct a new Schema.

	Schema(ValueType type);
	//! Construct a new Schema for an entry of the specified type.

	Schema(const GConfSchema* src);
	//!< Construct a new Schema from an existing GConfSchema.

	Schema(const Schema& src);
	//!< Copy constructor.

	Schema& operator=(const Schema& src);
	//!< Assignment operator.

//! @}
//! @name Accessors
//! @{

	GConfSchema* gconf_schema() const { return schema_; }
	//!< Get a pointer to the GConfSchema.

	ValueType get_type() const;
	//!< Get the type of the described entry.

	ValueType get_list_type() const;
	//!< Get the list type of the described entry.

	ValueType get_car_type() const;
	//!< Get the Pair car type of the described entry.

	ValueType get_cdr_type() const;
	//!< Get the Pair cdr type of the described entry.

	String get_locale() const;
	//!< Get the Schema locale.

	String get_short_desc() const;
	//!< Get the short description of the entry.

	String get_long_desc() const;
	//!< Get the long description of the entry.

	String get_owner() const;
	//!< Get the name of creating application.

	Value get_default_value() const;
	//!< Get the default value of the key.

//! @}
//! @name Methods
//! @{

	void set_type(ValueType type);
	//!< Set the type of the described entry.
	//!< @param type The ValueType.

	void set_list_type(ValueType type);
	//!< Set the list type of the described entry.
	//!< @param type The ValueType.

	void set_car_type(ValueType type);
	//!< Set the Pair car type of the described entry.
	//!< @param type The ValueType.

	void set_cdr_type(ValueType type);
	//!< Set the Pair cdr type of the described entry.
	//!< @param type The ValueType.

	void set_locale(const String& locale);
	//!< Set the Schema locale.
	//!< @param locale The name of the locale.

	void set_short_desc(const String& desc);
	//!< Set the short description of the entry, 40 characters or less and no newlines.
	//!< @param desc A short description text.

	void set_long_desc(const String& desc);
	//!< Set the long description of the entry, could be a paragraph or so.
	//!< @param desc A long description text.

	void set_owner(const String& owner);
	//!< Set the name of creating application.

	void set_default_value(const Value& value);
	//!< Set the default value of the key.

//! @}
};

} // namespace GConf

} // namespace Inti

#endif // INTI_GCONF_SCHEMA_H

