/* 
 * $Id: ctkcombo.c,v 1.27 2000/07/25 05:50:00 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <arr.h>

#include "ctk.h"
#include "ctkcolor.h"

void ctk_combo_open(CtkCombo* combo)
{
	CtkWidget* self = CTK_WIDGET(combo);
	
	if (combo->open)
	      return;

	combo->open = TRUE;
	combo->wasDragged = FALSE;
	combo->doClose = FALSE;
	ctk_widget_show(combo->window);
	
	/* Minimum size the window */
	ctk_size_set_minimum(combo->window);
	
	/* Ok, now, we will try to position it vertically */
	if (combo->list->min_height + 2 < ctk_scr_r - (self->row + 1))
	{ /* If the thing fits below us, that's where it goes */
		combo->window->height = combo->list->min_height + 2;
		combo->window->row = self->row + 1;
	}
	else if (combo->list->min_height + 2 < ctk_scr_r)
	{ /* If the thing fits vertically, position at bottom */
		combo->window->height = combo->list->min_height + 2;		  	
		combo->window->row = ctk_scr_r - combo->window->height;
	}
	else
	{ /* It won't fit, so use whole screen and let the scrolled window work */
		combo->window->height = ctk_scr_r;
		combo->window->row = 0;
	}
	
	combo->window->width = self->width;
	combo->window->col   = self->col;

	ctk_window_set_focus(CTK_WINDOW(combo->window), combo->list);
	ctk_window_raise(CTK_WINDOW(combo->window));
}

gboolean ctk_combo_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkCombo* combo = CTK_COMBO(widget);

      /* Put focus on the entry so that is focused on popup close */
      ctk_window_set_focus(ctk_window_container(widget), CTK_COMBO(widget)->entry);
      ctk_combo_open(combo);

      return TRUE;
}

gboolean ctk_combo_drag(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      /* If the user dragged their mouse on the combo box, move selection */
      CtkCombo* combo = CTK_COMBO(widget);
      combo->wasDragged = TRUE;

      ctk_list_move_cursor(CTK_LIST(combo->list), event->dy);

      return TRUE;
}

gboolean ctk_combo_button_release(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      /* If the button was released after dragging, we're done. */
      CtkCombo* combo = CTK_COMBO(widget);

      if (combo->wasDragged)
      {
      	CTK_COMBO(widget)->doClose = TRUE;
	    ctk_signal_emit_by_name(CTK_OBJECT(combo->list), "select_row");
	  }

      return TRUE;
}

gboolean ctk_combo_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
      /* This allows keys sent to the combo to make it to the entry 
       */
      ctk_signal_emit_by_name(CTK_OBJECT(CTK_COMBO(widget)->entry), "key_press_event");

      return TRUE;
}

gboolean ctk_combo_entry_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
      /* This is bound onto the text entry widget to catch an up/down keypress
       */

      if ((event->keyval == AK_ARROW_DOWN) || (event->keyval == AK_ARROW_UP)) {
	    ctk_combo_open(CTK_COMBO(data));
	    return TRUE;
      }

      return FALSE;
}

gboolean ctk_combo_list_lost_focus(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
      /* If the list loses focus, it must die */
      CTK_COMBO(data)->doClose = TRUE;
      ctk_signal_emit_by_name(CTK_OBJECT(widget), "select_row");

      return TRUE;
}

gboolean ctk_combo_list_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	if ((event->keyval == AK_ENTER) || (event->keyval == ' ')) {
		CTK_COMBO(data)->doClose = TRUE;
		// We rely on the combo box handler to send the select_row to cause death
	}
	return FALSE;
}

gboolean ctk_combo_list_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
	CTK_COMBO(data)->doClose = TRUE;
	// We rely on the combo box handler to send the select_row of death
	return FALSE;
}

gboolean ctk_combo_list_select_row(CtkWidget* widget, gint row, gint col, 
				    CdkEventButton* event, gpointer data)
{
      /* If the list has a row selected, it must die */
      CtkCombo*  combo = CTK_COMBO(data);
      CtkWidget* item;
      CtkWidget* label;

      // FIXME: use list signals when they're done
      ctk_clist_get_widget(CTK_CLIST(widget), row, col, &item);
      
      if (item)
      {
            // FIXME: hack hack hack
            label = CTK_WIDGET(item->node->children->data);
            ctk_entry_set_text(CTK_ENTRY(combo->entry), CTK_LABEL(label)->text);
      
            if (!combo->open || !combo->doClose) 
     	          return TRUE;

	      ctk_assert(CTK_CHECK_TYPE(item, CtkTypeListItem), "Not a list item in this combo!");

	      if (CTK_CHECK_TYPE(label, CtkTypeLabel))
	      {
			ctk_entry_set_text(CTK_ENTRY(combo->entry), CTK_LABEL(label)->text);
		  }
	  }

      ctk_widget_hide(combo->window);
      combo->open = FALSE;

      return TRUE;
}

gpointer ctk_combo_destroy(CtkObject *object);

void ctk_combo_list_bind(CtkCombo* combo)
{
      ctk_signal_connect(CTK_OBJECT(combo->list), "focus_out_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_list_lost_focus), combo);
      ctk_signal_connect(CTK_OBJECT(combo->list), "key_press_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_list_key_press), combo);
      ctk_signal_connect(CTK_OBJECT(combo->list), "button_press_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_list_button_press), combo);
      ctk_signal_connect(CTK_OBJECT(combo->list), "select_row",
			 CTK_SIGNAL_FUNC(&ctk_combo_list_select_row), combo);
}

/* Initialize the Combo */
void ctk_combo_init(CtkCombo* combo)
{
      ctk_hbox_init(&combo->hbox, FALSE, 0);
      CTK_OBJECT(combo)->type = CtkTypeCombo;
	
      combo->entry = ctk_entry_new();
      ctk_widget_show(combo->entry);
	
      combo->window = ctk_window_new(CTK_WINDOW_TOPLEVEL);
      CTK_WINDOW(combo->window)->OPTIONS_MASK = 0;
//      combo->window->main_col       = ctk_calculate_palette(CTK_COLOR_BRIGHT_BLUE, CTK_COLOR_BLUE);
//      combo->window->background_col = ctk_calculate_palette(CTK_COLOR_BLUE, CTK_COLOR_BLUE);
      
      combo->scrolledwindow = ctk_scrolled_window_new(
	    CTK_ADJUSTMENT(ctk_adjustment_new(0, 0, 0, 5, 0, 0)),
	    CTK_ADJUSTMENT(ctk_adjustment_new(0, 0, 0, 5, 0, 0)));
      ctk_container_add(CTK_CONTAINER(combo->window), combo->scrolledwindow);
      ctk_widget_show(combo->scrolledwindow);
      
      combo->list = ctk_list_new();
      ctk_container_add(CTK_CONTAINER(combo->scrolledwindow), combo->list);
      ctk_widget_show(combo->list);
      
      combo->open          = FALSE;
      combo->ok_if_empty   = TRUE;
      combo->value_in_list = FALSE;
      
      ctk_box_pack_start(CTK_BOX(combo), combo->entry, TRUE, TRUE, 0);
      
      ((CtkWidget *)combo)->sensitive   = TRUE;
      ((CtkWidget *)combo)->orig_width  = 1;
      ((CtkWidget *)combo)->orig_height = 1;
      ((CtkWidget *)combo)->main_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
      ((CtkWidget *)combo)->selected_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
      ((CtkWidget *)combo)->inverse_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_RED);
      
      ((CtkObject *)combo)->destroy_func = ctk_combo_destroy;
      
      CTK_WIDGET(combo)->set_min_size  = &ctk_combo_min_size;
      CTK_WIDGET(combo)->set_real_size = &ctk_combo_real_size;
      
      ctk_signal_connect(CTK_OBJECT(combo), "button_press_event", 
			 CTK_SIGNAL_FUNC(&ctk_combo_button_press), NULL);
      ctk_signal_connect(CTK_OBJECT(combo), "ctk_drag",
			 CTK_SIGNAL_FUNC(&ctk_combo_drag), NULL);
      ctk_signal_connect(CTK_OBJECT(combo), "button_release_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_button_release), NULL);
      ctk_signal_connect(CTK_OBJECT(combo), "key_press_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_key_press), NULL);
      ctk_signal_connect(CTK_OBJECT(combo->entry), "key_press_event",
			 CTK_SIGNAL_FUNC(&ctk_combo_entry_key_press), combo);
      ctk_combo_list_bind(combo);
}

/* New Combo Box */
CtkWidget* ctk_combo_new(void)
{
	CtkCombo *combo;
	
	combo = g_malloc(sizeof(CtkCombo));
	
	ctk_combo_init(combo);

	return ((CtkWidget *)combo);
}

/* Set pop down strings for a combo box */
void ctk_combo_set_popdown_strings(CtkCombo* combo, GList* strings)
{
	GList* list;
	GList* item;
	
	ctk_widget_destroy(combo->list);

	combo->list = ctk_list_new();
	ctk_combo_list_bind(combo);
	ctk_container_add(CTK_CONTAINER(combo->scrolledwindow), combo->list);
	ctk_widget_show(combo->list);
	
	item = g_list_alloc();
	
	for (list = strings; list; list = list->next)
	{
		item->data = ctk_list_item_new_with_label((gchar*)list->data);
		
		if (list == strings)
			ctk_entry_set_text(CTK_ENTRY(combo->entry), (gchar*)list->data);
		
		ctk_widget_show(CTK_WIDGET(item->data));
		ctk_list_append_items(CTK_LIST(combo->list), item);
	}
	
	ctk_size_mark_changed(CTK_WIDGET(combo));
}

/* Set the value_in_list and ok_if_empty booleans */
void ctk_combo_set_value_in_list(CtkCombo* combo, 
				gboolean val,
				gboolean ok_if_empty)
{
	if (!combo)
	    return;
	
	combo->value_in_list = val;
	combo->ok_if_empty = ok_if_empty;
}

gpointer ctk_combo_destroy(CtkObject* object)
{
	CtkCombo* combo;
	
	combo = CTK_COMBO(object);
	ctk_widget_destroy(combo->window);

	return NULL;
}

/* Combo get minimum size */
void ctk_combo_min_size(CtkWidget* widget)
{
	ctk_table_min_size(widget);
	
	widget->min_width += 1;
}

/* Combo get minimum size */
void ctk_combo_real_size(CtkWidget* widget)
{
	widget->width--;
	widget->min_width--;
	ctk_table_real_size(widget);
	widget->width++;
	widget->min_width++;
}
