/* 
**  mod_filter.c -- Apache filter module
*/ 

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "ap_config.h"
#include "http_log.h"
#include "fnmatch.h"
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>



#define UNSET (-1)
#define OFF (0)
#define ON (1)
#define isOff(a) ((a == ON) ? 0 : 1 )
#define isOn(a) ((a == ON) ? 1 : 0 )
#define WATCHPOINT printf("WATCHPOINT %s %d\n", __FILE__, __LINE__);


typedef struct {
	int state;
	int header;
	int proxy;
	int html_handler;
	char *directory;
	table *types;
	table *uris_ignore;
} filter_conf;

typedef struct {
	size_t size;
	char *file;
} mmap_data;

module MODULE_VAR_EXPORT filter_module;

static void *create_dir_mconfig(pool *p, char *dir) {
	filter_conf *cfg;

	cfg = ap_pcalloc(p, sizeof(filter_conf));
	cfg->state = UNSET;
	cfg->header = UNSET;
	cfg->proxy = UNSET;
	cfg->html_handler = ON;
	cfg->directory = ap_pstrdup(p,"/tmp");
	cfg->types = ap_make_table(p, 8);
	cfg->uris_ignore = ap_make_table(p, 8);

	return (void *) cfg;
}

void filter_cleanup_file(void *data) {
	const char *filename = (const char *)data;
	//unlink(filename);
	printf("Wanting to clean up %s\n", filename);
 	
}

void cleanup_mmap(void *data) {
	mmap_data *file = (mmap_data *)data;
	munmap(file->file, file->size);
}

static void *merge_dir_mconfig(pool *p, void *origin, void *new) {
	filter_conf *cfg;
	filter_conf *cfg_origin = (filter_conf *)origin;
	filter_conf *cfg_new = (filter_conf *)new;

	cfg = ap_pcalloc(p, sizeof(filter_conf));
	cfg->directory = ap_pstrdup(p,"/tmp");
	cfg->types = ap_make_table(p, 8);
	cfg->uris_ignore = ap_make_table(p, 8);
	
	cfg->state = (cfg_new->state == UNSET) ? cfg_origin->state : cfg_new->state;
	cfg->header = (cfg_new->header == UNSET) ? cfg_origin->header : cfg_new->header;
	cfg->proxy = (cfg_new->proxy == UNSET) ? cfg_origin->proxy : cfg_new->proxy;


	if(strcmp(cfg_new->directory, "/tmp")){
		cfg->directory = ap_pstrdup(p, cfg_new->directory);
	} else if (strcmp(cfg_origin->directory, "/tmp")){
		cfg->directory = ap_pstrdup(p, cfg_origin->directory);
	}

	cfg->types = ap_overlay_tables(p, cfg_new->types, cfg_origin->types);
	cfg->uris_ignore = ap_overlay_tables(p, cfg_new->uris_ignore, cfg_origin->uris_ignore);

														
	return (void *) cfg;
}

int check_table(const char *a) {
  if (a == NULL)
		return 0;
	if('1' == a[0])
		return 1;

	return 0;
}

int table_find(const table * t, const char *key) {
  array_header *hdrs_arr = ap_table_elts(t);
	table_entry *elts = (table_entry *) hdrs_arr->elts;
	int i;

	if (key == NULL)
		return 0;

	for (i = 0; i < hdrs_arr->nelts; ++i) {
	if (!ap_fnmatch(elts[i].key, key, FNM_PATHNAME | FNM_CASE_BLIND))
		if(check_table(elts[i].val))
			return 1;
	}

	return 0;
}

void set_env(request_rec *r, request_rec *subr, const char *filtercache) {
	ap_table_set(subr->subprocess_env, "FILTER_SCRIPT_NAME", r->uri);
	if(r->path_info) 
		ap_table_set(subr->subprocess_env, "FILTER_INFO", r->path_info);
	if(r->args) 
		ap_table_set(subr->subprocess_env, "FILTER_QUERY_STRING", r->args);
	ap_table_set(subr->subprocess_env, "FILTER_CACHE", filtercache); 
}

static int call_main(request_rec *r, int assbackwards) {
	int status = OK;
	request_rec *subr;

	subr = (request_rec *) ap_sub_req_method_uri((char *) r->method, r->uri, r);

	subr->assbackwards = assbackwards; 
	subr->args = r->args;
	ap_bsetflag(subr->connection->client, B_CHUNK, 0);
	status = ap_run_sub_req(subr);
	ap_bflush(subr->connection->client);
	r->status_line = ap_pstrdup(r->pool, subr->status_line);
	r->status = subr->status;
	ap_destroy_sub_req(subr);

	return status;
}

static int call_container(request_rec *r, const char *uri, const char *filtercache, const char *content_length) {
	int status = OK;
	request_rec *subr;

	r->remaining = atoi(content_length);
	r->read_length = 0;
	r->read_chunked = 0;
	lseek(r->connection->client->fd_in, 0, SEEK_SET);
	subr = (request_rec *) ap_sub_req_method_uri("POST", uri, r);
	subr->assbackwards = 0;
/* 
 So you are asking, what is up with Content-Length? Well to make CGI's
 work we have to spoof it a bit. Namely, if Content-Length is set when
 mod_cgi runs, mod_cgi will try to read the request. Now if your CGI
 gets it contents through a POST method this of course is a no go since
 all of the contents will have already been read (and Apache will deadlock
 trying to read from a stream with no data in it. To get around this we
 spoof the content length till the original request runs 
*/
	set_env(r, subr, filtercache);

	ap_table_set(subr->headers_in, "Content-Length", content_length);
	status = ap_run_sub_req(subr);
	ap_bflush(subr->connection->client);
	ap_destroy_sub_req(subr);

	return status;
}

static int call_ssi(request_rec *r, const char *filtercache) {
	int status = OK;
	request_rec *subr;

	subr = (request_rec *) ap_sub_req_method_uri("GET", filtercache, r);
	subr->assbackwards = 0;
	ap_table_set(subr->headers_in, "Content-Length", "0");
	set_env(r, subr, filtercache);
	subr->filename = ap_pstrdup(subr->pool, filtercache);
	subr->handler = "server-parsed";
	/* This really should only be needed if SSI is the last of the called handlers*/
	subr->content_type = "text/html";
	/* We fake it */
	subr->finfo.st_mode = 1;
	status = ap_run_sub_req(subr);
	ap_bflush(subr->connection->client);
	ap_destroy_sub_req(subr);

	return status;
}

static int filter_fixup(request_rec *r) {
	filter_conf *cfg = ap_get_module_config(r->per_dir_config, &filter_module);
	request_rec *subr;
	const char *type = NULL;
	const char *handler = NULL;
	int length;

	if (cfg->state < ON) {
		return DECLINED;
	}
	if (r->main) {
		return DECLINED;
	}

	/* If this is a HEAD only, we really don't need to involve ourselves. */
	if (r->header_only) {
		return DECLINED;
	}

/* Now, lets fix it if some goof has called a URL that 
   should have had a / in it */
	if(ap_is_directory(r->filename)) {
		if(r->uri[0] == '\0' || r->uri[strlen(r->uri) - 1] != '/') {
/* Now at this point we know things are not going to
   go over well, so lets just let it all die in some module
   designed to take care of this sort of thing */
			return DECLINED;
		}
	}


	/* So why switch to doing this? Somewhere since 1.3.6 something
		 has changed about the way that CGI's are done. Not sure what
		 it is, but this is now needed */
	/* First, we check to see if this is SSI, mod_perl or cgi */
	if(r->handler) {
		type = ap_pstrdup(r->pool, r->handler);
	} else {
		type = ap_pstrdup(r->pool, r->content_type);
	}

	/* If we support proxies, we need to do a special
		 lookups.
	*/
	if ((cfg->proxy > OFF) && (r->proxyreq || !strcmp(type,"proxy-server"))) {
		length = strlen(r->uri);
/* This is a guess. */
		if(r->uri[length - 1] == '/' ) {
			type = "text/html";
		} else {
			subr = (request_rec *) ap_sub_req_lookup_file(r->uri, r);
			type = ap_pstrdup(r->pool, subr->content_type);
		}
#ifdef DEBUG
		printf("MIME %s\nHandler %s\nProxy %d\nURI %s\nTYPE %s\n", r->handler, r->content_type, r->proxyreq, r->uri, type);
#endif
	}

	if (handler = ap_table_get(cfg->types, type)){
		if(strcmp(handler, "OFF")) {
			ap_table_set(r->notes, "FILTER_URI", handler);
		} else {
			return DECLINED;
		}
	} else {
		return DECLINED;
	}

	/* Now we exclude */
	if (table_find(cfg->uris_ignore, r->uri))
		return DECLINED;

	ap_table_set(r->notes, "FILTER_TYPE", type);
	r->handler = "filter";

	return DECLINED;
}

int get_fd_out(request_rec *r, char *filename, BUFF *buff) {
	int status = OK;
	int temp_fd = 0;

	if ((temp_fd = ap_popenf(r->pool, filename,O_WRONLY|O_CREAT|O_TRUNC,S_IRWXU)) == -1) {
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		"mod_filter couldn't create a file descriptor at HTTP : %s", filename);

		return HTTP_INTERNAL_SERVER_ERROR;
	}
	WATCHPOINT
	ap_register_cleanup(r->pool, filename, filter_cleanup_file, ap_null_cleanup);

	buff->fd = temp_fd;

	return status;
}

int check_type(const char *type) {
	if(!strcmp(type, "text/plain"))
		return 1;
	if(!strcmp(type, "text/html"))
		return 1;
	return 0;
}

int send_file(request_rec *r, char *filename) {
	int status = OK;
	int temp_fd = 0;
	mmap_data *map_data = NULL;
	struct stat sbuf;


	if ((temp_fd = ap_popenf(r->pool, filename,O_RDONLY,S_IRWXU)) < 0) {
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		"mod_filter couldn't open a file descriptor for : %s", filename);

		return HTTP_NOT_FOUND;
	}
	(void)fstat(temp_fd, &sbuf);
	map_data = ap_pcalloc (r->pool, sizeof (mmap_data));
	map_data->size = sbuf.st_size;
	map_data->file = (char *)mmap(NULL, map_data->size, PROT_READ, MAP_PRIVATE, temp_fd, 0);
	ap_register_cleanup(r->pool, map_data, cleanup_mmap, ap_null_cleanup);
	ap_send_mmap(map_data->file, r, 0, map_data->size);
	ap_rflush(r);

	return status;
}

static int filter_handler(request_rec *r) {
	int status=0;
	int temp_fd, fd_out;
	int assbackwards;
	char string[HUGE_STRING_LEN];
	char *filename = NULL;
	const char *handler = NULL;
	const char *content_length = NULL;
	const char *type = NULL;
	filter_conf *cfg;
	struct stat sbuf;
	BUFF *buff = NULL;
	BUFF *obuff = NULL;

	if (r->main) {
		return DECLINED;
	}

	if (r->header_only) {
		ap_send_http_header(r);

		return OK;
	}

	ap_table_setn(r->headers_out, "ModFilter", "1.4");
	cfg = ap_get_module_config(r->per_dir_config, &filter_module);
/* Logic is reversed for assbackwards 
   One of these days I am going to ask why this 
   variable is named this. 
*/
	if (cfg->header == ON) {
		assbackwards = 0;
	} else {
		assbackwards = 1;
	}
	handler = ap_table_get(r->notes, "FILTER_URI");
	type = ap_table_get(r->notes, "FILTER_TYPE");
	if(isOn(cfg->html_handler) && check_type(type)) {
		WATCHPOINT
		filename = r->filename;
	} else {
		WATCHPOINT
		buff = ap_bcreate(r->pool, B_RDWR);
		obuff = r->connection->client;
		buff->fd_in = r->connection->client->fd_in;
		buff->incnt = r->connection->client->incnt;
		buff->inptr = r->connection->client->inptr;

		filename = ap_psprintf(r->pool, "%s/.mod_filter.%d", cfg->directory,  r->connection->child_num);

		if((status = get_fd_out(r, filename, buff)) != OK) {
			return status;
		}
		r->connection->client = buff;

		status = call_main(r, assbackwards);
		ap_rflush(r);

		buff->fd_in = -1;
		ap_bclose(buff);
		r->connection->client = obuff;
		if(status != OK) {
			send_file(r, filename);
			return status;
		}
	}

	if ((temp_fd = ap_popenf(r->pool, filename, O_RDONLY, S_IRWXU)) < 0) {
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r, 
		"Bad mojo, mod_filter couldn't open file : %s(%s)", filename, strerror(errno));

		return HTTP_NOT_FOUND; 
	}
	if(fstat(temp_fd, &sbuf)) {
		/* This would be very bad */
		status = HTTP_INTERNAL_SERVER_ERROR;
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r, "fstat blew chunks in mod_filter: %d", status);
		return status;
	}
	content_length = ap_psprintf(r->pool, "%d", sbuf.st_size);
	r->connection->client->fd_in = temp_fd;

/*****************************************************************/
	/* This is where we decide which file to run */

	if(strcmp(handler, "SSIFILTER")) {
		if ((status = call_container(r, handler, filename, content_length)) != OK) {
			ap_log_rerror(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r, "The following error occured while processing the filter : %d", status);
			return status;
		}
	} else {
		if ((status = call_ssi(r, filename)) != OK) {
			ap_log_rerror(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r, "The following error occured while processing the filter : %d", status);
			return status;
		}
	}
/*****************************************************************/

	return OK;
}

static const char *add_filter(cmd_parms * cmd, void *mconfig, char *mime_type, char *filter) {
  filter_conf *cfg = (filter_conf *) mconfig;

	ap_table_set(cfg->types, mime_type, filter);

	return NULL;
}

static const char *ignore_uri(cmd_parms * cmd, void *mconfig, char *uri) {
  filter_conf *cfg = (filter_conf *) mconfig;

	ap_table_set(cfg->uris_ignore, uri, "1");

	return NULL;
}

/* Dispatch list of content handlers */
static const handler_rec filter_handlers[] = { 
    { "filter", filter_handler }, 
    { NULL, NULL }
};

static const command_rec filter_cmds[] = {
	{"Filter", add_filter, NULL, OR_ALL, TAKE2, "Takes two parameters, the mime type/handler and the uri to call on it."},
	{"FilterEngine", ap_set_flag_slot, (void *) XtOffsetOf(filter_conf, state), OR_ALL, FLAG, "This can either be On or Off (default it Off)."}, 
	{"FilterProxy", ap_set_flag_slot, (void *) XtOffsetOf(filter_conf, proxy), OR_ALL, FLAG, "Enable support of filtering mod_proxy(default it Off)."}, 
	{"FilterHeader", ap_set_flag_slot, (void *) XtOffsetOf(filter_conf, header), OR_ALL, FLAG, "This can either be On or Off (default it Off)."}, 
	{"FilterHTMLHandler", ap_set_flag_slot, (void *) XtOffsetOf(filter_conf, html_handler), OR_ALL, FLAG, "This can either be On or Off (default it On). This makes use of the internal html handler."}, 
	{"FilterCache", ap_set_string_slot, (void *) XtOffsetOf(filter_conf, directory), OR_ALL, TAKE1, "Change the default directory from /tmp."},
	{"FilterIgnore", ignore_uri, NULL, OR_ALL, TAKE1, "Change the default directory from /tmp."},
	{NULL},
};

static void filter_init(server_rec * s, pool * p) {
	/* Tell apache we're here */
	ap_add_version_component("mod_filter/1.4");
}

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT filter_module = {
    STANDARD_MODULE_STUFF, 
    filter_init,           /* module initializer                  */
    create_dir_mconfig,    /* create per-dir    config structures */
    merge_dir_mconfig,     /* merge  per-dir    config structures */
    NULL,                  /* create per-server config structures */
    NULL,                  /* merge  per-server config structures */
    filter_cmds,           /* table of config file commands       */
    filter_handlers,       /* [#8] MIME-typed-dispatched handlers */
    NULL,                  /* [#1] URI to filename translation    */
    NULL,                  /* [#4] validate user id from request  */
    NULL,                  /* [#5] check if the user is ok _here_ */
    NULL,                  /* [#3] check access by host address   */
    NULL,                  /* [#6] determine MIME type            */
    filter_fixup,          /* [#7] pre-run fixups                 */
    NULL,                  /* [#9] log a transaction              */
    NULL,                  /* [#2] header parser                  */
    NULL,                  /* child_init                          */
    NULL,                  /* child_exit                          */
    NULL                   /* [#0] post read-request              */
};

