/* $Id: offline.c,v 1.11 2001/07/16 11:38:45 linus Exp $
 *
 * Copyright (C) 1999  Lysator Computer Club,
 *                     Linkoping University, Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author: Linus Tolke
 */

#include <config.h>

#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include <sys/param.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef HAVE_CUSERID
#include <stdio.h>
#endif
#include <errno.h>

#include <kom-errno.h>

#include <misc-parser.h>
#include <parse.h>
#include <zmalloc.h>

#include "offline.h"
#include "xoutput.h"
#include "copy.h"
#include "error.h"
#include "cache.h"
#include "quit.h"
#include "main.h"

#define Export

/* Static functions prototypes: */
static FILE * open_cache_file(const char * kind, unsigned long nr, const char * rw);
static int unlink_cache_file(const char * kind, unsigned long nr);
static void cache_file_name(char * path, const char * kind, unsigned long nr);


/*
 * This file contains commands for offline reading.
 * Offline reading is a reduced form of reading when it is only possible to:
 * - List conferences with unread.
 * - Go to a conference that is downloaded.
 * - Read unread articles.
 * - Reread articles that is already downloaded.
 * - Mark an article.
 * - Comment articles.
 * - Write a new article in a downloaded conference.
 * - Write a letter to a downloaded person.
 *
 * Normally we work like this:
 *  1. Connect to the remote server (dialup...)
 *  2. Start the client and login.
 *  3. Issue a download command and wait for completion.
 *  4. Exit the client.
 *  5. Disconnect from the remote server (hangup...)
 *  6. Enter the client in offline-reading mode.
 *  7. Read and comment.
 *  8. Exit the client.
 *  9. Connect to the remote server (dialup...)
 * 10. Start the client and login.
 * 11. Issue an upload command and wait for completion.
 * 12. Exit the client.
 * 13. Disconnect from the remote server (hangup...)
 */

/* 
 * The offline mechanism requires a database that saves articles, 
 * persons and commands to disk for later processing.
 * The files of this database are:
 * $HOME/.lyskom-tty-offline-HOST:PORT/confstat-NR
 * $HOME/.lyskom-tty-offline-HOST:PORT/map-NR (includes timestamp).
 * $HOME/.lyskom-tty-offline-HOST:PORT/membership
 * $HOME/.lyskom-tty-offline-HOST:PORT/text-NR
 * $HOME/.lyskom-tty-offline-HOST:PORT/textstat-NR
 * $HOME/.lyskom-tty-offline-HOST:PORT/command-NR
 *
 * Alternativ r /tmp/<cuserid>-lyskom-tty-offline-HOST:PORT/.
 *
 * Commands are:
 * - mark as read
 * - create text (in conference, comment to, footnote to...)
 * - mark/unmark
 *
 * The database is implemented by the store_* and fetch_* functions here.
 */
static
void cache_file_name(char * path, const char * kind, unsigned long nr)
{
    static char basic_path[MAXPATHLEN];
    static int basic_path_tested = 0;

    if (!basic_path_tested)
    {
	char * home;
	int alternative;

	for (alternative = 1; alternative < 4; alternative++)
	{
	    switch (alternative)
	    {
	    case 1:
		home = getenv("HOME");
		if (home != NULL)
		{
		    sprintf(basic_path, "%s/.lyskom-tty-offline-%s:%d", 
			    home, server_name, portno);
		    break;
		}
		continue;

	    case 2:
	        /*
		 * There is a problem on cygwin is to have directories 
		 * with : in them. Lets give cygwin systems an alternative.
		 */
		home = getenv("HOME");
		if (home != NULL)
		{
		    sprintf(basic_path, "%s/.lyskom-tty-offline-%s-%d", 
			    home, server_name, portno);
		    break;
		}
		continue;

	    case 3:
		sprintf(basic_path, "/tmp/%s-lyskom-tty-offline-%s:%d",
			getlogin(),
			server_name, portno);
		break;
	    }

	    if (mkdir(basic_path, 0700) == 0
		|| errno == EEXIST)
	    {
		/* Ok! */
		/* Lets create a file and see if that works. */
		FILE * testfp = NULL;

		sprintf(path, "%sTESTING", basic_path);
		if ((testfp = fopen(path, "w")) == NULL)
		    continue;

		fclose(testfp);
		unlink(path);

		basic_path_tested = 1;
		break;
	    }
	}
	
	if (!basic_path_tested)
	{
	    /* We could not find any working alternative. */
	    fprintf(stderr, "Cannot find or create a working cache directory.\n");
	    abort();
	}
    }

    /*
     * The first try was the following:
     * sprintf(path, "%s/%s-%ld", basic_path, kind, nr);
     *
     * This had a problem because it is an extremely flat structure that
     * easily can cause problems already noticed at ~ 40000 entries.
     * Some file systems have problems already at 1000 entries in the same
     * directory.
     *
     * Theory for the second try:
     * Since every entry is a name and a number, lets build a hash mechanism
     * based on the numbers. The goal is to split the files more or less
     * evenly over the directories. Since the numbers currently range from
     * 0 - 5000000 and the number of entries probably in the range of 
     * 100000 - 5000000 lets do this as a fixed hash with two levels of 
     * directories.
     * The first level the directory name is nr%100.
     * The second level the directory name is (nr/100)%100.
     */
    sprintf(path, "%s/%02ld/%02ld/%s-%ld", 
	    basic_path, 
	    nr%100, (nr/100)%100, 
	    kind, nr);
}


static
FILE *
open_cache_file(const char * kind, unsigned long nr, const char * rw)
{
#define FUNCTION "open_cache_file"
    char path[MAXPATHLEN];

    cache_file_name(path, kind, nr);

    if (strcmp(rw, "w") == 0
	|| strcmp(rw, "a") == 0)
    {
	/*
	 * Make sure that the directories in the path (two last levels of 
	 * directories only) are created or create them.
	 */
	char copy[MAXPATHLEN];
	char * ep2;
	char * ep1;

	strcpy(copy, path);
	ep2 = strrchr(copy, '/');
	*ep2 = '\0';
	ep1 = strrchr(copy, '/');
	*ep1 = '\0';
	if (mkdir(copy, 0700) == 0
	    || errno == EEXIST)
	{
	    /* Ok created. */
	}
	else
	{
	    fprintf(stderr, "Cannot find or create a cache subdirectory.\n");
	    abort();
	}

	*ep1 = '/';
	if (mkdir(copy, 0700) == 0
	    || errno == EEXIST)
	{
	    /* Ok created. */
	}
	else
	{
	    fprintf(stderr, "Cannot find or create a cache subdirectory.\n");
	    abort();
	}
    }
    else if (strcmp(rw, "r") == 0)
    {
	/*
	 * Don't do anything.
	 * We will get an ENOENT whether it is the file that does not exist
	 * or a directory in the path.
	 */
    }
    else
    {
	fatal2(CLIENT_SHOULDNT_HAPPEN,
	       "Internal error. Opening type %s is not supported.\n", rw);
	/*NOTREACHED*/
    }

    return fopen(path, rw);
#undef FUNCTION
}

static
int
unlink_cache_file(const char * kind, unsigned long nr)
{
    char path[MAXPATHLEN];

    cache_file_name(path, kind, nr);

    return unlink(path);
}


/* Routines to write different objects. */
void
fprintf_Time(FILE * fp, const Time * t)
{
    fprintf(fp, "%u %u %u %u %u %u %u %u %u",
	    t->tm_sec,
	    t->tm_min,
	    t->tm_hour,
	    t->tm_mday,
	    t->tm_mon,
	    t->tm_year,
	    t->tm_wday,
	    t->tm_yday,
	    t->tm_isdst);
}

void
fprintf_Membership(FILE * fp, const Membership * ms)
{
    fprintf_Time(fp, &ms->last_time_read);
    fprintf(fp, " %u %u ", ms->conf_no, ms->priority);
    fprintf(fp, "%lu %u ", ms->last_text_read, ms->no_of_read);
    if (ms->no_of_read > 0)
    {
	int i;

	fprintf(fp, "{ ");
	for (i = 0; i < ms->no_of_read; i++)
	{
	    fprintf(fp, "%lu ", ms->read_texts[i]);
	}
	fprintf(fp, "}");
    }
    else
	fprintf(fp, "*");
}

void
fprintf_Membership_list(FILE * fp, const Membership_list * ml)
{
    int i;

    fprintf(fp, "%u { ", ml->no_of_confs);
    for (i = 0; i < ml->no_of_confs; i++)
    {
	fprintf_Membership(fp, &ml->confs[i]);
	fprintf(fp, " ");
    }
    fprintf(fp, "}");
}

void
fprintf_String(FILE * fp, const String * s)
{
    fprintf(fp, "%luH", s->len);
    s_fputs(fp, *s);
}

void
fprintf_Conf_type(FILE * fp, const Conf_type * ct)
{
    fprintf(fp, "%1u%1u%1u%1u",
	    ct->rd_prot,
	    ct->original,
	    ct->secret,
	    ct->letter_box);
}

void
fprintf_Conference(FILE * fp, const Conference * cs)
{
    fprintf_String(fp, &cs->name);
    fprintf(fp, " ");
    fprintf_Conf_type(fp, &cs->type);
    fprintf(fp, " ");
    fprintf_Time(fp, &cs->creation_time);
    fprintf(fp, " ");
    fprintf_Time(fp, &cs->last_written);
    fprintf(fp, " %u %lu %u %u %u %lu %lu %u %lu %lu",
	    cs->creator,
	    cs->presentation,
	    cs->supervisor,
	    cs->permitted_submitters,
	    cs->super_conf,
	    cs->msg_of_day,
	    cs->nice,
	    cs->no_of_members,
	    cs->first_local_no,
	    cs->no_of_texts);
}


void
fprintf_Text_list(FILE * fp, const Text_list * tl)
{
    fprintf(fp, "%lu %lu ", tl->first_local_no, tl->no_of_texts);
    if (tl->no_of_texts > 0)
    {
        unsigned int i;

	fprintf(fp, "{ ");
	for (i = 0; i < tl->no_of_texts; i++)
	    fprintf(fp, "%lu ", tl->texts[i]);
	fprintf(fp, "}");
    }
    else
	fprintf(fp, "*");
}


void
fprintf_Misc_infos(FILE * fp, 
		   unsigned short no_of_miscs, const Misc_info * misc_items)
{
    fprintf(fp, "%u ", no_of_miscs);

    if (no_of_miscs > 0)
    {
        int i;

	fprintf(fp, "{ ");

	for (i = 0; i < no_of_miscs; i++)
	{
	    switch (misc_items[i].type)
	    {
	    case recpt:
		fprintf(fp, "%u %u ", 
			misc_items[i].type, 
			misc_items[i].datum.recipient);
		break;
	    case cc_recpt:
		fprintf(fp, "%u %u ", 
			misc_items[i].type, 
			misc_items[i].datum.cc_recipient);
		break;
	    case loc_no:
		fprintf(fp, "%u %lu ", 
			misc_items[i].type, 
			misc_items[i].datum.local_no);
		break;
	    case rec_time:
		fprintf(fp, "%u ", misc_items[i].type);
		fprintf_Time(fp, &misc_items[i].datum.received_at);
		fprintf(fp, " ");
		break;
	    case comm_to:
		fprintf(fp, "%u %lu ", 
			misc_items[i].type, 
			misc_items[i].datum.comment_to);
		break;
	    case comm_in:
		fprintf(fp, "%u %lu ", 
			misc_items[i].type, 
			misc_items[i].datum.commented_in);
		break;
	    case footn_to:
		fprintf(fp, "%u %lu ", 
			misc_items[i].type, 
			misc_items[i].datum.footnote_to);
		break;
	    case footn_in:
		fprintf(fp, "%u %lu ", 
			misc_items[i].type, 
			misc_items[i].datum.footnoted_in);
		break;
	    case sent_by:
		fprintf(fp, "%u %u ", 
			misc_items[i].type, 
			misc_items[i].datum.sender);
		break;
	    case sent_at:
		fprintf(fp, "%u ", misc_items[i].type);
		fprintf_Time(fp, &misc_items[i].datum.sent_at);
		fprintf(fp, " ");
		break;
	    default:
		fprintf(stderr, "Unknown text-stat type: %u\n", 
			misc_items[i].type);
		break;
	    }
	}
	fprintf(fp, "}");
    }
    else
	fprintf(fp, "*");
}

/* Routines to store and fetch. */
int
store_person_number(Pers_no pers)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(1))
	xprintf("store_person_number\n");
#endif

    if ((fp = open_cache_file("person", 0, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf(fp, "%u", pers);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

int
fetch_person_number(Pers_no * pers)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_person_number\n");
#endif
    
    if ((fp = open_cache_file("person", 0, "r")) == NULL)
    {
	return FAILURE;
    }

    *pers = parse_long(fp);
    fclose(fp);
    return *pers != 0;
}


int
store_Membership_list(Membership_list * ml)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(1))
	xprintf("store_Membership_list\n");
#endif

    if ((fp = open_cache_file("membership", 0, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf_Membership_list(fp, ml);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

int
fetch_Membership_list(Membership_list * ml)
{
    int res;
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_Membership_list\n");
#endif
    
    if ((fp = open_cache_file("membership", 0, "r")) == NULL)
    {
	return FAILURE;
    }

    res = parse_membership_list(fp, ml);
    fclose(fp);
    return res;
}

int
store_Conference(Conf_no cno, Conference * cs)
{
    FILE * fp;
#ifdef TRACE
    if (TRACE(1))
	xprintf("store_Conference %ld\n", cno);
#endif

    if ((fp = open_cache_file("confstat", cno, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf_Conference(fp, cs);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

int
fetch_Conference(Conf_no cno, Conference * cs)
{
    int res;
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_Conference %ld\n", cno);
#endif

    if ((fp = open_cache_file("confstat", cno, "r")) == NULL)
    {
	return FAILURE;
    }

    res = parse_conference(fp, cs);
    fclose(fp);
    return res;
}

int
store_map(Conf_no cno, Text_list * tl, Time * ts)
{
    FILE * fp;
#ifdef TRACE
    if (TRACE(1))
	xprintf("store_map %ld\n", cno);
#endif

    if ((fp = open_cache_file("map", cno, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf_Text_list(fp, tl);
    fprintf(fp, "\n");
    fprintf_Time(fp, ts);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}
int
fetch_map(Conf_no cno, Text_list * tl, Time * ts)
{
    int res;
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_map %ld\n", cno);
#endif

    if ((fp = open_cache_file("map", cno, "r")) == NULL)
    {
	return FAILURE;
    }

    res = parse_text_list(fp, tl);
    if (res == FAILURE)
    {
	fclose(fp);
	return FAILURE;
    }
    res = parse_time(fp, ts);
    fclose(fp);
    return res;
}

int
store_text(Text_no tno, String * tm)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(1))
	xprintf("store_text %ld\n", tno);
#endif

    if ((fp = open_cache_file("text", tno, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf_String(fp, tm);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

int
fetch_text(Text_no tno, String * tm)
{
    int res;
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_text %ld\n", tno);
#endif

    if ((fp = open_cache_file("text", tno, "r")) == NULL)
    {
	return FAILURE;
    }

    res = parse_string(fp, tm);
    fclose(fp);
    return res;
}

int
store_Text_stat(Text_no tno, Text_stat * ts)
{
    FILE * fp;
#ifdef TRACE
    if (TRACE(1))
	xprintf("store_Text_stat %ld\n", tno);
#endif

    if ((fp = open_cache_file("textstat", tno, "w")) == NULL)
    {
	return FAILURE;
    }

    fprintf_Time(fp, &ts->creation_time);
    fprintf(fp, " %u %u %lu %u ",
	    ts->author,
	    ts->no_of_lines,
	    ts->no_of_chars,
	    ts->no_of_marks);

    fprintf_Misc_infos(fp, ts->no_of_misc, ts->misc_items);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

int
fetch_Text_stat(Text_no tno, Text_stat * ts)
{
    int res;
    FILE * fp;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_Text_stat %ld\n", tno);
#endif

    if ((fp = open_cache_file("textstat", tno, "r")) == NULL)
    {
	return FAILURE;
    }

    res = parse_text_stat(fp, ts);
    fclose(fp);
    return res;
}

/* ================================================================
 * Here are functions that look exactly like the access functions
 * above. The difference is that they are to keep track of what we
 * have read.
 * ================================================================
 */
/*
 * store_mark_in_conf appends to the file.
 */
int
store_mark_in_conf(Conf_no cno, Local_text_no ltno)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(1))
	xprintf("store_mark_in_conf\n");
#endif

    if ((fp = open_cache_file("marks-in-conf", cno, "a")) == NULL)
    {
	return FAILURE;
    }

    fprintf(fp, "%lu", ltno);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}

/*
 * fetch_marks_in_conf
 * Returns a list of all texts to be marked as read in this conference.
 *
 * If the file is present an area will be zmalloc:ed containing the result.
 * It will be stored in *LTNOS that has to be zfree:d by the caller.
 * If the file is not present the area will not me zmalloc:ed and the
 * routine returns FAILURE.
 */
int
fetch_marks_in_conf(Conf_no cno, int * len, Local_text_no ** ltnos)
{
    FILE * fp;
    int maxlen = 100;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_marks_in_conf\n");
#endif
    
    *len = 0;

    if ((fp = open_cache_file("marks-in-conf", cno, "r")) == NULL)
    {
	return FAILURE;
    }

    *ltnos = zmalloc(sizeof(Local_text_no) * maxlen);

    while (((*ltnos)[*len] = parse_long(fp)))
    {
	(*len) ++;

	if (*len >= maxlen)
	{
	    maxlen *= 2;
	    *ltnos = zrealloc(*ltnos, sizeof(Local_text_no) * maxlen);
	}
    }

    fclose(fp);
    return OK;
}

/*
 * done_with_marks_in_conf
 */
int
done_with_marks_in_conf(Conf_no cno)
{
    if (unlink_cache_file("marks-in-conf", cno) < 0)
    {
        return FAILURE;
    }

    return OK;
}


/* ================================================================
 * Here are functions that keep track of the texts that we write.
 *
 * The numbers of these created texts are generated from 1 and up.
 * The texts are then always handled in the way that they are numbered.
 * If we find that a number is "used" we go to the next one.
 * In the odd case that first there are some "unused" numbers, then
 * there are some used the order in which the texts are sent in when
 * uploading will then not correspond with the order in which they are
 * written.
 * We don't try very hard in keeping track of what is the last text.
 * Instead we search for the texts in the number sequence. If there is a 
 * whole in the number sequence, then we might miss the texts after the
 * whole if the whole is big enough (100) numbers. 
 */
static int stored_created_text_index = 1;
/*
 * store_created_text
 */
int
store_created_text(const String message,
		   unsigned short no_of_miscs,
		   const Misc_info * misc)
{
    FILE * fp;

#ifdef TRACE
    if (TRACE(1))
	xprintf("store_created_text\n");
#endif

    /* Finding files that are in the way. */
    while ((fp = open_cache_file("created-text", stored_created_text_index, "r"))
	   != NULL)
    {
	/* The file exists already. Goto the next number. */
	fclose(fp);
	stored_created_text_index++;
    }

    if ((fp = open_cache_file("created-text", stored_created_text_index, "w"))
	== NULL)
    {
	return FAILURE;
    }

    fprintf_Misc_infos(fp, no_of_miscs, misc);
    fprintf(fp, "\n");
    fprintf_String(fp, &message);
    fprintf(fp, "\n");
    fclose(fp);
    return OK;
}


/*
 * Returns the index number of the text.
 * If it fails, it returns 0.
 */
static int fetched_created_text_index = 1;
int
fetch_created_text(String * message,
		   unsigned short * no_of_miscs,
		   Misc_info ** misc)
{
    FILE * fp;
    int try_index = fetched_created_text_index;

#ifdef TRACE
    if (TRACE(2))
	xprintf("fetch_created_text\n");
#endif

    /* Try for texts. */
    while ((fp = open_cache_file("created-text", try_index, "r")) == NULL
	   && try_index < fetched_created_text_index + 100)
    {
	try_index++;
    }

    if (fp == NULL)
    {
	return 0;
    }

    fetched_created_text_index = try_index + 1;

    if (parse_Misc_infos(fp, no_of_miscs, misc) == OK
	&& parse_string(fp, message) == OK)
    {
	/* All went well */
    }
    else
    {
	try_index = 0;
    }
    fclose(fp);
    return try_index;
}


/*
 * done_with_created_text
 */
int
done_with_created_text(int ind)
{
    if (unlink_cache_file("created-text", ind) < 0)
    {
        return FAILURE;
    }

    return OK;
}


/* ================================================================
 * Functions that "replace" complex functions.
 *
 * ================================================================
 */
static int		memb_list_read = 0;
static Membership_list	memb_list = EMPTY_MEMBERSHIP_LIST_i;
static int		get_unread_confs_pos;

/*
 * offline_get_unread_confs_start
 *
 * Initiate the reading of the membership.
 */
int
offline_get_unread_confs_start(void)
{
  /*BUG: Inte bra! Mste lsa saved_membership_list frn cache.c */ 
    if (!memb_list_read)
    {
	if (fetch_Membership_list(&memb_list) != OK)
	{
	    return 0;
	}
    }

    get_unread_confs_pos = -1;
    return 1;
}

/*
 * offline_get_unread_confs_next
 *
 * Returns the conf no of the next conference with unread news.
 * If there is none, return 0.
 */
Conf_no
offline_get_unread_confs_next(void)
{
    while (++get_unread_confs_pos < memb_list.no_of_confs)
    {
        Text_list 	ltl = EMPTY_TEXT_LIST_i;
	Time		lt = NULL_TIME_i;
	Conference	conf = EMPTY_CONFERENCE_i;
	Conf_no		confno = memb_list.confs[get_unread_confs_pos].conf_no;

	if (memb_list.confs[get_unread_confs_pos].no_of_read > 0)
	{
	    if (fetch_map(confno, &ltl, &lt) == FAILURE)
	    {
	        /* Disqualify conferences that we havn't fetched the map for.
		 */
	        continue;
	    }
	    else
	    {
		release_text_list(&ltl);
	    }

	    return memb_list.confs[get_unread_confs_pos].conf_no;
	}

	if (fetch_Conference(confno, &conf) == OK)
	{
	    if (memb_list.confs[get_unread_confs_pos].last_text_read < 
		conf.first_local_no + conf.no_of_texts - 1)
	    {
		release_conf(&conf);

		if (fetch_map(confno, &ltl, &lt) == FAILURE)
		{
		    /* Disqualify conferences that we havn't fetched
		     * the map for.
		     */
		    continue;
		}
		else
		{
		    release_text_list(&ltl);
		}

		return memb_list.confs[get_unread_confs_pos].conf_no;
	    }
	    else
	    {
		release_conf(&conf);
	    }
	}
    }

    return (Conf_no)0;
}

/*
 * offline_get_unread_confs_count
 *
 * Calculates how many unread confs we have.
 * The algorithm is:
 * Read the membership, for each of the conferences read the conf-stat and
 * if that conf has unread add one.
 * Since texts that are marked as read by this or earlier sessions of 
 * offline reading shall not count, then we have to read the map for that
 * conference and see if any of those texts are read.
 * This we will leave out since the test is complicated and I have to write
 * the read marking code first. BUG:
 * Another solution is to update the membership as we read.
 */
int
offline_get_unread_confs_count(void)
{
    int			count = 0;

    if (!offline_get_unread_confs_start())
    {
	return 0;
    }

    while (offline_get_unread_confs_next())
    {
	count++;
    }
    return count;
}


/*
 * offline_get_unread_confs
 *
 * Returns a list of conferences with unread.
 */
Success
offline_get_unread_confs(Conf_no_list * list)
{
#define FUNCTION "offline_get_unread_confs()"

    Conf_no cno;

    release_conf_no_list(list);

    offline_get_unread_confs_start();

    list->no_of_confs = 0;
    /* This will always be enough. */
    list->conf_nos =
        (Conf_no*)zmalloc(memb_list.no_of_confs * sizeof(Conf_no));
    if (list->conf_nos == NULL)
        fatal2(CLIENT_OUT_OF_MEMORY, "zmalloc(%d) failed.",
	       memb_list.no_of_confs * sizeof(Conf_no));

    while ((cno = offline_get_unread_confs_next()) != (Conf_no)0)
    {
        list->conf_nos[list->no_of_confs++] = cno;
    }

    return OK;
#undef FUNCTION
}
