/*
 * dmnt.c -- Linux mount functions for /proc-based lslk
 *
 * V. Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1998 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1998 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.2 99/11/10 15:02:43 abe Exp $";
#endif


#include "lslk.h"


/*
 * Externals
 */

int HasNFS = 0;			/* 1 == there's a mounted NFS file system */


/*
 * get_fields() - separate a line into fields
 */

int
get_fields(ln, sep, fr)
	char *ln;			/* input line */
	char *sep;			/* separator list */
	char ***fr;			/* field pointer return address */
{
	char *bp, *cp, *sp;
	MALLOC_S len;
	int n;
	static char **fp = (char **)NULL;
	static int nfpa = 0;

	for (cp = ln, n = 0; cp && *cp;) {
	    for (bp = cp; *bp && (*bp == ' ' || *bp == '\t'); bp++);
		;
	    if (!*bp || *bp == '\n')
		break;
	    for (cp = bp; *cp; cp++) {
		if (*cp == '\n') {
		    *cp = '\0';
		    break;
		}
		if (*cp == ' ' || *cp == '\t')
		    break;
		if (sep) {
		    for (sp = sep; *sp; sp++) {
			if (*sp == *cp)
			    break;
		    }
		    if (*sp)
			break;
		}
	    }
	    if (*cp)
		*cp++ = '\0';
	    if (n >= nfpa) {
		nfpa += 32;
		len = (MALLOC_S)(nfpa * sizeof(char *));
		if (fp)
		    fp = (char **)realloc((MALLOC_P *)fp, len);
		else
		    fp = (char **)malloc(len);
		if (!fp) {
		    (void) fprintf(stderr,
			"%s: can't allocate %d bytes for field pointers.\n",
			Pn, len);
		    Exit(1);
		}
	    }
	    fp[n++] = bp;
	}
	*fr = fp;
	return(n);
}


/*
 * readmnt() -- read mount table information
 */

int
readmnt()
{
	char buf[MAXPATHLEN], *cp, **fp;
	char *dn = (char *)NULL;
	MALLOC_S len;
	char *ln;
	int na = 0;
	FILE *mfs;
	int rv = 0;
	struct stat sb;
/*
 * Open access to /proc/mounts.
 */
	(void) sprintf(buf, "%s/mounts", PROCFS);
	if (!(mfs = fopen(buf, "r"))) {
	    (void) fprintf(stderr, "%s: can't fopen(%s)\n", Pn, buf);
	    return(1);
	}
/*
 * Read mount lines.
 */
	while (fgets(buf, sizeof(buf), mfs)) {
	    if (get_fields(buf, (char *)NULL, &fp) < 3
	    ||  !fp[0] || !fp[1] || !fp[2])
		continue;
	/*
	 * Ignore an autmounter entry -- it has a colon in the device name,
	 * followed by "(pid*".
	 */
	    if ((cp = strchr(fp[0], ':')) && !strncasecmp(++cp, "(pid", 4))
		continue;
	/*
	 * Interpolate a possible symbolic directory link.
	 */
	    if (dn)
		(void) free((MALLOC_P *)dn);
	    len = (MALLOC_S)(strlen(fp[1]) + 1);
	    if (!(dn = (char *)malloc(len))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for directory: %s\n",
		    Pn, len, fp[1]);
		(void) fclose(mfs);
		return(1);
	    }
	    (void) strcpy(dn, fp[1]);
	    if (!(ln = Readlink(dn))) {
		if (Owarn){
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		rv = 1;
		continue;
	    }
	    if (ln != dn) {
		(void) free((MALLOC_P *)dn);
		dn = ln;
	    }
	/*
	 * Stat() the directory.
	 */
	    if (statsafely(dn, &sb)) {
		if (Owarn) {
		    (void) fprintf(stderr,
			"%s: WARNING: can't stat() %s file system %s\n",
			Pn, fp[2], fp[1]);
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		rv = 1;
		continue;
	    }
	/*
	 * See if there's room in the local mount table for another entry.
	 */
	    if (NMnt >= na) {

	    /*
	     * Increase the space allocated to the local mount table and
	     * reserve space for the increase.
	     */
		na += 10;
		len = (MALLOC_S)(na * sizeof(struct mounts));
		if (Mnt)
		    Mnt = (struct mounts *)realloc((MALLOC_P *)Mnt, len);
		else
		    Mnt = (struct mounts *)malloc(len);
		if (!Mnt) {
		    (void) fprintf(stderr,
			"%s: no space for %d mount structures at %s: %s\n",
			Pn, na, fp[0], fp[1]);
		    (void) fclose(mfs);
		    return(1);
		}
	    }
	/*
	 * Allocate space for what's been mounted.
	 */
	    len = (MALLOC_S)(strlen(fp[0]) + 1);
	    if (!(Mnt[NMnt].sp = (char *)malloc(len))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for file system: %s\n",
		    Pn, len, fp[0]);
		(void) fclose(mfs);
		return(1);
	    }
	    (void) strcpy(Mnt[NMnt].sp, fp[0]);
	/*
	 * Create the new local mount information entry.
	 */
	    Mnt[NMnt].dev = sb.st_dev;
	    Mnt[NMnt].mntp = dn;
	    Mnt[NMnt].priv = (caddr_t)NULL;
	/*
	 * Record that an NFS file system is mounted.
	 */
	    if (!HasNFS && fp[2] && strcasecmp(fp[2], "nfs") == 0) {
		HasNFS = 1;
		Mnt[NMnt].priv = (caddr_t)"nfs";
	    } else
		Mnt[NMnt].priv = (caddr_t)NULL;
	    dn = (char *)NULL;
	    NMnt++;
	}
	(void) fclose(mfs);
	return(rv);
}
