/* Copyright (C) 2002 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#include <string.h>
#include "lopster.h"
#include "utils.h"
#include "buffer.h"

buffer_t* buffer_new(int size) {
  buffer_t* result;
  
  if (size < 0) return NULL;
  result = l_malloc(sizeof(buffer_t));
  if (!result) return NULL;
  result->data = l_malloc(size);
  if (size && !result->data) {
    l_free(result);
    return NULL;
  }
  result->datamax = size;
  result->datasize = 0;
  result->consumed = 0;

  return result;
}

void buffer_destroy(buffer_t* buffer) {
  if (!buffer) return;
  if (buffer->data) l_free(buffer->data);
  l_free(buffer);
}

int buffer_resize(buffer_t* buffer, int by) {
  int new_size;

  if (!buffer) return 0;
  if (buffer->datamax+by < 0) by = -buffer->datamax;
  new_size = buffer->datamax + by;
  buffer->data = l_realloc(buffer->data, new_size);
  if (new_size && !buffer->data) {
    buffer->consumed = 0;
    buffer->datamax = 0;
    buffer->datasize = 0;
    return 0;
  } else {
    buffer->datamax = new_size;
  }
  if (buffer->datasize > buffer->datamax)
    buffer->datasize = buffer->datamax;
  if (buffer->consumed > buffer->datasize)
    buffer->consumed = buffer->datasize;
  
  return 1;
}

int buffer_consume_virt(buffer_t* buffer, int how_many) {
  if (!buffer) return 0;
  if (how_many <= 0) return 0;

  if (buffer->consumed + how_many > buffer->datasize)
    how_many = buffer->datasize - buffer->consumed;
  buffer->consumed += how_many;
  return 1;
}

int buffer_consume(buffer_t* buffer, int how_many) {
  if (!buffer) return 0;
  if (how_many >= buffer->datasize) {
    // nothing to shift
    buffer->datasize = 0;
    buffer->consumed = 0;
    return 1;
  }
  if (how_many <= 0) return 0;
  memmove(buffer->data, buffer->data + how_many,
	  buffer->datasize - how_many);
  buffer->datasize -= how_many;
  buffer->consumed -= how_many;
  if (buffer->consumed < 0) buffer->consumed = 0;

  return 1;
}

void buffer_clear(buffer_t* buffer) {
  if (!buffer) return;
  buffer_consume(buffer, buffer->datasize);
}
