package Lire::ChartTypes::Lines;

use strict;

use base qw/Lire::ChartType/;

use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::ChartTypes::Lines - Wrapper for the Ploticus lines prefab.

=head1 SYNOPSIS

  my $type = $chart_cfg->get( 'type' )->as_value();
  $type->write_chart( $chart_cfg, $subreport );

=head1 DESCRIPTION

This Lire::ChartType wraps the Ploticus lines prefab.

=cut

sub name {
    return 'lines';
}

sub title {
    return 'Lines chart';
}

sub description {
    return '<para>' . __( 'Plot one to four variable as lines against another.' ) . '</para>';
}

sub prefab {
    return '-prefab lines'
}

sub get_vars {
    my ( $self, $chart_cfg ) = @_;

    my @vars = ();
    my $props = $chart_cfg->get('type')->get_properties();
    push @vars, $props->get( 'x' )->as_value()
      if $props->get( 'x' )->is_valid();
    push @vars, $props->get( 'y' )->as_value();
    push @vars, $props->get( 'y2' )->as_value()
      if $props->get( 'y2' )->is_valid();
    push @vars, $props->get( 'y3' )->as_value()
      if $props->get( 'y3' )->is_valid();
    push @vars, $props->get( 'y4' )->as_value()
      if $props->get( 'y4' )->is_valid();

    return \@vars;
}

sub write_parameters {
    my ( $self, $chart_cfg, $fh ) = @_;

    $self->SUPER::write_parameters( $chart_cfg, $fh );

    my $props = $chart_cfg->get('type')->get_properties();
    my $idx = 2;
    if ( $props->get( 'x' )->is_valid() ) {
        print $fh "#set cats = no\n#set x = ", $idx++, "\n";
    } else {
        print $fh "#set cats = yes\n#set stubvert = yes\n#set legend = max+0.1 max\n";
    }
    print $fh "#set y = ", $idx++, "\n";
    print $fh "#set y2 = ", $idx++, "\n"
      if $props->get( 'y2' )->is_valid();
    print $fh "#set y3 = ", $idx++, "\n"
      if $props->get( 'y3' )->is_valid();
    print $fh "#set y4 = ", $idx++, "\n"
      if $props->get( 'y4' )->is_valid();
    print $fh "#set numbers = yes\n"
      if $props->get( 'numbers' )->as_value();

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

 Lire::PluginManager(3pm) Lire::ChartType(3pm), ploticus(1)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Lines.pm,v 1.1 2004/07/17 20:15:53 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

