/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Moon.h"
#include "Inline.h"

#include <cmath>
using namespace std;

const double R = 6.378137E3;

CMoon::CMoon() :
m_az(0.0),
m_el(0.0),
m_ra(0.0),
m_dec(0.0),
m_lha(0.0),
m_gha(0.0),
m_lat(0.0),
m_lon(0.0),
m_ro(0.0),
m_mm(0.0),
m_dx(0.0),
m_frequency(1296.0)
{
}

CMoon::~CMoon()
{
}

void CMoon::findMoon(double dayNum)
{
	/* This function determines the position of the moon, including
	   the azimuth and elevation headings, relative to the latitude
	   and longitude of the tracking station.  This code was derived
	   from a Javascript implementation of the Meeus method for
	   determining the exact position of the Moon found at:
	   http://www.geocities.com/s_perona/ingles/poslun.htm. */

	double jd = dayNum + 2444238.5;

	double  t = (jd - 2415020.0) / 36525.0;
	double t2 = t * t;
	double t3 =t2 * t;
	double l1 = 270.434164 + 481267.8831 * t - 0.001133 * t2 + 0.0000019 * t3;
	double  m = 358.475833 + 35999.0498 * t - 0.00015 * t2 - 0.0000033 * t3;
	double m1 = 296.104608 + 477198.8491 * t + 0.009192 * t2 + 0.0000144 * t3;
	double  d = 350.737486 + 445267.1142 * t - 0.001436 * t2 + 0.0000019 * t3;
	double ff = 11.250889 + 483202.0251 * t - 0.003211 * t2 - 0.0000003 * t3;
	double om = RAD(259.183275 - 1934.142 * t + 0.002078 * t2 + 0.0000022 * t3);

	/* Additive terms */

	double ss, ex;
	l1 += 0.000233 * ::sin(RAD(51.2 + 20.2 * t));
	ss  = 0.003964 * ::sin(RAD(346.56 + 132.87 * t - 0.0091731 * t2));
        l1 += ss + 0.001964 * ::sin(om);
	m  -= 0.001778 * ::sin(RAD(51.2 + 20.2 * t));
	m1 += 0.000817 * ::sin(RAD(51.2 + 20.2 * t));
	m1 += ss + 0.002541 * ::sin(om);
	d  += 0.002011 * ::sin(RAD(51.2 + 20.2 * t));
	d  += ss + 0.001964 * ::sin(om);
	ff += ss - 0.024691 * ::sin(om);
	ff -= 0.004328 * ::sin(om + RAD(275.05 - 2.3 * t));
        ex  = 1.0 - 0.002495 * t - 0.00000752 * t2;
        om = RAD(om);		// XXXXX

	l1 = primeAngle(l1);
	m  = RAD(primeAngle(m));
	m1 = RAD(primeAngle(m1));
	d  = RAD(primeAngle(d));
	ff = RAD(primeAngle(ff));
	om = primeAngle(om);

        /* Ecliptic Longitude */

	double l;
	l = l1 + 6.28875 * ::sin(m1) + 1.274018 * ::sin(2.0 * d - m1) + 0.658309 * ::sin(2.0 * d);
	l = l + 0.213616 * ::sin(2.0 * m1) - ex * 0.185596 * ::sin(m) - 0.114336 * ::sin(2.0 * ff);
	l = l + 0.058793 * ::sin(2.0 * d - 2.0 * m1) + ex * 0.057212 * ::sin(2.0 * d - m - m1) + 0.05332 * ::sin(2.0 * d + m1);
	l = l + ex * 0.045874 * ::sin(2.0 * d - m) + ex * 0.041024 * ::sin(m1 - m) - 0.034718 * ::sin(d);
	l = l - ex * 0.030465 * ::sin(m + m1) + 0.015326 * ::sin(2.0 * d - 2.0 * ff) - 0.012528 * ::sin(2.0 * ff + m1);

	l = l - 0.01098 * ::sin(2.0 * ff - m1) + 0.010674 * ::sin(4.0 * d - m1) + 0.010034 * ::sin(3.0 * m1);
        l = l + 0.008548 * ::sin(4.0 * d - 2.0 * m1) - ex * 0.00791 * ::sin(m - m1 + 2.0 * d) - ex * 0.006783 * ::sin(2.0 * d + m);

	l = l + 0.005162 * ::sin(m1 - d) + ex * 0.005 * ::sin(m + d) + ex * 0.004049 * ::sin(m1 - m + 2.0 * d);
	l = l + 0.003996 * ::sin(2.0 * m1 + 2.0 * d) + 0.003862 * ::sin(4.0 * d) + 0.003665 * ::sin(2.0 * d - 3.0 * m1);

	l = l + ex * 0.002695 * ::sin(2.0 * m1 - m) + 0.002602 * ::sin(m1 - 2.0 * ff - 2.0 * d) + ex * 0.002396 * ::sin(2.0 * d - m - 2.0 * m1);

	l = l - 0.002349 * ::sin(m1 + d) + ex * ex * 0.002249 * ::sin(2.0 * d - 2.0 * m) - ex* 0.002125 * ::sin(2.0 * m1 + m);

	l = l - ex * ex * 0.002079 * ::sin(2.0 * m) + ex * ex * 0.002059 * ::sin(2.0 * d - m1 - 2.0 * m) - 0.001773 * ::sin(m1 + 2.0 * d - 2.0 * ff);

	l = l + ex * 0.00122 * ::sin(4.0 * d - m - m1) - 0.00111 * ::sin(2.0 * m1 + 2.0 * ff) + 0.000892 * ::sin(m1 - 3.0 * d);

	l = l - ex * 0.000811 * ::sin(m + m1 + 2.0 * d) + ex * 0.000761 * ::sin(4.0 * d - m - 2.0 * m1) + ex * ex * 0.000717 * ::sin(m1 - 2.0 * m);

	l = l + ex * ex * 0.000704 * ::sin(m1 - 2.0 * m -2.0 * d) + ex * 0.000693 * ::sin(m - 2.0 * m1 + 2.0 * d) + ex * 0.000598 * ::sin(2.0 * d - m - 2.0 * ff) + 0.00055 * ::sin(m1 + 4.0 * d);

	l = l + 0.000538 * ::sin(4.0 * m1) + ex * 0.000521 * ::sin(4.0 * d - m) + 0.000486 * ::sin(2.0 * m1 - d);

	l = l - 0.001595 * ::sin(2.0 * ff + 2.0 * d);

	/* Ecliptic latitude */

	double b;
        b = 5.128189 * ::sin(ff) + 0.280606 * ::sin(m1 + ff) + 0.277693 * ::sin(m1 - ff) + 0.173238 * ::sin(2.0 * d - ff);
        b = b + 0.055413 * ::sin(2.0 * d + ff - m1) + 0.046272 * ::sin(2.0 * d - ff - m1) + 0.032573 * ::sin(2.0 * d + ff);

	b = b + 0.017198 * ::sin(2.0 * m1 + ff) + 9.266999e-03 * ::sin(2.0 * d + m1 - ff) + 0.008823 * ::sin(2.0 * m1 - ff);
	b = b + ex*0.008247 * ::sin(2.0 * d - m - ff) + 0.004323 * ::sin(2.0 * d - ff - 2.0 * m1) + 0.0042 * ::sin(2.0 * d + ff + m1);

	b = b + ex * 0.003372 * ::sin(ff - m - 2.0 * d) + ex * 0.002472 * ::sin(2.0 * d + ff - m - m1) + ex * 0.002222 * ::sin(2.0 * d + ff - m);

	b = b + 0.002072 * ::sin(2.0 * d - ff - m - m1) + ex * 0.001877 * ::sin(ff - m + m1) + 0.001828 * ::sin(4.0 * d - ff - m1);

	b = b - ex * 0.001803 * ::sin(ff + m) - 0.00175 * ::sin(3.0 * ff) + ex * 0.00157 * ::sin(m1 - m - ff) - 0.001487 * ::sin(ff + d) - ex * 0.001481 * ::sin(ff + m + m1) + ex * 0.001417 * ::sin(ff - m - m1) + ex * 0.00135 * ::sin(ff - m) + 0.00133 * ::sin(ff - d);

	b = b + 0.001106 * ::sin(ff + 3.0 * m1) + 0.00102 * ::sin(4.0 * d - ff) + 0.000833 * ::sin(ff + 4.0 * d - m1);

	b = b + 0.000781 * ::sin(m1 - 3.0 * ff) + 0.00067 * ::sin(ff + 4.0 * d - 2.0 * m1) + 0.000606 * ::sin(2.0 * d - 3.0 * ff);

	b = b + 0.000597 * ::sin(2.0 * d + 2.0 * m1 - ff) + ex * 0.000492 * ::sin(2.0 * d + m1 - m - ff) + 0.00045 * ::sin(2.0 * m1 - ff - 2.0 * d);

	b = b + 0.000439 * ::sin(3.0 * m1 - ff) + 0.000423 * ::sin(ff + 2.0 * d + 2.0 * m1) + 0.000422 * ::sin(2.0 * d - ff - 3.0 * m1);

	b = b - ex * 0.000367 * ::sin(m + ff + 2.0 * d - m1) - ex * 0.000353 * ::sin(m + ff + 2.0 * d) + 0.000331 * ::sin(ff + 4.0 * d);

	b = b + ex * 0.000317 * ::sin(2.0 * d + ff - m + m1) + ex * ex * 0.000306 * ::sin(2.0 * d - 2.0 * m - ff) - 0.000283 * ::sin(m1 + 3.0 * ff);

	double w1 = 0.0004664 * ::cos(RAD(om));
	double w2 = 0.0000754 * cos(RAD(om + 275.05 - 2.3 * t));
	double bt = b * (1.0 - w1 - w2);

	/* Parallax calculations */

	double p;
	p = 0.950724 + 0.051818 * ::cos(m1) + 0.009531 * ::cos(2.0 * d - m1) + 0.007843 * ::cos(2.0 * d) + 0.002824 * ::cos(2.0 * m1) + 0.000857 * ::cos(2.0 * d + m1) + ex * 0.000533 * ::cos(2.0 * d - m) + ex * 0.000401 * ::cos(2.0 * d - m - m1);

	p = p + 0.000173 * ::cos(3.0 * m1) + 0.000167 * ::cos(4.0 * d - m1) - ex * 0.000111 * ::cos(m) + 0.000103 * ::cos(4.0 * d - 2.0 * m1) - 0.000084 * ::cos(2.0 * m1 - 2.0 * d) - ex * 0.000083 * ::cos(2.0 * d + m) + 0.000079 * ::cos(2.0 * d + 2.0 * m1);

	p = p + 0.000072 * ::cos(4.0 * d) + ex * 0.000064 * ::cos(2.0 * d - m + m1) - ex * 0.000063 * ::cos(2.0 * d + m - m1);

	p = p + ex * 0.000041 * ::cos(m + d) + ex * 0.000035 * ::cos(2.0 * m1 - m) - 0.000033 * ::cos(3.0 * m1 - 2.0 * d);

	p = p - 0.00003 * ::cos(m1 + d) - 0.000029 * ::cos(2.0 * ff - 2.0 * d) - ex * 0.000029 * ::cos(2.0 * m1 + m);

	p = p + ex * ex * 0.000026 * ::cos(2.0 * d - 2.0 * m) - 0.000023 * ::cos(2.0 * ff - 2.0 * d + m1) + ex * 0.000019 * ::cos(4.0 * d - m - m1);

	b  = RAD(bt);
	double lm = RAD(l);

	/* Convert ecliptic coordinates to equatorial coordinates */

	double z, ob;
	z  = (jd - 2415020.5) / 365.2422;
	ob = RAD(23.452294 - (0.46845 * z + 5.9e-07 * z * z) / 3600.0);
	m_dec = ::asin(::sin(b) * ::cos(ob) + ::cos(b) * ::sin(ob) * ::sin(lm));
	m_ra  = ::acos(::cos(b) * ::cos(lm) / ::cos(m_dec));

	if (lm > M_PI)
		m_ra = 2.0 * M_PI - m_ra;

	/* Find siderial time in radians */

	double teg;
	t   = (jd - 2451545.0) / 36525.0;
	teg = 280.46061837 + 360.98564736629 * (jd - 2451545.0) + (0.000387933 * t - t * t / 38710000.0) * t;

	while (teg > 360.0) teg -= 360.0;

	double th = fixAngle(RAD(teg) + m_lon);
        double h  = th - m_ra;
	m_gha     = RAD(teg) - m_ra;

        if (m_gha < 0.0) m_gha += 2.0 * M_PI;

	m_lha = m_gha + m_lon;

        if (m_lha < 0.0) m_lha += 2.0 * M_PI;

	m_az = ::atan2(::sin(h), ::cos(h) * ::sin(m_lat) - ::tan(m_dec) * ::cos(m_lat)) + M_PI;
	m_el = ::asin(::sin(m_lat) * ::sin(m_dec) + ::cos(m_lat) * ::cos(m_dec) * ::cos(h));

	m_ro = 0.996986 / (1.0 + 0.0549 * ::cos(m_mm + 0.10976 * ::sin(m_mm)));

	double t1;
	t1   = R * 1000.0;
	t2   = 384401.0 * m_ro * 1000.0;
	m_dx = ::sqrt(t2 * t2 + 2.0 * t1 * t2 * ::sin(m_el));
}

double CMoon::getAzimuth() const
{
	return DEG(m_az);
}

double CMoon::getElevation() const
{
	return DEG(m_el);
}

double CMoon::getRA() const
{
	return DEG(m_ra);
}

double CMoon::getDec() const
{
	return DEG(m_dec);
}

double CMoon::getLHA() const
{
	return DEG(m_lha);
}

double CMoon::getGHA() const
{
	return DEG(m_gha);
}

double CMoon::getDoppler() const
{
	double t2 = 0.10976;
	double t1 = m_mm + t2 + ::sin(m_mm);
	double dmdt = 0.01255 * m_ro * m_ro * ::sin(t1) * (1.0 + t2 * ::cos(m_mm));
	dmdt *= 4449.0;

	t1 = R * 1000.0;
	t2 = 384401.0 * m_ro * 1000.0;

	double drdt1 = dmdt * (t2 + (t1 * ::sin(m_el))) / m_dx;
	double dlhdt = 7.53125e-5 * (::sin(m_lha) * ::cos(m_dec) * ::cos(m_lat));

	double    dcMax = RAD(20.0);
	double ddcdtMax = RAD(0.0000716);

	double t3 = m_dec / dcMax;
	if (t3 < 1.0) {
		t3 = ::sin(t3);
		double ddcdt = ddcdtMax * ::sqrt(1.0 - t3 * t3);
		t3 = ddcdt * (::cos(m_dec) * ::sin(m_lat) - ::cos(m_lha) * ::sin(m_dec) * ::cos(m_lat));
	} else {
		t3 = 0.0;
	}

	double dsineldt = dlhdt + t3;
	double    drdt2 = t1 * t2 * dsineldt / m_dx;

	double dv = drdt1 + drdt2;

	return -dv * m_frequency / 299.810;
}

double CMoon::getPathLoss() const
{
	return -17.37 * ::log10(m_ro);
}

double CMoon::getRange() const
{
	return m_dx;
}

void CMoon::setLocation(double latitude, double longitude)
{
	m_lat = RAD(latitude);
	m_lon = RAD(longitude);
}

void CMoon::setFrequency(double frequency)
{
	m_frequency = frequency;
}

double CMoon::fixAngle(double angle) const
{
	/* This function reduces angles greater than
	   two pi by subtracting two pi from the angle */

	while (angle > (2.0 * M_PI))
		angle -= 2.0 * M_PI;

	return angle;
}

double CMoon::primeAngle(double x) const
{
	return x - 360.0 * ::floor(x / 360.0);
}

double CMoon::getDayNum(const wxDateTime& dateTime) const
{
	dateTime.ToGMT(true);

	int  secs = dateTime.GetTicks();
	int msecs = dateTime.GetMillisecond();

	return ((double(secs) + 0.001 * double(msecs)) / 86400.0) - 3651.0;
}
