/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qdatastream.h>
#include <kdebug.h>
#include "patterncanvas.h"
#include "stitch.h"

PatternCanvas::PatternCanvas()
  : m_width(0),
    m_height(0),
    m_pCanvas(new QPtrVector<Stitch::Queue>),
    m_pKnot(new QPtrList<Knot>),
    m_pBackstitch(new QPtrList<BackStitch>)
{
  m_pCanvas->setAutoDelete(true);
  m_pKnot->setAutoDelete(true);
  m_pBackstitch->setAutoDelete(true);
}

PatternCanvas::~PatternCanvas()
{
  delete m_pCanvas;
  delete m_pKnot;
  delete m_pBackstitch;
}

void PatternCanvas::clear()
{
  m_pBackstitch->clear();
  m_pKnot->clear();
  m_pCanvas->clear();
  m_width = 0;
  m_height = 0;
}

bool PatternCanvas::isEmpty()
{
  if (m_pBackstitch->isEmpty() && m_pKnot->isEmpty() && m_pCanvas->isEmpty())
    return true;
  else
    return false;
}

void PatternCanvas::resize(int width,int height)
{
  m_pCanvas->resize(width*height);
  m_width = width;
  m_height = height;
}

void PatternCanvas::extendTop(int rows)
{
  int w = m_width;
  int h = m_height;
  int size = w*h;
  resize(w,h+rows);
  int newSize = m_width*m_height;
  int count = size;
  QPoint snapOffset = QPoint(0,rows*2);
  while (count--)
    m_pCanvas->insert(--newSize, m_pCanvas->take(--size));
  for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    k->pos += snapOffset;
  for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
  {
    bs->start += snapOffset;
    bs->end += snapOffset;
  }
}

void PatternCanvas::extendLeft(int cols)
{
  int w = m_width;
  int h = m_height;
  int size = w*h;
  resize(m_width+cols, m_height);
  int newSize = m_width*m_height;
  QPoint snapOffset = QPoint(cols*2,0);
  while (newSize)
  {
    for (int c = w ; c ; c--)
      m_pCanvas->insert(--newSize, m_pCanvas->take(--size));
    newSize -= cols;
  }
  for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    k->pos += snapOffset;
  for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
  {
    bs->start += snapOffset;
    bs->end += snapOffset;
  }
}

void PatternCanvas::extendRight(int cols)
{
  int w = m_width;
  int h = m_height;
  int size = w*h;
  resize(m_width+cols, m_height);
  int newSize = m_width*m_height;
  while (newSize)
  {
    for (int c = w ; c ; c--)
      m_pCanvas->insert((--newSize)-cols, m_pCanvas->take(--size));
    newSize -= cols;
  }
}

void PatternCanvas::extendBottom(int rows)
{
  resize(m_width, m_height+rows);
}

QRect PatternCanvas::patternBoundingRect()
{
  int canvasW = m_width;
  int canvasH = m_height;
  int canvasSize = canvasW*canvasH;

  QRect stitchesRect;
  if (!m_pCanvas->isEmpty())
    for (int i = 0 ; i < canvasSize ; i++)
    {
      int dx = i%canvasW;
      int dy = i/canvasW;
      if (m_pCanvas->at(i))
        stitchesRect|=QRect(dx,dy,1,1);
    }

  if (!m_pKnot->isEmpty())
  {
    int x1 = canvasW*2+1;
    int y1 = canvasH*2+1;
    int x2 = 0;
    int y2 = 0;
    for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    {
      x1 = x1 <? k->pos.x();
      y1 = y1 <? k->pos.y();
      x2 = x2 >? k->pos.x();
      y2 = y2 >? k->pos.y();
    }
    stitchesRect|=QRect(x1/2,y1/2,(x2-x1)/2,(y2-y1)/2);
  }

  if (!m_pBackstitch->isEmpty())
  {
    for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
    {
      // create a normalised rectangle that bounds the backstitch
      QPoint s = bs->start;
      QPoint e = bs->end;
      int x1,y1,x2,y2;
      x1 = s.x() <? e.x();
      x2 = s.x() >? e.x();
      y1 = s.y() <? e.y();
      y2 = s.y() >? e.y();
      if (x1 == x2 || y1 == y2)
      {
        // special case as a QRect would show as zero m_width or m_height (ie invalid)
        // so work out were this backstitch is in relation to the canvas
        // and extend the bounding rectangle to suit.
        // worst case would be if there was only 1 vertical or horizontal backstitch and nothing else, but this is unlikely
        if (x1 == x2)
        {
          // vertical line
          if (x1/2 < canvasW/2)
          {
            // its to the left of canvas
            x2+=2; // backstitch coordinates
          }
          else if (x1/2 == canvasW/2)
          {
            // its in the middle of the canvas
            x1-=2;
            x2+=2;
          }
          else
          {
            // its to the right of canvas
            x2-=2;
          }
        }
        if (y1 == y2)
        {
          // horizontal line
          if (y1/2 < canvasH/2)
          {
            // its above the canvas center
            y2+=2;
          }
          else if (y1/2 == canvasH/2)
          {
            // its in the middle of the canvas
            y1-=2;
            y2+=2;
          }
          else
          {
            // its below the canvas center
            y1-=2;
          }
        }
      }
      stitchesRect|=QRect(x1/2,y1/2,(x2-x1)/2,(y2-y1)/2);
    }
  }
  return stitchesRect;
}

void PatternCanvas::centerPattern()
{
  int canvasW = m_width;
  int canvasH = m_height;
  int canvasSize = canvasW*canvasH;

  QRect stitchesRect = patternBoundingRect();
  // stitchesRect now bounds stitches and backstitches
  // if it is still invalid then the pattern is empty
  if (stitchesRect.isValid())
  {
    // center the rectangle in the pattern
    int patternW = stitchesRect.width();
    int patternH = stitchesRect.height();
    int destinationX = (canvasW-patternW)/2;
    int destinationY = (canvasH-patternH)/2; // destinationX,destinationY is the position of the top left of the destination rectangle

    int destinationOrigin = destinationY*canvasW+destinationX; // index to QVector of top left of destination rectangle
    int sourceOrigin = stitchesRect.y()*canvasW+stitchesRect.x(); // index to QVector of top left of source rectangle
    QPoint offset((destinationX-stitchesRect.x())*2,(destinationY-stitchesRect.y())*2);
    if (destinationOrigin != sourceOrigin)
    {
      if (destinationOrigin < sourceOrigin)
      {
        // moving backward
        while (sourceOrigin < canvasSize)
          m_pCanvas->insert(destinationOrigin++, m_pCanvas->take(sourceOrigin++));
      }
      else
      {
        // moving forward
        int offset = destinationOrigin-sourceOrigin;
        sourceOrigin = stitchesRect.bottom()*canvasW+stitchesRect.right();
        destinationOrigin = sourceOrigin+offset;
        while (sourceOrigin >= 0)
          m_pCanvas->insert(destinationOrigin--, m_pCanvas->take(sourceOrigin--));
      }
      // offset the knots
      for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
        k->pos += offset;
      // offset the backstitches
      for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
      {
        bs->start += offset;
        bs->end += offset;
      }
    }
    // else no move is required
  }
}

void PatternCanvas::cropCanvasToRect(QRect r)
{
  int rx = r.x();
  int ry = r.y();
  int rW = r.width();
  int rH = r.height();
  int destinationIndex = 0;
  for (int y = 0 ; y < rH ; y++)
    for (int x = 0 ; x < rW ; x++)
      m_pCanvas->insert(destinationIndex++, m_pCanvas->take((y+ry)*m_width+rx+x));
  resize(rW,rH);
  Knot* kn = m_pKnot->first();
  while (kn)
  {
    kn->pos -= QPoint(rx*2,ry*2);
    if (kn->pos.x() < 0 || kn->pos.y() < 0 || kn->pos.x() > rW*2 || kn->pos.y() > rH*2)
      m_pKnot->remove();
    else
      m_pKnot->next();
    kn = m_pKnot->current();
  }
  BackStitch* bs = m_pBackstitch->first();
  while (bs)
  {
    bs->start -= QPoint(rx*2,ry*2);
    bs->end -= QPoint(rx*2,ry*2);
    if (bs->start.x() < 0 || bs->start.y() < 0 || bs->end.x() < 0 || bs->end.y() < 0 || bs->start.x() > rW*2 || bs->start.y() > rH*2 || bs->end.x() > rW*2 || bs->end.y() > rH*2)
      m_pBackstitch->remove();
    else
      m_pBackstitch->next();
    bs = m_pBackstitch->current();
  }
}

int PatternCanvas::patternWidth()
{
  return m_width;
}

int PatternCanvas::patternHeight()
{
  return m_height;
}

uint PatternCanvas::index(QPoint c)
{
  return c.y()*m_width+c.x();
}

Stitch::Queue *PatternCanvas::stitchAt(QPoint cell)
{
  return (m_pCanvas && validateCell(cell))?m_pCanvas->at(index(cell)):0;
}

bool PatternCanvas::validateCell(QPoint c)
{
  if (c.x() < 0) return false;
  if (c.y() < 0) return false;
  if (m_width <= c.x()) return false;
  if (m_height <= c.y()) return false;
  return true;
}

bool PatternCanvas::validateSnap(QPoint s)
{
  if (s.x() < 0) return false;
  if (s.y() < 0) return false;
  if (m_width*2+1 <= s.x()) return false;
  if (m_height*2+1 <= s.y()) return false;
  return true;
}

QPtrListIterator<BackStitch> *PatternCanvas::backstitches()
{
  return new QPtrListIterator<BackStitch>(*m_pBackstitch);
}

QPtrListIterator<Knot> *PatternCanvas::knots()
{
  return new QPtrListIterator<Knot>(*m_pKnot);
}

bool PatternCanvas::deleteStitch(QPoint c,Stitch::Type t,int f)
{
  if (!validateCell(c))
    return false;
  uint id = index(c);
  if (t == Stitch::Delete && f == -1)
  {
    m_pCanvas->remove(id);
    return true;
  }
  Stitch::Queue *q = m_pCanvas->at(id);
  if (q)
  {
    int n = q->count();
    while (n--)
    {
      Stitch *s = q->dequeue();
      if (!(t == Stitch::Delete ^ s->type == t) || !(f == -1 ^ s->floss == f))
        q->enqueue(s);
      else
        delete s;
    }
    if (q->count() == 0)
      m_pCanvas->remove(id);
    return true;
  }
  return false;
}

bool PatternCanvas::addStitch(QPoint c, Stitch::Type t, int i)
{
  if (!validateCell(c))
    return false;
  uint id = index(c);

  Stitch::Queue *pStitches = m_pCanvas->at(id); // get the pointer to any existing queue of stitches for this location
  if (pStitches == 0)
  {
    /** no stitch queue currently exists for this location */
    pStitches = new Stitch::Queue();
    m_pCanvas->insert(id, pStitches); 				// insert a new stitch queue
  }

  uint nStitches = pStitches->count();			// get the number of stitches in the queue (may be none)
  pStitches->enqueue(new Stitch(t, i));			// add the new stitch to the end of the queue

  /** iterate the queue of existing stitches for any that have been overwritten by the new stitch */
  while (nStitches--)											// while there are existing stitches
  {
    Stitch *pStitch = pStitches->dequeue();	// get the stitch at the head of the queue
    Stitch::Type cst = pStitch->type;					// and find its type
    int cfi = pStitch->floss;								// and colour
    Stitch::Type interferenceMask = (Stitch::Type)(cst & t);
                                            // interferenceMask now contains a mask of which bits are affected by new stitch
    if (interferenceMask)
    {
                                            // Some parts of the current stitch are being overwritten
                                            // but a check needs to be made for illegal values
      Stitch::Type changeMask = (Stitch::Type)(cst ^ interferenceMask);
      switch (changeMask)
      {
                                            // changeMask contains what is left of the original stitch after being overwritten
                                            // it may contain illegal values, so these are checked for
        case 3:
          pStitches->enqueue(new Stitch(Stitch::TLQtr, cfi));
          pStitches->enqueue(new Stitch(Stitch::TRQtr, cfi));
          changeMask = Stitch::Delete;
          break;
        case 5:
          pStitches->enqueue(new Stitch(Stitch::TLQtr, cfi));
          pStitches->enqueue(new Stitch(Stitch::BLQtr, cfi));
          changeMask = Stitch::Delete;
          break;
        case 10:
          pStitches->enqueue(new Stitch(Stitch::TRQtr, cfi));
          pStitches->enqueue(new Stitch(Stitch::BRQtr, cfi));
          changeMask = Stitch::Delete;
          break;
        case 12:
          pStitches->enqueue(new Stitch(Stitch::BLQtr, cfi));
          pStitches->enqueue(new Stitch(Stitch::BRQtr, cfi));
          changeMask = Stitch::Delete;
          break;
        default:
          // other values are acceptable as is
          break;
      }
      if (changeMask)
      {
                                            // Check there is something left of the original stitch
        pStitch->type = changeMask;					// and change stitch type to the changeMask value
        pStitches->enqueue(pStitch);				// and then add it back to the queue
      }
      else
      {
                                            // if changeMask is NULL, it does not get requeued, effectively deleting it from the pattern
        delete pStitch;											// delete the Stitch as it is no longer required
      }
    }
    else
    {
      pStitches->enqueue(pStitch);
    }
  }
  return true;
}

int PatternCanvas::findColor(QPoint cell, Stitch::Type stitchType)
{
  int colorIndex = -1;

  if (validateCell(cell))
  {
    uint id = index(cell);
    Stitch::Queue *pStitches = m_pCanvas->at(id); // get the pointer to any existing queue of stitches for this location
    if (pStitches)
    {
      uint nStitches = pStitches->count();			// get the number of stitches in the queue (may be none)
      /** iterate the queue of existing stitches */
      while (nStitches--)											// while there are existing stitches
      {
        Stitch *pStitch = pStitches->dequeue();	// get the stitch at the head of the queue
        Stitch::Type cst = pStitch->type;					// and find its type
        int cfi = pStitch->floss;								// and colour
        if (colorIndex == -1)
          colorIndex = cfi;											// set the colorIndex to the first color found
        if (cst & stitchType)
          colorIndex = cfi;											// change the colorIndex if the current stitch type found
        pStitches->enqueue(pStitch);						// put the stitch back in the queue
      }
    }
  }
  return colorIndex;
}

void PatternCanvas::replaceColor(int original, int replacement)
{
  int sq = m_pCanvas->size();
  for (int i = 0 ; i < sq ; i++)
  {
    Stitch::Queue *q = m_pCanvas->at(i);
    if (q)
    {
      int c = q->count();
      while (c--)
      {
        Stitch *s = q->dequeue();
        if (s)
          // shouldn't really happen, but check for it anyway
          if (s->floss == original)
            s->floss = replacement;
        q->enqueue(s);
      }
    }
  }
  for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
    if (bs->floss == original)
      bs->floss = replacement;
  for (Knot* k = m_pKnot->first() ; k ; k = m_pKnot->next())
    if (k->floss == original)
      k->floss = replacement;
}

QValueList<int> PatternCanvas::usedColors()
{
  QValueList<int> uc;
  int sq = m_pCanvas->size();
  for (int i = 0 ; i < sq ; i++)
  {
    Stitch::Queue *pStitches = m_pCanvas->at(i);
    if (pStitches)
    {
      int c = pStitches->count();
      while (c--)
      {
        Stitch *s = pStitches->dequeue();
        if (s)
        { // shouldn't really happen, but check for it anyway
          if (!uc.contains(s->floss))
            uc << s->floss;
          pStitches->enqueue(s);
        }
      }
    }
  }
  for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    if (!uc.contains(k->floss))
      uc << k->floss;
  for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
    if (!uc.contains(bs->floss))
      uc << bs->floss;
  return uc;
}

void PatternCanvas::addBackstitch(QPoint start, QPoint end, int i)
{
  if (validateSnap(start) && validateSnap(end))
    m_pBackstitch->append(new BackStitch(start, end, i));
}

void PatternCanvas::deleteBackstitch(BackStitch *bs)
{
  m_pBackstitch->removeRef(bs);
}

void PatternCanvas::addFrenchKnot(QPoint snap, int i)
{
  if (validateSnap(snap))
  {
    deleteFrenchKnot(snap,-1); // just in case there is one there already
    m_pKnot->append(new Knot(snap,i));
  }
}

void PatternCanvas::deleteFrenchKnot(QPoint p, int f)
{
  for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    if (k->pos == p && (f == -1 ^ f == k->floss))
    {
      m_pKnot->remove(k);
      return;
    }
}

void PatternCanvas::writeCanvas(QDataStream &s)
{
  s << (Q_INT32)m_width;
  s << (Q_INT32)m_height;

  int sq = m_pCanvas->size();
  for (int i = 0 ; i < sq ; i++)
  {
    Stitch::Queue *psq = m_pCanvas->at(i);
    if (psq)
    {
      int stitches = psq->count();  // the number of stitches in the queue, may be 0
      s << (Q_INT8)stitches;
      while (stitches--)
      {
        Stitch *stitch = psq->dequeue();
        s << (Q_INT8)(stitch->type);
        s << (Q_INT16)(stitch->floss);
        psq->enqueue(stitch);
      }
    }
    else
      s << (Q_INT8)0;
  }
  sq = m_pKnot->count();
  s << (Q_INT32)sq;
  for (Knot *k = m_pKnot->first() ; k ; k = m_pKnot->next())
    s << k->pos << (Q_INT16)(k->floss);   // QPoint, Q_INT16
  sq = m_pBackstitch->count();
  s << (Q_INT32)sq;
  for (BackStitch* bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
    s << bs->start << bs->end << (Q_INT16)(bs->floss);		// QPoint, QPoint, Q_INT16
}

bool PatternCanvas::readPCStitch5Canvas(QDataStream& s)
{
  Stitch::Type  stitchType[] = {Stitch::Delete,Stitch::Full,Stitch::TL3Qtr,Stitch::TR3Qtr,Stitch::BL3Qtr,Stitch::BR3Qtr,Stitch::TBHalf,Stitch::BTHalf,Stitch::FRKnot,Stitch::TLQtr,Stitch::TRQtr,Stitch::BLQtr,Stitch::BRQtr}; // conversion of PCStitch to KXStitch
  Q_INT32       width;
  Q_INT32       height;
  Q_INT32       cells;
  Q_INT32       backstitches;
  Q_INT32       extras;
  Q_INT32       knots;
  Q_INT16       cellCount;
  Q_INT16       x, y;
  Q_UINT8       type;
  Q_UINT8       color;
  QIODevice::Offset fileSize = s.device()->size();
  // KXStitch file format has the m_width and m_height available at this point
  // PCStitch does not, so it has been set by KXStitchDoc::openDocument()
  // and the canvas has been cleared before calling this function
  width = m_width;
  height = m_height;
  cells = width*height; // total number of cells
  for (int i = 0 ; i < cells ; i+= cellCount)
  {
    if (s.device()->at()+4 > fileSize) return false;
    s >> cellCount;
    s >> color;
    s >> type;
    if (type != 0xff)
    {
      for (int c = 0 ; c < cellCount ; c++)
      {
        int xc = (i+c)/height;
        int yc = (i+c)%height;
        addStitch(QPoint(xc,yc),stitchType[type],color-1); // color-1 because PCStitch uses 1 based array
                                                            // KXStitch uses 0 based arrays
      }
    }
  }
  // check for additional stitches
  if (s.device()->at()+4 > fileSize) return false;
  s >> extras; // assuming a 32 bit value, time will tell
  while (extras--)
  {
    if (s.device()->at()+4 > fileSize) return false;
    s >> x;
    s >> y;
    for (int dx = 0 ; dx < 4 ; dx++)
    {
      if (s.device()->at()+2 > fileSize) return false;
      s >> color;
      s >> type;
      if (type != 0xff)
        addStitch(QPoint(x-1,y-1),stitchType[type],color-1); // values -1 for same reason as above
    }
  }
  // read french knots
  if (s.device()->at()+4 > fileSize) return false;
  s >> knots;
  while (knots--)
  {
    if (s.device()->at()+5 > fileSize) return false;
    s >> x;
    s >> y;
    s >> color;
    addFrenchKnot(QPoint(x-1,y-1),color-1);
  }
  // read backstitches
  if (s.device()->at()+4 > fileSize) return false;
  s >> backstitches;
  while (backstitches--)
  {
    if (s.device()->at()+13 > fileSize) return false;
    Q_INT16 sx, sy, sp, ex, ey, ep;
    s >> sx;
    s >> sy;
    s >> sp;
    s >> ex;
    s >> ey;
    s >> ep;
    s >> color;
    m_pBackstitch->append(new BackStitch(QPoint(--sx*2+((sp-1)%3),--sy*2+((sp-1)/3)),QPoint(--ex*2+((ep-1)%3),--ey*2+((ep-1)/3)),color-1));
  }
  return true;
}

bool PatternCanvas::readKXStitchCanvas(QDataStream& s,int fileFormatVersion)
{
  Q_INT32     width;
  Q_INT32     height;
  Q_INT32     cells;
  Q_INT32     backstitches;
  Q_INT32     knots;
  Q_INT8      stitches;
  Q_INT8      type;
  Q_INT16     color;
  QPoint      start;
  QPoint      end;
  QIODevice*  device = s.device();
  int         fileSize = device->size();
  switch (fileFormatVersion)
  {
    case 5:
    case 4:
      if (device->at()+8 > fileSize) return false;
      s >> width >> height;
      cells = width*height;
      m_pCanvas->clear();
      resize(width, height);
      for (int i = 0 ; i < cells ; i++)
      {
        if (device->at()+1 > fileSize) return false;
        s >> stitches;
        if (stitches)
        {
          Stitch::Queue* psq = new Stitch::Queue();
          m_pCanvas->insert(i,psq);
          while (stitches--)
          {
            if (device->at()+3 > fileSize) return false;
            s >> type;
            s >> color;
            psq->enqueue(new Stitch((Stitch::Type)type, color));
          }
        }
      }
      if (device->at()+4 > fileSize) return false;
      s >> knots;
      while (knots--)
      {
        if (device->at()+10 > fileSize) return false;
        s >> start >> color;
        m_pKnot->append(new Knot(start,color));
      }
      if (device->at()+4 > fileSize) return false;
      s >> backstitches;
      while (backstitches--)
      {
      if (device->at()+17 > fileSize) return false;
        s >> start >> end >> color;
        m_pBackstitch->append(new BackStitch(start,end,color));
      }
      break;
    case 3:
    case 2:
      if (device->at()+8 > fileSize) return false;
      s >> width >> height;
      cells = width*height;
      m_pCanvas->clear();
      m_pKnot->clear();
      m_pBackstitch->clear();
      resize(width, height);
      for (int i = 0 ; i < cells ; i++)
      {
        if (device->at()+1 > fileSize) return false;
        s >> stitches;
        if (stitches)
        {
          Stitch::Queue *psq = new Stitch::Queue();
          m_pCanvas->insert(i,psq);
          while (stitches--)
          {
            if (device->at()+3 > fileSize) return false;
            s >> type;
            s >> color;
            if (type == Stitch::FRKnot)
              addFrenchKnot(QPoint((i%width)*2+1,(i/width)*2+1),color);
            else
              psq->enqueue(new Stitch((Stitch::Type)type, color));
          }
        }
      }
      if (device->at()+4 > fileSize) return false;
      s >> backstitches;
      while (backstitches--)
      {
        if (device->at()+10 > fileSize) return false;
        s >> start >> end >> color;
        m_pBackstitch->append(new BackStitch(start,end,color));
      }
      break;
  }
  return true;
}

UsageMap PatternCanvas::calculateUsage()
{
  // Calculate size of stitch cell relative to pattern properties
  // and calculate relative size of a half stitch and quarter stitch
  // Iterate canvas cells, iterating stitches in cell totalling the quantity of 
  // thread required
  // Required ClothCount, ClothCountUnits
  UsageMap usageMap;
  const double halfStitch = 2.4142135; // using 2.4142135 = sqrt(2)+1 representing the diagonal of a cell + 1 side, e.g.  /|/|/|/|
  const double qtrStitch = 1.4142135;  // using 1.4142135 = sqrt(2) representing corner to middle to another corner
  int sq = m_pCanvas->size();
  for (int i = 0 ; i < sq ; i++)
  {
    Stitch::Queue *q = m_pCanvas->at(i);
    if (q)
    {
      int c = q->count();
      while (c--)
      {
        Stitch *s = q->dequeue();
        if (s)
          // shouldn't really happen, but check for it anyway
        {
          switch (s->type)
          {
            case Stitch::TLQtr:
            case Stitch::TRQtr:
            case Stitch::BLQtr:
            case Stitch::BRQtr:
              usageMap[s->floss].quarter += 1;
              usageMap[s->floss].stitchLength += qtrStitch;
              break;
            case Stitch::BTHalf:
            case Stitch::TBHalf:
              usageMap[s->floss].half += 1;
              usageMap[s->floss].stitchLength += halfStitch;
              break;
            case Stitch::TL3Qtr:
            case Stitch::TR3Qtr:
            case Stitch::BL3Qtr:
            case Stitch::BR3Qtr:
              usageMap[s->floss].threeQuarter += 1;
              usageMap[s->floss].stitchLength += (halfStitch+qtrStitch);
              break;
            case Stitch::Full:
              usageMap[s->floss].full += 1;
              usageMap[s->floss].stitchLength += (halfStitch+halfStitch);
              break;
          }
          q->enqueue(s);
        }
      }
    }
  }
  for (BackStitch *bs = m_pBackstitch->first() ; bs ; bs = m_pBackstitch->next())
  {
    QPoint p = bs->end - bs->start;
    usageMap[bs->floss].backstitches += 1;
    double manhattanLength = p.manhattanLength()/2.0;
    usageMap[bs->floss].backstitchLength += manhattanLength;
  }
  for (Knot* k = m_pKnot->first() ; k ; k = m_pKnot->next())
  {
    usageMap[k->floss].knots += 1;
    usageMap[k->floss].stitchLength += halfStitch;   // estimated length of floss required for french knots
  }
  return usageMap;
}
