/* kastasker.h
**
** Copyright (C) 2001-2004 Richard Moore <rich@kde.org>
** Contributor: Mosfet
**     All rights reserved.
**
** KasBar is dual-licensed: you can choose the GPL or the BSD license.
** Short forms of both licenses are included below.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
** MA 02111-1307, USA.
*/

/*
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
// -*- c++ -*-


#ifndef KASTASKER_H
#define KASTASKER_H

#include "kasbar.h"

class KConfig;
class KPixmap;

class TaskManager;
class Task;
class Startup;

class KasTaskItem;
class KasStartupItem;
class KasGroupItem;

/**
 * A KasBar that provides a taskbar using the TaskManager API.
 *
 * @author Richard Moore, rich@kde.org
 */
class KasTasker : public KasBar
{
    Q_OBJECT

public:
    /**
     * Create a KasTasker widget.
     */
    KasTasker( Orientation o, QWidget* parent = 0, const char* name = 0, WFlags f = 0 );

    /**
     * Create a KasTasker widget that is slaved to another KasTasker. The
     * created widget will inherit the settings of the parent, but will
     * not connect to the signals of the TaskManager.
     */
    KasTasker( Orientation o, KasTasker *master,
	       QWidget *parent=0, const char *name=0, WFlags f=0 );
    /**
     * Cleans up.
     */
    virtual ~KasTasker();

    /**
     * Finds the item representing a task (if there is one).
     */
    KasTaskItem *findItem( Task * );

    /**
     * Finds the item representing a startup (if there is one).
     */
    KasStartupItem *findItem( Startup *s );

    /**
     * Accessor for the min icon (singleton).
     */
    QBitmap *minIcon();

    /**
     * Accessor for the max icon (singleton).
     */
    QBitmap *maxIcon();

    /**
     * Accessor for the shade icon (singleton).
     */
    QBitmap *shadeIcon();

    /**
     * Accessor for the micro min icon (singleton).
     */
    QPixmap *microMinIcon();

    /**
     * Accessor for the micro max icon (singleton).
     */
    QPixmap *microMaxIcon();

    /**
     * Accessor for the micro shade icon (singleton).
     */
    QPixmap *microShadeIcon();

    /**
     * Returns true iff thumbnails are enabled.
     */
    bool thumbnailsEnabled() const { return enableThumbs_; }

    /**
     * Returns the proportions of the window thumbnails.
     */
    double thumbnailSize() const { return thumbnailSize_; }

    /**
     * Returns true iff the startup notifier is enabled.
     */
    bool notifierEnabled() const { return enableNotifier_; }

    /**
     * Returns true iff the modified flag should be shown.
     */
    bool showModified() const { return showModified_; }

    bool showProgress() const { return showProgress_; }

    /**
     * Returns true iff windows from all desktops should be displays.
     */
    bool showAllWindows() const { return showAllWindows_; }

    /**
     * Returns the delay between thumbnail updates (in seconds).
     */
    int thumbnailUpdateDelay() const { return thumbUpdateDelay_; }

    /**
     * Returns true iff windows should be grouped together.
     */
    bool groupWindows() const { return groupWindows_; }

    //
    // Internal stuff
    //

    /**
     * Returns the TaskManager object.
     */
    TaskManager *taskManager() const { return manager; }

    /**
     * Converts the item for a task into a group item to which additional
     * tasks can be added.
     */
    KasGroupItem *convertToGroup( Task *t );

    /**
     * Moves an item from a group into the main bar.
     */
    void moveToMain( KasGroupItem *gi, Task *t );

    /**
     * If the task should be part of a group then make it so.
     */
    KasItem *maybeAddToGroup( Task *t );

public slots:
    /**
     * Adds a task to the bar.
     */
    void addTask( Task * );

    /**
     * Removes a task from the bar.
     */
    void removeTask( Task * );

    /**
     * Adds a startup item to the bar.
     */
    void addStartup( Startup * );

    /**
     * Removes a startup item from the bar.
     */
    void removeStartup( Startup * );

    void refreshAll();
    void refreshIconGeometry();

    void setNotifierEnabled( bool enable );
    void setThumbnailSize( double size );
    void setThumbnailSize( int percent );
    void setThumbnailsEnabled( bool enable );
    void setShowModified( bool enable );
    void setShowProgress( bool enable );
    void setShowAllWindows( bool enable );
    void setThumbnailUpdateDelay( int secs );
    void setGroupWindows( bool enable );

    void showPreferences();
    void showAbout();

    /**
     * Sets the current KConfig object.
     */
    void setConfig( KConfig *config );
    KConfig *config() const { return conf; }


    /**
     * Reads the settings from the current KConfig.
     */
    void readConfig();

    /**
     * Rereads the configuration of the master Kasbar.
     */
    void rereadMaster();

protected slots:
    /**
     * Load settings from the specified configuration.
     */
    void readConfig( KConfig *conf );

private:
    KasTasker *master_;
    TaskManager *manager;
    QBitmap *minPix;
    QBitmap *maxPix;
    QBitmap *shadePix;
    QPixmap *microShadePix;
    QPixmap *microMaxPix;
    QPixmap *microMinPix;
    bool passive_;
    bool enableThumbs_;
    double thumbnailSize_;
    bool enableNotifier_;
    bool showModified_;
    bool showProgress_;
    bool showAllWindows_;
    int thumbUpdateDelay_;
    bool groupWindows_;
    KConfig *conf;
};

#endif // KASTASKER_H

