% Various programming utils that are used by most of my other modes.
% (I would really like them in site.sl :-)

% Version 1.0 first public release
%         1.1 new: max(), contract_filename()
%         1.2 new: normalize_modename(), what_line_if_wide()

%!%+
%\function{push_defaults}
%\synopsis{Push the n last of the args to the stack}
%\usage{(a1, ..., an) = push_defaults(a1, ..., am, n)}
%\description
% Helps to define a slang function with optional arguments.
%\example
%#v+
% define fun() % (a1, a2, a3="d3", a4=whatbuf())
% {
%    variable a1, a2, a3, a4;
%    (a1, a2, a3, a4) = push_defaults( , , "d3", whatbuf(), _NARGS);
%    show(a1,a2,a3,a4);
% }
%#v-
%\seealso{__push_args, __pop_args, _NARGS }
%!%-
public define push_defaults() % args, n
{
   variable n = ();
   variable args = __pop_args(_NARGS-1);
   __push_args(args[[n:]]);
}

%!%+
%\function{push_array}
%\synopsis{Push an ordinary array on stack}
%\usage{(a[0], ..., a[-1])  push_array(Array a)}
%\description
% Push the elements of an array to the stack. This works like
% __push_args(args) but with an ordinary array (all types)
%\example
%#v+
%   variable a = ["message", "hello world"];
%   runhooks(push_array(a));
%#v-
%\notes
%   Elements of an Any_Type-array are references. They are dereferenced
%   in order to get type-independend behaviour.
%\seealso{array, pop2array, __push_args, __pop_args}
%!%-
public define push_array(a)
{
   if (_typeof(a) == Any_Type)
	foreach (a)
	     if (dup == NULL)
	       ();
	     else
	      @();
   else
     foreach (a)
       ();
}

%!%+
%\function{get_blocal}
%\synopsis{return value of blocal variable or default value}
%\usage{Any get_blocal (String name, [Any default=NULL])}
%\description
% This function is similar to get_blocal_var, but if the local variable 
% "name" doesnot exist, it returns the default value instead of an error.
% Default defaults to NULL.
%\example
%#v+
%    if (get_blocal(foo), 0)
%      message("this buffer is fooish");  
%#v-
% will print the message if foo is a blocal variable with nonzero value.
%\seealso{get_blocal_var, blocal_var_exists}
%!%-
define get_blocal() % (name, default=NULL)
{
   variable name, default;
   (name, default) = push_defaults( , NULL, _NARGS);
     
   if (blocal_var_exists(name))
     return get_blocal_var(name);
   return default;
}

%!%+
%\function{run_function}
%\synopsis{Run a function if it exists.}
%\usage{Int_Type run_function(fun, [args])}
%\description
% Run a function if it exists. Return whether it exists or not
% The function can be given by name or by reference (this allows both:
% yet undefined function (as string) as well as static functions 
% (as reference)
% Any arguments following the function argument will be passed to the 
% function. 
%\example
%#v+
%
%    !if (run_function("foo"))
% 	message("\"foo\" is not defined");
% 	
%    !if (run_function(&foo))
% 	message("\"foo\" is not defined");
%#v-
%\notes
% If fun is (only) an internal function, the optional arguments will
% be popped.
%\seealso{runhooks, run_blocal_hook}
%!%-
define run_function()  % (fun, [args])
{
   variable args = __pop_args(_NARGS-1);
   variable fun = ();
   if (typeof(fun) == String_Type)
     {
	if (is_defined(fun) > 0)
	  fun = __get_reference(fun);
	else if (is_internal(fun))
	  {
	     call(fun);
	     return 1;
	  }
     }
   if (typeof(fun) == Ref_Type)
     {
	@fun(__push_args(args));
	return 1;
     }
   return 0;
}

% max is a slang intrinsic since 1.4.6.
% Return the maximum of the arguments (must be numeric values)
define max() % (args)
{
   variable max = (), arg;
   loop(_NARGS-1)
     {
	arg = ();
	if (arg > max)
	  max = arg;
     }
   return max;
}

%!%+
%\function{contract_filename}
%\synopsis{Make a filename as short as possible}
%\usage{ contract_filename(file, cwd=getcwd())}
%\description
%  The opposite of expand_filename (in some case of view)
%  Make a filename as short as possible while
%  expand_filname will restore it to the previous value.
%\notes
%  * If the path starts with the working dir, strip it. 
%  (This maight fail on case insensitive filesystems).
%  (Be carefull to give the trailing path separator to the cwd argument)
%  * If the path starts with the home-dir, replace it with "~".
%  (Be carefull not to give the trailing path separator to the 
%   HOME environment variable)
%\seealso{expand_filename}
%!%-
define contract_filename() % (file, cwd=getcwd())
{
   variable file, cwd;
   (file, cwd) = push_defaults( , getcwd(), _NARGS);
   variable home = getenv("HOME");
   if (is_substr(file, cwd) == 1)
     file = file[[strlen(cwd):]];
   if (andelse
      {home != NULL}
	{strlen(home)}
	{is_substr(file, home) == 1})
     file = "~" + file[[strlen(home):]];
   return file;
}

% when a buffer is folded what_line may give the false number
public define what_line_if_wide ()
{
  if (count_narrows ())
    {
      push_narrow ();
      widen_buffer ();
      what_line ();
      pop_narrow ();
    }
  else
    what_line ();
}

