/****************************************************************************
 *                          IviRemoteSim.cc
 *
 * Author: Matthew Ballance
 * Desc:   The IviRemoteSim object is an interface for the IVI application 
 *         to the remote simulator.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "IviRemoteSim.h"
/* #include "IviRemoteSimInfo.h" */
#include "IviRemoteProc.h"
#include "IviRemoteProto.h"
#include "IviRemoteSDBSrv.h"
#include "SimMgr.h"
#include "SdbMgr.h"
#include "DFIOMgr.h"
#include "ConfigDB.h"
#include "ivi_String.h"
#include "RemoteAppConnectionListener.h"
#include "IviRemoteDesignDBSrv.h"
#include "iviSplitString.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include "ivi_print.h"


#undef DEBUG_IVI_REMOTE_SIM

#define FP stderr
#ifdef DEBUG_IVI_REMOTE_SIM
#define DBG_MSG(x) fprintf x ; fflush(FP)
#else
#define DBG_MSG(x)
#endif

/********************************************************************/
/** \class IviRemoteSimRecvChannel
 *  \brief Implements message listener for the simulator comm channel
 ********************************************************************/
class IviRemoteSimRecvChannel : public RemoteAppConnectionListener {

    public:
        IviRemoteSimRecvChannel(IviRemoteSim *sim) :
            RemoteAppConnectionListener(REMOTE_RESP_IDX)
        {
            d_sim = sim;
        }

        virtual void Receive(Uint32 idx, Uint32 len, Uchar *data)
        {
            Uint32 cmd = data[0]|(data[1]<<8)|(data[2]<<16)|(data[3]<<24);

            DBG_MSG((FP, "----> IviRemoteSimRecvChannel::Receive(cmd=%d)\n",
                    cmd));
            d_sim->d_recv = true;
            d_sim->d_cmd  = cmd;
            d_sim->d_len  = len;
            memcpy(d_sim->d_data, data, len);

            DBG_MSG((FP, "<---- IviRemoteSimRecvChannel::Receive(cmd=%d)\n",
                    cmd));
        }

    private:
        IviRemoteSim    *d_sim;
};

/********************************************************************/
class IviRemoteSimCmdChannel : public RemoteAppConnectionListener {
    friend class IviRemoteSim;

    public:

        IviRemoteSimCmdChannel(IviRemoteSim *rsim) : 
            RemoteAppConnectionListener(REMOTE_CMD_IDX) 
        {
            d_sim = rsim;
        }

        virtual void Receive(Uint32 idx, Uint32 len, Uchar *data)
        {
            Uint32 cmd = data[0]|(data[1]<<8)|(data[2]<<16)|(data[3]<<24);

            DBG_MSG((FP, "----> IviRemoteSimCmdChannel::Receive(cmd=%d)\n",
                    cmd));
            d_sim->d_recv = true;
            d_sim->d_cmd  = cmd;
            d_sim->d_len  = len;
            memcpy(d_sim->d_data, data, len);

            DBG_MSG((FP, "<---- IviRemoteSimCmdChannel::Receive(cmd=%d)\n",
                    cmd));
        }

    private:
        IviRemoteSim    *d_sim;
};


/********************************************************************
 * IviRemoteSim()
 ********************************************************************/
IviRemoteSim::IviRemoteSim(Tcl_Interp *interp, int argc, char **argv) :
    IviSim(interp, argc, argv)
{
    d_interp   = interp;
    d_vvpProc  = new IviRemoteProc(interp, argc, argv);
    d_conn     = 0;

    d_currTime.low  = 0;
    d_currTime.high = 0;

    ok = 1;
}

/********************************************************************
 * ~IviRemoteSim()
 ********************************************************************/
IviRemoteSim::~IviRemoteSim()
{
    DBG_MSG((FP, "----> IviRemoteSim::~IviRemoteSim()\n"));

    delete d_vvpProc;
    delete d_sdb;

    delete d_cmdChan;
    delete d_conn;

    delete d_dfio;

    DBG_MSG((FP, "<---- IviRemoteSim::~IviRemoteSim()\n"));
}

/********************************************************************
 * LoadDesign()
 ********************************************************************/
int IviRemoteSim::LoadDesign(char *design_path)
{
    Tcl_AppendResult(d_interp, "load_design unimplemented", 0);

    return TCL_ERROR;
}

/********************************************************************
 * CreateDDB()
 ********************************************************************/
DesignDB *IviRemoteSim::CreateDDB()
{
    char    buf[512];
    int     argc;
    char  **argv;

    sprintf(buf, "ddb %s.ddb -sim %s", getInstName(), getInstName());
    iviSplitString(buf, &argc, &argv);

    DesignDB *ddb = new IviRemoteDesignDBSrv(d_cmdChan, d_interp, argc, argv);
    iviFreeStringArr(argv);

    return ddb;
}

/********************************************************************
 * CreateDFIO()
 ********************************************************************/
DFIO *IviRemoteSim::CreateDFIO()
{
    char    buf[512];
    int     argc;
    char  **argv;
    DFIO   *dfio;

    sprintf(buf, "shm_dfio %s.dfio -file ivi.wav", getInstName());
    iviSplitString(buf, &argc, &argv);

    dfio = DFIOMgr_NewDFIO("shm_dfio_reader", d_interp, argc, argv);
    iviFreeStringArr(argv);

    return dfio;
}

/********************************************************************
 * CreateSDB()
 ********************************************************************/
SigDB *IviRemoteSim::CreateSDB()
{
    char    buf[512];
    int     argc;
    char  **argv;
    SigDB  *sdb;

    sprintf(buf, "sdb %s.sdb -dfio %s.dfio -sdb_id Sim -ddb %s.ddb",
            getInstName(), getInstName(), getInstName());

    iviSplitString(buf, &argc, &argv);
    sdb = new IviRemoteSDBSrv(d_conn, d_interp, argc, argv);
    iviFreeStringArr(argv);

    return sdb;
}

/********************************************************************
 * StartSimulator()
 ********************************************************************/
int IviRemoteSim::StartSimulator(const char *command)
{
    int       ret;
    Char      buf[64];
    Uint32    idx=0;

/*** Do some pre-invoke setup for Windows ***/
#ifdef __MINGW32__
    {
        const char *size_str;
        char *size_env;
        Uint32      shm_size;

        if (!(size_str = ConfigDB_GetCurrent("App.win9xShmFileSize"))) {
            shm_size = 0x01000000; /* 16MB file */
            ivi_print("WARNING: IviApp.win9xShmFileSize isn't set\n");
        } else {
            shm_size = ((1024*1024) * strtoul(size_str, 0, 0));
        }

        size_env = (char *)malloc(strlen("=0x01000000")+
                strlen("IVI_WIN9X_SHM_SIZE")+16);
        sprintf(size_env, "IVI_WIN9X_SHM_SIZE=0x%08x", shm_size);

        putenv(size_env);
    }
#endif

    d_vvpProc->setInvokeString(command);
    d_vvpProc->invoke();

    while (!(d_conn = d_vvpProc->getCommConnect())) {
        Tcl_DoOneEvent(0);
    }

    while (!d_conn->isConnected()) {
        DBG_MSG((stderr, "Polling while not connected\n"));
        if (!d_vvpProc->getProcRunning()) {
            fprintf(stderr, "ERROR: vvp process exited abnormally\n");
            break;
        }

        Tcl_DoOneEvent(0);
    }

    d_conn->AddListener(new IviRemoteSimCmdChannel(this));
    d_conn->AddListener(new IviRemoteSimRecvChannel(this));

    d_cmdChan = new RemoteAppTclCmdListener(d_interp);
    d_conn->AddListener(d_cmdChan);

    d_conn->Connect();

    /**** Okay, wait for 'IVI_REMOTE_INIT' msg ****/
    if ((ret = WaitCmd()) < 0) {
        fprintf(stderr, "ERROR: problem initializing simulator\n");
        return TCL_ERROR;
    } else {
        DBG_MSG((FP, "Simulator Connected\n"));
    }

    /**** Send the instance name to the plugin ****/
    DBG_MSG((stderr, "----> IVI_REMOTE_INST_NAME\n"));
    RemoteAppConnection::WriteUint32(IVI_REMOTE_INST_NAME, (Uchar *)buf, idx);
    strcpy(&buf[idx], getInstName());
    d_conn->Send(REMOTE_CMD_IDX, 5+strlen(getInstName()), (Uchar *)buf);
    DBG_MSG((stderr, "<---- IVI_REMOTE_INST_NAME\n"));

    /**** Wait for the 'IVI_REMOTE_LOAD_DONE' msg ****/
    if ((ret = WaitCmd()) < 0) {
        fprintf(stderr, "ERROR: problem loading design\n");
        return TCL_ERROR;
    } else {
        DBG_MSG((FP, "Design Loaded\n"));
    }

    if (d_cmd != IVI_REMOTE_LOAD_DONE) {
        fprintf(stderr, "ERROR: Expecting packet IVI_REMOTE_LOAD_DONE - got "
                "%d\n", d_cmd);
    }

    idx=4;

    Uint32 res  = RemoteAppConnection::ReadUint32(d_data, idx);
    Uint32 mult = RemoteAppConnection::ReadUint32(d_data, idx);

    setSimResolution(res);
    setResMultiplier(mult);

    SetupRemoteObjs();

    DBG_MSG((FP, "Pid %d done connecting\n", getpid()));

    return ret;
}

/********************************************************************
 * GetStdoutChannel()
 ********************************************************************/
const char *IviRemoteSim::GetStdoutChannel()
{
    return d_vvpProc->getStdoutChannelName();
}

/********************************************************************
 * Close()
 ********************************************************************/
int IviRemoteSim::Close()
{
    Uint32   idx=0;
    int      ret;

    /**** First, try to close down the remote simulator... ****/
    RemoteAppConnection::WriteUint32(IVI_REMOTE_CLOSE, d_buf, idx);

    if ((ret = d_conn->Send(REMOTE_CMD_IDX, idx, d_buf)) < 0) {
        fprintf(stderr, "ERROR: Sending of IVI_REMOTE_CLOSE "
                "command failed\n");
        return TCL_ERROR;
    }

    do {
        if ((ret = WaitCmd()) < 0) {
            fprintf(stderr, "ERROR: WaitCmd() for CLOSE_DONE failed\n");
            break;
        }

        if (d_cmd == IVI_REMOTE_CLOSE_DONE) {
            break;
        }
    } while (1);

    idx = 0;
    RemoteAppConnection::WriteUint32(IVI_REMOTE_CLOSE_EXIT, d_buf, idx);
    if ((ret = d_conn->Send(REMOTE_CMD_IDX, idx, d_buf)) < 0) {
        fprintf(stderr, "ERROR: Sending of IVI_REMOTE_CLOSE_EXIT "
                "command failed\n");
        return TCL_ERROR;
    }

    /**** now, wait for a short time to see if the process exits
     ****/
    for (Uint32 y=0; y<1024; y++) {
#ifndef __MINGW32__
        struct timeval  tv;

        if (!d_vvpProc->isRunning()) {
            break;
        }

        tv.tv_sec  = 0;
        tv.tv_usec = 1000;

        select(0, 0, 0, 0, &tv);
#else
        if (!d_vvpProc->isRunning(1)) {
            break;
        }
#endif
    }

    Delete();

    return TCL_OK;
}

/********************************************************************
 * Run()
 ********************************************************************/
int IviRemoteSim::Run(Uint32 max_time)
{
    Uint32 idx=0, rlen;
    int    ret;
    Uint32 finished = 0;

    DBG_MSG((FP, "----> ::Run(%d)\n", max_time));

    RemoteAppConnection::WriteUint32(IVI_REMOTE_RUN, d_buf, idx);
    RemoteAppConnection::WriteUint32(max_time, d_buf, idx);

    if ((ret = d_conn->Send(REMOTE_CMD_IDX, idx, d_buf)) < 0) {
        fprintf(stderr, "Send failed\n");
        return TCL_ERROR;
    }

    /**** Now, wait for a STOP response... ****/
    if ((ret = WaitCmd()) < 0) {
        fprintf(stderr, "WaitCmd() failed\n");
        return TCL_ERROR;
    }

    DBG_MSG((FP, "Received %d \n", d_cmd));

    if (d_cmd != IVI_REMOTE_STOP) {
        DBG_MSG((FP, "ERROR: Bad RUN response \"%d\"\n",
                d_cmd));
    } else {
        /**** Packet format is:
         **** - Uint32  stop?
         **** - Uint64  time {low, high}
         ****/
        Uint32 done     = d_data[4]|(d_data[5]<<8)|
            (d_data[6]<<16)|(d_data[7]<<24);

        if (done) {
            fprintf(stderr, "NOTE: IVI says simulation complete\n");
            finished = 1;
        }

        DBG_MSG((FP, "NOTE: end-time is %d\n", end_time));
        d_currTime.low = d_data[8]|(d_data[9]<<8)|
            (d_data[10]<<16)|(d_data[11]<<24);
        d_currTime.high = d_data[25]|(d_data[26]<<8)|
            (d_data[27]<<16)|(d_data[28]<<24);

        Tcl_SetObjResult(d_interp, Tcl_NewIntObj(d_currTime.low));
    }

    DBG_MSG((FP, "<---- ::Run(%d)\n", max_time));
    return finished;
}

class DFIO;
/********************************************************************
 * SetupRemoteObjs()
 ********************************************************************/
int IviRemoteSim::SetupRemoteObjs()
{
    int     ret;
    char    buf[512];
    char   *args[16];
    int     argc;
    char  **argv;

    /**** Construct the command-reflector stub for the DDB ****/
    DBG_MSG((stderr, "----> IviRemoteDesignDBSrv()\n"));
    d_ddb = CreateDDB();
#if 0
    sprintf(buf, "ddb %s.ddb -sim %s", getInstName(), getInstName());
    iviSplitString(buf, &argc, &argv);
    d_ddb = new IviRemoteDesignDBSrv(d_cmdChan, d_interp, argc, argv);
    iviFreeStringArr(argv);
#endif
    DBG_MSG((stderr, "<---- IviRemoteDesignDBSrv()\n"));

    /**** Construct the ShmDFIOReader ****/
    DBG_MSG((stderr, "----> DFIOMgr_NewDFIO()\n"));
    d_dfio = CreateDFIO();

#if 0
    sprintf(buf, "shm_dfio %s.dfio -file ivi.wav", getInstName());
    iviSplitString(buf, &argc, &argv);
    d_dfio = DFIOMgr_NewDFIO("shm_dfio_reader", d_interp, argc, argv);
    iviFreeStringArr(argv);
#endif
    DBG_MSG((stderr, "<---- DFIOMgr_NewDFIO()\n"));

    DBG_MSG((stderr, "----> IviRemoteSDBSrv()\n"));
    d_sdb = CreateSDB();
#if 0
    sprintf(buf, "sdb %s.sdb -dfio %s.dfio -sdb_id Sim -ddb %s.ddb",
            getInstName(), getInstName(), getInstName());
    iviSplitString(buf, &argc, &argv);
    d_sdbSrv = new IviRemoteSDBSrv(d_conn, d_interp, argc, argv);
    iviFreeStringArr(argv);
#endif
    SdbMgr_AddSDB(d_sdb);
    DBG_MSG((stderr, "<---- IviRemoteSDBSrv()\n"));

    DBG_MSG((stderr, "----> Configure(sim)\n"));
    sprintf(buf, "configure -ddb %s.ddb", getInstName());
    iviSplitString(buf, &argc, &argv);
    if (InstCmd(argc, argv) != TCL_OK) {
        fprintf(stderr, "error while configuring IviSim: %s\n",
                Tcl_GetStringResult(d_interp));
    }
    iviFreeStringArr(argv);
    DBG_MSG((stderr, "<---- Configure(sim)\n"));

    sprintf(buf, "%s.ddb scan_design", getInstName());
    if ((ret = d_cmdChan->Command(d_interp, buf)) != TCL_OK) {
        fprintf(stderr, "ddb scan_design command failed: %s\n",
                Tcl_GetStringResult(d_interp));
    }

    return TCL_OK;
}

/********************************************************************
 * vpi_get_time()
 ********************************************************************/
void IviRemoteSim::vpi_get_time(vpiHandle obj, s_vpi_time *t)
{
    t->low  = d_currTime.low;
    t->high = d_currTime.high;
}

/********************************************************************
 * vpi_get()
 ********************************************************************/
Int32 IviRemoteSim::vpi_get(Int32 prop, vpiHandle ref)
{
    fprintf(stderr, "Unknown vpi_get property: %d\n", prop);
    return 0;
}

/********************************************************************
 * WaitCmd()
 ********************************************************************/
int IviRemoteSim::WaitCmd()
{
    int ret = 0;

    DBG_MSG((FP, "----> IviRemoteSim::WaitCmd()\n"));

    d_recv = false;
    do {
        if ((ret = Tcl_DoOneEvent(0)) < 0)  {
            fprintf(stderr, "ERROR: Tcl_DoOneEvent() returned -1\n");
        }

        if (!d_vvpProc->getProcRunning()) {
            fprintf(stderr, "ERROR: Process stopped\n");
            ret = -1;
        }

        if (!d_conn->isConnected()) {
            fprintf(stderr, "ERROR: not connected\n");
            ret = -1;
        }
    } while (ret > 0 && !d_recv);

    DBG_MSG((FP, "<---- IviRemoteSim::WaitCmd(cmd=%d)\n", d_cmd));
    return ret;
}
#if 0

/********************************************************************
 * Vvp_remote_sim_Init()
 ********************************************************************/
extern "C" int Vvp_remote_sim_Init(Tcl_Interp *interp);
int Vvp_remote_sim_Init(Tcl_Interp *interp)
{
    IviRemoteSimInfo    *simInfo;
    IviRemoteSim        *tmp;

    simInfo = new IviRemoteSimInfo("IviRemoteSim",
            "IviSim interface to the VVP Verilog Simulator. "
            "The RemoteSim interface doesn't require a patched "
            "verilog distribution",
            "remote_vvp");

    SimMgr_RegisterSim(simInfo);

    Tcl_PkgProvide(interp, "vvp_remote_sim", "1.0");

    return TCL_OK;
}


extern "C" int Vvp_remote_sim_SafeInit(Tcl_Interp *interp)
{
    return Vvp_remote_sim_Init(interp);
}
#endif

