/****************************************************************************
 *                              GCObj.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a GC object. Drawing on a pixmap is done through
 *         a GC object...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "GCObj.h"
#include "PixmapObj.h"
#include <string.h>
#include <X11/Xlib.h>

/********************************************************************
 * GCObj()
 ********************************************************************/
GCObj::GCObj()
{
    memset(&d_GCValues, 0, sizeof(XGCValues));
    d_ValueMask = 0;
    d_GC        = 0;
    pObj        = 0;
    d_attached  = 0;
}

/********************************************************************
 * GCObj()
 ********************************************************************/
GCObj::GCObj(PixmapObj *pixObj)
{
    memset(&d_GCValues, 0, sizeof(XGCValues));
    d_ValueMask = 0;
    d_GC = 0;
    pObj = pixObj;
    d_attached = 1;
}

/********************************************************************
 * getGC()
 ********************************************************************/
GC GCObj::getGC()
{
    if (!d_GC) {
        d_GC = pObj->createGC(*(this));
    }

    return d_GC;
}

/********************************************************************
 * invalidate()
 ********************************************************************/
void GCObj::invalidate()
{
    if (d_GC && pObj) {
        pObj->freeGC(*(this));
    }

    d_GC = 0;
}

/********************************************************************
 * invalidate()
 ********************************************************************/
void GCObj::invalidate(GCObj *objs, Uint32 num_objs)
{
    for (Uint32 i=0; i<num_objs; i++) {
        objs[i].invalidate();
    }
}

/********************************************************************
 * setFgColor()
 ********************************************************************/
void GCObj::setFgColor(Uint32 fg)
{
    invalidate();
    d_GCValues.foreground = fg;
    d_ValueMask |= GCForeground;
}

/********************************************************************
 * setBgColor()
 ********************************************************************/
void GCObj::setBgColor(Uint32 bg)
{
    invalidate();
    d_GCValues.background = bg;
    d_ValueMask |= GCBackground;
}

/********************************************************************
 * setStipple()
 ********************************************************************/
void GCObj::setStipple(UChar *data, Uint32 width, Uint32 height)
{
    d_ValueMask |= GCStipple | GCFillStyle;
    d_GCValues.fill_style = FillStippled;
    stipple(data, width, height);
}

/********************************************************************
 * clrStipple()
 ********************************************************************/

/********************************************************************
 * setFont()
 ********************************************************************/
void GCObj::setFont(Tk_Font font)
{
    invalidate();
    d_GCValues.font = Tk_FontId(font);
    d_font = font;
    d_ValueMask |= GCFont;
}

/********************************************************************
 * setLineWidth()
 ********************************************************************/
void GCObj::setLineWidth(Uint32 width)
{
    invalidate();
    d_ValueMask |= GCLineWidth;
    d_GCValues.line_width = width;
}

/********************************************************************
 * setLineStyle()
 ********************************************************************/
void GCObj::setLineStyle(Uint32 style)
{
    invalidate();
    d_ValueMask |= GCLineStyle;
    d_GCValues.line_style = style;
}

/********************************************************************
 * setPixmap()
 ********************************************************************/
void GCObj::setPixmap(PixmapObj *pixmap)
{
    /**** Invalidate the gc if there was is no current pixmap
     **** or if the type of the new pixmap is different than
     **** that of the old...
     ****/
    if (!pObj  || !pixmap || (pObj->getType() != pixmap->getType())) {
        invalidate();
    }

    pObj = pixmap;

    if (pObj) {
        d_attached = 1;
    }
}

/********************************************************************
 * setPixmap()
 ********************************************************************/
void GCObj::setPixmap(GCObj *objs, Uint32 num_objs, PixmapObj *pixmap)
{

    for (Uint32 i=0; i<num_objs; i++) {
        objs[i].setPixmap(pixmap);
    }
}

/********************************************************************
 * line()
 ********************************************************************/
void GCObj::line(Uint32 x1, Uint32 y1, Uint32 x2, Uint32 y2)
{
    pObj->line(getGC(), x1, y1, x2, y2);
}

/********************************************************************
 * rect()
 ********************************************************************/
void GCObj::rect(Uint32 x, Uint32 y, Uint32 width, Uint32 height)
{
    pObj->rect(getGC(), x, y, width, height);
}

/********************************************************************
 * fill_rect()
 ********************************************************************/
void GCObj::fill_rect(Uint32 x, Uint32 y, Uint32 width, Uint32 height)
{
    pObj->fill_rect(getGC(), x, y, width, height);
}

/********************************************************************
 * ComputeTextLayout()
 ********************************************************************/
Tk_TextLayout GCObj::ComputeTextLayout(
        Char *str, Int32 len, Tk_Justify justify, Int32 *width, Int32 *height)
{
    return pObj->ComputeTextLayout(d_font, str, len, 0,
            justify, 0, width, height);
}

/********************************************************************
 * DrawTextLayout()
 ********************************************************************/
void GCObj::DrawTextLayout(Tk_TextLayout layout, Uint32 x, Uint32 y)
{
    pObj->DrawTextLayout(getGC(), layout, x, y, 0, -1);
}

/********************************************************************
 * FreeTextLayout()
 ********************************************************************/
void GCObj::FreeTextLayout(Tk_TextLayout layout)
{
    pObj->FreeTextLayout(layout);
}

